/*	$Id: elfmark.c,v 1.3 2001/04/29 13:23:58 hch Exp $	*/
/*
 * elfmark - set ELF header e_flags value to an abi-specific value.
 *
 * This utility is used for marking ELF binaries (including shared libraries)
 * for use by the UW Linux kernel module that provides exec domain for PER_UW7 
 * personality. Run only on IA32 architectures.
 *
 * Authors - Tigran Aivazian <tigran@veritas.com>,
 *		Christoph Hellwig <hch@caldera.de>
 *
 * This software is under GPL
 */

#define _XOPEN_SOURCE		600
#define _FILE_OFFSET_BITS	64

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <stdio.h>
#include <unistd.h>
#include <elf.h>

#include "elfmark.h"


static void	xperror(char *, char *);
static int	seteflags(char *, int);


const char	progname[] = "elfmark";
const char	usage[] = "usage: elfmark [-t uw7|udk|osr5|none] file...\n";


int
main(int argc, char **argv)
{
	int			eflags = -1, err = 1, c, i;

	while ((c = getopt(argc, argv, "t:")) != EOF) {
		switch (c) {
		case 't':
			if (!strcmp(optarg, "uw7") || !strcmp(optarg, "udk")) {
				eflags = EF_386_UW7;
				break;
			} else if (!strcmp(optarg, "osr5")) {
				eflags = EF_386_OSR5;
				break;
			} else if (!strcmp(optarg, "none")) {
				eflags = EF_386_NONE;
				break;
			}
			/* FALLTHROUGH */
		default:
			goto err;
		}
	}

	if (eflags != -1) {
		for (i = optind; i < argc; i++)
			if (!seteflags(argv[i], eflags))
				err = 0;
	}

	exit(err);
err:
	fprintf(stderr, usage);
	exit(1);
}

static void
xperror(char *cause, char *file)
{
	fprintf(stderr, "%s: %s %s, errno=%d (%s)\n",
			progname, cause, file, errno, strerror(errno));
}

static int
seteflags(char *filename, int value)
{
	Elf32_Ehdr		ex;
	int			fd, err;
	
	fd = open(filename, O_RDWR);
	if (fd == -1) {
		xperror("open() of ", filename);
		return 1;
	}

	err = read(fd, &ex, sizeof(Elf32_Ehdr));
	if (err != sizeof(Elf32_Ehdr)) {
		xperror("read() from", filename);
		return 2;
	}

	if (strncmp((char *)ex.e_ident, ELFMAG, SELFMAG) ||
	   (ex.e_ident[EI_CLASS] != ELFCLASS32) ||
	   (ex.e_ident[EI_DATA] != ELFDATA2LSB) ||
	   (ex.e_ident[EI_VERSION] != EV_CURRENT) ||
	   (ex.e_machine != EM_386)) {
		fprintf(stderr, "%s: %s is not a valid ELF file, skipping.\n",
				progname, filename);
		return 4;
	}

	ex.e_flags = (Elf32_Word)value;
	err = pwrite(fd, &ex, sizeof(Elf32_Ehdr), 0);
	if (err != sizeof(Elf32_Ehdr)) {
		xperror("pwrite() to", filename);
		return 5;
	}

	return (close(fd));
}
