/*
 * $Id: cap_alloc.c,v 1.2 1997/04/28 00:57:11 morgan Exp $
 *
 * Copyright (c) 1997 Andrew G Morgan <morgan@parc.power.net>
 *
 * See end of file for Log.
 *
 * This file deals with allocation and deallocation of internal
 * capability sets as specified by POSIX.1e (formerlly, POSIX 6).
 */

#include "libcap.h"

/*
 * This function duplicates an internal capability set (x3) with
 * malloc()'d memory. It is the responsibility of the user to call
 * cap_free() to liberate it.
 */

cap_t cap_dup(cap_t cap_d)
{
    cap_t result;

    if (!good_cap_t(cap_d)) {
	_cap_debug("bad argument");
	errno = EINVAL;
	return NULL;
    }

    result = (cap_t) malloc( sizeof(*cap_d) );
    if (result == NULL) {
	_cap_debug("out of memory");
	errno = ENOMEM;
	return NULL;
    }

    memcpy(result, cap_d, sizeof(*cap_d));

    return result;
}


/*
 * Scrub and then liberate an internal capability set.
 */

int cap_free(cap_t *cap_d_p)
{
    if ( cap_d_p && good_cap_t(*cap_d_p) ) {
	memset(*cap_d_p, 0, sizeof(**cap_d_p));
	free(*cap_d_p);
	*cap_d_p = NULL;

	return 0;
    } else {
	_cap_debug("no capability to liberate");
	errno = EINVAL;
	return -1;
    }
}

/*
 * Obtain a blank set of capabilities
 */

cap_t cap_init(void)
{
    cap_t result = (cap_t) malloc( sizeof(*result) );

    if (result) {
	result->magic = CAP_T_MAGIC;
	memset(&result->set, 0, 3*sizeof(__cap_s));
    } else {
	errno = ENOMEM;
    }
    return result;
}

/*
 * $Log: cap_alloc.c,v $
 * Revision 1.2  1997/04/28 00:57:11  morgan
 * fixes and zefram's patches
 *
 * Revision 1.1  1997/04/21 04:32:52  morgan
 * Initial revision
 *
 */
