/*
 *  dir.c
 *
 *  Copyright (C) 1995, 1996 by Paal-Kr. Engstad and Volker Lendecke
 *  Copyright (C) 1997 by Volker Lendecke
 *
 *  Please add a note about your changes to davfs in the ChangeLog file.
 */

#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/ctype.h>
#include <linux/smp_lock.h>

#include "davfs.h"
#include "dav_debug.h"

static int dav_readdir(struct file *, void *, filldir_t);
static int dav_dir_open(struct inode *, struct file *);

static struct dentry *dav_lookup(struct inode *, struct dentry *);
static int dav_create(struct inode *, struct dentry *, int);
static int dav_mkdir(struct inode *, struct dentry *, int);
static int dav_rmdir(struct inode *, struct dentry *);
static int dav_unlink(struct inode *, struct dentry *);
static int dav_rename(struct inode *, struct dentry *,
		      struct inode *, struct dentry *);

struct file_operations dav_dir_operations =
{
	read:		generic_read_dir,
	readdir:	dav_readdir,
	ioctl:		dav_ioctl,
	open:		dav_dir_open,
};

struct inode_operations dav_dir_inode_operations =
{
	create:		dav_create,
	lookup:		dav_lookup,
	unlink:		dav_unlink,
	mkdir:		dav_mkdir,
	rmdir:		dav_rmdir,
	rename:		dav_rename,
	revalidate:	dav_revalidate_inode,
	setattr:	dav_notify_change,
};

/*
 * Read a directory, using filldir to fill the dirent memory.
 * dav_proc_readdir does the actual reading from the dav server.
 *
 * The cache code is almost directly taken from ncpfs
 */
static int 
dav_readdir(struct file *filp, void *dirent, filldir_t filldir)
{
	struct dentry *dentry = filp->f_dentry;
	struct inode *dir = dentry->d_inode;
	struct dav_sb_info *server = dav_server_from_dentry(dentry);
	union  dav_dir_cache *cache = NULL;
	struct dav_cache_control ctl;
	struct page *page = NULL;
	int result;

	ctl.page  = NULL;
	ctl.cache = NULL;
	
	TRACE();

	VERBOSE("reading %s/%s, f_pos=%d\n",
		DENTRY_PATH(dentry),  (int) filp->f_pos);

	result = 0;
	switch ((unsigned int) filp->f_pos) {
	case 0:
	    if (filldir(dirent, ".", 1, 0, dir->i_ino, DT_DIR) < 0)
		    goto out;
		filp->f_pos = 1;
		/* fallthrough */
	case 1:
		if (filldir(dirent, "..", 2, 1,
			    dentry->d_parent->d_inode->i_ino, DT_DIR) < 0)
			goto out;
		filp->f_pos = 2;
	}

	/*
	 * Make sure our inode is up-to-date.
	 */
	result = dav_revalidate_inode(dentry);
	if (result)
		goto out;


	page = grab_cache_page(&dir->i_data, 0);
	if (!page)
		goto read_really;

	ctl.cache = cache = kmap(page);
	ctl.head  = cache->head;

	if (!Page_Uptodate(page) || !ctl.head.eof) {
		VERBOSE("%s/%s, page uptodate=%d, eof=%d\n",
			 DENTRY_PATH(dentry), Page_Uptodate(page),ctl.head.eof);
		goto init_cache;
	}

	if (filp->f_pos == 2) {
		if (jiffies - ctl.head.time >= DAV_MAX_AGE(server))
			goto init_cache;

		/*
		 * N.B. ncpfs checks mtime of dentry too here, we don't.
		 *   1. common dav servers do not update mtime on dir changes
		 *   2. it requires an extra dav request
		 *      (revalidate has the same timeout as ctl.head.time)
		 *
		 * Instead davfs invalidates its own cache on local changes
		 * and remote changes are not seen until timeout.
		 */
	}

	if (filp->f_pos > ctl.head.end)
		goto finished;

	ctl.fpos = filp->f_pos + (DAV_DIRCACHE_START - 2);
	ctl.ofs  = ctl.fpos / DAV_DIRCACHE_SIZE;
	ctl.idx  = ctl.fpos % DAV_DIRCACHE_SIZE;

	for (;;) {
		if (ctl.ofs != 0) {
			ctl.page = find_lock_page(&dir->i_data, ctl.ofs);
			if (!ctl.page)
				goto invalid_cache;
			ctl.cache = kmap(ctl.page);
			if (!Page_Uptodate(ctl.page))
				goto invalid_cache;
		}
		while (ctl.idx < DAV_DIRCACHE_SIZE) {
			struct dentry *dent;
			int res;

			dent = dav_dget_fpos(ctl.cache->dentry[ctl.idx], dentry, filp->f_pos);
			if (!dent)
				goto invalid_cache;

			res = filldir(dirent, dent->d_name.name,
				      dent->d_name.len, filp->f_pos,
				      dent->d_inode->i_ino, DT_UNKNOWN);
			dput(dent);
			if (res)
				goto finished;
			filp->f_pos += 1;
			ctl.idx += 1;
			if (filp->f_pos > ctl.head.end)
				goto finished;
		}
		if (ctl.page) {
			kunmap(ctl.page);
			SetPageUptodate(ctl.page);
			UnlockPage(ctl.page);
			page_cache_release(ctl.page);
			ctl.page = NULL;
		}
		ctl.idx  = 0;
		ctl.ofs += 1;
	}
invalid_cache:
	if (ctl.page) {
		kunmap(ctl.page);
		UnlockPage(ctl.page);
		page_cache_release(ctl.page);
		ctl.page = NULL;
	}
	ctl.cache = cache;
init_cache:
	dav_invalidate_dircache_entries(dentry);
	ctl.head.time = jiffies;
	ctl.head.eof = 0;
	ctl.fpos = 2;
	ctl.ofs = 0;
	ctl.idx = DAV_DIRCACHE_START;
	ctl.filled = 0;
	ctl.valid  = 1;
read_really:
	result = dav_proc_readdir(filp, dirent, filldir, &ctl);
	if (ctl.idx == -1)
		goto invalid_cache;	/* retry */
	ctl.head.end = ctl.fpos - 1;
	ctl.head.eof = ctl.valid;
finished:
	if (page) {
		cache->head = ctl.head;
		kunmap(page);
		SetPageUptodate(page);
		UnlockPage(page);
		page_cache_release(page);
	}
	if (ctl.page) {
		kunmap(ctl.page);
		SetPageUptodate(ctl.page);
		UnlockPage(ctl.page);
		page_cache_release(ctl.page);
	}
out:
	return result;
}

/*
 * Note: in order to allow the davmount process to open the
 * mount point, we don't revalidate if conn_pid is NULL.
 *
 * return 0 for next
 */
static int
dav_dir_open(struct inode *dir, struct file *file)
{
	struct dentry *dentry = file->f_dentry;
	struct dav_sb_info *server;
	int error = 0;

	TRACE();

	VERBOSE("(%s/%s)\n", dentry->d_parent->d_name.name,
		file->f_dentry->d_name.name);

	/*
	 * Directory timestamps in the core protocol aren't updated
	 * when a file is added, so we give them a very short TTL.
	 */
	lock_kernel();
	server = dav_server_from_dentry(dentry);

	/*
	if (server->conn_pid)
		error = dav_revalidate_inode(dentry);
	*/
	DEBUG1("conn_pid : %d\n", server->conn_pid);
	unlock_kernel();
	return error;
}

/*
 * Dentry operations routines
 */
static int dav_lookup_validate(struct dentry *, int);
static int dav_hash_dentry(struct dentry *, struct qstr *);
static int dav_compare_dentry(struct dentry *, struct qstr *, struct qstr *);
static int dav_delete_dentry(struct dentry *);

static struct dentry_operations davfs_dentry_operations =
{
	d_revalidate:	dav_lookup_validate,
	d_hash:		dav_hash_dentry,
	d_compare:	dav_compare_dentry,
	d_delete:	dav_delete_dentry,
};

/*
static struct dentry_operations davfs_dentry_operations_case =
{
	d_revalidate:	dav_lookup_validate,
	d_delete:	dav_delete_dentry,
};
*/

/*
 * This is the callback when the dcache has a lookup hit.
 */
static int
dav_lookup_validate(struct dentry * dentry, int flags)
{
	struct dav_sb_info *server = dav_server_from_dentry(dentry);
	struct inode * inode = dentry->d_inode;
	unsigned long age = jiffies - dentry->d_time;
	int valid = 1;

	TRACE();

	/*
	 * The default validation is based on dentry age:
	 * we believe in dentries for a few seconds.  (But each
	 * successful server lookup renews the timestamp.)
	 */
	valid = (age <= DAV_MAX_AGE(server));
#ifdef davFS_DEBUG_VERBOSE
	if (!valid)
		VERBOSE("%s/%s not valid, age=%lu\n", 
			DENTRY_PATH(dentry), age);
#endif

	if (inode) {
		lock_kernel();
		if (is_bad_inode(inode)) {
			PARANOIA("%s/%s has dud inode\n", DENTRY_PATH(dentry));
			valid = 0;
		} else if (!valid)
			valid = (dav_revalidate_inode(dentry) == 0);
		unlock_kernel();
	} else {
		/*
		 * What should we do for negative dentries?
		 */
	}
	return valid;
}

static int 
dav_hash_dentry(struct dentry *dir, struct qstr *this)
{
	unsigned long hash;
	int i;

	TRACE();

	hash = init_name_hash();
	for (i=0; i < this->len ; i++)
		hash = partial_name_hash(tolower(this->name[i]), hash);
	this->hash = end_name_hash(hash);
  
	return 0;
}

static int
dav_compare_dentry(struct dentry *dir, struct qstr *a, struct qstr *b)
{
	int i, result = 1;

	TRACE();

	if (a->len != b->len)
		goto out;
	for (i=0; i < a->len; i++) {
		if (tolower(a->name[i]) != tolower(b->name[i]))
			goto out;
	}
	result = 0;
out:
	return result;
}

/*
 * This is the callback from dput() when d_count is going to 0.
 * We use this to unhash dentries with bad inodes.
 */
static int
dav_delete_dentry(struct dentry * dentry)
{

	TRACE();
	if (dentry->d_inode) {
		if (is_bad_inode(dentry->d_inode)) {
			PARANOIA("bad inode, unhashing %s/%s\n",
				 DENTRY_PATH(dentry));
			return 1;
		}
	} else {
		/* N.B. Unhash negative dentries? */
	}
	return 0;
}

/*
 * Initialize a new dentry
 */
void
dav_new_dentry(struct dentry *dentry)
{
    //	struct dav_sb_info *server = dav_server_from_dentry(dentry);

	TRACE();

	dentry->d_op = &davfs_dentry_operations;
	
	dentry->d_time = jiffies;
}


/*
 * Whenever a lookup succeeds, we know the parent directories
 * are all valid, so we want to update the dentry timestamps.
 * N.B. Move this to dcache?
 */
void
dav_renew_times(struct dentry * dentry)
{
	TRACE();

	for (;;) {
		dentry->d_time = jiffies;
		if (IS_ROOT(dentry))
			break;
		dentry = dentry->d_parent;
	}
}

static struct dentry *
dav_lookup(struct inode *dir, struct dentry *dentry)
{
	struct dav_fattr finfo;
	struct inode *inode;
	int error;
	struct dav_sb_info *server;

	TRACE();

	error = -ENAMETOOLONG;
	if (dentry->d_name.len > DAV_MAXNAMELEN)
		goto out;

	error = dav_proc_getattr(dentry, &finfo);
#ifdef davFS_PARANOIA
	if (error && error != -ENOENT)
		PARANOIA("find %s/%s failed, error=%d\n",
			 DENTRY_PATH(dentry), error);
#endif
	DEBUG1("=== getattr result : %d\n", error);

	inode = NULL;
	if (error == -ENOENT)
		goto add_entry;
	if (!error) {
		error = -EACCES;
		finfo.f_ino = iunique(dentry->d_sb, 2);
		inode = dav_iget(dir->i_sb, &finfo);
		if (inode) {
	add_entry:
			server = dav_server_from_dentry(dentry);
			dentry->d_op = &davfs_dentry_operations;

			d_add(dentry, inode);
			dav_renew_times(dentry);
			error = 0;
		}
	}
out:
	return ERR_PTR(error);
}

/*
 * This code is common to all routines creating a new inode.
 */
static int
dav_instantiate(struct dentry *dentry, long fileid, int have_id)
{
    //    struct dav_sb_info *server = dav_server_from_dentry(dentry);
    struct inode *inode;
    int error;
    struct dav_fattr fattr;
    
    TRACE();
    
    VERBOSE("file %s/%s, fileid=%ld\n", DENTRY_PATH(dentry), fileid);
    
    
    error = dav_proc_getattr(dentry, &fattr);
    if (error)
	goto out_close;
    
    dav_renew_times(dentry);
    fattr.f_ino = iunique(dentry->d_sb, 2);
    inode = dav_iget(dentry->d_sb, &fattr);
    if (!inode)
	goto out_no_inode;
    
    if (have_id)
      {
	  /* we don't have inode before here */
	  /* bug fix 05/09/01 hunkim         */
	  struct dav_i_info *dii = DAV_INOP(inode);
	  dii->fileid = fileid;
	  DEBUG2("FILEID = %ld\n", dii->fileid);
	  dii->access = DAV_O_RDWR;
	  dii->open++;
      }
    
    d_instantiate(dentry, inode);
 out:
    return error;
    
 out_no_inode:
    error = -EACCES;
 out_close:
    if (have_id)
      {
	  PARANOIA("%s/%s failed, error=%d, closing %ld\n",
		   DENTRY_PATH(dentry), error, fileid);
	  dav_close_fileid(dentry, fileid);
      }
    goto out;
}

/* N.B. How should the mode argument be used? */
static int
dav_create(struct inode *dir, struct dentry *dentry, int mode)
{
	/* fake file id */
	long fileid=0;
	int error;

	TRACE();

	VERBOSE("creating %s/%s, mode=%d\n", DENTRY_PATH(dentry), mode);

	dav_invalid_dir_cache(dir);
	error = dav_proc_create(dentry, 0, CURRENT_TIME, &fileid);
	if (!error) {
		error = dav_instantiate(dentry, fileid, 1);
	} else {
		PARANOIA("%s/%s failed, error=%d\n",
			 DENTRY_PATH(dentry), error);
	}

	DEBUG2("FILEID = %ld\n", fileid);
	return error;
}

/* N.B. How should the mode argument be used? */
static int
dav_mkdir(struct inode *dir, struct dentry *dentry, int mode)
{
	int error;

	TRACE();

	dav_invalid_dir_cache(dir);
	error = dav_proc_mkdir(dentry);
	if (!error) {
		error = dav_instantiate(dentry, 0, 0);
	}
	return error;
}

static int
dav_rmdir(struct inode *dir, struct dentry *dentry)
{
	struct inode *inode = dentry->d_inode;
	int error;

	TRACE();

	/*
	 * Close the directory if it's open.
	 */
	dav_close(inode);

	/*
	 * Check that nobody else is using the directory..
	 */
	error = -EBUSY;
	if (!d_unhashed(dentry))
		goto out;

	dav_invalid_dir_cache(dir);
	error = dav_proc_rmdir(dentry);

out:
	return error;
}

static int
dav_unlink(struct inode *dir, struct dentry *dentry)
{
	int error;

	TRACE();

	/*
	 * Close the file if it's open.
	 */
	dav_close(dentry->d_inode);

	dav_invalid_dir_cache(dir);
	error = dav_proc_unlink(dentry);
	if (!error)
		dav_renew_times(dentry);
	return error;
}

static int
dav_rename(struct inode *old_dir, struct dentry *old_dentry,
	   struct inode *new_dir, struct dentry *new_dentry)
{
	int error;

	TRACE();

	/*
	 * Close any open files, and check whether to delete the
	 * target before attempting the rename.
	 */
	if (old_dentry->d_inode)
		dav_close(old_dentry->d_inode);
	if (new_dentry->d_inode)
	{
		dav_close(new_dentry->d_inode);
		error = dav_proc_unlink(new_dentry);
		if (error)
		{
			VERBOSE("unlink %s/%s, error=%d\n",
				DENTRY_PATH(new_dentry), error);
			goto out;
		}
		/* FIXME */
		d_delete(new_dentry);
	}

	dav_invalid_dir_cache(old_dir);
	dav_invalid_dir_cache(new_dir);
	error = dav_proc_mv(old_dentry, new_dentry);
	if (!error)
	{
		dav_renew_times(old_dentry);
		dav_renew_times(new_dentry);
	}
out:
	return error;
}






