/*******************************************************************
 * This file is part of the Emulex Linux Device Driver for         *
 * Enterprise Fibre Channel Host Bus Adapters.                     *
 * Refer to the README file included with this package for         *
 * driver version and adapter support.                             *
 * Copyright (C) 2003 Emulex Corporation.                          *
 * www.emulex.com                                                  *
 *                                                                 *
 * This program is free software; you can redistribute it and/or   *
 * modify it under the terms of the GNU General Public License     *
 * as published by the Free Software Foundation; either version 2  *
 * of the License, or (at your option) any later version.          *
 *                                                                 *
 * This program is distributed in the hope that it will be useful, *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the   *
 * GNU General Public License for more details, a copy of which    *
 * can be found in the file COPYING included with this package.    *
 *******************************************************************/

#ifndef _H_FC_HW
#define _H_FC_HW

typedef unsigned        u32bit;
typedef unsigned        u16bit;
typedef unsigned        u8bit;

#define FC_MAX_TRANSFER    0x40000 /* Maximum transfer size per operation */

#define MAX_CONFIGURED_RINGS     4 /* # rings currently used */

#define IOCB_CMD_R0_ENTRIES      5 /* ELS command ring entries */
#define IOCB_RSP_R0_ENTRIES      5 /* ELS response ring entries */
#define IOCB_CMD_R1_ENTRIES     27 /* IP command ring entries */
#define IOCB_RSP_R1_ENTRIES     28 /* IP response ring entries */
#define IOCB_CMD_R2_ENTRIES     45 /* FCP command ring entries */
#define IOCB_RSP_R2_ENTRIES     10 /* FCP response ring entries */
#define MAX_BIOCB              120 /* max# of BIU IOCBs in shared memory */

#define SLI2_IOCB_CMD_R0_ENTRIES      6 /* SLI-2 ELS command ring entries */
#define SLI2_IOCB_RSP_R0_ENTRIES      6 /* SLI-2 ELS response ring entries */
#define SLI2_IOCB_CMD_R1_ENTRIES     24 /* SLI-2 IP command ring entries */
#define SLI2_IOCB_RSP_R1_ENTRIES     30 /* SLI-2 IP response ring entries */
#define SLI2_IOCB_CMD_R1XTRA_ENTRIES 18 /* SLI-2 extra FCP cmd ring entries */
#define SLI2_IOCB_RSP_R1XTRA_ENTRIES 24 /* SLI-2 extra FCP rsp ring entries */
#define SLI2_IOCB_CMD_R2_ENTRIES     30 /* SLI-2 FCP command ring entries */
#define SLI2_IOCB_RSP_R2_ENTRIES     20 /* SLI-2 FCP response ring entries */
#define SLI2_IOCB_CMD_R2XTRA_ENTRIES 22 /* SLI-2 extra FCP cmd ring entries */
#define SLI2_IOCB_RSP_R2XTRA_ENTRIES 20 /* SLI-2 extra FCP rsp ring entries */
#define SLI2_IOCB_CMD_R3_ENTRIES     0 /* SLI-2 FCP command ring entries */
#define SLI2_IOCB_RSP_R3_ENTRIES     0 /* SLI-2 FCP response ring entries */
#define MAX_SLI2_IOCB                SLI2_IOCB_CMD_R0_ENTRIES  + \
                                     SLI2_IOCB_RSP_R0_ENTRIES  + \
                                     SLI2_IOCB_CMD_R1_ENTRIES  + \
                                     SLI2_IOCB_RSP_R1_ENTRIES  + \
                                     SLI2_IOCB_CMD_R2_ENTRIES  + \
                                     SLI2_IOCB_RSP_R2_ENTRIES  + \
                                     SLI2_IOCB_CMD_R3_ENTRIES  + \
                                     SLI2_IOCB_RSP_R3_ENTRIES

#define FCELSSIZE             1024 /* maximum ELS transfer size */

#define FC_MAXRETRY              3 /* max retries for ELS commands */
#define FC_ELS_RING              0 /* use ring 0 for ELS commands */
#define FC_IP_RING               1 /* use ring 1 for IP commands */
#define FC_FCP_RING              2 /* use ring 2 for FCP initiator commands */

#define FF_DEF_EDTOV          2000 /* Default E_D_TOV (2000ms) */ 
#define FF_DEF_ALTOV            15 /* Default AL_TIME (15ms) */ 
#define FF_DEF_RATOV             2 /* Default RA_TOV (2s) */ 
#define FF_DEF_ARBTOV         1900 /* Default ARB_TOV (1900ms) */
#define MB_WAIT_PERIOD         500 /* Wait period in usec inbetween MB polls */
#define MAX_MB_COMPLETION     1000 /* # MB_WAIT_PERIODs to wait for MB cmplt */
#define MAX_MSG_DATA            28 /* max msg data in CMD_ADAPTER_MSG iocb */

#define FF_REG_AREA_SIZE       256 /* size, in bytes, of i/o register area */
#define FF_SLIM_SIZE          4096 /* size, in bytes, of SLIM */

/*
 * Miscellaneous stuff....
 */
/* HBA Mgmt */
#define FDMI_DID        ((uint32)0xfffffa)
#define NameServer_DID  ((uint32)0xfffffc)
#define SCR_DID         ((uint32)0xfffffd)
#define Fabric_DID      ((uint32)0xfffffe)
#define Bcast_DID       ((uint32)0xffffff)
#define Mask_DID        ((uint32)0xffffff)
#define CT_DID_MASK     ((uint32)0xffff00)
#define Fabric_DID_MASK ((uint32)0xfff000)

#define PT2PT_LocalID   ((uint32)1)
#define PT2PT_RemoteID  ((uint32)2)

#define OWN_CHIP        1        /* IOCB / Mailbox is owned by Hba */
#define OWN_HOST        0        /* IOCB / Mailbox is owned by Host */
#define END_OF_CHAIN    0
#define IOCB_WORD_SZ    8        /* # of words in ULP BIU XCB */
#define MAX_RINGS       3        /* Max # of supported rings */

/* defines for type field in fc header */
#define FC_ELS_DATA     0x1
#define FC_LLC_SNAP     0x5             
#define FC_FCP_DATA     0x8             
#define FC_COMMON_TRANSPORT_ULP 0x20

/* defines for rctl field in fc header */
#define FC_DEV_DATA     0x0             
#define FC_UNSOL_CTL    0x2             
#define FC_SOL_CTL      0x3
#define FC_UNSOL_DATA   0x4             
#define FC_FCP_CMND     0x6             
#define FC_ELS_REQ      0x22
#define FC_ELS_RSP      0x23
#define FC_NET_HDR      0x20    /* network headers for Dfctl field */

/*
 *  Common Transport structures and definitions
 *
 */

union CtRevisionId {
   /* Structure is in Big Endian format */
   struct {
      u32bit Revision:    8;
      u32bit InId:       24;
   } bits;
   uint32 word;
};

union CtCommandResponse {
   /* Structure is in Big Endian format */
   struct {
      u32bit CmdRsp:  16;
      u32bit Size:    16;
   } bits;
   uint32 word;
};

typedef struct SliCtRequest {
   /* Structure is in Big Endian format */
   union CtRevisionId RevisionId;
   uchar FsType;
   uchar FsSubType;
   uchar Options;
   uchar Rsrvd1;
   union CtCommandResponse CommandResponse;
   uchar Rsrvd2;
   uchar ReasonCode;
   uchar Explanation;
   uchar VendorUnique;

   union {
      uint32   PortID;
      struct gid {
         uchar PortType;            /* for GID_PT requests */
         uchar DomainScope;
         uchar AreaScope;
         uchar Fc4Type;             /* for GID_FT requests */
      } gid;
      struct rft {
         uint32 PortId;             /* For RFT_ID requests */
#if BIG_ENDIAN_HW
         u32bit rsvd0:   16;
         u32bit rsvd1:   7;
         u32bit fcpReg:  1;           /* Type 8 */
         u32bit rsvd2:   2;
         u32bit ipReg:   1;           /* Type 5 */
         u32bit rsvd3:   5;
#endif
#if LITTLE_ENDIAN_HW
         u32bit rsvd0:   16;
         u32bit fcpReg:  1;           /* Type 8 */
         u32bit rsvd1:   7;
         u32bit rsvd3:   5;
         u32bit ipReg:   1;           /* Type 5 */
         u32bit rsvd2:   2;
#endif
         uint32 rsvd[7];
      } rft;
   } un;
} SLI_CT_REQUEST, *PSLI_CT_REQUEST;

#define  SLI_CT_REVISION        1
#define  GID_REQUEST_SZ         (sizeof(SLI_CT_REQUEST) - 32)
#define  RFT_REQUEST_SZ         (sizeof(SLI_CT_REQUEST))


/*
 * FsType Definitions
 */

#define  SLI_CT_MANAGEMENT_SERVICE        0xFA
#define  SLI_CT_TIME_SERVICE              0xFB
#define  SLI_CT_DIRECTORY_SERVICE         0xFC
#define  SLI_CT_FABRIC_CONTROLLER_SERVICE 0xFD

/*
 * Directory Service Subtypes
 */

#define  SLI_CT_DIRECTORY_NAME_SERVER     0x02

/*
 * Response Codes
 */

#define  SLI_CT_RESPONSE_FS_RJT           0x8001
#define  SLI_CT_RESPONSE_FS_ACC           0x8002

/*
 * Reason Codes
 */

#define  SLI_CT_NO_ADDITIONAL_EXPL    0x0
#define  SLI_CT_INVALID_COMMAND           0x01
#define  SLI_CT_INVALID_VERSION           0x02
#define  SLI_CT_LOGICAL_ERROR             0x03
#define  SLI_CT_INVALID_IU_SIZE           0x04
#define  SLI_CT_LOGICAL_BUSY              0x05
#define  SLI_CT_PROTOCOL_ERROR            0x07
#define  SLI_CT_UNABLE_TO_PERFORM_REQ     0x09
#define  SLI_CT_REQ_NOT_SUPPORTED         0x0b
#define  SLI_CT_HBA_INFO_NOT_REGISTERED   0x10
#define  SLI_CT_MULTIPLE_HBA_ATTR_OF_SAME_TYPE  0x11
#define  SLI_CT_INVALID_HBA_ATTR_BLOCK_LEN      0x12
#define  SLI_CT_HBA_ATTR_NOT_PRESENT      0x13
#define  SLI_CT_PORT_INFO_NOT_REGISTERED  0x20
#define  SLI_CT_MULTIPLE_PORT_ATTR_OF_SAME_TYPE 0x21
#define  SLI_CT_INVALID_PORT_ATTR_BLOCK_LEN     0x22
#define  SLI_CT_VENDOR_UNIQUE             0xff

/*
 * Name Server SLI_CT_UNABLE_TO_PERFORM_REQ Explanations
 */

#define  SLI_CT_NO_PORT_ID                0x01
#define  SLI_CT_NO_PORT_NAME              0x02
#define  SLI_CT_NO_NODE_NAME              0x03
#define  SLI_CT_NO_CLASS_OF_SERVICE       0x04
#define  SLI_CT_NO_IP_ADDRESS             0x05
#define  SLI_CT_NO_IPA                    0x06
#define  SLI_CT_NO_FC4_TYPES              0x07
#define  SLI_CT_NO_SYMBOLIC_PORT_NAME     0x08
#define  SLI_CT_NO_SYMBOLIC_NODE_NAME     0x09
#define  SLI_CT_NO_PORT_TYPE              0x0A
#define  SLI_CT_ACCESS_DENIED             0x10
#define  SLI_CT_INVALID_PORT_ID           0x11
#define  SLI_CT_DATABASE_EMPTY            0x12



/*
 * Name Server Command Codes
 */

#define  SLI_CTNS_GA_NXT      0x0100
#define  SLI_CTNS_GPN_ID      0x0112
#define  SLI_CTNS_GNN_ID      0x0113      
#define  SLI_CTNS_GCS_ID      0x0114
#define  SLI_CTNS_GFT_ID      0x0117
#define  SLI_CTNS_GSPN_ID     0x0118
#define  SLI_CTNS_GPT_ID      0x011A
#define  SLI_CTNS_GID_PN      0x0121
#define  SLI_CTNS_GID_NN      0x0131
#define  SLI_CTNS_GIP_NN      0x0135
#define  SLI_CTNS_GIPA_NN     0x0136
#define  SLI_CTNS_GSNN_NN     0x0139
#define  SLI_CTNS_GNN_IP      0x0153
#define  SLI_CTNS_GIPA_IP     0x0156
#define  SLI_CTNS_GID_FT      0x0171
#define  SLI_CTNS_GID_PT      0x01A1
#define  SLI_CTNS_RPN_ID      0x0212
#define  SLI_CTNS_RNN_ID      0x0213
#define  SLI_CTNS_RCS_ID      0x0214
#define  SLI_CTNS_RFT_ID      0x0217
#define  SLI_CTNS_RSPN_ID     0x0218
#define  SLI_CTNS_RPT_ID      0x021A
#define  SLI_CTNS_RIP_NN      0x0235
#define  SLI_CTNS_RIPA_NN     0x0236
#define  SLI_CTNS_RSNN_NN     0x0239
#define  SLI_CTNS_DA_ID       0x0300

/*
 * Port Types
 */

#define  SLI_CTPT_N_PORT      0x01
#define  SLI_CTPT_NL_PORT     0x02
#define  SLI_CTPT_FNL_PORT    0x03
#define  SLI_CTPT_IP          0x04
#define  SLI_CTPT_FCP         0x08
#define  SLI_CTPT_NX_PORT     0x7F
#define  SLI_CTPT_F_PORT      0x81
#define  SLI_CTPT_FL_PORT     0x82
#define  SLI_CTPT_E_PORT      0x84

#define SLI_CT_LAST_ENTRY     0x80000000

/*=====================================================================*/

#ifdef LP6000
/* PCI register offsets */
#define MEM_ADDR_OFFSET 0x10    /* SLIM base memory address */
#define MEMH_OFFSET     0x14    /* SLIM base memory high address */
#define REG_ADDR_OFFSET 0x18    /* REGISTER base memory address */
#define REGH_OFFSET     0x1c    /* REGISTER base memory high address */
#define IO_ADDR_OFFSET  0x20    /* BIU I/O registers */
#define REGIOH_OFFSET   0x24    /* REGISTER base io high address */
#endif

#define CMD_REG_OFFSET  0x4     /* PCI command configuration */

/* General PCI Register Definitions */
/* Refer To The PCI Specification For Detailed Explanations */

/* Register Offsets in little endian format */
#define PCI_VENDOR_ID_REGISTER      0x00    /* PCI Vendor ID Register*/
#define PCI_DEVICE_ID_REGISTER      0x02    /* PCI Device ID Register*/
#define PCI_CONFIG_ID_REGISTER      0x00    /* PCI Configuration ID Register*/
#define PCI_COMMAND_REGISTER        0x04    /* PCI Command Register*/
#define PCI_STATUS_REGISTER         0x06    /* PCI Status Register*/
#define PCI_REV_ID_REGISTER         0x08    /* PCI Revision ID Register*/
#define PCI_CLASS_CODE_REGISTER     0x09    /* PCI Class Code Register*/
#define PCI_CACHE_LINE_REGISTER     0x0C    /* PCI Cache Line Register*/
#define PCI_LATENCY_TMR_REGISTER    0x0D    /* PCI Latency Timer Register*/
#define PCI_HEADER_TYPE_REGISTER    0x0E    /* PCI Header Type Register*/
#define PCI_BIST_REGISTER           0x0F    /* PCI Built-In SelfTest Register*/
#define PCI_BAR_0_REGISTER          0x10    /* PCI Base Address Register 0*/
#define PCI_BAR_1_REGISTER          0x14    /* PCI Base Address Register 1*/
#define PCI_BAR_2_REGISTER          0x18    /* PCI Base Address Register 2*/
#define PCI_BAR_3_REGISTER          0x1C    /* PCI Base Address Register 3*/
#define PCI_BAR_4_REGISTER          0x20    /* PCI Base Address Register 4*/
#define PCI_BAR_5_REGISTER          0x24    /* PCI Base Address Register 5*/
#define PCI_EXPANSION_ROM           0x30    /* PCI Expansion ROM Base Register*/
#define PCI_INTR_LINE_REGISTER      0x3C    /* PCI Interrupt Line Register*/
#define PCI_INTR_PIN_REGISTER       0x3D    /* PCI Interrupt Pin Register*/
#define PCI_MIN_GNT_REGISTER        0x3E    /* PCI Min-Gnt Register*/
#define PCI_MAX_LAT_REGISTER        0x3F    /* PCI Max_Lat Register*/
#define PCI_NODE_ADDR_REGISTER      0x40    /* PCI Node Address Register*/

/* PCI access methods */
#define P_CONF_T1       1
#define P_CONF_T2       2

/* max number of pci buses */
#define MAX_PCI_BUSES   0xFF

/* number of PCI config bytes to access */
#define PCI_BYTE        1
#define PCI_WORD        2
#define PCI_DWORD       4

/* PCI related constants */
#define CMD_IO_ENBL     0x0001
#define CMD_MEM_ENBL    0x0002
#define CMD_BUS_MASTER  0x0004
#define CMD_MWI         0x0010
#define CMD_PARITY_CHK  0x0040
#define CMD_SERR_ENBL   0x0100

#define CMD_CFG_VALUE   0x156   /* mem enable, master, MWI, SERR, PERR */

/* PCI addresses */
#define PCI_SPACE_ENABLE            0x0CF8
#define CF1_CONFIG_ADDR_REGISTER    0x0CF8
#define CF1_CONFIG_DATA_REGISTER    0x0CFC
#define CF2_FORWARD_REGISTER        0x0CFA
#define CF2_BASE_ADDRESS            0xC000

#define PCI_VENDOR_ID_EMULEX        0x10df  

#define PCI_DEVICE_ID_SUPERFLY      0xf700
#define PCI_DEVICE_ID_DRAGONFLY     0xf800
#define PCI_DEVICE_ID_CENTAUR       0xf900
#define PCI_DEVICE_ID_PFLY          0xf098
#define PCI_DEVICE_ID_PEGASUS       0xf980
#define PCI_DEVICE_ID_TFLY          0xf0a5
#define PCI_DEVICE_ID_THOR          0xfa00

#define JEDEC_ID_ADDRESS            0x0080001c
#define SUPERFLY_JEDEC_ID           0x0020
#define DRAGONFLY_JEDEC_ID          0x0021
#define DRAGONFLY_V2_JEDEC_ID       0x0025
#define CENTAUR_2G_JEDEC_ID         0x0026
#define CENTAUR_1G_JEDEC_ID         0x0028
#define JEDEC_ID_MASK               0x0FFFF000
#define JEDEC_ID_SHIFT              12
#define FC_JEDEC_ID(id)             ((id & JEDEC_ID_MASK) >> JEDEC_ID_SHIFT)

#define DEFAULT_PCI_LATENCY_CLOCKS  0xf8       /* 0xF8 is a special value for
                                                * FF11.1N6 firmware.  Use
                                                * 0x80 for pre-FF11.1N6 &N7, etc
                                                */ 
#define PCI_LATENCY_VALUE           0xf8

#ifdef LP6000
typedef struct {             /* BIU registers */
   uint32  hostAtt;          /* See definitions for Host Attention register */
   uint32  chipAtt;          /* See definitions for Chip Attention register */
   uint32  hostStatus;       /* See definitions for Host Status register */
   uint32  hostControl;      /* See definitions for Host Control register */
   uint32  buiConfig;        /* See definitions for BIU configuration register*/
} FF_REGS, *PFF_REGS;

/* Host Attention Register */

#define HA_REG_OFFSET  0 /* Word offset from register base address */

#define HA_R0RE_REQ    0x00000001  /* Bit  0 */
#define HA_R0CE_RSP    0x00000002  /* Bit  1 */
#define HA_R0ATT       0x00000008  /* Bit  3 */
#define HA_R1RE_REQ    0x00000010  /* Bit  4 */
#define HA_R1CE_RSP    0x00000020  /* Bit  5 */
#define HA_R1ATT       0x00000080  /* Bit  7 */
#define HA_R2RE_REQ    0x00000100  /* Bit  8 */
#define HA_R2CE_RSP    0x00000200  /* Bit  9 */
#define HA_R2ATT       0x00000800  /* Bit 11 */
#define HA_R3RE_REQ    0x00001000  /* Bit 12 */
#define HA_R3CE_RSP    0x00002000  /* Bit 13 */
#define HA_R3ATT       0x00008000  /* Bit 15 */
#define HA_LATT        0x20000000  /* Bit 29 */
#define HA_MBATT       0x40000000  /* Bit 30 */
#define HA_ERATT       0x80000000  /* Bit 31 */


/* Chip Attention Register */

#define CA_REG_OFFSET  1 /* Word offset from register base address */

#define CA_R0CE_REQ    0x00000001  /* Bit  0 */
#define CA_R0RE_RSP    0x00000002  /* Bit  1 */
#define CA_R0ATT       0x00000008  /* Bit  3 */
#define CA_R1CE_REQ    0x00000010  /* Bit  4 */
#define CA_R1RE_RSP    0x00000020  /* Bit  5 */
#define CA_R1ATT       0x00000080  /* Bit  7 */
#define CA_R2CE_REQ    0x00000100  /* Bit  8 */
#define CA_R2RE_RSP    0x00000200  /* Bit  9 */
#define CA_R2ATT       0x00000800  /* Bit 11 */
#define CA_R3CE_REQ    0x00001000  /* Bit 12 */
#define CA_R3RE_RSP    0x00002000  /* Bit 13 */
#define CA_R3ATT       0x00008000  /* Bit 15 */
#define CA_MBATT       0x40000000  /* Bit 30 */


/* Host Status Register */

#define HS_REG_OFFSET  2 /* Word offset from register base address */

#define HS_MBRDY       0x00400000  /* Bit 22 */
#define HS_FFRDY       0x00800000  /* Bit 23 */
#define HS_FFER8       0x01000000  /* Bit 24 */
#define HS_FFER7       0x02000000  /* Bit 25 */
#define HS_FFER6       0x04000000  /* Bit 26 */
#define HS_FFER5       0x08000000  /* Bit 27 */
#define HS_FFER4       0x10000000  /* Bit 28 */
#define HS_FFER3       0x20000000  /* Bit 29 */
#define HS_FFER2       0x40000000  /* Bit 30 */
#define HS_FFER1       0x80000000  /* Bit 31 */
#define HS_FFERM       0xFF000000  /* Mask for error bits 31:24 */


/* Host Control Register */

#define HC_REG_OFFSET  3 /* Word offset from register base address */

#define HC_MBINT_ENA   0x00000001  /* Bit  0 */
#define HC_R0INT_ENA   0x00000002  /* Bit  1 */
#define HC_R1INT_ENA   0x00000004  /* Bit  2 */
#define HC_R2INT_ENA   0x00000008  /* Bit  3 */
#define HC_R3INT_ENA   0x00000010  /* Bit  4 */
#define HC_INITHBI     0x02000000  /* Bit 25 */
#define HC_INITMB      0x04000000  /* Bit 26 */
#define HC_INITFF      0x08000000  /* Bit 27 */
#define HC_LAINT_ENA   0x20000000  /* Bit 29 */
#define HC_ERINT_ENA   0x80000000  /* Bit 31 */

/* BIU Configuration Register */

#define BC_REG_OFFSET  4 /* Word offset from register base address */

#define BC_BSE         0x00000001  /* Bit 0 */
#define BC_BSE_SWAP    0x01000000  /* Bit 0 - swapped */

#endif  /* LP6000 */

/*=====================================================================*/

/*
 *  Start of FCP specific structures
 */

/*
 *  Definition of FCP_RSP Packet
 */

typedef struct  _FCP_RSP {
   uint32   rspRsvd1;    /* FC Word 0, byte 0:3 */
   uint32   rspRsvd2;    /* FC Word 1, byte 0:3 */

   uchar   rspStatus0;  /* FCP_STATUS byte 0 (reserved) */
   uchar   rspStatus1;  /* FCP_STATUS byte 1 (reserved) */
   uchar   rspStatus2;  /* FCP_STATUS byte 2 field validity */
#define RSP_LEN_VALID  0x01 /* bit 0 */
#define SNS_LEN_VALID  0x02 /* bit 1 */
#define RESID_OVER     0x04 /* bit 2 */
#define RESID_UNDER    0x08 /* bit 3 */
   uchar   rspStatus3;  /* FCP_STATUS byte 3 SCSI status byte */
#define SCSI_STAT_GOOD        0x00
#define SCSI_STAT_CHECK_COND  0x02
#define SCSI_STAT_COND_MET    0x04
#define SCSI_STAT_BUSY        0x08
#define SCSI_STAT_INTERMED    0x10
#define SCSI_STAT_INTERMED_CM 0x14
#define SCSI_STAT_RES_CNFLCT  0x18
#define SCSI_STAT_CMD_TERM    0x22
#define SCSI_STAT_QUE_FULL    0x28

   uint32   rspResId;    /* Residual xfer if RESID_xxxx set in fcpStatus2 */
                         /* Received in Big Endian format */
   uint32   rspSnsLen;   /* Length of sense data in fcpSnsInfo */
                         /* Received in Big Endian format */
   uint32   rspRspLen;   /* Length of FCP response data in fcpRspInfo */
                         /* Received in Big Endian format */

   uchar   rspInfo0;    /* FCP_RSP_INFO byte 0 (reserved) */
   uchar   rspInfo1;    /* FCP_RSP_INFO byte 1 (reserved) */
   uchar   rspInfo2;    /* FCP_RSP_INFO byte 2 (reserved) */
   uchar   rspInfo3;    /* FCP_RSP_INFO RSP_CODE byte 3 */

#define RSP_NO_FAILURE       0x00
#define RSP_DATA_BURST_ERR   0x01
#define RSP_CMD_FIELD_ERR    0x02
#define RSP_RO_MISMATCH_ERR  0x03
#define RSP_TM_NOT_SUPPORTED 0x04 /* Task mgmt function not supported */
#define RSP_TM_NOT_COMPLETED 0x05 /* Task mgmt function not performed */

   uint32   rspInfoRsvd; /* FCP_RSP_INFO bytes 4-7 (reserved) */

#define MAX_FCP_SNS  128   
   uchar   rspSnsInfo[MAX_FCP_SNS];
} FCP_RSP, *PFCP_RSP;

/*
 *  Definition of FCP_CMND Packet
 */

typedef struct  _FCP_CMND {
   uint32  fcpLunMsl;   /* most  significant lun word (32 bits) */
   uint32  fcpLunLsl;   /* least significant lun word (32 bits) */
 /* # of bits to shift lun id to end up in right
  * payload word, little endian = 8, big = 16.
  */
#if LITTLE_ENDIAN_HW
#define FC_LUN_SHIFT         8
#define FC_ADDR_MODE_SHIFT   0
#endif
#if BIG_ENDIAN_HW
#define FC_LUN_SHIFT         16
#define FC_ADDR_MODE_SHIFT   24 
#endif

   uchar  fcpCntl0;    /* FCP_CNTL byte 0 (reserved) */
   uchar  fcpCntl1;    /* FCP_CNTL byte 1 task codes */
#define  SIMPLE_Q        0x00
#define  HEAD_OF_Q       0x01
#define  ORDERED_Q       0x02
#define  ACA_Q           0x04
#define  UNTAGGED        0x05
   uchar  fcpCntl2;    /* FCP_CTL byte 2 task management codes */
#define  ABORT_TASK_SET  0x02 /* Bit 1 */
#define  CLEAR_TASK_SET  0x04 /* bit 2 */
#define  LUN_RESET       0x10 /* bit 4 */
#define  TARGET_RESET    0x20 /* bit 5 */
#define  CLEAR_ACA       0x40 /* bit 6 */
#define  TERMINATE_TASK  0x80 /* bit 7 */
   uchar  fcpCntl3;
#define  WRITE_DATA      0x01 /* Bit 0 */
#define  READ_DATA       0x02 /* Bit 1 */

   uchar  fcpCdb[16];  /* SRB cdb field is copied here */
   uint32 fcpDl;       /* Total transfer length */

} FCP_CMND, *PFCP_CMND;

/* SCSI INQUIRY Command Structure */

typedef struct inquiryDataType {
    u8bit DeviceType : 5;
    u8bit DeviceTypeQualifier : 3;

    u8bit DeviceTypeModifier : 7;
    u8bit RemovableMedia : 1;

    uchar Versions;
    uchar ResponseDataFormat;
    uchar AdditionalLength;
    uchar Reserved[2];

    u8bit SoftReset : 1;
    u8bit CommandQueue : 1;
    u8bit Reserved2 : 1;
    u8bit LinkedCommands : 1;
    u8bit Synchronous : 1;
    u8bit Wide16Bit : 1;
    u8bit Wide32Bit : 1;
    u8bit RelativeAddressing : 1;

    uchar VendorId[8];
    uchar ProductId[16];
    uchar ProductRevisionLevel[4];
    uchar VendorSpecific[20];
    uchar Reserved3[40];
} INQUIRY_DATA_DEF;

typedef struct _READ_CAPACITY_DATA {
    ulong LogicalBlockAddress;
    ulong BytesPerBlock;
} READ_CAPACITY_DATA_DEF;

typedef struct _REPORT_LUNS_DATA {
    union {
       uchar  cB[8];
       uint32 cL[2];
    } control;
    union {
       uchar  eB[8];
       uint32 eL[2];
    } entry [1];
} REPORT_LUNS_DATA_DEF;

/* SCSI CDB command codes */
#define FCP_SCSI_FORMAT_UNIT                  0x04
#define FCP_SCSI_INQUIRY                      0x12
#define FCP_SCSI_MODE_SELECT                  0x15
#define FCP_SCSI_MODE_SENSE                   0x1A
#define FCP_SCSI_PAUSE_RESUME                 0x4B
#define FCP_SCSI_PLAY_AUDIO                   0x45
#define FCP_SCSI_PLAY_AUDIO_EXT               0xA5
#define FCP_SCSI_PLAY_AUDIO_MSF               0x47
#define FCP_SCSI_PLAY_AUDIO_TRK_INDX          0x48
#define FCP_SCSI_PREVENT_ALLOW_REMOVAL        0x1E
#define FCP_SCSI_READ                         0x08
#define FCP_SCSI_READ_BUFFER                  0x3C
#define FCP_SCSI_READ_CAPACITY                0x25
#define FCP_SCSI_READ_DEFECT_LIST             0x37
#define FCP_SCSI_READ_EXTENDED                0x28
#define FCP_SCSI_READ_HEADER                  0x44
#define FCP_SCSI_READ_LONG                    0xE8
#define FCP_SCSI_READ_SUB_CHANNEL             0x42
#define FCP_SCSI_READ_TOC                     0x43
#define FCP_SCSI_REASSIGN_BLOCK               0x07
#define FCP_SCSI_RECEIVE_DIAGNOSTIC_RESULTS   0x1C
#define FCP_SCSI_RELEASE_UNIT                 0x17
#define FCP_SCSI_REPORT_LUNS                  0xa0
#define FCP_SCSI_REQUEST_SENSE                0x03
#define FCP_SCSI_RESERVE_UNIT                 0x16
#define FCP_SCSI_REZERO_UNIT                  0x01
#define FCP_SCSI_SEEK                         0x0B
#define FCP_SCSI_SEEK_EXTENDED                0x2B
#define FCP_SCSI_SEND_DIAGNOSTIC              0x1D
#define FCP_SCSI_START_STOP_UNIT              0x1B
#define FCP_SCSI_TEST_UNIT_READY              0x00
#define FCP_SCSI_VERIFY                       0x2F
#define FCP_SCSI_WRITE                        0x0A
#define FCP_SCSI_WRITE_AND_VERIFY             0x2E
#define FCP_SCSI_WRITE_BUFFER                 0x3B
#define FCP_SCSI_WRITE_EXTENDED               0x2A
#define FCP_SCSI_WRITE_LONG                   0xEA
#define FCP_SCSI_RELEASE_LUNR                 0xBB
#define FCP_SCSI_RELEASE_LUNV                 0xBF

#define HPVA_SETPASSTHROUGHMODE               0x27
#define HPVA_EXECUTEPASSTHROUGH               0x29
#define HPVA_CREATELUN                        0xE2
#define HPVA_SETLUNSECURITYLIST               0xED
#define HPVA_SETCLOCK                         0xF9
#define HPVA_RECOVER                          0xFA
#define HPVA_GENERICSERVICEOUT                0xFD

#define DMEP_EXPORT_IN                        0x85
#define DMEP_EXPORT_OUT                       0x89

#define MDACIOCTL_DIRECT_CMD                  0x22
#define MDACIOCTL_STOREIMAGE                  0x2C
#define MDACIOCTL_WRITESIGNATURE              0xA6
#define MDACIOCTL_SETREALTIMECLOCK            0xAC
#define MDACIOCTL_PASS_THRU_CDB               0xAD
#define MDACIOCTL_PASS_THRU_INITIATE          0xAE
#define MDACIOCTL_CREATENEWCONF               0xC0
#define MDACIOCTL_ADDNEWCONF                  0xC4
#define MDACIOCTL_MORE                        0xC6
#define MDACIOCTL_SETPHYSDEVPARAMETER         0xC8
#define MDACIOCTL_SETLOGDEVPARAMETER          0xCF
#define MDACIOCTL_SETCONTROLLERPARAMETER      0xD1
#define MDACIOCTL_WRITESANMAP                 0xD4
#define MDACIOCTL_SETMACADDRESS               0xD5

/*
 *  End of FCP specific structures
 */

#define FL_ALPA    0x00      /* AL_PA of FL_Port */

/* Fibre Channel Service Parameter definitions */

#define FC_PH_4_0   6  /* FC-PH version 4.0 */
#define FC_PH_4_1   7  /* FC-PH version 4.1 */
#define FC_PH_4_2   8  /* FC-PH version 4.2 */
#define FC_PH_4_3   9  /* FC-PH version 4.3 */

#define FC_PH_LOW   8  /* Lowest supported FC-PH version */ 
#define FC_PH_HIGH  9  /* Highest supported FC-PH version */
#define FC_PH3   0x20  /* FC-PH-3 version */

#define FF_FRAME_SIZE     2048


/* ==== Mailbox Commands ==== */
#define MBX_SHUTDOWN        0x00     /* terminate testing */
#define MBX_LOAD_SM         0x01
#define MBX_READ_NV         0x02
#define MBX_WRITE_NV        0x03
#define MBX_RUN_BIU_DIAG    0x04
#define MBX_INIT_LINK       0x05
#define MBX_DOWN_LINK       0x06
#define MBX_CONFIG_LINK     0x07
#define MBX_PART_SLIM       0x08
#define MBX_CONFIG_RING     0x09
#define MBX_RESET_RING      0x0A
#define MBX_READ_CONFIG     0x0B
#define MBX_READ_RCONFIG    0x0C
#define MBX_READ_SPARM      0x0D
#define MBX_READ_STATUS     0x0E
#define MBX_READ_RPI        0x0F
#define MBX_READ_XRI        0x10
#define MBX_READ_REV        0x11
#define MBX_READ_LNK_STAT   0x12
#define MBX_REG_LOGIN       0x13
#define MBX_UNREG_LOGIN     0x14
#define MBX_READ_LA         0x15
#define MBX_CLEAR_LA        0x16
#define MBX_DUMP_MEMORY     0x17
#define MBX_DUMP_CONTEXT    0x18
#define MBX_RUN_DIAGS       0x19
#define MBX_RESTART         0x1A
#define MBX_UPDATE_CFG      0x1B
#define MBX_DOWN_LOAD       0x1C
#define MBX_DEL_LD_ENTRY    0x1D
#define MBX_RUN_PROGRAM     0x1E
#define MBX_SET_MASK        0x20
#define MBX_SET_SLIM        0x21
#define MBX_UNREG_D_ID      0x23
#define MBX_CONFIG_FARP     0x25

#define MBX_LOAD_AREA       0x81
#define MBX_RUN_BIU_DIAG64  0x84
#define MBX_CONFIG_PORT     0x88
#define MBX_READ_SPARM64    0x8D
#define MBX_READ_RPI64      0x8F
#define MBX_REG_LOGIN64     0x93
#define MBX_READ_LA64       0x95

#define MBX_FLASH_WR_ULA    0x98
#define MBX_SET_DEBUG       0x99
#define MBX_LOAD_EXP_ROM    0x9C

#define MBX_MAX_CMDS        0x9D
#define MBX_SLI2_CMD_MASK   0x80


/* ==== IOCB Commands ==== */

#define CMD_RCV_SEQUENCE_CX     0x01
#define CMD_XMIT_SEQUENCE_CR    0x02
#define CMD_XMIT_SEQUENCE_CX    0x03
#define CMD_XMIT_BCAST_CN       0x04
#define CMD_XMIT_BCAST_CX       0x05
#define CMD_QUE_RING_BUF_CN     0x06
#define CMD_QUE_XRI_BUF_CX      0x07
#define CMD_IOCB_CONTINUE_CN    0x08
#define CMD_RET_XRI_BUF_CX      0x09
#define CMD_ELS_REQUEST_CR      0x0A
#define CMD_ELS_REQUEST_CX      0x0B
#define CMD_RCV_ELS_REQ_CX      0x0D
#define CMD_ABORT_XRI_CN        0x0E
#define CMD_ABORT_XRI_CX        0x0F
#define CMD_CLOSE_XRI_CR        0x10
#define CMD_CLOSE_XRI_CX        0x11
#define CMD_CREATE_XRI_CR       0x12
#define CMD_CREATE_XRI_CX       0x13
#define CMD_GET_RPI_CN          0x14
#define CMD_XMIT_ELS_RSP_CX     0x15
#define CMD_GET_RPI_CR          0x16
#define CMD_XRI_ABORTED_CX      0x17
#define CMD_FCP_IWRITE_CR       0x18
#define CMD_FCP_IWRITE_CX       0x19
#define CMD_FCP_IREAD_CR        0x1A
#define CMD_FCP_IREAD_CX        0x1B
#define CMD_FCP_ICMND_CR        0x1C
#define CMD_FCP_ICMND_CX        0x1D
#define CMD_ADAPTER_MSG         0x20
#define CMD_ADAPTER_DUMP        0x22
#define CMD_BPL_IWRITE_CR       0x48
#define CMD_BPL_IWRITE_CX       0x49
#define CMD_BPL_IREAD_CR        0x4A
#define CMD_BPL_IREAD_CX        0x4B
#define CMD_BPL_ICMND_CR        0x4C
#define CMD_BPL_ICMND_CX        0x4D

/*  SLI_2 IOCB Command Set */

#define CMD_RCV_SEQUENCE64_CX   0x81
#define CMD_XMIT_SEQUENCE64_CR  0x82
#define CMD_XMIT_SEQUENCE64_CX  0x83
#define CMD_XMIT_BCAST64_CN     0x84
#define CMD_XMIT_BCAST64_CX     0x85
#define CMD_QUE_RING_BUF64_CN   0x86
#define CMD_QUE_XRI_BUF64_CX    0x87
#define CMD_IOCB_CONTINUE64_CN  0x88
#define CMD_RET_XRI_BUF64_CX    0x89
#define CMD_ELS_REQUEST64_CR    0x8A
#define CMD_ELS_REQUEST64_CX    0x8B
#define CMD_RCV_ELS_REQ64_CX    0x8D
#define CMD_XMIT_ELS_RSP64_CX   0x95
#define CMD_FCP_IWRITE64_CR     0x98
#define CMD_FCP_IWRITE64_CX     0x99
#define CMD_FCP_IREAD64_CR      0x9A
#define CMD_FCP_IREAD64_CX      0x9B
#define CMD_FCP_ICMND64_CR      0x9C
#define CMD_FCP_ICMND64_CX      0x9D
#define CMD_GEN_REQUEST64_CR    0xC2
#define CMD_GEN_REQUEST64_CX    0xC3


/*
 *  Define Status
 */
#define MBX_SUCCESS                 0
#define MBXERR_NUM_RINGS            1
#define MBXERR_NUM_IOCBS            2
#define MBXERR_IOCBS_EXCEEDED       3
#define MBXERR_BAD_RING_NUMBER      4
#define MBXERR_MASK_ENTRIES_RANGE   5
#define MBXERR_MASKS_EXCEEDED       6
#define MBXERR_BAD_PROFILE          7
#define MBXERR_BAD_DEF_CLASS        8
#define MBXERR_BAD_MAX_RESPONDER    9
#define MBXERR_BAD_MAX_ORIGINATOR   10
#define MBXERR_RPI_REGISTERED       11
#define MBXERR_RPI_FULL             12
#define MBXERR_NO_RESOURCES         13
#define MBXERR_BAD_RCV_LENGTH       14
#define MBXERR_DMA_ERROR            15
#define MBXERR_ERROR                16
#define MBX_NOT_FINISHED           255
/*
 * Error codes returned by issue_mb_cmd()
 */
#define MBX_BUSY                   0xffffff /* Attempted cmd to a busy Mailbox */
#define MBX_TIMEOUT                0xfffffe /* Max time-out expired waiting for */
/* synch. Mailbox operation */
/*
 * flags for issue_mb_cmd()
 */
#define MBX_POLL        1  /* poll mailbox till command done, then return */
#define MBX_SLEEP       2  /* sleep till mailbox intr cmpl wakes thread up */
#define MBX_NOWAIT      3  /* issue command then return immediately */

typedef struct  {
#if BIG_ENDIAN_HW
   u32bit    crReserved :16;
   u32bit    crBegin    : 8;
   u32bit    crEnd      : 8;    /* Low order bit first word */
   u32bit    rrReserved :16;
   u32bit    rrBegin    : 8;
   u32bit    rrEnd      : 8;    /* Low order bit second word */
#endif
#if LITTLE_ENDIAN_HW
   u32bit    crEnd      : 8;    /* Low order bit first word */
   u32bit    crBegin    : 8;
   u32bit    crReserved :16;
   u32bit    rrEnd      : 8;    /* Low order bit second word */
   u32bit    rrBegin    : 8;
   u32bit    rrReserved :16;
#endif
} RINGS;


typedef struct  {
#if BIG_ENDIAN_HW
   ushort  offCiocb;
   ushort  numCiocb;
   ushort  offRiocb;
   ushort  numRiocb;
#endif
#if LITTLE_ENDIAN_HW
   ushort  numCiocb;
   ushort  offCiocb;
   ushort  numRiocb;
   ushort  offRiocb;
#endif
} RING_DEF;


/*
 *  The following F.C. frame stuctures are defined in Big Endian format.
 */

typedef struct  _NAME_TYPE {
#if BIG_ENDIAN_HW
   u8bit    nameType    : 4; /* FC Word 0, bit 28:31 */
   u8bit    IEEEextMsn  : 4; /* FC Word 0, bit 24:27, bit 8:11 of IEEE ext */
#endif
#if LITTLE_ENDIAN_HW
   u8bit    IEEEextMsn  : 4; /* FC Word 0, bit 24:27, bit 8:11 of IEEE ext */
   u8bit    nameType    : 4; /* FC Word 0, bit 28:31 */
#endif
#define NAME_IEEE           0x1 /* IEEE name - nameType */
#define NAME_IEEE_EXT       0x2 /* IEEE extended name */
#define NAME_FC_TYPE        0x3 /* FC native name type */
#define NAME_IP_TYPE        0x4 /* IP address */
#define NAME_CCITT_TYPE     0xC
#define NAME_CCITT_GR_TYPE  0xE
   uchar   IEEEextLsb;      /* FC Word 0, bit 16:23, IEEE extended Lsb */
   uchar   IEEE[6];         /* FC IEEE address */
} NAME_TYPE;


typedef struct  _CSP {
   uchar   fcphHigh;               /* FC Word 0, byte 0 */
   uchar   fcphLow;
   uchar   bbCreditMsb;
   uchar   bbCreditlsb;            /* FC Word 0, byte 3 */
#if BIG_ENDIAN_HW
   u16bit    increasingOffset  : 1;  /* FC Word 1, bit 31 */
   u16bit    randomOffset      : 1;  /* FC Word 1, bit 30 */
   u16bit    word1Reserved2    : 1;  /* FC Word 1, bit 29 */
   u16bit    fPort             : 1;  /* FC Word 1, bit 28 */
   u16bit    altBbCredit       : 1;  /* FC Word 1, bit 27 */
   u16bit    edtovResolution   : 1;  /* FC Word 1, bit 26 */
   u16bit    multicast         : 1;  /* FC Word 1, bit 25 */
   u16bit    broadcast         : 1;  /* FC Word 1, bit 24 */

   u16bit    huntgroup         : 1;  /* FC Word 1, bit 23 */
   u16bit    simplex           : 1;  /* FC Word 1, bit 22 */
   u16bit    word1Reserved1    : 3;  /* FC Word 1, bit 21:19 */
   u16bit    dhd               : 1;  /* FC Word 1, bit 18 */
   u16bit    contIncSeqCnt     : 1;  /* FC Word 1, bit 17 */
   u16bit    payloadlength     : 1;  /* FC Word 1, bit 16 */
#endif
#if LITTLE_ENDIAN_HW
   u16bit    broadcast         : 1;  /* FC Word 1, bit 24 */
   u16bit    multicast         : 1;  /* FC Word 1, bit 25 */
   u16bit    edtovResolution   : 1;  /* FC Word 1, bit 26 */
   u16bit    altBbCredit       : 1;  /* FC Word 1, bit 27 */
   u16bit    fPort             : 1;  /* FC Word 1, bit 28 */
   u16bit    word1Reserved2    : 1;  /* FC Word 1, bit 29 */
   u16bit    randomOffset      : 1;  /* FC Word 1, bit 30 */
   u16bit    increasingOffset  : 1;  /* FC Word 1, bit 31 */

   u16bit    payloadlength     : 1;  /* FC Word 1, bit 16 */
   u16bit    contIncSeqCnt     : 1;  /* FC Word 1, bit 17 */
   u16bit    dhd               : 1;  /* FC Word 1, bit 18 */
   u16bit    word1Reserved1    : 3;  /* FC Word 1, bit 21:19 */
   u16bit    simplex           : 1;  /* FC Word 1, bit 22 */
   u16bit    huntgroup         : 1;  /* FC Word 1, bit 23 */
#endif
   uchar   bbRcvSizeMsb;           /* Upper nibble is reserved */

   uchar   bbRcvSizeLsb;           /* FC Word 1, byte 3 */
   union {
      struct {
         uchar    word2Reserved1;   /* FC Word 2 byte 0 */

         uchar    totalConcurrSeq;  /* FC Word 2 byte 1 */
         uchar    roByCategoryMsb;  /* FC Word 2 byte 2 */

         uchar    roByCategoryLsb;  /* FC Word 2 byte 3 */
      } nPort;
      uint32 r_a_tov;               /* R_A_TOV must be in B.E. format */
   } w2;

   uint32   e_d_tov;                /* E_D_TOV must be in B.E. format */
} CSP;


typedef struct  _CLASS_PARMS {
#if BIG_ENDIAN_HW
   u8bit    classValid      : 1; /* FC Word 0, bit 31 */
   u8bit    intermix        : 1; /* FC Word 0, bit 30 */
   u8bit    stackedXparent  : 1; /* FC Word 0, bit 29 */
   u8bit    stackedLockDown : 1; /* FC Word 0, bit 28 */
   u8bit    seqDelivery     : 1; /* FC Word 0, bit 27 */
   u8bit    word0Reserved1  : 3; /* FC Word 0, bit 24:26 */
#endif
#if LITTLE_ENDIAN_HW
   u8bit    word0Reserved1  : 3; /* FC Word 0, bit 24:26 */
   u8bit    seqDelivery     : 1; /* FC Word 0, bit 27 */
   u8bit    stackedLockDown : 1; /* FC Word 0, bit 28 */
   u8bit    stackedXparent  : 1; /* FC Word 0, bit 29 */
   u8bit    intermix        : 1; /* FC Word 0, bit 30 */
   u8bit    classValid      : 1; /* FC Word 0, bit 31 */

#endif
   uchar   word0Reserved2;      /* FC Word 0, bit 16:23 */
#if BIG_ENDIAN_HW
   u8bit    iCtlXidReAssgn  : 2; /* FC Word 0, Bit 14:15 */
   u8bit    iCtlInitialPa   : 2; /* FC Word 0, bit 12:13 */
   u8bit    iCtlAck0capable : 1; /* FC Word 0, bit 11 */
   u8bit    iCtlAckNcapable : 1; /* FC Word 0, bit 10 */
   u8bit    word0Reserved3  : 2; /* FC Word 0, bit  8: 9 */
#endif
#if LITTLE_ENDIAN_HW
   u8bit    word0Reserved3  : 2; /* FC Word 0, bit  8: 9 */
   u8bit    iCtlAckNcapable : 1; /* FC Word 0, bit 10 */
   u8bit    iCtlAck0capable : 1; /* FC Word 0, bit 11 */
   u8bit    iCtlInitialPa   : 2; /* FC Word 0, bit 12:13 */
   u8bit    iCtlXidReAssgn  : 2; /* FC Word 0, Bit 14:15 */
#endif
   uchar   word0Reserved4;      /* FC Word 0, bit  0: 7 */
#if BIG_ENDIAN_HW
   u8bit    rCtlAck0capable : 1; /* FC Word 1, bit 31 */
   u8bit    rCtlAckNcapable : 1; /* FC Word 1, bit 30 */
   u8bit    rCtlXidInterlck : 1; /* FC Word 1, bit 29 */
   u8bit    rCtlErrorPolicy : 2; /* FC Word 1, bit 27:28 */
   u8bit    word1Reserved1  : 1; /* FC Word 1, bit 26 */
   u8bit    rCtlCatPerSeq   : 2; /* FC Word 1, bit 24:25 */
#endif
#if LITTLE_ENDIAN_HW
   u8bit    rCtlCatPerSeq   : 2; /* FC Word 1, bit 24:25 */
   u8bit    word1Reserved1  : 1; /* FC Word 1, bit 26 */
   u8bit    rCtlErrorPolicy : 2; /* FC Word 1, bit 27:28 */
   u8bit    rCtlXidInterlck : 1; /* FC Word 1, bit 29 */
   u8bit    rCtlAckNcapable : 1; /* FC Word 1, bit 30 */
   u8bit    rCtlAck0capable : 1; /* FC Word 1, bit 31 */
#endif
   uchar   word1Reserved2;      /* FC Word 1, bit 16:23 */
   uchar   rcvDataSizeMsb;      /* FC Word 1, bit  8:15 */
   uchar   rcvDataSizeLsb;      /* FC Word 1, bit  0: 7 */

   uchar   concurrentSeqMsb;    /* FC Word 2, bit 24:31 */
   uchar   concurrentSeqLsb;    /* FC Word 2, bit 16:23 */
   uchar   EeCreditSeqMsb;      /* FC Word 2, bit  8:15 */
   uchar   EeCreditSeqLsb;      /* FC Word 2, bit  0: 7 */

   uchar   openSeqPerXchgMsb;   /* FC Word 3, bit 24:31 */
   uchar   openSeqPerXchgLsb;   /* FC Word 3, bit 16:23 */
   uchar   word3Reserved1;      /* Fc Word 3, bit  8:15 */
   uchar   word3Reserved2;      /* Fc Word 3, bit  0: 7 */
} CLASS_PARMS;


typedef struct  _SERV_PARM { /* Structure is in Big Endian format */
   CSP          cmn;
   NAME_TYPE    portName;
   NAME_TYPE    nodeName;
   CLASS_PARMS  cls1;
   CLASS_PARMS  cls2;
   CLASS_PARMS  cls3;
   CLASS_PARMS  cls4;
   uchar        vendorVersion[16];
} SERV_PARM, *PSERV_PARM;


/*
 *  Extended Link Service LS_COMMAND codes (Payload Word 0)
 */
#if BIG_ENDIAN_HW
#define ELS_CMD_MASK      0xffff0000
#define ELS_RSP_MASK      0xff000000
#define ELS_CMD_LS_RJT    0x01000000
#define ELS_CMD_ACC       0x02000000
#define ELS_CMD_PLOGI     0x03000000
#define ELS_CMD_FLOGI     0x04000000
#define ELS_CMD_LOGO      0x05000000
#define ELS_CMD_ABTX      0x06000000
#define ELS_CMD_RCS       0x07000000
#define ELS_CMD_RES       0x08000000
#define ELS_CMD_RSS       0x09000000
#define ELS_CMD_RSI       0x0A000000
#define ELS_CMD_ESTS      0x0B000000
#define ELS_CMD_ESTC      0x0C000000
#define ELS_CMD_ADVC      0x0D000000
#define ELS_CMD_RTV       0x0E000000
#define ELS_CMD_RLS       0x0F000000
#define ELS_CMD_ECHO      0x10000000
#define ELS_CMD_TEST      0x11000000
#define ELS_CMD_RRQ       0x12000000
#define ELS_CMD_PRLI      0x20100014
#define ELS_CMD_PRLO      0x21100014
#define ELS_CMD_PDISC     0x50000000
#define ELS_CMD_FDISC     0x51000000
#define ELS_CMD_ADISC     0x52000000
#define ELS_CMD_FARP      0x54000000
#define ELS_CMD_FARPR     0x55000000
#define ELS_CMD_FAN       0x60000000
#define ELS_CMD_RSCN      0x61040000
#define ELS_CMD_SCR       0x62000000
#define ELS_CMD_RNID      0x78000000
#endif
#if LITTLE_ENDIAN_HW
#define ELS_CMD_MASK      0xffff
#define ELS_RSP_MASK      0xff
#define ELS_CMD_LS_RJT    0x01
#define ELS_CMD_ACC       0x02
#define ELS_CMD_PLOGI     0x03
#define ELS_CMD_FLOGI     0x04
#define ELS_CMD_LOGO      0x05
#define ELS_CMD_ABTX      0x06
#define ELS_CMD_RCS       0x07
#define ELS_CMD_RES       0x08
#define ELS_CMD_RSS       0x09
#define ELS_CMD_RSI       0x0A
#define ELS_CMD_ESTS      0x0B
#define ELS_CMD_ESTC      0x0C
#define ELS_CMD_ADVC      0x0D
#define ELS_CMD_RTV       0x0E
#define ELS_CMD_RLS       0x0F
#define ELS_CMD_ECHO      0x10
#define ELS_CMD_TEST      0x11
#define ELS_CMD_RRQ       0x12
#define ELS_CMD_PRLI      0x14001020
#define ELS_CMD_PRLO      0x14001021
#define ELS_CMD_PDISC     0x50
#define ELS_CMD_FDISC     0x51
#define ELS_CMD_ADISC     0x52
#define ELS_CMD_FARP      0x54
#define ELS_CMD_FARPR     0x55
#define ELS_CMD_FAN       0x60
#define ELS_CMD_RSCN      0x0461
#define ELS_CMD_SCR       0x62
#define ELS_CMD_RNID      0x78
#endif


/*
 *  LS_RJT Payload Definition
 */

typedef struct  _LS_RJT { /* Structure is in Big Endian format */
   union {
      uint32 lsRjtError;
      struct {
         uchar  lsRjtRsvd0;                    /* FC Word 0, bit 24:31 */

         uchar  lsRjtRsnCode;                  /* FC Word 0, bit 16:23 */
         /* LS_RJT reason codes */
#define LSRJT_INVALID_CMD     0x01
#define LSRJT_LOGICAL_ERR     0x03
#define LSRJT_LOGICAL_BSY     0x05
#define LSRJT_PROTOCOL_ERR    0x07
#define LSRJT_UNABLE_TPC      0x09              /* Unable to perform command */
#define LSRJT_CMD_UNSUPPORTED 0x0B
#define LSRJT_VENDOR_UNIQUE   0xFF              /* See Byte 3 */

         uchar  lsRjtRsnCodeExp;               /* FC Word 0, bit  8:15 */
         /* LS_RJT reason explanation */
#define LSEXP_NOTHING_MORE      0x00
#define LSEXP_SPARM_OPTIONS     0x01
#define LSEXP_SPARM_ICTL        0x03
#define LSEXP_SPARM_RCTL        0x05 
#define LSEXP_SPARM_RCV_SIZE    0x07
#define LSEXP_SPARM_CONCUR_SEQ  0x09
#define LSEXP_SPARM_CREDIT      0x0B
#define LSEXP_INVALID_PNAME     0x0D
#define LSEXP_INVALID_NNAME     0x0E
#define LSEXP_INVALID_CSP       0x0F
#define LSEXP_INVALID_ASSOC_HDR 0x11
#define LSEXP_ASSOC_HDR_REQ     0x13
#define LSEXP_INVALID_O_SID     0x15
#define LSEXP_INVALID_OX_RX     0x17
#define LSEXP_CMD_IN_PROGRESS   0x19
#define LSEXP_INVALID_NPORT_ID  0x1F
#define LSEXP_INVALID_SEQ_ID    0x21
#define LSEXP_INVALID_XCHG      0x23
#define LSEXP_INACTIVE_XCHG     0x25
#define LSEXP_RQ_REQUIRED       0x27 
#define LSEXP_OUT_OF_RESOURCE   0x29
#define LSEXP_CANT_GIVE_DATA    0x2A
#define LSEXP_REQ_UNSUPPORTED   0x2C
         uchar  vendorUnique;                  /* FC Word 0, bit  0: 7 */
      } b;
   } un;
} LS_RJT;


/*
 *  N_Port Login (FLOGO/PLOGO Request) Payload Definition
 */

typedef struct  _LOGO { /* Structure is in Big Endian format */
   union {
      uint32 nPortId32;              /* Access nPortId as a word */
      struct {
         uchar   word1Reserved1;    /* FC Word 1, bit 31:24 */
         uchar   nPortIdByte0;      /* N_port  ID bit 16:23 */
         uchar   nPortIdByte1;      /* N_port  ID bit  8:15 */
         uchar   nPortIdByte2;      /* N_port  ID bit  0: 7 */
      } b;
   } un;
   NAME_TYPE portName;              /* N_port name field */
} LOGO;


/*
 *  FCP Login (PRLI Request / ACC) Payload Definition
 */

#define PRLX_PAGE_LEN   0x10
#define TPRLO_PAGE_LEN  0x14

typedef struct  _PRLI { /* Structure is in Big Endian format */
   uchar   prliType;              /* FC Parm Word 0, bit 24:31 */

#define PRLI_FCP_TYPE 0x08
   uchar   word0Reserved1;        /* FC Parm Word 0, bit 16:23 */

#if BIG_ENDIAN_HW
   u8bit    origProcAssocV  :  1;  /* FC Parm Word 0, bit 15 */
   u8bit    respProcAssocV  :  1;  /* FC Parm Word 0, bit 14 */
   u8bit    estabImagePair  :  1;  /* FC Parm Word 0, bit 13 */

   u8bit    word0Reserved2  :  1;  /* FC Parm Word 0, bit 12 */
   u8bit    acceptRspCode   :  4;  /* FC Parm Word 0, bit 8:11, ACC ONLY */
#endif
#if LITTLE_ENDIAN_HW
   u8bit    acceptRspCode   :  4;  /* FC Parm Word 0, bit 8:11, ACC ONLY */
   u8bit    word0Reserved2  :  1;  /* FC Parm Word 0, bit 12 */
   u8bit    estabImagePair  :  1;  /* FC Parm Word 0, bit 13 */
   u8bit    respProcAssocV  :  1;  /* FC Parm Word 0, bit 14 */
   u8bit    origProcAssocV  :  1;  /* FC Parm Word 0, bit 15 */
#endif
#define PRLI_REQ_EXECUTED     0x1  /* acceptRspCode */
#define PRLI_NO_RESOURCES     0x2
#define PRLI_INIT_INCOMPLETE  0x3
#define PRLI_NO_SUCH_PA       0x4
#define PRLI_PREDEF_CONFIG    0x5
#define PRLI_PARTIAL_SUCCESS  0x6
#define PRLI_INVALID_PAGE_CNT 0x7
   uchar   word0Reserved3;        /* FC Parm Word 0, bit 0:7 */

   uint32   origProcAssoc;        /* FC Parm Word 1, bit 0:31 */

   uint32   respProcAssoc;        /* FC Parm Word 2, bit 0:31 */

   uchar   word3Reserved1;        /* FC Parm Word 3, bit 24:31 */
   uchar   word3Reserved2;        /* FC Parm Word 3, bit 16:23 */
#if BIG_ENDIAN_HW
   u16bit    Word3bit15Resved :  1;  /* FC Parm Word 3, bit 15 */
   u16bit    Word3bit14Resved :  1;  /* FC Parm Word 3, bit 14 */
   u16bit    Word3bit13Resved :  1;  /* FC Parm Word 3, bit 13 */
   u16bit    Word3bit12Resved :  1;  /* FC Parm Word 3, bit 12 */
   u16bit    Word3bit11Resved :  1;  /* FC Parm Word 3, bit 11 */
   u16bit    Word3bit10Resved :  1;  /* FC Parm Word 3, bit 10 */
   u16bit    TaskRetryIdReq   :  1;  /* FC Parm Word 3, bit  9 */
   u16bit    Retry            :  1;  /* FC Parm Word 3, bit  8 */
   u16bit    ConfmComplAllowed : 1;  /* FC Parm Word 3, bit  7 */
   u16bit    dataOverLay      :  1;  /* FC Parm Word 3, bit  6 */
   u16bit    initiatorFunc    :  1;  /* FC Parm Word 3, bit  5 */
   u16bit    targetFunc       :  1;  /* FC Parm Word 3, bit  4 */
   u16bit    cmdDataMixEna    :  1;  /* FC Parm Word 3, bit  3 */
   u16bit    dataRspMixEna    :  1;  /* FC Parm Word 3, bit  2 */
   u16bit    readXferRdyDis   :  1;  /* FC Parm Word 3, bit  1 */
   u16bit    writeXferRdyDis  :  1;  /* FC Parm Word 3, bit  0 */
#endif
#if LITTLE_ENDIAN_HW
   u16bit    Retry            :  1;  /* FC Parm Word 3, bit  8 */
   u16bit    TaskRetryIdReq   :  1;  /* FC Parm Word 3, bit  9 */
   u16bit    Word3bit10Resved :  1;  /* FC Parm Word 3, bit 10 */
   u16bit    Word3bit11Resved :  1;  /* FC Parm Word 3, bit 11 */
   u16bit    Word3bit12Resved :  1;  /* FC Parm Word 3, bit 12 */
   u16bit    Word3bit13Resved :  1;  /* FC Parm Word 3, bit 13 */
   u16bit    Word3bit14Resved :  1;  /* FC Parm Word 3, bit 14 */
   u16bit    Word3bit15Resved :  1;  /* FC Parm Word 3, bit 15 */
   u16bit    writeXferRdyDis  :  1;  /* FC Parm Word 3, bit  0 */
   u16bit    readXferRdyDis   :  1;  /* FC Parm Word 3, bit  1 */
   u16bit    dataRspMixEna    :  1;  /* FC Parm Word 3, bit  2 */
   u16bit    cmdDataMixEna    :  1;  /* FC Parm Word 3, bit  3 */
   u16bit    targetFunc       :  1;  /* FC Parm Word 3, bit  4 */
   u16bit    initiatorFunc    :  1;  /* FC Parm Word 3, bit  5 */
   u16bit    dataOverLay      :  1;  /* FC Parm Word 3, bit  6 */
   u16bit    ConfmComplAllowed : 1;  /* FC Parm Word 3, bit  7 */
#endif
} PRLI;

/*
 *  FCP Logout (PRLO Request / ACC) Payload Definition
 */

typedef struct  _PRLO { /* Structure is in Big Endian format */
   uchar   prloType;              /* FC Parm Word 0, bit 24:31 */

#define PRLO_FCP_TYPE  0x08
   uchar   word0Reserved1;        /* FC Parm Word 0, bit 16:23 */

#if BIG_ENDIAN_HW
   u8bit    origProcAssocV  :  1;  /* FC Parm Word 0, bit 15 */
   u8bit    respProcAssocV  :  1;  /* FC Parm Word 0, bit 14 */
   u8bit    word0Reserved2  :  2;  /* FC Parm Word 0, bit 12:13 */
   u8bit    acceptRspCode   :  4;  /* FC Parm Word 0, bit 8:11, ACC ONLY */
#endif
#if LITTLE_ENDIAN_HW
   u8bit    acceptRspCode   :  4;  /* FC Parm Word 0, bit 8:11, ACC ONLY */
   u8bit    word0Reserved2  :  2;  /* FC Parm Word 0, bit 12:13 */
   u8bit    respProcAssocV  :  1;  /* FC Parm Word 0, bit 14 */
   u8bit    origProcAssocV  :  1;  /* FC Parm Word 0, bit 15 */
#endif
#define PRLO_REQ_EXECUTED     0x1  /* acceptRspCode */
#define PRLO_NO_SUCH_IMAGE    0x4
#define PRLO_INVALID_PAGE_CNT 0x7

   uchar   word0Reserved3;        /* FC Parm Word 0, bit 0:7 */

   uint32   origProcAssoc;         /* FC Parm Word 1, bit 0:31 */

   uint32   respProcAssoc;         /* FC Parm Word 2, bit 0:31 */

   uint32   word3Reserved1;        /* FC Parm Word 3, bit 0:31 */
} PRLO;


typedef struct  _ADISC { /* Structure is in Big Endian format */
   uint32       hardAL_PA;
   NAME_TYPE    portName;
   NAME_TYPE    nodeName;
   uint32       DID;
} ADISC;


typedef struct  _FARP { /* Structure is in Big Endian format */
   u32bit         Mflags :  8;
   u32bit         Odid   : 24;
#define FARP_NO_ACTION          0    /* FARP information enclosed, no action */
#define FARP_MATCH_PORT         0x1  /* Match on Responder Port Name */
#define FARP_MATCH_NODE         0x2  /* Match on Responder Node Name */
#define FARP_MATCH_IP           0x4  /* Match on IP address, not supported */
#define FARP_MATCH_IPV4         0x5  /* Match on IPV4 address, not supported */
#define FARP_MATCH_IPV6         0x6  /* Match on IPV6 address, not supported */
   u32bit         Rflags :  8;
   u32bit         Rdid   : 24;
#define FARP_REQUEST_PLOGI      0x1  /* Request for PLOGI */
#define FARP_REQUEST_FARPR      0x2  /* Request for FARP Response */
   NAME_TYPE    OportName;
   NAME_TYPE    OnodeName;
   NAME_TYPE    RportName;
   NAME_TYPE    RnodeName;
   uchar        Oipaddr[16];
   uchar        Ripaddr[16];
} FARP;

typedef struct  _FAN { /* Structure is in Big Endian format */
   uint32       Fdid;
   NAME_TYPE    FportName;
   NAME_TYPE    FnodeName;
} FAN;

typedef struct  _SCR { /* Structure is in Big Endian format */
   uchar resvd1;
   uchar resvd2;
   uchar resvd3;
   uchar Function;
#define  SCR_FUNC_FABRIC     0x01
#define  SCR_FUNC_NPORT      0x02
#define  SCR_FUNC_FULL       0x03
#define  SCR_CLEAR           0xff
} SCR;

typedef struct _RNID_TOP_DISC {
   NAME_TYPE    portName;
   uchar         resvd[8];
   uint32       unitType;
#define RNID_HBA            0x7
#define RNID_HOST           0xa
#define RNID_DRIVER         0xd
   uint32       physPort;
   uint32       attachedNodes;
   ushort       ipVersion;
#define RNID_IPV4           0x1
#define RNID_IPV6           0x2
   ushort       UDPport;
   uchar        ipAddr[16];
   ushort       resvd1;
   ushort       flags;
#define RNID_TD_SUPPORT     0x1
#define RNID_LP_VALID       0x2
} RNID_TOP_DISC;

typedef struct  _RNID { /* Structure is in Big Endian format */
   uchar Format;
#define RNID_TOPOLOGY_DISC  0xdf
   uchar CommonLen;
   uchar resvd1;
   uchar SpecificLen;
   NAME_TYPE    portName;
   NAME_TYPE    nodeName;
   union {
      RNID_TOP_DISC       topologyDisc;    /* topology disc (0xdf) */
   } un;
} RNID;

typedef struct  _RRQ { /* Structure is in Big Endian format */
   uint32       SID;
   ushort       Oxid;
   ushort       Rxid;
   uchar        resv[32];       /* optional association hdr */
} RRQ;


/* This is used for RSCN command */
typedef struct  _D_ID { /* Structure is in Big Endian format */
   union {
      uint32   word;
      struct {
#if BIG_ENDIAN_HW
         uchar  resv;
         uchar  domain;
         uchar  area;
         uchar  id;
#endif
#if LITTLE_ENDIAN_HW
         uchar  id;
         uchar  area;
         uchar  domain;
         uchar  resv;
#endif
      } b;
   } un;
} D_ID;

/*
 *  Structure to define all ELS Payload types
 */

typedef struct  _ELS_PKT { /* Structure is in Big Endian format */
   uchar     elsCode;           /* FC Word 0, bit 24:31 */
   uchar     elsByte1;
   uchar     elsByte2;
   uchar     elsByte3;
   union {
      LS_RJT        lsRjt;   /* Payload for LS_RJT ELS response */
      SERV_PARM     logi;    /* Payload for PLOGI/FLOGI/PDISC/ACC */
      LOGO          logo;    /* Payload for PLOGO/FLOGO/ACC */
      PRLI          prli;    /* Payload for PRLI/ACC */
      PRLO          prlo;    /* Payload for PRLO/ACC */
      ADISC         adisc;   /* Payload for ADISC/ACC */
      FARP          farp;    /* Payload for FARP/ACC */
      FAN           fan;     /* Payload for FAN */
      SCR           scr;     /* Payload for SCR/ACC */
      RRQ           rrq;     /* Payload for RRQ */
      RNID          rnid;    /* Payload for RNID */
      uchar         pad[128-4]; /* Pad out to payload of 128 bytes */
   } un;
} ELS_PKT;


/*
 *    Begin Structure Definitions for Mailbox Commands
 */

typedef struct  {
#if BIG_ENDIAN_HW
   uchar   tval;
   uchar   tmask;
   uchar   rval;
   uchar   rmask;
#endif
#if LITTLE_ENDIAN_HW
   uchar   rmask;
   uchar   rval;
   uchar   tmask;
   uchar   tval;
#endif
} RR_REG;

typedef struct  {
   uint32   bdeAddress;
#if BIG_ENDIAN_HW
   u32bit    bdeReserved :  4;
   u32bit    bdeAddrHigh :  4;
   u32bit    bdeSize     : 24;
#endif
#if LITTLE_ENDIAN_HW
   u32bit    bdeSize     : 24;
   u32bit    bdeAddrHigh :  4;
   u32bit    bdeReserved :  4;
#endif
} ULP_BDE;

typedef struct ULP_BDE_64 {    /* SLI-2 */
   union ULP_BDE_TUS {
      uint32    w;
      struct  {
#if BIG_ENDIAN_HW
         u32bit  bdeFlags :  8;  
         u32bit  bdeSize  : 24;  /* Size of buffer (in bytes) */
#endif
#if LITTLE_ENDIAN_HW
         u32bit  bdeSize  : 24;  /* Size of buffer (in bytes) */
         u32bit  bdeFlags :  8;  
#endif
#define BUFF_USE_RSVD       0x01    /* bdeFlags */
#define BUFF_USE_INTRPT     0x02   
#define BUFF_USE_CMND       0x04    /* Optional, 1=cmd/rsp 0=data buffer */
#define BUFF_USE_RCV        0x08    /*  ""  "",  1=rcv buffer, 0=xmit buffer */
#define BUFF_TYPE_32BIT     0x10    /*  ""  "",  1=32 bit addr 0=64 bit addr */
#define BUFF_TYPE_SPECIAL   0x20    
#define BUFF_TYPE_BDL       0x40    /* Optional,  may be set in BDL */
#define BUFF_TYPE_INVALID   0x80    /*  ""  "" */
      } f;
   } tus;
   uint32       addrLow;
   uint32       addrHigh;
} ULP_BDE64;
#define BDE64_SIZE_WORD 0
#define BPL64_SIZE_WORD 0x40

typedef struct ULP_BDL {        /* SLI-2 */
#if BIG_ENDIAN_HW
   u32bit    bdeFlags     :  8;  /* BDL Flags */
   u32bit    bdeSize      : 24;  /* Size of BDL array in host memory (bytes) */
#endif
#if LITTLE_ENDIAN_HW
   u32bit    bdeSize      : 24;  /* Size of BDL array in host memory (bytes) */
   u32bit    bdeFlags     :  8;  /* BDL Flags */
#endif
   uint32  addrLow;            /* Address 0:31 */
   uint32  addrHigh;           /* Address 32:63 */
   uint32  ulpIoTag32;         /* Can be used for 32 bit I/O Tag */
} ULP_BDL;


/* Structure for MB Command LOAD_SM and DOWN_LOAD */

typedef struct  {
#if BIG_ENDIAN_HW
   u32bit rsvd2           :25;
   u32bit acknowledgment  : 1;
   u32bit version         : 1;
   u32bit erase_or_prog   : 1;
   u32bit update_flash    : 1;
   u32bit update_ram      : 1;
   u32bit method          : 1;
   u32bit load_cmplt      : 1;
#endif
#if LITTLE_ENDIAN_HW
   u32bit load_cmplt      : 1;
   u32bit method          : 1;
   u32bit update_ram      : 1;
   u32bit update_flash    : 1;
   u32bit erase_or_prog   : 1;
   u32bit version         : 1;
   u32bit acknowledgment  : 1;
   u32bit rsvd2           :25;
#endif

#define DL_FROM_BDE     0       /* method */
#define DL_FROM_SLIM    1

   uint32       dl_to_adr_low;
   uint32       dl_to_adr_high;
   uint32       dl_len;
   union {
      uint32    dl_from_mbx_offset;
      ULP_BDE   dl_from_bde;
      ULP_BDE64 dl_from_bde64;
   } un;

} LOAD_SM_VAR;


/* Structure for MB Command READ_NVPARM (02) */

typedef struct  {
   uint32   rsvd1[3];          /* Read as all one's */
   uint32   rsvd2;             /* Read as all zero's */
   uint32   portname[2];       /* N_PORT name */
   uint32   nodename[2];       /* NODE name */
#if BIG_ENDIAN_HW
   u32bit    pref_DID  : 24;
   u32bit    hardAL_PA :  8;
#endif
#if LITTLE_ENDIAN_HW
   u32bit    hardAL_PA :  8;
   u32bit    pref_DID  : 24;
#endif
   uint32   rsvd3[21];          /* Read as all one's */
} READ_NV_VAR;


/* Structure for MB Command WRITE_NVPARMS (03) */

typedef struct  {
   uint32   rsvd1[3];          /* Must be all one's */
   uint32   rsvd2;             /* Must be all zero's */
   uint32   portname[2];       /* N_PORT name */
   uint32   nodename[2];       /* NODE name */
#if BIG_ENDIAN_HW
   u32bit    pref_DID  : 24;
   u32bit    hardAL_PA :  8;
#endif
#if LITTLE_ENDIAN_HW
   u32bit    hardAL_PA :  8;
   u32bit    pref_DID  : 24;
#endif
   uint32   rsvd3[21];          /* Must be all one's */
} WRITE_NV_VAR;


/* Structure for MB Command RUN_BIU_DIAG (04) */
/* Structure for MB Command RUN_BIU_DIAG64 (0x84) */

typedef struct  {
   uint32   rsvd1;
   union {
      struct  {
         ULP_BDE        xmit_bde;
         ULP_BDE        rcv_bde;
      } s1;
      struct  {
         ULP_BDE64      xmit_bde64;
         ULP_BDE64      rcv_bde64;
      } s2;
   } un;
} BIU_DIAG_VAR;


/* Structure for MB Command INIT_LINK (05) */

typedef struct  {
#if BIG_ENDIAN_HW
   u32bit    rsvd1       : 24;
   u32bit    lipsr_AL_PA :  8; /* AL_PA to issue Lip Selective Reset to */
#endif
#if LITTLE_ENDIAN_HW
   u32bit    lipsr_AL_PA :  8; /* AL_PA to issue Lip Selective Reset to */
   u32bit    rsvd1       : 24;
#endif

#if BIG_ENDIAN_HW
   uchar   fabric_AL_PA;     /* If using a Fabric Assigned AL_PA */
   uchar   rsvd2;
   ushort  link_flags;
#endif
#if LITTLE_ENDIAN_HW
   ushort  link_flags;
   uchar   rsvd2;
   uchar   fabric_AL_PA;     /* If using a Fabric Assigned AL_PA */
#endif
#define FLAGS_LOCAL_LB               0x01 /* link_flags (=1) ENDEC loopback */
#define FLAGS_TOPOLOGY_MODE_LOOP_PT  0x00 /* Attempt loop then pt-pt */
#define FLAGS_TOPOLOGY_MODE_PT_PT    0x02 /* Attempt pt-pt only */
#define FLAGS_TOPOLOGY_MODE_LOOP     0x04 /* Attempt loop only */
#define FLAGS_TOPOLOGY_MODE_PT_LOOP  0x06 /* Attempt pt-pt then loop */
#define FLAGS_LIRP_LILP              0x80 /* LIRP / LILP is disabled */

#define FLAGS_TOPOLOGY_FAILOVER      0x0400 /* Bit 10 */
#define FLAGS_LINK_SPEED             0x0800 /* Bit 11 */

   uint32  link_speed;            
#define LINK_SPEED_AUTO 0                 /* Auto selection */
#define LINK_SPEED_1G   1                 /* 1 Gigabaud */
#define LINK_SPEED_2G   2                 /* 2 Gigabaud */

} INIT_LINK_VAR;


/* Structure for MB Command DOWN_LINK (06) */

typedef struct  {
   uint32   rsvd1;
} DOWN_LINK_VAR;


/* Structure for MB Command CONFIG_LINK (07) */

typedef struct  {
#if BIG_ENDIAN_HW
   u32bit   cr          : 1;
   u32bit   ci          : 1;
   u32bit   cr_delay    : 6;
   u32bit   cr_count    : 8;
   u32bit   rsvd1       : 8;
   u32bit   MaxBBC      : 8;
#endif
#if LITTLE_ENDIAN_HW
   u32bit   MaxBBC      : 8;
   u32bit   rsvd1       : 8;
   u32bit   cr_count    : 8;
   u32bit   cr_delay    : 6;
   u32bit   ci          : 1;
   u32bit   cr          : 1;
#endif
   uint32   myId;
   uint32   rsvd2;
   uint32   edtov;
   uint32   arbtov;
   uint32   ratov;
   uint32   rttov;
   uint32   altov;
   uint32   crtov;
   uint32   citov;
#if BIG_ENDIAN_HW
   u32bit    rrq_enable  : 1;
   u32bit    rrq_immed   : 1;
   u32bit    rsvd4       : 29;
   u32bit    ack0_enable : 1;
#endif
#if LITTLE_ENDIAN_HW
   u32bit    ack0_enable : 1;
   u32bit    rsvd4       : 29;
   u32bit    rrq_immed   : 1;
   u32bit    rrq_enable  : 1;
#endif
} CONFIG_LINK;


/* Structure for MB Command PART_SLIM (08) */

typedef struct  {
#if BIG_ENDIAN_HW
   u32bit     unused1  : 24;
   u32bit     numRing  :  8;
#endif
#if LITTLE_ENDIAN_HW
   u32bit     numRing  :  8;
   u32bit     unused1  : 24;
#endif
   RING_DEF   ringdef[4];
   u32bit     hbainit;
} PART_SLIM_VAR;


/* Structure for MB Command CONFIG_RING (09) */

typedef struct  {
#if BIG_ENDIAN_HW
   u32bit    unused2   :  6;
   u32bit    recvSeq   :  1;
   u32bit    recvNotify:  1;
   u32bit    numMask   :  8;
   u32bit    profile   :  8;
   u32bit    unused1   :  4;
   u32bit    ring      :  4;
#endif
#if LITTLE_ENDIAN_HW
   u32bit    ring      :  4;
   u32bit    unused1   :  4;
   u32bit    profile   :  8;
   u32bit    numMask   :  8;
   u32bit    recvNotify:  1;
   u32bit    recvSeq   :  1;
   u32bit    unused2   :  6;
#endif
#if BIG_ENDIAN_HW
   ushort  maxRespXchg;
   ushort  maxOrigXchg;
#endif
#if LITTLE_ENDIAN_HW
   ushort  maxOrigXchg;
   ushort  maxRespXchg;
#endif
   RR_REG  rrRegs[6];
} CONFIG_RING_VAR;


/* Structure for MB Command RESET_RING (10) */

typedef struct  {
   uint32   ring_no;
} RESET_RING_VAR;


/* Structure for MB Command READ_CONFIG (11) */

typedef struct  {
#if BIG_ENDIAN_HW
   u32bit   cr          : 1;
   u32bit   ci          : 1;
   u32bit   cr_delay    : 6;
   u32bit   cr_count    : 8;
   u32bit   InitBBC     : 8;
   u32bit   MaxBBC      : 8;
#endif
#if LITTLE_ENDIAN_HW
   u32bit   MaxBBC      : 8;
   u32bit   InitBBC     : 8;
   u32bit   cr_count    : 8;
   u32bit   cr_delay    : 6;
   u32bit   ci          : 1;
   u32bit   cr          : 1;
#endif
#if BIG_ENDIAN_HW
   u32bit   topology :  8;
   u32bit   myDid    : 24;
#endif
#if LITTLE_ENDIAN_HW
   u32bit   myDid    : 24;
   u32bit   topology :  8;
#endif
   /* Defines for topology (defined previously) */
#if BIG_ENDIAN_HW
   u32bit   AR      :  1;
   u32bit   IR      :  1; 
   u32bit   rsvd1   : 29;
   u32bit   ack0    :  1;
#endif
#if LITTLE_ENDIAN_HW
   u32bit   ack0    :  1;
   u32bit   rsvd1   : 29;
   u32bit   IR      :  1; 
   u32bit   AR      :  1;
#endif
   uint32   edtov;
   uint32   arbtov;
   uint32   ratov;
   uint32   rttov;
   uint32   altov;
   uint32   lmt;
#define LMT_RESERVED    0x0    /* Not used */
#define LMT_266_10bit   0x1    /*  265.625 Mbaud 10 bit iface */
#define LMT_532_10bit   0x2    /*  531.25  Mbaud 10 bit iface */
#define LMT_1063_10bit  0x3    /* 1062.5   Mbaud 20 bit iface */
#define LMT_2125_10bit  0x8    /* 2125     Mbaud 10 bit iface */

   uint32   rsvd2;
   uint32   rsvd3;
   uint32   max_xri;
   uint32   max_iocb;
   uint32   max_rpi;
   uint32   avail_xri;
   uint32   avail_iocb;
   uint32   avail_rpi;
   uint32   default_rpi;
} READ_CONFIG_VAR;


/* Structure for MB Command READ_RCONFIG (12) */

typedef struct  {
#if BIG_ENDIAN_HW
   u32bit    rsvd2      :  7;
   u32bit    recvNotify :  1;
   u32bit    numMask    :  8;
   u32bit    profile    :  8;
   u32bit    rsvd1      :  4;
   u32bit    ring       :  4;
#endif
#if LITTLE_ENDIAN_HW
   u32bit    ring       :  4;
   u32bit    rsvd1      :  4;
   u32bit    profile    :  8;
   u32bit    numMask    :  8;
   u32bit    recvNotify :  1;
   u32bit    rsvd2      :  7;
#endif
#if BIG_ENDIAN_HW
   ushort  maxResp;
   ushort  maxOrig;
#endif
#if LITTLE_ENDIAN_HW
   ushort  maxOrig;
   ushort  maxResp;
#endif
   RR_REG  rrRegs[6];
#if BIG_ENDIAN_HW
   ushort  cmdRingOffset;
   ushort  cmdEntryCnt;
   ushort  rspRingOffset;
   ushort  rspEntryCnt;
   ushort  nextCmdOffset;
   ushort  rsvd3;
   ushort  nextRspOffset;
   ushort  rsvd4;
#endif
#if LITTLE_ENDIAN_HW
   ushort  cmdEntryCnt;
   ushort  cmdRingOffset;
   ushort  rspEntryCnt;
   ushort  rspRingOffset;
   ushort  rsvd3;
   ushort  nextCmdOffset;
   ushort  rsvd4;
   ushort  nextRspOffset;
#endif
} READ_RCONF_VAR;


/* Structure for MB Command READ_SPARM (13) */
/* Structure for MB Command READ_SPARM64 (0x8D) */

typedef struct  {
   uint32   rsvd1;
   uint32   rsvd2;
   union {
      ULP_BDE sp;         /* This BDE points to SERV_PARM structure */
      ULP_BDE64 sp64;
   } un;
}  READ_SPARM_VAR;


/* Structure for MB Command READ_STATUS (14) */

typedef struct  {
#if BIG_ENDIAN_HW
   u32bit    rsvd1        : 31;
   u32bit    clrCounters  :  1;
   ushort  activeXriCnt;
   ushort  activeRpiCnt;
#endif
#if LITTLE_ENDIAN_HW
   u32bit    clrCounters  :  1;
   u32bit    rsvd1        : 31;
   ushort  activeRpiCnt;
   ushort  activeXriCnt;
#endif
   uint32   xmitByteCnt;
   uint32   rcvbyteCnt;
   uint32   xmitFrameCnt;
   uint32   rcvFrameCnt;
   uint32   xmitSeqCnt;
   uint32   rcvSeqCnt;
   uint32   totalOrigExchanges;
   uint32   totalRespExchanges;
   uint32   rcvPbsyCnt;
   uint32   rcvFbsyCnt;
} READ_STATUS_VAR;


/* Structure for MB Command READ_RPI (15) */
/* Structure for MB Command READ_RPI64 (0x8F) */

typedef struct  {
#if BIG_ENDIAN_HW
   ushort  nextRpi;
   ushort  reqRpi;
   u32bit    rsvd2 :  8;
   u32bit    DID   : 24;
#endif
#if LITTLE_ENDIAN_HW
   ushort  reqRpi;
   ushort  nextRpi;
   u32bit    DID   : 24;
   u32bit    rsvd2 :  8;
#endif
   union {
      ULP_BDE   sp;
      ULP_BDE64 sp64;
   } un;

}  READ_RPI_VAR;


/* Structure for MB Command READ_XRI (16) */

typedef struct  {
#if BIG_ENDIAN_HW
   ushort  nextXri;
   ushort  reqXri;
   ushort  rsvd1;
   ushort  rpi;
   u32bit    rsvd2     :  8;
   u32bit    DID       : 24;
   u32bit    rsvd3     :  8;
   u32bit    SID       : 24;
   uint32   rsvd4;
   uchar   seqId;
   uchar   rsvd5;
   ushort  seqCount;
   ushort  oxId;
   ushort  rxId;
   u32bit    rsvd6     : 30;
   u32bit    si        :  1;
   u32bit    exchOrig  :  1;
#endif
#if LITTLE_ENDIAN_HW
   ushort  reqXri;
   ushort  nextXri;
   ushort  rpi;
   ushort  rsvd1;
   u32bit    DID       : 24;
   u32bit    rsvd2     :  8;
   u32bit    SID       : 24;
   u32bit    rsvd3     :  8;
   uint32   rsvd4;
   ushort  seqCount;
   uchar   rsvd5;
   uchar   seqId;
   ushort  rxId;
   ushort  oxId;
   u32bit    exchOrig  :  1;
   u32bit    si        :  1;
   u32bit    rsvd6     : 30;
#endif
} READ_XRI_VAR;


/* Structure for MB Command READ_REV (17) */

typedef struct  {
#if BIG_ENDIAN_HW
   u32bit    cv        :  1;
   u32bit    rr        :  1;
   u32bit    rsvd1     : 29;
   u32bit    rv        :  1;
#endif
#if LITTLE_ENDIAN_HW
   u32bit    rv        :  1;
   u32bit    rsvd1     : 29;
   u32bit    rr        :  1;
   u32bit    cv        :  1;
#endif
   uint32   biuRev;
   uint32   smRev;
   union {
      uint32   smFwRev;
      struct {
#if BIG_ENDIAN_HW
         uchar   ProgType;
         uchar   ProgId;
         u16bit    ProgVer        : 4;
         u16bit    ProgRev        : 4;
         u16bit    ProgFixLvl     : 2;
         u16bit    ProgDistType   : 2;
         u16bit    DistCnt        : 4;
#endif
#if LITTLE_ENDIAN_HW
         u16bit    DistCnt        : 4;
         u16bit    ProgDistType   : 2;
         u16bit    ProgFixLvl     : 2;
         u16bit    ProgRev        : 4;
         u16bit    ProgVer        : 4;
         uchar   ProgId;
         uchar   ProgType;
#endif
      } b;
   } un;
   uint32   endecRev;
#if BIG_ENDIAN_HW
   uchar   feaLevelHigh;
   uchar   feaLevelLow;
   uchar   fcphHigh;
   uchar   fcphLow;
#endif
#if LITTLE_ENDIAN_HW
   uchar   fcphLow;
   uchar   fcphHigh;
   uchar   feaLevelLow;
   uchar   feaLevelHigh;
#endif
   uint32   postKernRev;
   uint32   opFwRev;
   uchar   opFwName[16];
   uint32   sli1FwRev;
   uchar   sli1FwName[16];
   uint32   sli2FwRev;
   uchar   sli2FwName[16];
   uint32   rsvd2;
   uint32   RandomData[7];
} READ_REV_VAR;

#define rxSeqRev postKernRev 
#define txSeqRev opFwRev 

/* Structure for MB Command READ_LINK_STAT (18) */

typedef struct  {
   uint32   rsvd1;
   uint32   linkFailureCnt;
   uint32   lossSyncCnt;

   uint32   lossSignalCnt;
   uint32   primSeqErrCnt;
   uint32   invalidXmitWord;
   uint32   crcCnt;
   uint32   primSeqTimeout;
   uint32   elasticOverrun;
   uint32   arbTimeout;
} READ_LNK_VAR;


/* Structure for MB Command REG_LOGIN (19) */
/* Structure for MB Command REG_LOGIN64 (0x93) */

typedef struct  {
#if BIG_ENDIAN_HW
   ushort  rsvd1;
   ushort  rpi;
   u32bit    rsvd2 : 8;
   u32bit    did   : 24;
#endif
#if LITTLE_ENDIAN_HW
   ushort  rpi;
   ushort  rsvd1;
   u32bit    did   : 24;
   u32bit    rsvd2 : 8;
#endif
   union {
      ULP_BDE   sp;
      ULP_BDE64 sp64;
   } un;

}  REG_LOGIN_VAR;

/* Word 30 contents for REG_LOGIN */
typedef union {
   struct {
#if BIG_ENDIAN_HW
      u16bit    rsvd1 : 12;
      u16bit    class : 4;
      ushort  xri;
#endif
#if LITTLE_ENDIAN_HW
      ushort  xri;
      u16bit    class : 4;
      u16bit    rsvd1 : 12;
#endif
   } f;
   uint32 word;
}  REG_WD30;


/* Structure for MB Command UNREG_LOGIN (20) */

typedef struct  {
#if BIG_ENDIAN_HW
   ushort  rsvd1;
   ushort  rpi;
#endif
#if LITTLE_ENDIAN_HW
   ushort  rpi;
   ushort  rsvd1;
#endif
} UNREG_LOGIN_VAR;


/* Structure for MB Command UNREG_D_ID (0x23) */

typedef struct  {
   uint32  did;
} UNREG_D_ID_VAR;


/* Structure for MB Command READ_LA (21) */
/* Structure for MB Command READ_LA64 (0x95) */

typedef struct  {
   uint32   eventTag;         /* Event tag */
#if BIG_ENDIAN_HW
   u32bit    rsvd1   : 22;
   u32bit    pb      :  1;
   u32bit    il      :  1;
   u32bit    attType :  8;
#endif
#if LITTLE_ENDIAN_HW
   u32bit    attType :  8;
   u32bit    il      :  1;
   u32bit    pb      :  1;
   u32bit    rsvd1   : 22;
#endif
#define AT_RESERVED    0x00  /* Reserved - attType */
#define AT_LINK_UP     0x01  /* Link is up */
#define AT_LINK_DOWN   0x02  /* Link is down */
#if BIG_ENDIAN_HW
   uchar   granted_AL_PA;
   uchar   lipAlPs;
   uchar   lipType;
   uchar   topology;
#endif
#if LITTLE_ENDIAN_HW
   uchar   topology;
   uchar   lipType;
   uchar   lipAlPs;
   uchar   granted_AL_PA;
#endif
#define LT_PORT_INIT    0x00 /* An L_PORT initing (F7, AL_PS) - lipType */
#define LT_PORT_ERR     0x01 /* Err @L_PORT rcv'er (F8, AL_PS) */
#define LT_RESET_APORT  0x02 /* Lip Reset of some other port */
#define LT_RESET_MYPORT 0x03 /* Lip Reset of my port */
#define TOPOLOGY_PT_PT 0x01  /* Topology is pt-pt / pt-fabric */
#define TOPOLOGY_LOOP  0x02  /* Topology is FC-AL */

   union {
      ULP_BDE lilpBde;         /* This BDE points to a 128 byte buffer to */
      /* store the LILP AL_PA position map into */
      ULP_BDE64 lilpBde64;
   } un;
#if BIG_ENDIAN_HW
   u32bit    Dlu       :  1;
   u32bit    Dtf       :  1;
   u32bit    Drsvd2    : 14;
   u32bit    DlnkSpeed :  8;
   u32bit    DnlPort   :  4;
   u32bit    Dtx       :  2;
   u32bit    Drx       :  2;
#endif
#if LITTLE_ENDIAN_HW
   u32bit    Drx       :  2;
   u32bit    Dtx       :  2;
   u32bit    DnlPort   :  4;
   u32bit    DlnkSpeed :  8;
   u32bit    Drsvd2    : 14;
   u32bit    Dtf       :  1;
   u32bit    Dlu       :  1;
#endif
#if BIG_ENDIAN_HW
   u32bit    Ulu       :  1;
   u32bit    Utf       :  1;
   u32bit    Ursvd2    : 14;
   u32bit    UlnkSpeed :  8;
   u32bit    UnlPort   :  4;
   u32bit    Utx       :  2;
   u32bit    Urx       :  2;
#endif
#if LITTLE_ENDIAN_HW
   u32bit    Urx       :  2;
   u32bit    Utx       :  2;
   u32bit    UnlPort   :  4;
   u32bit    UlnkSpeed :  8;
   u32bit    Ursvd2    : 14;
   u32bit    Utf       :  1;
   u32bit    Ulu       :  1;
#endif
#define LA_1GHZ_LINK   4    /* lnkSpeed */
#define LA_2GHZ_LINK   8    /* lnkSpeed */

} READ_LA_VAR;


/* Structure for MB Command CLEAR_LA (22) */

typedef struct  {
   uint32   eventTag;         /* Event tag */
   uint32   rsvd1;
} CLEAR_LA_VAR;

/* Structure for MB Command DUMP */

typedef struct {
#if BIG_ENDIAN_HW
   u32bit rsvd           : 25 ;
   u32bit ra             : 1 ;
   u32bit co             : 1 ;
   u32bit cv             : 1 ;
   u32bit type           : 4 ;
   u32bit entry_index    : 16 ;
   u32bit region_id      : 16 ;
#endif
#if LITTLE_ENDIAN_HW
   u32bit type           : 4 ;
   u32bit cv             : 1 ;
   u32bit co             : 1 ;
   u32bit ra             : 1 ;
   u32bit rsvd           : 25 ;
   u32bit region_id      : 16 ;
   u32bit entry_index    : 16 ;
#endif
   uint32 rsvd1;
   uint32 word_cnt ;
   uint32 resp_offset ;
}  DUMP_VAR ;

#define  DMP_MEM_REG             0x1
#define  DMP_NV_PARAMS           0x2

#define  DMP_REGION_VPD          0xe
#define  DMP_VPD_SIZE            0x100

/* Structure for MB Command CONFIG_PORT (0x88) */

typedef struct  {
   uint32    pcbLen;
   uint32    pcbLow;     /* bit 31:0  of memory based port config block */
   uint32    pcbHigh;    /* bit 63:32 of memory based port config block */
   uint32    hbainit[5];
} CONFIG_PORT_VAR;


/* SLI-2 Port Control Block */

/* SLIM POINTER */
#define SLIMOFF 0x30  /* WORD */

typedef struct  _SLI2_RDSC {
   uint32  cmdEntries;
   uint32  cmdAddrLow;
   uint32  cmdAddrHigh;

   uint32  rspEntries;
   uint32  rspAddrLow;
   uint32  rspAddrHigh;
} SLI2_RDSC;

typedef struct  _PCB {
#if BIG_ENDIAN_HW
   u32bit       type    :  8;
#define TYPE_NATIVE_SLI2       0x01;
   u32bit       feature :  8;
#define FEATURE_INITIAL_SLI2   0x01;
   u32bit       rsvd    : 12;
   u32bit       maxRing :  4;
#endif
#if LITTLE_ENDIAN_HW
   u32bit       maxRing :  4;
   u32bit       rsvd    : 12;
   u32bit       feature :  8;
#define FEATURE_INITIAL_SLI2   0x01;
   u32bit       type    :  8;
#define TYPE_NATIVE_SLI2       0x01;
#endif

   uint32      mailBoxSize;
   uint32      mbAddrLow;
   uint32      mbAddrHigh;

   uint32      hgpAddrLow;
   uint32      hgpAddrHigh;

   uint32      pgpAddrLow;
   uint32      pgpAddrHigh;
   SLI2_RDSC  rdsc[ MAX_RINGS];
} PCB;

typedef struct  {
#if BIG_ENDIAN_HW
   u32bit      rsvd0        : 27;
   u32bit      discardFarp  : 1;
   u32bit      IPEnable     : 1;
   u32bit      nodeName     : 1;
   u32bit      portName     : 1;
   u32bit      filterEnable : 1;
#endif
#if LITTLE_ENDIAN_HW
   u32bit      filterEnable : 1;
   u32bit      portName     : 1;
   u32bit      nodeName     : 1;
   u32bit      IPEnable     : 1;
   u32bit      discardFarp  : 1;
   u32bit      rsvd         : 27;
#endif
   NAME_TYPE   portname;
   NAME_TYPE   nodename;
   uint32      rsvd1;
   uint32      rsvd2;
   uint32      rsvd3;
   uint32      IPAddress;
} CONFIG_FARP_VAR;


/* Union of all Mailbox Command types */

typedef union  {
   uint32          varWords[31];
   LOAD_SM_VAR     varLdSM;       /* cmd =  1 (LOAD_SM)        */
   READ_NV_VAR     varRDnvp;      /* cmd =  2 (READ_NVPARMS)   */
   WRITE_NV_VAR    varWTnvp;      /* cmd =  3 (WRITE_NVPARMS)  */
   BIU_DIAG_VAR    varBIUdiag;    /* cmd =  4 (RUN_BIU_DIAG)   */
   INIT_LINK_VAR   varInitLnk;    /* cmd =  5 (INIT_LINK)      */
   DOWN_LINK_VAR   varDwnLnk;     /* cmd =  6 (DOWN_LINK)      */
   CONFIG_LINK     varCfgLnk;     /* cmd =  7 (CONFIG_LINK)    */
   PART_SLIM_VAR   varSlim;       /* cmd =  8 (PART_SLIM)      */
   CONFIG_RING_VAR varCfgRing;    /* cmd =  9 (CONFIG_RING)    */
   RESET_RING_VAR  varRstRing;    /* cmd = 10 (RESET_RING)     */
   READ_CONFIG_VAR varRdConfig;   /* cmd = 11 (READ_CONFIG)    */
   READ_RCONF_VAR  varRdRConfig;  /* cmd = 12 (READ_RCONFIG)   */
   READ_SPARM_VAR  varRdSparm;    /* cmd = 13 (READ_SPARM(64)) */
   READ_STATUS_VAR varRdStatus;   /* cmd = 14 (READ_STATUS)    */
   READ_RPI_VAR    varRdRPI;      /* cmd = 15 (READ_RPI(64))   */
   READ_XRI_VAR    varRdXRI;      /* cmd = 16 (READ_XRI)       */
   READ_REV_VAR    varRdRev;      /* cmd = 17 (READ_REV)       */
   READ_LNK_VAR    varRdLnk;      /* cmd = 18 (READ_LNK_STAT)  */
   REG_LOGIN_VAR   varRegLogin;   /* cmd = 19 (REG_LOGIN(64))  */
   UNREG_LOGIN_VAR varUnregLogin; /* cmd = 20 (UNREG_LOGIN)    */
   READ_LA_VAR     varReadLA;     /* cmd = 21 (READ_LA(64))    */
   CLEAR_LA_VAR    varClearLA;    /* cmd = 22 (CLEAR_LA)       */
   DUMP_VAR        varDmp ;       /* Warm Start DUMP mbx cmd   */  
   UNREG_D_ID_VAR  varUnregDID;   /* cmd = 0x23 (UNREG_D_ID)   */
   CONFIG_PORT_VAR varCfgPort;    /* cmd = 0x88 (CONFIG_PORT)  */
   CONFIG_FARP_VAR varCfgFarp;    /* cmd = 0x25 (CONFIG_FARP) */
} MAILVARIANTS;

#define MAILBOX_CMD_WSIZE    32

/*
 * SLI-2 specific structures
 */

typedef struct  _SLI1_DESC {
   RINGS   mbxCring[ 4];
   uint32   mbxUnused[ 24];
} SLI1_DESC;

typedef struct  {
   uint32   cmdPutInx;
   uint32   rspGetInx;
} HGP;

typedef struct  {
   uint32   cmdGetInx;
   uint32   rspPutInx;
} PGP;

typedef struct  _SLI2_DESC {
   HGP     host[ MAX_RINGS];
   uint32   unused[ 16];
   PGP     port[ MAX_RINGS];
} SLI2_DESC;

typedef union  {
   SLI1_DESC s1;
   SLI2_DESC s2;
} SLI_VAR;

typedef volatile struct  {
#if BIG_ENDIAN_HW
   ushort  mbxStatus;
   uchar   mbxCommand;
   u8bit    mbxReserved   :  6;
   u8bit    mbxHc         :  1;
   u8bit    mbxOwner      :  1;    /* Low order bit first word */
#endif
#if LITTLE_ENDIAN_HW
   u8bit    mbxOwner      :  1;    /* Low order bit first word */
   u8bit    mbxHc         :  1;
   u8bit    mbxReserved   :  6;
   uchar   mbxCommand;
   ushort  mbxStatus;
#endif
   MAILVARIANTS un;
   SLI_VAR      us;
} MAILBOX, *PMAILBOX;

/*
 *    End Structure Definitions for Mailbox Commands
 */


/*
 *    Begin Structure Definitions for IOCB Commands
 */

typedef struct  {
#if BIG_ENDIAN_HW
   uchar   statAction;
   uchar   statRsn;
   uchar   statBaExp;
   uchar   statLocalError;
#endif
#if LITTLE_ENDIAN_HW
   uchar   statLocalError;
   uchar   statBaExp;
   uchar   statRsn;
   uchar   statAction;
#endif
   /* statAction  FBSY reason codes */
#define FBSY_RSN_MASK   0xF0  /* Rsn stored in upper nibble */
#define FBSY_FABRIC_BSY 0x10  /* F_bsy due to Fabric BSY */
#define FBSY_NPORT_BSY  0x30  /* F_bsy due to N_port BSY */

   /* statAction  PBSY action codes */
#define PBSY_ACTION1    0x01  /* Sequence terminated - retry */
#define PBSY_ACTION2    0x02  /* Sequence active - retry */

   /* statAction  P/FRJT action codes */
#define RJT_RETRYABLE   0x01  /* Retryable class of error */
#define RJT_NO_RETRY    0x02  /* Non-Retryable class of error */

   /* statRsn  LS_RJT reason codes defined in LS_RJT structure */

   /* statRsn  P_BSY reason codes */
#define PBSY_NPORT_BSY  0x01  /* Physical N_port BSY */
#define PBSY_RESRCE_BSY 0x03  /* N_port resource BSY */
#define PBSY_VU_BSY     0xFF  /* See VU field for rsn */

   /* statRsn  P/F_RJT reason codes */
#define RJT_BAD_D_ID       0x01  /* Invalid D_ID field */
#define RJT_BAD_S_ID       0x02  /* Invalid S_ID field */
#define RJT_UNAVAIL_TEMP   0x03  /* N_Port unavailable temp. */
#define RJT_UNAVAIL_PERM   0x04  /* N_Port unavailable perm. */
#define RJT_UNSUP_CLASS    0x05  /* Class not supported */
#define RJT_DELIM_ERR      0x06  /* Delimiter usage error */
#define RJT_UNSUP_TYPE     0x07  /* Type not supported */
#define RJT_BAD_CONTROL    0x08  /* Invalid link conrtol */
#define RJT_BAD_RCTL       0x09  /* R_CTL invalid */
#define RJT_BAD_FCTL       0x0A  /* F_CTL invalid */
#define RJT_BAD_OXID       0x0B  /* OX_ID invalid */
#define RJT_BAD_RXID       0x0C  /* RX_ID invalid */
#define RJT_BAD_SEQID      0x0D  /* SEQ_ID invalid */
#define RJT_BAD_DFCTL      0x0E  /* DF_CTL invalid */
#define RJT_BAD_SEQCNT     0x0F  /* SEQ_CNT invalid */
#define RJT_BAD_PARM       0x10  /* Param. field invalid */
#define RJT_XCHG_ERR       0x11  /* Exchange error */
#define RJT_PROT_ERR       0x12  /* Protocol error */
#define RJT_BAD_LENGTH     0x13  /* Invalid Length */
#define RJT_UNEXPECTED_ACK 0x14  /* Unexpected ACK */
#define RJT_LOGIN_REQUIRED 0x16  /* Login required */
#define RJT_TOO_MANY_SEQ   0x17  /* Excessive sequences */
#define RJT_XCHG_NOT_STRT  0x18  /* Exchange not started */
#define RJT_UNSUP_SEC_HDR  0x19  /* Security hdr not supported */
#define RJT_UNAVAIL_PATH   0x1A  /* Fabric Path not available */
#define RJT_VENDOR_UNIQUE  0xFF  /* Vendor unique error */

   /* statRsn  BA_RJT reason codes */
#define BARJT_BAD_CMD_CODE 0x01  /* Invalid command code */ 
#define BARJT_LOGICAL_ERR  0x03  /* Logical error */ 
#define BARJT_LOGICAL_BSY  0x05  /* Logical busy */ 
#define BARJT_PROTOCOL_ERR 0x07  /* Protocol error */ 
#define BARJT_VU_ERR       0xFF  /* Vendor unique error */ 

   /* LS_RJT reason explanation defined in LS_RJT structure */

   /* BA_RJT reason explanation */
#define BARJT_EXP_INVALID_ID  0x01  /* Invalid OX_ID/RX_ID */
#define BARJT_EXP_ABORT_SEQ   0x05  /* Abort SEQ, no more info */

   /* Localy detected errors */
#define IOERR_SUCCESS                 0x00  /* statLocalError */
#define IOERR_MISSING_CONTINUE        0x01
#define IOERR_SEQUENCE_TIMEOUT        0x02
#define IOERR_INTERNAL_ERROR          0x03
#define IOERR_INVALID_RPI             0x04
#define IOERR_NO_XRI                  0x05
#define IOERR_ILLEGAL_COMMAND         0x06
#define IOERR_XCHG_DROPPED            0x07
#define IOERR_ILLEGAL_FIELD           0x08
#define IOERR_BAD_CONTINUE            0x09
#define IOERR_TOO_MANY_BUFFERS        0x0A
#define IOERR_RCV_BUFFER_WAITING      0x0B
#define IOERR_NO_CONNECTION           0x0C
#define IOERR_TX_DMA_FAILED           0x0D
#define IOERR_RX_DMA_FAILED           0x0E
#define IOERR_ILLEGAL_FRAME           0x0F
#define IOERR_EXTRA_DATA              0x10
#define IOERR_NO_RESOURCES            0x11
#define IOERR_RESERVED                0x12
#define IOERR_ILLEGAL_LENGTH          0x13
#define IOERR_UNSUPPORTED_FEATURE     0x14
#define IOERR_ABORT_IN_PROGRESS       0x15
#define IOERR_ABORT_REQUESTED         0x16
#define IOERR_RECEIVE_BUFFER_TIMEOUT  0x17
#define IOERR_LOOP_OPEN_FAILURE       0x18
#define IOERR_RING_RESET              0x19
#define IOERR_LINK_DOWN               0x1A
#define IOERR_CORRUPTED_DATA          0x1B
#define IOERR_CORRUPTED_RPI           0x1C
#define IOERR_OUT_OF_ORDER_DATA       0x1D
#define IOERR_OUT_OF_ORDER_ACK        0x1E
#define IOERR_DUP_FRAME               0x1F
#define IOERR_LINK_CONTROL_FRAME      0x20      /* ACK_N received */
#define IOERR_BAD_HOST_ADDRESS        0x21
#define IOERR_RCV_HDRBUF_WAITING      0x22
#define IOERR_MISSING_HDR_BUFFER      0x23
#define IOERR_MSEQ_CHAIN_CORRUPTED    0x24
#define IOERR_ABORTMULT_REQUESTED     0x25
#define IOERR_BUFFER_SHORTAGE         0x28
} PARM_ERR;

typedef union  {
   struct {
#if BIG_ENDIAN_HW
      uchar   Rctl;                      /* R_CTL field */
      uchar   Type;                      /* TYPE field */
      uchar   Dfctl;                     /* DF_CTL field */
      uchar   Fctl;                      /* Bits 0-7 of IOCB word 5 */
#endif
#if LITTLE_ENDIAN_HW
      uchar   Fctl;                      /* Bits 0-7 of IOCB word 5 */
      uchar   Dfctl;                     /* DF_CTL field */
      uchar   Type;                      /* TYPE field */
      uchar   Rctl;                      /* R_CTL field */
#endif

#define BC      0x02     /* Broadcast Received  - Fctl */
#define SI      0x04     /* Sequence Initiative */
#define LA      0x08     /* Ignore Link Attention state */
#define LS      0x80     /* Last Sequence */
   } hcsw;
   uint32  reserved;
} WORD5;


/* IOCB Command template for a generic response */
typedef struct  {
   uint32    reserved[4];
   PARM_ERR perr;
} GENERIC_RSP;


/* IOCB Command template for XMIT / XMIT_BCAST / RCV_SEQUENCE / XMIT_ELS */
typedef struct  {
   ULP_BDE xrsqbde[2];
   uint32   xrsqRo;              /* Starting Relative Offset */
   WORD5   w5;                  /* Header control/status word */
} XR_SEQ_FIELDS;

/* IOCB Command template for ELS_REQUEST */
typedef struct  {
   ULP_BDE  elsReq;
   ULP_BDE  elsRsp;
#if BIG_ENDIAN_HW
   u32bit     word4Rsvd :  7;
   u32bit     fl        :  1;
   u32bit     myID      : 24;
   u32bit     word5Rsvd :  8;
   u32bit     remoteID  : 24;
#endif
#if LITTLE_ENDIAN_HW
   u32bit     myID      : 24;
   u32bit     fl        :  1;
   u32bit     word4Rsvd :  7;
   u32bit     remoteID  : 24;
   u32bit     word5Rsvd :  8;
#endif
} ELS_REQUEST;

/* IOCB Command template for RCV_ELS_REQ */
typedef struct  {
   ULP_BDE  elsReq[2];
   uint32    parmRo;
#if BIG_ENDIAN_HW
   u32bit     word5Rsvd :  8;
   u32bit     remoteID  : 24;
#endif
#if LITTLE_ENDIAN_HW
   u32bit     remoteID  : 24;
   u32bit     word5Rsvd :  8;
#endif
} RCV_ELS_REQ;

/* IOCB Command template for ABORT / CLOSE_XRI */
typedef struct  {
   uint32    rsvd[3];
   uint32    abortType;
#define ABORT_TYPE_ABTX  0x00000000
#define ABORT_TYPE_ABTS  0x00000001
   uint32    parm;
#if BIG_ENDIAN_HW
   ushort   abortContextTag;      /* ulpContext from command to abort/close */
   ushort   abortIoTag;           /* ulpIoTag from command to abort/close */
#endif
#if LITTLE_ENDIAN_HW
   ushort   abortIoTag;           /* ulpIoTag from command to abort/close */
   ushort   abortContextTag;      /* ulpContext from command to abort/close */
#endif
} AC_XRI;

/* IOCB Command template for GET_RPI */
typedef struct  {
   uint32    rsvd[4];
   uint32    parmRo;
#if BIG_ENDIAN_HW
   u32bit     word5Rsvd :  8;
   u32bit     remoteID  : 24;
#endif
#if LITTLE_ENDIAN_HW
   u32bit     remoteID  : 24;
   u32bit     word5Rsvd :  8;
#endif
} GET_RPI;

/* IOCB Command template for all FCPI commands */
typedef struct  {
   ULP_BDE fcpi_cmnd;           /* FCP_CMND payload descriptor */
   ULP_BDE fcpi_rsp;            /* Rcv buffer */
   uint32   fcpi_parm;
   uint32   fcpi_XRdy;           /* transfer ready for IWRITE */
} FCPI_FIELDS;

/* IOCB Command template for all FCPT commands */
typedef struct  {
   ULP_BDE fcpt_Buffer[2];      /* FCP_CMND payload descriptor */
   uint32   fcpt_Offset;
   uint32   fcpt_Length;         /* transfer ready for IWRITE */
} FCPT_FIELDS;

/* SLI-2 IOCB structure definitions */

/* IOCB Command template for 64 bit XMIT / XMIT_BCAST / XMIT_ELS */
typedef struct  {
   ULP_BDL bdl;
   uint32   xrsqRo;              /* Starting Relative Offset */
   WORD5   w5;                  /* Header control/status word */
} XMT_SEQ_FIELDS64;

/* IOCB Command template for 64 bit RCV_SEQUENCE64 */
typedef struct  {
   ULP_BDE64 rcvBde;
   uint32   rsvd1;
   uint32   xrsqRo;              /* Starting Relative Offset */
   WORD5   w5;                  /* Header control/status word */
} RCV_SEQ_FIELDS64;

/* IOCB Command template for ELS_REQUEST64 */
typedef struct  {
   ULP_BDL      bdl;
#if BIG_ENDIAN_HW
   u32bit     word4Rsvd :  7;
   u32bit     fl        :  1;
   u32bit     myID      : 24;
   u32bit     word5Rsvd :  8;
   u32bit     remoteID  : 24;
#endif
#if LITTLE_ENDIAN_HW
   u32bit     myID      : 24;
   u32bit     fl        :  1;
   u32bit     word4Rsvd :  7;
   u32bit     remoteID  : 24;
   u32bit     word5Rsvd :  8;
#endif
} ELS_REQUEST64;

/* IOCB Command template for GEN_REQUEST64 */
typedef struct  {
   ULP_BDL      bdl;
   uint32   xrsqRo;              /* Starting Relative Offset */
   WORD5    w5;                  /* Header control/status word */
} GEN_REQUEST64;

/* IOCB Command template for RCV_ELS_REQ64 */
typedef struct  {
   ULP_BDE64 elsReq;
   uint32    rcvd1;
   uint32    parmRo;
#if BIG_ENDIAN_HW
   u32bit     word5Rsvd :  8;
   u32bit     remoteID  : 24;
#endif
#if LITTLE_ENDIAN_HW
   u32bit     remoteID  : 24;
   u32bit     word5Rsvd :  8;
#endif
} RCV_ELS_REQ64;

/* IOCB Command template for all 64 bit FCPI commands */
typedef struct  {
   ULP_BDL bdl;
   uint32   fcpi_parm;
   uint32   fcpi_XRdy;           /* transfer ready for IWRITE */
} FCPI_FIELDS64;

/* IOCB Command template for all 64 bit FCPT commands */
typedef struct  {
   ULP_BDL bdl;
   uint32   fcpt_Offset;
   uint32   fcpt_Length;         /* transfer ready for IWRITE */
} FCPT_FIELDS64;

typedef volatile struct  _IOCB {      /* IOCB structure */
   union {
      GENERIC_RSP    grsp;            /* Generic response */
      XR_SEQ_FIELDS  xrseq;           /* XMIT / BCAST / RCV_SEQUENCE cmd */
      ULP_BDE        cont[3];         /* up to 3 continuation bdes */
      ELS_REQUEST    elsreq;          /* ELS_REQUEST template */
      RCV_ELS_REQ    rcvels;          /* RCV_ELS_REQ template */
      AC_XRI         acxri;           /* ABORT / CLOSE_XRI template */
      GET_RPI        getrpi;          /* GET_RPI template */
      FCPI_FIELDS    fcpi;            /* FCPI template */
      FCPT_FIELDS    fcpt;            /* FCPT template */

      /* SLI-2 structures */

      ULP_BDE64        cont64[ 2];    /* up to 2 64 bit continuation bde_64s */
      ELS_REQUEST64    elsreq64;      /* ELS_REQUEST template */
      GEN_REQUEST64    genreq64;      /* GEN_REQUEST template */
      RCV_ELS_REQ64    rcvels64;      /* RCV_ELS_REQ template */
      XMT_SEQ_FIELDS64 xseq64;        /* XMIT / BCAST cmd */
      FCPI_FIELDS64    fcpi64;        /* FCPI 64 bit template */
      FCPT_FIELDS64    fcpt64;        /* FCPT 64 bit template */

      uint32  ulpWord[IOCB_WORD_SZ-2]; /* generic 6 'words' */
   } un;
   union {
      struct  {
#if BIG_ENDIAN_HW
         ushort  ulpContext;                /* High order bits word 6 */
         ushort  ulpIoTag;                  /* Low  order bits word 6 */
#endif
#if LITTLE_ENDIAN_HW
         ushort  ulpIoTag;                  /* Low  order bits word 6 */
         ushort  ulpContext;                /* High order bits word 6 */
#endif
      } t1;
      struct  {
#if BIG_ENDIAN_HW
         ushort  ulpContext;                /* High order bits word 6 */
         u16bit    ulpIoTag1      :  2;       /* Low  order bits word 6 */
         u16bit    ulpIoTag0      : 14;       /* Low  order bits word 6 */
#endif
#if LITTLE_ENDIAN_HW
         u16bit    ulpIoTag0      : 14;       /* Low  order bits word 6 */
         u16bit    ulpIoTag1      :  2;       /* Low  order bits word 6 */
         ushort  ulpContext;                /* High order bits word 6 */
#endif
      } t2;
   } un1;
#define ulpContext un1.t1.ulpContext
#define ulpIoTag   un1.t1.ulpIoTag
#define ulpIoTag0  un1.t2.ulpIoTag0
#define ulpDelayXmit  un1.t2.ulpIoTag1
#define IOCB_DELAYXMIT_MSK 0x3000
#if BIG_ENDIAN_HW
   u32bit    ulpRsvdByte   :  8;
   u32bit    ulpXS         :  1;
   u32bit    ulpFCP2Rcvy   :  1;
   u32bit    ulpPU         :  2;
   u32bit    ulpIr         :  1;
   u32bit    ulpClass      :  3;
   u32bit    ulpCommand    :  8;
   u32bit    ulpStatus     :  4;
   u32bit    ulpBdeCount   :  2;
   u32bit    ulpLe         :  1;
   u32bit    ulpOwner      :  1;        /* Low order bit word 7 */
#endif
#if LITTLE_ENDIAN_HW
   u32bit    ulpOwner      :  1;        /* Low order bit word 7 */
   u32bit    ulpLe         :  1;
   u32bit    ulpBdeCount   :  2;
   u32bit    ulpStatus     :  4;
   u32bit    ulpCommand    :  8;
   u32bit    ulpClass      :  3;
   u32bit    ulpIr         :  1;
   u32bit    ulpPU         :  2;
   u32bit    ulpFCP2Rcvy   :  1;
   u32bit    ulpXS         :  1;
   u32bit    ulpRsvdByte   :  8;
#endif

#define ulpTimeout  ulpRsvdByte

#define IOCB_FCP           1  /* IOCB is used for FCP ELS cmds - ulpRsvByte */
#define IOCB_IP            2  /* IOCB is used for IP ELS cmds */
#define PARM_UNUSED        0  /* PU field (Word 4) not used */
#define PARM_REL_OFF       1  /* PU field (Word 4) = R. O. */
#define PARM_READ_CHECK    2  /* PU field (Word 4) = Data Transfer Length */
#define CLASS1             0  /* Class 1 */
#define CLASS2             1  /* Class 2 */
#define CLASS3             2  /* Class 3 */
#define CLASS_FCP_INTERMIX 7  /* FCP Data->Cls 1, all else->Cls 2 */

#define IOSTAT_SUCCESS         0x0  /* ulpStatus */
#define IOSTAT_FCP_RSP_ERROR   0x1
#define IOSTAT_REMOTE_STOP     0x2
#define IOSTAT_LOCAL_REJECT    0x3
#define IOSTAT_NPORT_RJT       0x4
#define IOSTAT_FABRIC_RJT      0x5
#define IOSTAT_NPORT_BSY       0x6
#define IOSTAT_FABRIC_BSY      0x7
#define IOSTAT_INTERMED_RSP    0x8
#define IOSTAT_LS_RJT          0x9
#define IOSTAT_BA_RJT          0xA

} IOCB, *PIOCB;

typedef struct  {
   IOCB   iocb;         /* iocb entry */
   uchar  * q;          /* ptr to next iocb entry */
   uchar  * bp;         /* ptr to data buffer structure */
   uchar  * info;       /* ptr to data information structure */
   uchar  * bpl;        /* ptr to data BPL structure */
   uchar  * ndlp;       /* ptr to the ndlp structure */
   uchar    retry;      /* retry counter for IOCB cmd - if needed */
   uchar    rsvd1;
   ushort   rsvd2;
} IOCBQ;

typedef struct  {
   volatile uint32 mb[MAILBOX_CMD_WSIZE];
   uchar  * q;
   uchar  * bp;         /* ptr to data buffer structure */
} MAILBOXQ;

/* Given a pointer to the start of the ring, and the slot number of
 * the desired iocb entry, calc a pointer to that entry.
 */
#define IOCB_ENTRY(ring,slot) ((IOCB *)(((uchar *)((ulong)ring)) + (((uint32)((ulong)slot))<< 5)))

/*
 *    End Structure Definitions for IOCB Commands
 */

typedef struct  {
   MAILBOX mbx;
   IOCB    IOCBs[MAX_BIOCB];
} SLIM;

typedef struct  {
   MAILBOX mbx;
   PCB     pcb;
   IOCB    IOCBs[MAX_SLI2_IOCB];
} SLI2_SLIM;

/*
* FDMI
* HBA MAnagement Operations Command Codes
*/
#define  SLI_MGMT_GRHL     0x100                   /* Get registered HBA list */
#define  SLI_MGMT_GHAT     0x101                   /* Get HBA attributes */
#define  SLI_MGMT_GRPL     0x102                   /* Get registered Port list */
#define  SLI_MGMT_GPAT     0x110                   /* Get Port attributes */
#define  SLI_MGMT_RHBA     0x200                   /* Register HBA */
#define  SLI_MGMT_RHAT     0x201                   /* Register HBA atttributes */
#define  SLI_MGMT_RPRT     0x210                   /* Register Port */
#define  SLI_MGMT_RPA      0x211                   /* Register Port attributes */
#define  SLI_MGMT_DHBA     0x300                   /* De-register HBA */
#define  SLI_MGMT_DPRT     0x310                   /* De-register Port */

/*
 * Management Service Subtypes
 */
#define  SLI_CT_FDMI_Subtypes     0x10

/*
 * HBA Management Service Reject Code
 */
#define  REJECT_CODE             0x9      /* Unable to perform command request */
/*
 * HBA Management Service Reject Reason Code
 * Please refer to the Reason Codes above
 */

/*
 * HBA Attribute Types
 */
#define  NODE_NAME               0x1
#define  MANUFACTURER            0x2
#define  SERIAL_NUMBER           0x3
#define  MODEL                   0x4
#define  MODEL_DESCRIPTION       0x5
#define  HARDWARE_VERSION        0x6
#define  DRIVER_VERSION          0x7
#define  OPTION_ROM_VERSION      0x8
#define  FIRMWARE_VERSION        0x9
#define  VENDOR_SPECIFIC         0xa
#define  DRIVER_NAME             0xb
#define  OS_NAME_VERSION     0xc
#define  MAX_CT_PAYLOAD_LEN  0xd

/*
 * Port Attrubute Types
 */
#define  SUPPORTED_FC4_TYPES     0x1
#define  SUPPORTED_SPEED         0x2
#define  PORT_SPEED              0x3
#define  MAX_FRAME_SIZE          0x4
#define  OS_DEVICE_NAME          0x5

union AttributesDef {
   /* Structure is in Big Endian format */
   struct {
      u32bit AttrType:    16;
      u32bit AttrLen:     16;
   } bits;
   uint32 word;
};

/*
 * HBA Attribute Entry (8 - 260 bytes)
 */
typedef struct 
{
   union AttributesDef  ad;
   union {
      uint32            VendorSpecific;
      uint32            SupportSpeed;
      uint32            PortSpeed;
      uint32            MaxFrameSize;
      uint32        MaxCTPayloadLen;
      uchar             SupportFC4Types[32];
      uchar             OsDeviceName[256];
      uchar             Manufacturer[64];
      uchar             SerialNumber[64];
      uchar             Model[256];
      uchar             ModelDescription[256];
      uchar             HardwareVersion[256];
      uchar             DriverVersion[256];
      uchar             OptionROMVersion[256];
      uchar             FirmwareVersion[256];
      uchar             DriverName[256];      
      NAME_TYPE         NodeName; 
   } un;
} ATTRIBUTE_ENTRY, *PATTRIBUTE_ENTRY;


/*
 * HBA Attribute Block
 */
typedef struct
{
    uint32                EntryCnt;          /* Number of HBA attribute entries */
    ATTRIBUTE_ENTRY       Entry;             /* Variable-length array */
} ATTRIBUTE_BLOCK, *PATTRIBUTE_BLOCK;


/*
 * Port Entry
 */
typedef struct
{
   NAME_TYPE   PortName;
} PORT_ENTRY, *PPORT_ENTRY;

/*
 * HBA Identifier
 */
typedef struct
{
   NAME_TYPE   PortName;
} HBA_IDENTIFIER, *PHBA_IDENTIFIER;

/*
 * Registered Port List Format
 */
typedef struct
{
   uint32       EntryCnt;
   PORT_ENTRY   pe;            /* Variable-length array */
} REG_PORT_LIST, *PREG_PORT_LIST;

/*
 * Register HBA(RHBA)
 */
typedef struct
{
   HBA_IDENTIFIER    hi;
   REG_PORT_LIST     rpl;        /* variable-length array */
} REG_HBA, *PREG_HBA;

/*
 * Register HBA Attributes (RHAT)
 */
typedef struct
{
   NAME_TYPE            HBA_PortName;
   ATTRIBUTE_BLOCK      ab;
} REG_HBA_ATTRIBUTE, *PREG_HBA_ATTRIBUTE;

/*
 * Register Port Attributes (RPA)
 */
typedef struct
{
   NAME_TYPE            HBA_PortName;
   NAME_TYPE            PortName;
   ATTRIBUTE_BLOCK      ab;
} REG_PORT_ATTRIBUTE, *PREG_PORT_ATTRIBUTE;

/*
 * Get Registered HBA List (GRHL) Accept Payload Format
 */
typedef struct
{
   uint32      HBA__Entry_Cnt;        /* Number of Registered HBA Identifiers */
   NAME_TYPE   HBA_PortName;          /* Variable-length array */
} GRHL_ACC_PAYLOAD, *PGRHL_ACC_PAYLOAD;

/*
 * Get Registered Port List (GRPL) Accept Payload Format
 */
typedef struct
{
   uint32           RPL_Entry_Cnt;          /* Number of Registered Port Entries */
   PORT_ENTRY       Reg_Port_Entry[1];      /* Variable-length array */
} GRPL_ACC_PAYLOAD, *PGRPL_ACC_PAYLOAD;

/*
 * Get Port Attributes (GPAT) Accept Payload Format
 */

typedef struct
{
   ATTRIBUTE_BLOCK      pab;
} GPAT_ACC_PAYLOAD, *PGPAT_ACC_PAYLOAD;
#endif  /* _H_FC_HW */
