/*
 * ac.c -- code to handle AC adapter related functions
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * Written by Sos Pter <sp@osb.hu>, 2002-2003
 */


#include <linux/config.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/types.h>
#include <linux/init.h>
#include <linux/proc_fs.h>

#include <asm/system.h>
#include <asm/errno.h>
#include <asm/io.h>
#include <asm/uaccess.h>

#ifdef OMNIBOOK_STANDALONE
#include "omnibook.h"
#else
#include <linux/omnibook.h>
#endif

#include "ec.h"

static struct proc_dir_entry *proc_ac;

int omnibook_get_ac(void)
{
	u8 ac;
	int retval;

	switch (omnibook_ectype) {
	case XE3GF:
		retval = omnibook_ec_read(XE3GF_ADP, &ac);
		if (retval)
			return retval;
		retval = (ac & XE3GF_ADP_MASK) ? 1 : 0;
		break;
	case XE3GC:
		retval = omnibook_ec_read(XE3GC_STA1, &ac);
		if (retval)
			return retval;
		retval = (ac & XE3GC_ADP_MASK) ? 1 : 0;
		break;
	case OB500:
	case OB510:
	case OB6000:
	case OB6100:
	case XE4500:
		retval = omnibook_ec_read(OB500_STA2, &ac);
		if (retval)
			return retval;
		retval = (ac & OB500_ADP_MASK) ? 1 : 0;
		break;
	case OB4150:
		retval = omnibook_ec_read(OB4150_ADP, &ac);
		if (retval)
			return retval;
		retval = (ac & OB4150_ADP_MASK) ? 1 : 0;
		break;
	default:
		printk(KERN_INFO "%s: AC adapter status monitoring is unsupported on this machine.\n", MODULE_NAME);
		retval = -ENODEV;
	}
	return retval;
}

static int omnibook_proc_ac(char *buffer, char **start, off_t off, int count, int *eof, void *data)
{
	int len;
	char *b = buffer;
	int ac;
	char *ac_str;
	
	ac = omnibook_get_ac();
	ac_str = (ac) ? "on-line" : "off-line";
	
	b += sprintf(b, "AC %s\n", ac_str);
	
	len = b - buffer;
	if (len <= off + count)
		*eof = 1;
	*start = buffer + off;
	len -= off;
	if (len > count)
		len = count;
	if (len < 0)
		len = 0;
	return len;
}

int __init omnibook_ac_init(void)
{
	mode_t pmode;

	switch (omnibook_ectype) {
	case XE3GF:
	case XE3GC:
	case OB500:
	case OB510:
	case OB6000:
	case OB6100:
	case XE4500:
	case OB4150:
		pmode = S_IFREG | S_IRUGO;
		proc_ac = create_proc_read_entry("ac", pmode, omnibook_proc_root, omnibook_proc_ac, NULL);
		break;
	default:
		printk(KERN_INFO "%s: AC adapter status monitoring is unsupported on this machine.\n", MODULE_NAME);
		return 0;
	}
	if (! proc_ac) {
		printk(KERN_ERR "%s: Unable to create /proc/%s/ac.\n", MODULE_NAME, MODULE_NAME);
		return -ENOENT;
	}
	printk(KERN_INFO "%s: AC adapter status monitoring is enabled.\n", MODULE_NAME);
	return 0;
}

void __exit omnibook_ac_cleanup(void)
{
	if (proc_ac)
		remove_proc_entry("ac", omnibook_proc_root);
}

EXPORT_SYMBOL(omnibook_get_ac);

/* End of file */
