#define UwsQueue_c


// **********************************************************************
// * uwsqueue.c                                                         *
// *                                                                    *
// * Written by Matthew DeLoera (deloera@us.ibm.com)                    *
// * (C) Copyright IBM Corporation, 1998-2001                           *
// *                                                                    *
// *  This software may be used and distributed according to the terms  *
// *  of the GNU Public License, incorporated herein by reference.      *
// **********************************************************************

// **********************************************************************
// * uwsqueue.c - This source file provides all the abstract queue      *
// *              management routines.                                  *
// *                                                                    *
// * The queue implementation consists of 2 void pointers in the global *
// * driver info struct (EventQueue/CmdQueue). These 2 void pointers are*
// * each the head pointer for one of the 2 driver queues.              *
// *                                                                    *
// * When a queue is not empty, its head pointer will point to a        *
// * PWORK_BLOCK. From this point, one can traverse the queue as a      *
// * linked list of self-referential PWORK_BLOCK structs via the        *
// * pNextBlock/pPrevBlock members (who are also void*).                *
// *                                                                    *
// * If a node is the end node for a queue, the appropriate member will *
// * be NULL.                                                           *
// *                                                                    *
// * The only complication in these routines, is when one of the global *
// * queue head pointers needs to be updated. Rather than having to     *
// * really complicate the parameter passing into these routines, I     *
// * figure out a way to determine whether this affects the event or the*
// * command node, and I explicitly update that global pointer here. It *
// * should still be pretty straightforward.                            *
// *                                                                    *
// * Exported Functions                                                 *
// * ------------------                                                 *
// * sysSpDisplayQueue               - display queue contents           *
// * sysSpDequeue                    - pop top node and return it to    *
// *                                   caller                           *
// * sysSpEnqueue                    - enqueue a new node onto end and  *
// *                                   return a pointer to the head     *
// * sysSpPeekQueue                  - return a pointer to the top node *
// * sysSpPurgeQueueByPID            - specialized purge routines, with *
// *                                   each one walking the list and    *
// *                                   purging each node meeting a      *
// * sysSpPurgeQueueAll                                                 *
// * sysSpPurgeQueueDeregisterEvents                                    *
// * sysSpSeekQueue                  - walks queue looking for a PID,   *
// *                                   & indicates whether it was found *
// *                                                                    *
// * Changes                                                            *
// * -------                                                            *
// * 01/29/2001 - Cleaned up for open source release.                   *
// **********************************************************************


// ************
// * INCLUDES *
// ************

#include "uwscmd.h"


void sysSpDisplayQueue(void *pHead);
PWORK_BLOCK sysSpRemoveFromQueue(void *pHead, PWORK_BLOCK pInputCmdBlock );
PWORK_BLOCK sysSpQueryQueue(void *pHead, PDD_IOCTL_INFO pDDIoctlInfo);           // PHR_174969
PWORK_BLOCK sysSpDequeue(PDRIVER_INFO pDriverNode, void *pHead);
PWORK_BLOCK sysSpEnqueue(PDRIVER_INFO pDriverNode, void *pHead, PWORK_BLOCK pBlock);
PWORK_BLOCK sysSpPeekQueue(void *pHead);
UCHAR sysSpSeekQueue(void *pHead, unsigned int PID);
void sysSpPurgeQueueByPID(PDRIVER_INFO pDriverNode, void *pHead, unsigned int PID);
void sysSpPurgeQueueAll(PDRIVER_INFO pDriverNode, void *pHead);
void sysSpPurgeQueueDeregisterEvents(PDRIVER_INFO pDriverNode, void *pHead, unsigned int PID);
void sysSpPurgeQueue(PDRIVER_INFO pDriverNode, void *pHead, unsigned int PID, int FunctionFlag, int EventQueueFlag, int InterruptionFlag);


// **************************************************************************
// * sysSpDequeue - Pop top node from queue and return it to caller.        *
// *                                                                        *
// * Parameters:                                                            *
// *      pDriverNode - Pointer th the active driver info structure         *
// *      pHead       - Pointer to a queue head                             *
// *                                                                        *
// * Returns:                                                               *
// *      Returns a pointer to the popped node, or NULL if queue was empty. *
// **************************************************************************
PWORK_BLOCK sysSpDequeue(PDRIVER_INFO pDriverNode, void *pHead)
{
     PWORK_BLOCK pCurr;

     pCurr = (PWORK_BLOCK)pHead;

     // IF QUEUE IS CURRENTLY EMPTY, RETURN NULL
     if (!pCurr)
     {
          return pCurr;
     }

     // POP TOP NODE AND UPDATE APPROPRIATE QUEUE HEAD IN GLOBAL DRIVER INFO STRUCT
     //      EVENT QUEUE   : pBlock->completionCode == DDERR_EV_INITIATED
     //      COMMAND QUEUE : pBlock->completionCode == DDERR_CMD_INITIATED  (assumed)
     if (DBG_FUNCENTEXT || DBG_QUEUES)
     {
        printk(KERN_CRIT "ibmasm: DBG_QUEUES - Popping top node and exiting.\n");
     }

     pHead = ((PWORK_BLOCK)pHead)->pNextBlock;

     if (pHead)
     {
         ((PWORK_BLOCK)pHead)->pPrevBlock = NULL;
     }

     if(pCurr->completionCode == DDERR_EV_INITIATED)
     {
       pDriverNode->EventQueue = pHead;
     }
     else if (pCurr->completionCode == DDERR_EV_PENDING)
     {
       pDriverNode->EventQueue = pHead;
     }
     else
     {
       pDriverNode->CmdQueue = pHead;
     }

     // PREPARE POPPED NODE AS AN ISOLATED NODE, AND RETURN IT
     pCurr->pNextBlock = NULL;
     pCurr->pPrevBlock = NULL;
     return pCurr;
}


// ******************************************************************************
// * sysSpEnqueue - Enqueue a node onto the end of a queue. Return a pointer to *
// *                the top node, to indicate whether the list was previously   *
// *                empty. If the list was empty, we return NULL.               *
// *                                                                            *
// * Parameters:                                                                *
// *      pDriverNode - Pointer th the active driver info structure             *
// *      pHead       - Pointer to a queue head                                 *
// *      pBlock      - Pointer to the node that we're enqueuing                *
// *                                                                            *
// * Returns:                                                                   *
// *      Returns a pointer to the head node, or NULL if queue was previously   *
// *      empty.                                                                *
// ******************************************************************************
PWORK_BLOCK sysSpEnqueue(PDRIVER_INFO pDriverNode, void *pHead, PWORK_BLOCK pBlock)
{
     PWORK_BLOCK pCurr     = (PWORK_BLOCK)pHead;
     PWORK_BLOCK pPrevHead = (PWORK_BLOCK)pHead;
     if (DBG_FUNCENTEXT) printk(KERN_CRIT "sysSpEnqueue: Entry.\n");
     
     // TRAVERSE QUEUE TO END
     if (pCurr)
     {
        while (pCurr->pNextBlock)
        {
           pCurr = (PWORK_BLOCK)(pCurr->pNextBlock);
        }
     }

     // APPEND NEW NODE
     pBlock->pNextBlock = NULL;
     pBlock->pPrevBlock = (void*)pCurr;

     if (!pCurr)
     {
         pHead  = (void*)pBlock;
     }
     else
     {
         pCurr->pNextBlock = (void*)pBlock;
     }

     // UPDATE APPROPRIATE QUEUE HEAD IN GLOBAL DRIVER INFO STRUCT
     //      EVENT QUEUE   : pBlock->completionCode == DDERR_EV_INITIATED
     //      COMMAND QUEUE : pBlock->completionCode == DDERR_CMD_INITIATED  (assumed)
     if (pBlock->completionCode == DDERR_EV_INITIATED)
     {
       pDriverNode->EventQueue = pHead;
     }
     else
     {
       pDriverNode->CmdQueue   = pHead;
     }

     // RETURN PREVIOUS QUEUE HEAD
     // printk(KERN_CRIT "sysSpEnqueue: Exit.\n");
     return pPrevHead;
}


// ******************************************************************************
// * sysSpQuereyQueue - Searches a queue for a specific node. Returns a pointer *
// *                    to that node or NULL if it's not found                  *
// *                                                                            *
// * Parameters:                                                                *
// *      pHead        - Pointer to a queue head                                *
// *      pDDIoctlInfo - Pointer to ioctl info                                  *
// *                                                                            *
// * Returns:                                                                   *
// *      Returns a pointer to the node being searched for, or NULL if the node *
// *      was not found.  PHR_174969                                            *
// ******************************************************************************
PWORK_BLOCK sysSpQueryQueue(void *pHead, PDD_IOCTL_INFO pDDIoctlInfo )
{
     PWORK_BLOCK pCurr = (PWORK_BLOCK)pHead;

     // TRAVERSE THE QUEUE UNTIL A MATCH ON DDIoctlInfo's POINTER IS FOUND
     // DIoctlInfo's POINTER IS BEING USED BECAUSE IT IS THE ONLY KNOWN 
     // UNIQUE CONSTANT 
     if (pCurr)
     {
         while ( 1 )
         {
            if ( pCurr->pDDIoctlInfo == pDDIoctlInfo )
            {
               break; 
            } 
            else
            {
                pCurr = (PWORK_BLOCK)(pCurr->pNextBlock);

                if ( !pCurr )
                {  
                   if (DBG_PHR) printk(KERN_CRIT "DBG_PHR: sysSpQueryQueue is returning NULL from within the while \n");
                   break;
                }
            }
         }
     }
     else 
     {
         if (DBG_PHR) printk(KERN_CRIT "DBG_PHR: sysSpQueryQueue is returning NULL \n");
     }

     return pCurr;
}

// ******************************************************************************
// * sysSpEnqueueToHead - Enqueue a node onto the head of a queue. If there is  *
// *                      a command on the queue that is currently busy attach  *
// *                      this command behind the busy command. Return a pointer*
// *                      to the top node of the queue or NULL if it was empty. *
// *                                                                            *
// * Parameters:                                                                *
// *      pHead  - Pointer to a queue head                                      *
// *      pBlock - Pointer to the node that we're enqueuing                     *
// *                                                                            *
// * Returns:                                                                   *
// *      Returns a pointer to the head node, or NULL if queue was previously   *
// *      empty.                                                                *
// ******************************************************************************
PWORK_BLOCK sysSpEnqueueToHead(void *pHead, PWORK_BLOCK pBlock)
{
    PWORK_BLOCK pPrevHead  = (PWORK_BLOCK)pHead;
    PWORK_BLOCK pTempBlock = (PWORK_BLOCK)(pPrevHead->pNextBlock);

    if (pHead)
    {
      if ( pPrevHead->completionCode == DDERR_CMD_PENDING )
      { 
          pTempBlock->pPrevBlock = (void *)pBlock;

          pBlock->pNextBlock = (void *)pTempBlock;
          pBlock->pPrevBlock = pHead;

          pPrevHead->pNextBlock  = (void *)pBlock;
      }
      else
      {
          pBlock->pPrevBlock = NULL;
          pBlock->pNextBlock = pHead;
          pPrevHead->pPrevBlock = (void *)pBlock;
      }

    }
    else
    {
        // APPEND NEW NODE
        pBlock->pPrevBlock = NULL;
        pBlock->pNextBlock = NULL;
        pHead = (void *)pBlock;
    }

    // RETURN PREVIOUS QUEUE HEAD
    return (PWORK_BLOCK)pPrevHead;
}

// **************************************************************************
// * sysSpPeekQueue - Returns a pointer to the top node of the queue. This  *
// *                  routine is basically here only to be cute, so that I  *
// *                  don't have to do any typecasting elsewhere.           *
// *                                                                        *
// * Parameters:                                                            *
// *      pHead - Pointer to a queue head                                   *
// *                                                                        *
// * Returns:                                                               *
// *      Returns a pointer to the top node, or NULL if the queue is empty. *
// **************************************************************************

PWORK_BLOCK sysSpPeekQueue(void *pHead)
{
     if (DBG_FUNCENTEXT) printk(KERN_CRIT "sysSpPeekQueue: Entry and exit.\n");
     return (PWORK_BLOCK)pHead;
}


// ***************************************************************************
// * sysSpSeekQueue - Returns a pointer to the top node of the queue. This   *
// *                  routine is basically here only to be cute, so that I   *
// *                  don't have to do any typecasting elsewhere.            *
// *                                                                         *
// * Parameters:                                                             *
// *      pHead - Pointer to a queue head                                    *
// *                                                                         *
// * Returns:                                                                *
// *      Returns a UCHAR indicating whether the PID was found or not found. *
// ***************************************************************************

UCHAR sysSpSeekQueue(void *pHead, unsigned int PID)
{
     PWORK_BLOCK pCurr = (PWORK_BLOCK)pHead;
     if (DBG_FUNCENTEXT) printk(KERN_CRIT "sysSpSeekQueue: Entry.\n");

     // TRAVERSE QUEUE TO END
     if (pCurr) while (pCurr)
     {
         if (pCurr->PID == PID)
         {
              if (DBG_QUEUES)     printk(KERN_CRIT "sysSpSeekQueue: PID found!\n");
              if (DBG_FUNCENTEXT) printk(KERN_CRIT "sysSpSeekQueue: Exit.\n");
              return 1;
         }
         pCurr = (PWORK_BLOCK)(pCurr->pNextBlock);
     }
     if (DBG_QUEUES)     printk(KERN_CRIT "sysSpSeekQueue: PID not found.\n");
     if (DBG_FUNCENTEXT) printk(KERN_CRIT "sysSpSeekQueue: Exit.\n");
     return 0;
}


// *************************************************************************************
// * sysSpPurgeQueueByPID                                                              *
// *           - Purges the queue of all nodes associated with a given PID             *
// *                                                                                   *
// * sysSpPurgeQueueAll                                                                *
// *           - Purges the queue of all nodes                                         *
// *                                                                                   *
// * sysSpPurgeQueueDeregisterEvents                                                   *
// *           - Same as ByPID, but sets the completion codes to DDERR_EVENT_CANCELLED *
// *             instead of DDERR_DRIVER_CLOSED.                                       *
// *                                                                                   *
// * Parameters:                                                                       *
// *      pHead                - Pointer to a queue head                               *
// *      PID                  - The PID to purge                                      *
// *************************************************************************************

#define QUEUEDEF_BYPID  1
#define QUEUEDEF_EXTERN 2
#define QUEUEDEF_DEREG  4
#define QUEUEDEF_ALL    8

void sysSpPurgeQueueByPID(PDRIVER_INFO pDriverNode, void *pHead, unsigned int PID)
{
     if ( pHead == pDriverNode->EventQueue )
          sysSpPurgeQueue(pDriverNode, pHead, PID, QUEUEDEF_BYPID, 1, 0);
     else
          sysSpPurgeQueue(pDriverNode, pHead, PID, QUEUEDEF_BYPID, 0, 0);
     return;
}

void sysSpPurgeQueueAll(PDRIVER_INFO pDriverNode, void *pHead)
{
     if (pHead == pDriverNode->EventQueue)
         sysSpPurgeQueue(pDriverNode, pHead, 0, QUEUEDEF_ALL, 1, 0);
     else
         sysSpPurgeQueue(pDriverNode, pHead, 0, QUEUEDEF_ALL, 0, 0);
     return;
}

void sysSpPurgeQueueDeregisterEvents(PDRIVER_INFO pDriverNode, void *pHead, unsigned int PID)
{
     if (pHead == pDriverNode->EventQueue)
         sysSpPurgeQueue(pDriverNode, pHead, PID, QUEUEDEF_DEREG, 1, 0);
     else
         sysSpPurgeQueue(pDriverNode, pHead, PID, QUEUEDEF_DEREG, 0, 0);
     return;
}


// *************************************************************************************
// * sysSpPurgeQueue - Purges the queue of all nodes as specified by FunctionFlag. I   *
// *                   *really* had to think about this one for a while. If we're      *
// *                   purging a command queue, and there's a top pending command, we  *
// *                   skip it. So, with a command queue, we'll never be purging the   *
// *                   top node.                                                       *
// *                                                                                   *
// *                   Because, if we released that blocked thread, and simply         *
// *                   discarded the response, we'd still be stuck waiting to send the *
// *                   next command until *after* the discardable response!! So, we    *
// *                   might as well wait.                                             *
// *                                                                                   *
// * Parameters:                                                                       *
// *      pHead                - Pointer to a queue head                               *
// *      PID                  - The PID to purge                                      *
// *      FunctionFlag         - Indicates the context of this purge operation         *
// *************************************************************************************
void sysSpPurgeQueue(PDRIVER_INFO pDriverNode, void *pHead, unsigned int PID, int FunctionFlag, int EventQueueFlag, int InterruptionFlag)
{
     UCHAR       QueueHeadModified = 0;
     PWORK_BLOCK pCurr             = (PWORK_BLOCK)pHead;
     PWORK_BLOCK pCachedHead       = NULL;
     PWORK_BLOCK pExtractedNode    = NULL;
     PWORK_BLOCK pNext             = NULL;
     if (DBG_FUNCENTEXT) printk(KERN_CRIT "sysSpPurgeQueue: Entry.\n");

     // IF QUEUE IS EMPTY, UNLOCK EVENT QUEUE AND EXIT
     if (!pCurr)
     {
          return;
     }

     // WATCH FOR A CURRENTLY PENDING (EXECUTING) TOP COMMAND BLOCK
     if (!(FunctionFlag & QUEUEDEF_DEREG) && (pCurr->completionCode == DDERR_CMD_PENDING))
     {
          pCachedHead = (PWORK_BLOCK)pHead;
          pCurr       = (PWORK_BLOCK)(pCurr->pNextBlock);
          if (DBG_QUEUES) printk(KERN_CRIT "sysSpPurgeQueue: Top command currently executing - skipping to next block.\n");
     }

     // SCAN QUEUE FOR INSTANCES OF THE SPECIFIED PID
     while (pCurr)
     {
          // DETERMINE WHETHER TO EXTRACT THIS NODE
          if ( (FunctionFlag & QUEUEDEF_ALL)                                               ||
               ((FunctionFlag & QUEUEDEF_BYPID)  && (pCurr->PID == PID))                   ||
               ((FunctionFlag & QUEUEDEF_EXTERN) &&
                     (pCurr->PID != PID_SEND_DRVR_VER) && (pCurr->PID != PID_OS_IS_UP) &&
                     (pCurr->PID != PID_STOP_DRIVER)   && (pCurr->PID != PID_HEART_BEAT))  ||
               ((FunctionFlag & QUEUEDEF_DEREG)  && (pCurr->PID == PID))
             )
          {
               // EXTRACT NODE FROM QUEUE
               pExtractedNode = pCurr;
               if (pExtractedNode->pPrevBlock)
               {
                    // THIS IS *NOT* THE HEAD NODE. EXTRACT AWAY!
                    if (DBG_QUEUES) printk(KERN_CRIT "sysSpPurgeQueue: Extracting a non-head node.\n");
                    pCurr = (PWORK_BLOCK)(pExtractedNode->pPrevBlock);
                    pNext = (PWORK_BLOCK)(pExtractedNode->pNextBlock);
                    if (pNext) pCurr->pNextBlock = (void*)pNext;
                    else       pCurr->pNextBlock = NULL;
                    if (pNext) pNext->pPrevBlock = (void*)pCurr;
               }
               else if (EventQueueFlag)
               {
                    // THIS *IS* THE HEAD NODE. BE CAREFUL!
                    if (DBG_QUEUES) printk(KERN_CRIT "sysSpPurgeQueue: Extracting the *head* node of an event queue.\n");
                    pCurr = (PWORK_BLOCK)(pExtractedNode->pNextBlock);
                    if (pCurr) pCurr->pPrevBlock = NULL;
         
                    // UPDATE APPROPRIATE QUEUE HEAD IN GLOBAL DRIVER INFO STRUCT
                    QueueHeadModified = 1;
                    pDriverNode->EventQueue = pCurr;
               }
               else
               {
                    // RARE CASE THAT WE'D BE TRYING TO POP THE TOP OF THE COMMAND QUEUE. I THINK THE ONLY
                    // WAY THIS WOULD OCCUR IS IF THE TOP COMMAND COMPLETED IN THE SPACE OF TIME IN WHICH
                    // WE'VE ACTIVELY BEEN PURGING. SO, WE SIMPLY JUMP AHEAD TO THE NEXT BLOCK, IN ORDER TO
                    // MAINTAIN A STATE WHERE WE'VE SKIPPED THE TOP COMMAND BLOCK....
                    if (DBG_QUEUES) printk(KERN_CRIT "sysSpPurgeQueue: Rare case - popping top of cmd queue.\n");
                    pCurr = (PWORK_BLOCK)(pCurr->pNextBlock);
                    continue;
               }

               // PREPARE EXTRACTED NODE AS AN ISOLATED NODE AND SET ITS COMPLETION CODE
               pExtractedNode->pPrevBlock = NULL;
               pExtractedNode->pNextBlock = NULL;
               if (InterruptionFlag && (FunctionFlag & QUEUEDEF_BYPID))
                                                       pExtractedNode->completionCode = DDERR_INTERRUPTED;
               else if (FunctionFlag & QUEUEDEF_DEREG) pExtractedNode->completionCode = DDERR_EVENT_CANCELLED;
               else                                    pExtractedNode->completionCode = DDERR_DRIVER_CLOSED;

               // RELEASE THE BLOCKED PROCESS ASSOCIATED WITH THE EXTRACTED NODE
               wake_up_interruptible(&pExtractedNode->pBlock);
               if (DBG_BLOCKS) printk(KERN_CRIT "sysSpPurgeQueue: Unblocking the associated thread of this popped node.\n");
          }
          else
          {
               if (DBG_QUEUES)
               {
                    if ((FunctionFlag & QUEUEDEF_BYPID)  && (pCurr->PID != PID))
                         printk(KERN_CRIT "sysSpPurgeQueue: Skipping node - Purging by PID, but a PID mismatch.\n");
                    else if ((FunctionFlag & QUEUEDEF_EXTERN) && (pCurr->PID != PID_SEND_DRVR_VER))
                         printk(KERN_CRIT "sysSpPurgeQueue: Skipping node - Purging all external blocks, but this is internal.\n");
                    else if ((FunctionFlag & QUEUEDEF_EXTERN) && (pCurr->PID != PID_OS_IS_UP))
                         printk(KERN_CRIT "sysSpPurgeQueue: Skipping node - Purging all external blocks, but this is internal.\n");
                    else if ((FunctionFlag & QUEUEDEF_EXTERN) && (pCurr->PID != PID_STOP_DRIVER))
                         printk(KERN_CRIT "sysSpPurgeQueue: Skipping node - Purging all external blocks, but this is internal.\n");
                    else if ((FunctionFlag & QUEUEDEF_EXTERN) && (pCurr->PID != PID_HEART_BEAT))
                         printk(KERN_CRIT "sysSpPurgeQueue: Skipping node - Purging all external blocks, but this is internal.\n");
                    else if ((FunctionFlag & QUEUEDEF_DEREG)  && (pCurr->PID != PID))
                         printk(KERN_CRIT "sysSpPurgeQueue: Skipping node - Purging all events for a particular PID, but a PID mismatch.\n");
                    else printk(KERN_CRIT "sysSpPurgeQueue: Skipping node - Unknown cause.\n");
               }
          }

          // ADVANCE TO NEXT NODE IN QUEUE (UNLESS THE QUEUE HEAD WAS MODIFIED)
          if (QueueHeadModified) QueueHeadModified = 0;
          else pCurr = (PWORK_BLOCK)(pCurr->pNextBlock);
     }
     if (DBG_FUNCENTEXT) printk(KERN_CRIT "sysSpPurgeQueue: Exit.\n");
     return;
}



PWORK_BLOCK sysSpRemoveFromQueue(void *pHead, PWORK_BLOCK pInputCmdBlock )
{
   PWORK_BLOCK pTempBlock;
   PWORK_BLOCK pPrevBlock;
   PWORK_BLOCK pNextBlock;

   pTempBlock = pHead;


   if (DBG_PHR) printk(KERN_CRIT "sysSpRemoveFromQueue: Searching the command queue.\n");

   while( (pTempBlock != pInputCmdBlock) && (pTempBlock != NULL) )
   { 
       pTempBlock = pTempBlock->pNextBlock;
   }
   
   if ( pTempBlock )
   {
       if (DBG_PHR) printk(KERN_CRIT "sysSpRemoveFromQueue: Command found and removing from queue.\n");

       pPrevBlock = pTempBlock->pPrevBlock;
       pNextBlock = pTempBlock->pNextBlock;

       if (pPrevBlock)
       {
           pPrevBlock->pNextBlock = pTempBlock->pNextBlock;
       }

       if (pNextBlock)
       {
           pNextBlock->pPrevBlock = pTempBlock->pPrevBlock;
       }
   }
   else
   {
       if (DBG_PHR) printk(KERN_CRIT "sysSpRemoveFromQueue: Command not found and cannot be removed from queue.\n");
   }
      
   return pTempBlock;
}

#undef UwsQueue_c


