/* ----------------------------------------------------------------------- *
 *   
 *   Copyright 2003 Andrew de Quincey - All Rights Reserved
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, Inc., 675 Mass Ave, Cambridge MA 02139,
 *   USA; either version 2 of the License, or (at your option) any later
 *   version; incorporated herein by reference.
 *
 * ----------------------------------------------------------------------- */

#include <linux/mm.h>
#include <linux/irq.h>
#include <linux/init.h>
#include <linux/acpi.h>
#include <linux/delay.h>
#include <linux/config.h>
#include <linux/bootmem.h>
#include <linux/smp_lock.h>
#include <linux/kernel_stat.h>

#include <asm/acpi.h>

#ifdef CONFIG_ACPI_PCI

extern void eisa_set_level_irq(unsigned int irq);

int __init pic_parse_prt (void)
{
	struct list_head	*node = NULL;
	struct acpi_prt_entry	*entry = NULL;
	struct acpi_prt_list	*prt_list = NULL;
	int			edge_level = 0;
	int			active_high_low = 0;
	int			irq = 0;
	int			programmed[16];

	/* Get the current PRT */
	prt_list = acpi_pci_get_prt_list();
   
	if (!prt_list->count) {
		acpi_pci_destroy_prt_list(prt_list);
		printk(KERN_WARNING "ACPI tables contain no PIC PCI IRQ "
			"routing entries\n");
		return_VALUE(-ENODEV);
	}

	/* mark all IRQs as unprogrammed */
	memset(programmed, 0, sizeof(programmed));
   
	/*
	 * Parsing through the PCI Interrupt Routing Table (PRT) and program
	 * IRQs if necessary.
	 */
	list_for_each(node, &prt_list->entries) {
		entry = list_entry(node, struct acpi_prt_entry, node);

		/* Need to get irq for dynamic entry */
		if (entry->link.handle) {
			irq = acpi_pci_link_get_irq(entry->link.handle, entry->link.index, &edge_level, &active_high_low);
			if (irq < 0) {
				acpi_pci_destroy_prt_list(prt_list);
				return -ENODEV;
			}
 			if (!irq) 
				continue;
		}
	   
		/* sanity check + update entry */
		if ((irq < 0) || (irq > 15)) {
			printk(KERN_ERR "Invalid IRQ (%i) passed to PIC programming code\n", irq);
			entry->irq = 0;
			continue;
		}
		entry->irq = irq;

		/* check if it has already been dealt with */
		if (programmed[irq]) {
			printk(KERN_DEBUG "PIC: IRQ (%i) already programmed\n", irq);
			continue;
		}
		programmed[irq] = 1;

		/* program it */
		if (edge_level) {
			eisa_set_level_irq(irq);
		}
	   
		printk(KERN_DEBUG "%02x:%02x:%02x[%c] -> IRQ %d Mode %d Trigger %d\n",
			entry->id.segment, entry->id.bus, 
			entry->id.device, ('A' + entry->pin), 
			entry->irq, edge_level, active_high_low);
	}
	
	/* if we get here, the PRT was fine. commit it */
	acpi_pci_commit_prt_list(prt_list);
   
	return 0;
}

#endif /*CONFIG_ACPI_PCI*/
