/* 
 * lib.c
 * Copyright (C) 2002, AVM GmbH. All rights reserved.
 * 
 * This Software is  free software. You can redistribute and/or
 * modify such free software under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * The free software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this Software; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA, or see
 * http://www.opensource.org/licenses/lgpl-license.html
 * 
 * Contact: AVM GmbH, Alt-Moabit 95, 10559 Berlin, Germany, email: info@avm.de
 */

#include <asm/param.h>
#include <asm/io.h>
#include <linux/mm.h>
#include <linux/kernel.h>
#include <linux/wait.h>
#include <linux/sched.h>
#include <stdarg.h>
#include "main.h"
#include "driver.h"
#include "devif.h"
#include "queue.h" 
#include "defs.h"
#include "tools.h"
#include "libstub.h"
#include "lib.h"

#define	PRINTF_BUFFER_SIZE	256	/* see: os_debug_printf */
#define	CA_PRINTF_BUFFER_SIZE	1024	/* see: os_printf */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static unsigned long long	j64 = 0;
static unsigned long		j32 = 0;

static DECLARE_WAIT_QUEUE_HEAD(delay);

#define	TEN_MSECS		(HZ/100)
#define JIFF2MSEC		(1000/HZ)

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void os__enter_critical (const char * f, int l) { 

	f = (void *) l;
	enter_critical (); 
} /* os__enter_critical */

static void os__leave_critical (const char * f, int l) {

	f = (void *) l;
	leave_critical ();
} /* os__leave_critical */

static void os_enter_critical (void) { enter_critical (); }

static void os_leave_critical (void) { leave_critical (); } 

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void os_enter_cache_sensitive_code (void) { /* NOP */ }

static void os_leave_cache_sensitive_code (void) { /* NOP */ }

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void os_init (unsigned len, register_t reg, release_t rel, down_t down) {

	init (len, reg, rel, down);
} /* os_init */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static char * os_params (void) {
	static char	parmbuf[128];
	size_t		parmlen = 0;
	size_t		n;
	int		i	= 0;
	
	static struct {
	  char *	name;
	  short *	var;
	} 		X[] = {
		{ "ATMVCC",	&VCC },
		{ "ATMVPI",	&VPI },
		{ "ATMVCI",	&VCI },
		{ NULL,		NULL }
	} ;

#define	GUARD(n)	if ((parmlen + (n)) >= sizeof (parmbuf)) {	\
				log ("Parameter buffer overflow!\n");	\
				return NULL;				\
			}
	
	lprintf (
		KERN_INFO, 
		"Using VCC/VPI/VCI = 0x%x/0x%x/0x%x\n", 
		VCC, 
		VPI, 
		VCI
	);
	while (X[i].name != NULL) {
		GUARD(1);
		parmbuf[parmlen++] = ':';
		n = strlen (X[i].name);
		GUARD(n + 1)
		memcpy (&parmbuf[parmlen], X[i].name, n);
		parmlen += n;
		parmbuf[parmlen++] = (char) 0;
		GUARD (7);
		n = sprintf (&parmbuf[parmlen], "\\x%04x", *X[i].var);
		assert (n == 6);
		parmlen += n;
		parmbuf[parmlen++] = (char) 0;
		++i;
	}
	GUARD (1);
	parmbuf[parmlen++] = (char) 0;
	
#undef GUARD

	return parmbuf;
} /* os_params */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int os_get_message (unsigned char * msg) {
	int res;

	assert (msg);
	res = msg2stack (msg); 
	return res;
} /* os_get_message */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void os_put_message (unsigned char * msg) {

	assert (msg);
	msg2capi (msg);
} /* os_put_message */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static unsigned char * os_get_data_block (
	unsigned	appl, 
	unsigned long	ncci, 
	unsigned	index
) {
	char * res;

	res = data_block (appl, ncci, index);
	return res;
} /* os_get_data_block */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void os_free_data_block (unsigned appl, unsigned char * data) {

	data = (void *) appl;	/* Avoid "unused arg" warning */
} /* os_free_data_block */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int os_new_ncci (
	unsigned	appl,
	unsigned long	ncci, 
	unsigned	win_size,
	unsigned	blk_size
) {
	
	new_ncci (appl, ncci, win_size, blk_size); 
	return 1;
} /* os_new_ncci */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void os_free_ncci (unsigned appl, unsigned long ncci) {

	free_ncci (appl, ncci);
} /* os_free_ncci */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static unsigned os_block_size (unsigned appl) {
	unsigned bs, dummy;

	appl_profile (appl, &bs, &dummy);
	return bs;
} /* os_block_size */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static unsigned os_window_size (unsigned appl) {
	unsigned bc, dummy;

	appl_profile (appl, &dummy, &bc);
	return bc;
} /* os_window_size */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static unsigned os_card (void) {

	return 0;
} /* os_card */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void * os_appl_data (unsigned appl) {
	void * res;

	res = data_by_id (appl);
	return res;
} /* os_appl_data */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static unsigned os_appl_attr (unsigned appl) {
	unsigned res;

	res = attr_by_id (appl);
	return res;
} /* os_appl_attr */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static appldata_t * os_appl_1st_data (appldata_t * s) {
	int		num;
	void *		data;
	static char	e[10]; 

	memset (&e, 0, 10);
	data = first_data (&num);
	s->num    = num;
	s->buffer = (NULL == data) ? e : data;
	return s;
} /* os_appl_1st_data */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static appldata_t * os_appl_next_data (appldata_t * s) {
	int		num;
	void *		data;
	static char	e[10]; 

	memset (&e, 0, 10);
	if ((num = s->num) < 0) {	
		return NULL;
	}
	data = next_data (&num);
	s->num    = num;
	s->buffer = (NULL == data) ? e : data;
	return s;
} /* os_appl_next_data */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void * os_malloc (unsigned len) {
	void * res;

	res = hcalloc (len); 
	return res;
} /* os_malloc */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void * os_malloc2 (unsigned len) {
	void * res;

	assert (0);	/* Not used */
	res = hcalloc_kernel (len); 
	return res;
} /* os_malloc2 */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void os_free (void * p) {

	assert (p != NULL);
	hfree (p);
} /* os_free */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void update_j64 (void) {
	static int	run1 = 1;
	unsigned long	j;
	long long	diff;

	j = jiffies;
	if (run1) {
		j32 = j;
		run1 = 0;
	}
	if ((diff = j - j32) < 0) {
		diff = -diff;
	}
	assert (diff >= 0);
	j32 = j;
	j64 += diff;
} /* update_j64 */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static unsigned long long os_msec64 (void) {
	unsigned long	tmp;

	update_j64 ();
	tmp = j64 * JIFF2MSEC;
	return tmp;
} /* os_msec64 */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static unsigned long os_msec (void) {
	unsigned long   tmp;

	update_j64 ();
	tmp = ((unsigned long) j64) * JIFF2MSEC;
	return tmp;
} /* os_msec */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void os_delay (unsigned msec) {

	assert (!in_interrupt ());
	info (msec > 9);
	msec = (msec > 9) ? (msec / 10) * TEN_MSECS : TEN_MSECS;
	interruptible_sleep_on_timeout (&delay, msec);
} /* os_delay */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
typedef struct {

	unsigned long	lstart;
	unsigned long	start;
	unsigned long	tics;
	unsigned long	arg;
	timerfunc_t	func;
} timer_rec_t;

static volatile timer_rec_t *	timer = 0;
static unsigned			timer_count = 0;

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int os_timer_new (unsigned ntimers) {
	unsigned size;

	assert (ntimers > 0); 
	timer_count = ntimers;
	size = sizeof (timer_rec_t) * ntimers;
	timer = (timer_rec_t *) hcalloc (size);
	info (timer != NULL);
	if (NULL == timer) {
		timer_count = 0;
	} 
	return timer == NULL;
} /* os_timer_new */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void os_timer_delete (void) {

	
	hfree ((void *) timer);
	timer = NULL;
	timer_count = 0;
} /* os_timer_delete */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int os_timer_start (
	unsigned	index,
	unsigned long	timeout,
	unsigned long	arg,
	timerfunc_t	func
) {
	assert (index < timer_count);
	if (index >= timer_count) {
		return 1;
	}
	enter_critical ();
	timer[index].start = os_msec ();
	timer[index].tics  = timeout + 1;
	timer[index].arg   = arg;
	timer[index].func  = func;
	leave_critical ();
	return 0;
} /* os_timer_start */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int os_timer_stop (unsigned index) {

	assert (index < timer_count);
	if (index >= timer_count) {
		return 1;
	}
	enter_critical ();
	timer[index].func = NULL;
	leave_critical ();
	return 0;
} /* os_timer_stop */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
void os_timer_poll (void) {
	unsigned  	i;
	unsigned long	msec;
	restart_t 	flag;

	if (NULL == timer) {
		return;
	}
	enter_critical ();
	msec = os_msec ();
	for (i = 0; i < timer_count; i++) {
		if (timer[i].func != 0) {
			if ((msec - timer[i].start) >= timer[i].tics) {
				leave_critical ();
				assert (timer[i].func != NULL);
				flag = (*timer[i].func) (timer[i].arg);
				enter_critical ();
				switch (flag) {
				
				case timer_end:
					timer[i].func = NULL;
					break;
				case timer_restart: 
					assert (timer[i].func != NULL);
					timer[i].start = msec;
					break;
				case timer_renew:
					assert (timer[i].func != NULL);
					break;
				default:
					assert (0);
					break;
				}
			}
		}
	}
	leave_critical ();
} /* os_timer_poll */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int os_gettimeofday (struct timeval * tv) {

	if (NULL != tv) {
		do_gettimeofday (tv);
	}
	return 0;
} /* os_gettimeofday */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static int	nl_needed = 0;

static void os_printf (char * s, va_list args) {
#if !defined (NDEBUG)
	char	buffer[CA_PRINTF_BUFFER_SIZE];
	char *	bufptr = buffer;
	int	count;

	if (nl_needed) {
		nl_needed = 0;
		printk ("\n");
	}	
	count = VSNPRINTF (bufptr, sizeof (buffer), s, args);
	if ('\n' == buffer[0]) {
		bufptr++;
	}
	if ('\n' != buffer[count - 1]) {
		assert (count < (int) (sizeof (buffer) - 2));
		buffer[count++] = '\n';
		buffer[count]   = (char) 0;
	}
	printk (KERN_INFO TARGET ": %s", bufptr);
#else
	s = s;
	args = args;
#endif
} /* os_printf */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void os_puts (char * str, ...) { 
	va_list dummy;

	va_start (dummy, str);
	os_printf (str, dummy);
	va_end (dummy);
} /* os_puts */
 
static void os_putl (long l) { 

	nl_needed = 1; 
	lprintf (KERN_INFO, "%ld", l); 
}  /* os_putl */

static void os_puti (int i) { 

	nl_needed = 1; 
	lprintf (KERN_INFO, "%d", i); 
} /* os_puti */

static void os_putnl (void) { 

	nl_needed = 0; 
	lprintf (KERN_INFO, "\n"); 
}  /* os_putnl */

static void os_putc (char c) {
	char buffer[10];
    
	nl_needed = 1;
	if ((31 < c) && (c < 127)) {
	        sprintf (buffer, "'%c' (0x%02x)", c, (unsigned char) c);
	} else {
		sprintf (buffer, "0x%02x", (unsigned char) c);
	}
	lprintf (KERN_INFO, "%s", buffer);
} /* os_putc */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static void os_debug_printf (const char * fmt, va_list args) {
	char	buffer [PRINTF_BUFFER_SIZE];

	VSNPRINTF (buffer, sizeof (buffer), fmt, args);
	printk (KERN_INFO TARGET ": %s", buffer);
} /* os_debug_printf */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/
static lib_callback_t *	lib	= NULL;
static lib_interface_t	libif	= {

	init:				&os_init,
	params:				&os_params,
	get_message:			&os_get_message,
	put_message:			&os_put_message,
	get_data_block:			&os_get_data_block,
	free_data_block:		&os_free_data_block,
	new_ncci:			&os_new_ncci,
	free_ncci:			&os_free_ncci,
	block_size:			&os_block_size,
	window_size:			&os_window_size,
	card:				&os_card,
	appl_data:			&os_appl_data,
	appl_attr:			&os_appl_attr,
	appl_1st_data:			&os_appl_1st_data,
	appl_next_data:			&os_appl_next_data,
	malloc:				&os_malloc,
	malloc2:			&os_malloc2,
	free:				&os_free,
	msec:				&os_msec,	
	msec64:				&os_msec64,
	delay:				&os_delay,
	timer_new:			&os_timer_new,
	timer_delete:			&os_timer_delete,
	timer_start:			&os_timer_start,
	timer_stop:			&os_timer_stop,
	timer_poll:			&os_timer_poll,
	get_time:			&os_gettimeofday,
	printf:				&os_debug_printf,
	putf:				&os_printf,
	puts:				(void (*) (char *)) &os_puts,
	putl:				&os_putl,
	puti:				&os_puti,
	putc:				&os_putc,
	putnl:				&os_putnl,
	_enter_critical:		&os__enter_critical,
	_leave_critical:		&os__leave_critical,
	enter_critical:			&os_enter_critical,
	leave_critical:			&os_leave_critical,
	enter_cache_sensitive_code:	&os_enter_cache_sensitive_code,	
	leave_cache_sensitive_code:	&os_leave_cache_sensitive_code,

	xfer_req:			&dif_xfer_requirements,
					
	name:				TARGET,
	udata:				0,
	pdata:				NULL
} ;

lib_callback_t * get_library (void) {

	return lib;
} /* get_library */

lib_callback_t * link_library (void * context) {

	return (lib = avm_lib_attach (&libif, context));
} /* link_library */

void free_library (void) {

	if (lib != NULL) {
		lib = 0;
		avm_lib_detach (&libif);
	}
} /* free_library */

/*---------------------------------------------------------------------------*\
\*---------------------------------------------------------------------------*/

