/*************************************************** */
/* Rule Set Based Access Control                     */
/* Implementation of the Access Control Decision     */
/* Facility (ADF) - JAIL module                      */
/* File: rsbac/adf/jail/syscalls.c                   */
/*                                                   */
/* Author and (c) 1999-2002: Amon Ott <ao@rsbac.org> */
/*                                                   */
/* Last modified: 24/Jun/2002                        */
/*************************************************** */

#include <linux/string.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include <rsbac/types.h>
#include <rsbac/aci.h>
#include <rsbac/error.h>
#include <rsbac/debug.h>
#include <rsbac/helpers.h>
#include <rsbac/network.h>
#include <asm/uaccess.h>

/************************************************* */
/*           Global Variables                      */
/************************************************* */

static rsbac_jail_id_t next_id = 1;

/************************************************* */
/*          Internal Help functions                */
/************************************************* */

/************************************************* */
/*          Externally visible functions           */
/************************************************* */

extern long sys_chroot(const char * filename);
extern long sys_chdir(const char * filename);

/* Create a jail for current process */
/* Note: It is allowed to create jails within jails, but with restrictions */
int rsbac_jail_sys_jail(rsbac_version_t version,
                        char * path,
                        rsbac_jail_ip_t ip,
                        rsbac_jail_flags_t flags)
  {
    union rsbac_target_id_t i_tid;
    union rsbac_attribute_value_t i_attr_val1;
    int err;
    int chk_addr_ret;
    mm_segment_t                    oldfs;

    if(version != RSBAC_JAIL_VERSION)
      return -RSBAC_EINVALIDVERSION;
    chk_addr_ret = inet_addr_type(ip);
    if (ip != INADDR_ANY &&
        chk_addr_ret != RTN_LOCAL &&
        chk_addr_ret != RTN_MULTICAST &&
        chk_addr_ret != RTN_BROADCAST)
      return -EADDRNOTAVAIL;

    err = rsbac_get_owner(&i_tid.user);
    if(err)
      return err;
    if (rsbac_get_attr(JAIL,
                       T_USER,
                       i_tid,
                       A_jail_role,
                       &i_attr_val1,
                       TRUE))
      {
        printk(KERN_WARNING
               "rsbac_jail_sys_jail(): rsbac_get_attr() returned error!\n");
               return(-RSBAC_EREADFAILED);
      }
    /* if user role, then deny */
    if (i_attr_val1.system_role == SR_user)
      return(-EPERM);

    /* Get jail_id for this process */
    i_tid.process = current->pid;
    if (rsbac_get_attr(JAIL,
                       T_PROCESS,
                       i_tid,
                       A_jail_id,
                       &i_attr_val1,
                       TRUE))
      {
        printk(KERN_WARNING "rsbac_jail_sys_jail(): rsbac_get_attr() returned error!\n");
        return(-RSBAC_EREADFAILED);
      }
    if(i_attr_val1.jail_id)
      { /* this process is already in a jail -> limit ip and flags */
        if (rsbac_get_attr(JAIL,
                           T_PROCESS,
                           i_tid,
                           A_jail_flags,
                           &i_attr_val1,
                           TRUE))
          {
            printk(KERN_WARNING "rsbac_jail_sys_jail(): rsbac_get_attr() returned error!\n");
            return(-RSBAC_EREADFAILED);
          }
        flags &= i_attr_val1.jail_flags;
        if (rsbac_get_attr(JAIL,
                           T_PROCESS,
                           i_tid,
                           A_jail_ip,
                           &i_attr_val1,
                           TRUE))
          {
            printk(KERN_WARNING "rsbac_jail_sys_jail(): rsbac_get_attr() returned error!\n");
            return(-RSBAC_EREADFAILED);
          }
        if(i_attr_val1.jail_ip)
          ip = i_attr_val1.jail_ip;
      }

    err = sys_chroot(path);
    if(err)
      return err;

    /* Set current user space to kernel space, because sys_chdir() takes name */
    /* from user space */
    oldfs = get_fs();
    set_fs(KERNEL_DS);
    err = sys_chdir("/");
    /* Set current user space back to user space */
    set_fs(oldfs);

    /* Set jail_id for this process */
    i_attr_val1.jail_id = next_id++;
    if (rsbac_set_attr(JAIL,
                       T_PROCESS,
                       i_tid,
                       A_jail_id,
                       i_attr_val1))
      {
        printk(KERN_WARNING "rsbac_jail_sys_jail(): rsbac_set_attr() returned error!\n");
        return(-RSBAC_EWRITEFAILED);
      }
    /* Set jail_ip for this process */
    i_attr_val1.jail_ip = ip;
    if (rsbac_set_attr(JAIL,
                       T_PROCESS,
                       i_tid,
                       A_jail_ip,
                       i_attr_val1))
      {
        printk(KERN_WARNING "rsbac_jail_sys_jail(): rsbac_set_attr() returned error!\n");
        return(-RSBAC_EWRITEFAILED);
      }
    /* Set jail_flags for this process */
    i_attr_val1.jail_flags = flags;
    if (rsbac_set_attr(JAIL,
                       T_PROCESS,
                       i_tid,
                       A_jail_flags,
                       i_attr_val1))
      {
        printk(KERN_WARNING "rsbac_jail_sys_jail(): rsbac_set_attr() returned error!\n");
        return(-RSBAC_EWRITEFAILED);
      }
    return err;
  }

/* end of rsbac/adf/jail/syscalls.c */
