
/***************************************************************************
*
*   Copyright (c) 1998, 1999 Jeff V. Merkey
*   895 West Center Street
*   Orem, Utah  84057
*   jmerkey@utah-nac.org
*
*   This program is free software; you can redistribute it and/or modify it
*   under the terms of the GNU General Public License as published by the
*   Free Software Foundation, version 2, or any later version.
*
*   This program is distributed in the hope that it will be useful, but
*   WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
*   General Public License for more details.
*
*   You are free to modify and re-distribute this program in accordance
*   with the terms specified in the GNU Public License.  The copyright
*   contained in this code is required to be present in any derivative
*   works and you are required to provide the source code for this
*   program as part of any commercial or non-commercial distribution.
*   You are required to respect the rights of the Copyright holders
*   named within this code.
*
*   jmerkey@utah-nac.org is the official maintainer of
*   this code.  You are encouraged to report any bugs, problems, fixes,
*   suggestions, and comments about this software to jmerkey@utah-nac.org
*   or linux-kernel@vger.kernel.org.  New releases, patches, bug fixes, and
*   technical documentation can be found at www.kernel.org.  We will
*   periodically post new releases of this software to www.kernel.org
*   that contain bug fixes and enhanced capabilities.
*
*   Original Authorship      :
*      source code written by Jeff V. Merkey
*
*   Original Contributors    :
*      Jeff V. Merkey
*      Darren Major
*      
*
****************************************************************************
*
*
*   AUTHOR   :  Jeff V. Merkey (jmerkey@utah-nac.org)
*   FILE     :  DISK.C
*   DESCRIP  :  NWFS PC Platform Disk Support
*   DATE     :  November 1, 1998
*
*
***************************************************************************/

#include "globals.h"


ULONG TotalDisks = 0;
ULONG TotalNWParts = 0;
ULONG MaximumDisks = 0;
ULONG MaximumNWParts = 0;
ULONG FirstValidDisk = 0;


#if	(MULTI_MACHINE_EMULATION)
#define	TotalDisks 	machine->TotalDisks
#define	TotalNWParts	machine->TotalNWParts
#define	MaximumDisks	machine->MaximumDisks
#define	MaximumNWParts	machine->MaximumNWParts
#define	FirstValidDisk	machine->FirstValidDisk
uint8	disk_file[] = "./m2machXX/m2diskXX.dat";
BYTE NwPartSignature[16] = { 0, 'N', 'w', '_', 'P', 'a', 'R', 't', 'I',
				 't', 'I', 'o', 'N', 0, 0, 0 };
#else

extern NWDISK	*SystemDisk[MAX_DISKS];
uint8	disk_file[] = "m2diskXX.dat";

#endif

//
//   Linux Block Device Assignments
//
//   Linux Defines 8 IDE + 2 ESDI + 16 SCSI = 26 disks total
//
//   block device 3  (ATA and IDE Disks and CD-ROMs, 1st IDE Adapter)
//             /dev/hda   3:0
//             /dev/hdb   3:64
//
//   block device 8 (SCSI Disks, 1st SCSI Adapter)
//            /dev/sda    8:0
//            /dev/sdb    8:16
//            /dev/sdc    8:32
//            /dev/sdd    8:48
//            /dev/sde    8:64
//            /dev/sdf    8:80
//            /dev/sdg    8:96
//            /dev/sdh    8:112
//
//            /dev/sdi    8:128
//            /dev/sdj    8:144
//            /dev/sdk    8:160
//            /dev/sdl    8:176
//            /dev/sdm    8:192
//            /dev/sdn    8:208
//            /dev/sdo    8:224
//            /dev/sdp    8:240
//
//   block device 22  (ATA and IDE Disks and CD-ROMs, 2nd IDE Adapter)
//             /dev/hdc   22:0
//             /dev/hdd   22:64
//
//   block device 33  (ATA and IDE Disks and CD-ROMs, 3rd IDE Adapter)
//             /dev/hde   33:0
//             /dev/hdf   33:64
//
//   block device 34  (ATA and IDE Disks and CD-ROMs, 4th IDE Adapter)
//             /dev/hdg   34:0
//             /dev/hdh   34:64
//
//   block device 36  (MCA ESDI Disks)
//             /dev/eda   36:0
//             /dev/edb   36:64
//
//   Partition Types
//
//       0x01  FAT 12 Partition
//       0x02  Xenix Root Partition
//       0x03  Xenix Usr Partition
//       0x04  FAT 16 Partition < 32MB
//       0x05  Extended DOS Partition
//       0x06  FAT 16 Partition >= 32MB
//       0x07  IFS (HPFS) Partition
//       0x08  AIX Partition
//       0x09  AIX Boot Partition
//       0x0A  OS/2 Boot Manager
//       0x0B  FAT 32 Partition
//       0x0C  FAT 32 Partition (Int 13 Extensions)
//       0x0E  FAT 16 Partition (Win95 Int 13 Extensions)
//       0x0F  Extended DOS Partition (Int 13 Extensions)
//       0x40  Venix 286 Partition
//       0x41  Power PC (PReP) Partition
//       0x51  Novell [???]
//       0x52  Microport/CPM Partition
//       0x57  VNDI Partition
//       0x63  Unix Partition (GNU Hurd)
//       0x64  Netware 286 Partition
//       0x65  Netware 386 Partition
//       0x66  Netware SMS Partition
//       0x69  Netware NSS Partition
//       0x75  PC/IX
//       0x77  VNDI Partition
//       0x80  MINIX Partition
//       0x81  Linux/MINIX Partition
//       0x82  Linux SWAP Partition
//       0x83  Linux Native Partition
//       0x93  Amoeba Partition
//       0x94  Amoeba Bad Block Table Partition
//       0xA5  BSD/386 Partition
//       0xB7  BSDI File System Partition
//       0xB8  BSDI Swap Partition
//       0xC0  NTFT Partition
//       0xC7  Syrinx Partition
//       0xDB  CP/M Partition
//       0xE1  DOS access Parititon
//       0xE3  DOS R/O Partition
//       0xF2  DOS Secondary Partition
//       0xFF  Bad Block Table Partition
//
//


// This function verifies the reported partition extants and checks
// for any overlapping or out of bounds partition definitions.

extern ULONG FirstValidDisk;

ULONG ValidatePartitionExtants(ULONG disk)
{
    register ULONG i;
    register ULONG cEnd, nEnd, cStart, nStart, cSize, nSize;

#if (MULTI_MACHINE_EMULATION)
	m2machine	*machine;
	NWDISK	**SystemDisk;
	m2group_get_context(&machine);
	SystemDisk = (NWDISK **) &machine->LSystemDisk;
#endif
    if ((!SystemDisk[disk]) || (!SystemDisk[disk]->PhysicalDiskHandle))
       return -1;

    for (i=0; i < 4; i++)
    {
       if (!SystemDisk[disk]->PartitionTable[i].SysFlag)
	  continue;

       if ((SystemDisk[disk]->PartitionTable[i].StartLBA +
	    SystemDisk[disk]->PartitionTable[i].nSectorsTotal) > (ULONG)
	    (SystemDisk[disk]->Cylinders *
	     SystemDisk[disk]->TracksPerCylinder *
	     SystemDisk[disk]->SectorsPerTrack))
	  return -1;

       if (SystemDisk[disk]->PartitionTable[i].StartLBA <
	   SystemDisk[disk]->SectorsPerTrack)
	  return -1;

       if ((i + 1) < 4)
       {
	  for (; (i + 1) < 4; i++)
	  {
	     if (SystemDisk[disk]->PartitionTable[i + 1].SysFlag)
	     {
		cEnd = SystemDisk[disk]->PartitionTable[i].StartLBA +
		       SystemDisk[disk]->PartitionTable[i].nSectorsTotal;
		nEnd = SystemDisk[disk]->PartitionTable[i + 1].StartLBA +
		       SystemDisk[disk]->PartitionTable[i + 1].nSectorsTotal;
		cStart = SystemDisk[disk]->PartitionTable[i].StartLBA;
		nStart = SystemDisk[disk]->PartitionTable[i + 1].StartLBA;
		cSize = SystemDisk[disk]->PartitionTable[i].nSectorsTotal;
		nSize = SystemDisk[disk]->PartitionTable[i + 1].nSectorsTotal;

		if ((cStart >= nStart) && (cStart < nEnd))
		   return -1;

		if ((cEnd > nStart) && (cEnd < nEnd))
		   return -1;

		if ((cStart < nStart) && (cEnd > nEnd))
		   return -1;
	     }
	  }
       }
    }
    return 0;

}

//
//  Netware partitions use their own unique method for recording
//  cyl/head/sector addressing for big partitions.  This method
//  is detailed below.
//

ULONG SetPartitionTableGeometry(BYTE *hd, BYTE *sec, BYTE *cyl,
				ULONG LBA, ULONG TracksPerCylinder,
				ULONG SectorsPerTrack)
{
    ULONG offset, cylinders, head, sector;

#if (MULTI_MACHINE_EMULATION)
	m2machine	*machine;
	NWDISK	**SystemDisk;
	m2group_get_context(&machine);
	SystemDisk = (NWDISK **) &machine->LSystemDisk;
#endif
    if (!cyl || !hd || !sec)
       return -1;

    cylinders = (LBA / (TracksPerCylinder * SectorsPerTrack));
    offset = LBA % (TracksPerCylinder * SectorsPerTrack);
    head = (WORD)(offset / SectorsPerTrack);
    sector = (WORD)(offset % SectorsPerTrack) + 1;

    if (cylinders < 1023)
    {
       *sec = (BYTE)sector;
       *hd = (BYTE)head;
       *cyl = (BYTE)(cylinders & 0xff);
       *sec |= (BYTE)((cylinders >> 2) & 0xC0);
    }
    else
    {
	*sec = (BYTE)(SectorsPerTrack | 0xC0);
	*hd = (BYTE)(TracksPerCylinder - 1);
	*cyl = (BYTE)0xFE;
    }

    return 0;
}

ULONG SetPartitionTableValues(struct PartitionTableEntry *Part,
			      ULONG Type,
			      ULONG StartingLBA,
			      ULONG EndingLBA,
			      ULONG Flag,
			      ULONG TracksPerCylinder,
			      ULONG SectorsPerTrack)
{

    Part->SysFlag = (BYTE) Type;
    Part->fBootable = (BYTE) Flag;
    Part->StartLBA = StartingLBA;
    Part->nSectorsTotal = (EndingLBA - StartingLBA) + 1;

    SetPartitionTableGeometry(&Part->HeadStart, &Part->SecStart, &Part->CylStart,
			      StartingLBA, TracksPerCylinder, SectorsPerTrack);

    SetPartitionTableGeometry(&Part->HeadEnd, &Part->SecEnd, &Part->CylEnd,
			      EndingLBA, TracksPerCylinder, SectorsPerTrack);

    return 0;

}

BYTE *get_partition_type(ULONG type)
{
    switch (type)
    {
       case 0x01:
	  return "FAT 12";

       case 0x02:
	  return "Xenix Root";

       case 0x03:
	  return "Xenix USR";

       case 0x04:
	  return "FAT 16 < 32MB";

       case 0x05:
	  return "Extended DOS";

       case 0x06:
	  return "FAT 16 >= 32MB";

       case 0x07:
	  return "IFS (HPFS/NTFS)";

       case 0x08:
	  return "AIX Partition";

       case 0x09:
	  return "AIX Boot";

       case 0x0A:
	  return "OS/2 Boot Manager";

       case 0x0B:
	  return "FAT 32";

       case 0x0C:
	  return "FAT 32 (Int 13 Ext)";

       case 0x0E:
	  return "FAT 16 (Int 13 Ext)";

       case 0x0F:
	  return "Ext DOS (Int 13 Ext)";

       case 0x40:
	  return "Venix 286";

       case 0x41:
	  return "Power PC (PReP)";

       case 0x51:
	  return "Novell [???]";

       case 0x52:
	  return "Microport/CPM";

       case 0x57:
	  return "NWFS";

       case 0x63:
	  return "Unix (GNU Hurd)";

       case 0x64:
	  return "Netware 286";

       case 0x65:
	  return "Netware 386";

       case 0x66:
	  return "Netware SMS";

       case 0x75:
	  return "PC/IX";

       case 0x77:
	  return "M2FS";

       case 0x80:
	  return "MINIX";

       case 0x81:
	  return "Linux/MINIX";

       case 0x82:
	  return "LINUX SWAP";

       case 0x83:
	  return "LINUX Native";

       case 0x93:
	  return "Amoeba";

       case 0x94:
	  return "Amoeba BBT";

       case 0xA5:
	  return "BSD/386";

       case 0xB7:
	  return "BSDI File System";

       case 0xB8:
	  return "BSDI Swap";

       case 0xC0:
	  return "NTFT";

       case 0xC7:
	  return "Syrinx";

       case 0xDB:
	  return "CP/M";

       case 0xE1:
	  return "DOS access";

       case 0xE3:
	  return "DOS R/O";

       case 0xF2:
	  return "DOS Secondary";

       case 0xFF:
	  return "BBT";

       default:
	  return "Unknown";
    }
}


void GetNumberOfDisks(ULONG *disk_count)
{
#if (MULTI_MACHINE_EMULATION)
	m2machine	*machine;
	m2group_get_context(&machine);
#endif
    if (disk_count)
       *disk_count = TotalDisks;
}

ULONG GetDiskInfo(ULONG disk_number, DISK_INFO *disk_info)
{
    register ULONG i;
#if (MULTI_MACHINE_EMULATION)
	m2machine	*machine;
	NWDISK	**SystemDisk;
	m2group_get_context(&machine);
	SystemDisk = (NWDISK **) &machine->LSystemDisk;
#endif

    if (SystemDisk[disk_number] != 0)
    {
       disk_info->disk_number = disk_number;
       disk_info->partition_count = 0;

       for (i=0; i < 4; i++)
       {
	  if (SystemDisk[disk_number]->PartitionTable[i].nSectorsTotal != 0)
	     disk_info->partition_count ++;
       }

       disk_info->total_sectors = (ULONG)
	    (SystemDisk[disk_number]->Cylinders *
	     SystemDisk[disk_number]->TracksPerCylinder *
	     SystemDisk[disk_number]->SectorsPerTrack);

       if ((disk_info->cylinder_size = (ULONG)
			  (SystemDisk[disk_number]->TracksPerCylinder *
			  SystemDisk[disk_number]->SectorsPerTrack)) == 0)
	  disk_info->cylinder_size = 0x40;

       disk_info->track_size = (ULONG) SystemDisk[disk_number]->SectorsPerTrack;
       disk_info->bytes_per_sector = (ULONG) SystemDisk[disk_number]->BytesPerSector;

       return (0);
    }
    return (-1);
}

ULONG GetPartitionInfo(ULONG disk_number,
		       ULONG partition_number,
		       PARTITION_INFO *part_info)
{
#if (MULTI_MACHINE_EMULATION)
	m2machine	*machine;
	NWDISK	**SystemDisk;
	m2group_get_context(&machine);
	SystemDisk = (NWDISK **) &machine->LSystemDisk;
#endif
    if (SystemDisk[disk_number] != 0)
    {
       if (SystemDisk[disk_number]->PartitionTable[partition_number].nSectorsTotal != 0)
       {
	  part_info->disk_number = disk_number;
	  part_info->partition_number = partition_number;

	  part_info->total_sectors = (ULONG)
	       (SystemDisk[disk_number]->Cylinders *
		SystemDisk[disk_number]->TracksPerCylinder *
		SystemDisk[disk_number]->SectorsPerTrack);

	  if ((part_info->cylinder_size = (ULONG)
			(SystemDisk[disk_number]->TracksPerCylinder *
			SystemDisk[disk_number]->SectorsPerTrack)) == 0)
	     part_info->cylinder_size = 0x40;

	  part_info->partition_type = SystemDisk[disk_number]->PartitionTable[partition_number].SysFlag;
	  part_info->boot_flag = SystemDisk[disk_number]->PartitionTable[partition_number].fBootable;
	  part_info->sector_offset = SystemDisk[disk_number]->PartitionTable[partition_number].StartLBA;
	  part_info->sector_count = SystemDisk[disk_number]->PartitionTable[partition_number].nSectorsTotal;

	  return (0);
       }
    }
    return (-1);
}
#if (FILE_DISK_EMULATION)

void RemoveDiskDevices(void)
{
	register ULONG j;
#if (MULTI_MACHINE_EMULATION)
	m2machine	*machine;
	NWDISK		**SystemDisk;
	m2group_get_context(&machine);
	SystemDisk = (NWDISK **) &machine->LSystemDisk;
#endif
	for (j=0; j < MAX_DISKS; j++)
    {
	   if (SystemDisk[j])
	   {
		m2_fclose((uint32) SystemDisk[j]->PhysicalDiskHandle);
	  NWFSFree(SystemDisk[j]);
	  SystemDisk[j] = 0;
	  if (TotalDisks)
		 TotalDisks--;
	   }
	}
}

uint32	quick_parse_disk_file(
	uint8	*bptr,
	uint32	*capacity,
	uint32	*part_info)
{
	uint32	index = 0;
	uint32	value;
	uint32	base_index;
	NWFSSet(part_info, 0, 16 * 4);
	*capacity = 0;
	while ((bptr[0] != 0) && (index < 16))
	{
		if ((bptr[0] == '0') && ((bptr[1] == 'x') || (bptr[1] == 'X')))
		{
			bptr += 2;
			value = 0;
			while (1)
			{
				if ((bptr[0] >= '0') && (bptr[0] <= '9'))
					value = (value * 16) + (bptr[0] - '0');
				else
				if ((bptr[0] >= 'a') && (bptr[0] <= 'f'))
					value = (value * 16) + (bptr[0] - 'a') + 10;
				else
				if ((bptr[0] >= 'A') && (bptr[0] <= 'F'))
					value = (value * 16) + (bptr[0] - 'A') + 10;
				else
					break;
				bptr ++;
			}
			if ((index % 4) == 0)
			{
				base_index = value;
			}
			if (base_index < 4)
			{
				part_info[(base_index * 4) + (index % 4)] = value;
				if ((index % 4) == 3)
				{
					if ((part_info[(base_index * 4) + 2] + part_info[(base_index * 4) + 3]) > *capacity)
						*capacity = part_info[(base_index * 4) + 2] + part_info[(base_index * 4) + 3];
				}

			}
			index ++;
		}
		bptr ++;
	}

	return((*capacity == 0) ? -1 : 0);
}

uint32	scan_init_flag = 0;
void ScanDiskDevices(void)
{
	uint32	i, j, k;
	uint32	free_index;
	uint32	retCode;
	uint32	file_handle;
	uint32	capacity;
	uint8	*Sector;
	uint32	part_info[16];
#if (MULTI_MACHINE_EMULATION)
	m2machine	*machine;
	NWDISK		**SystemDisk;
	m2group_get_context(&machine);
	SystemDisk = (NWDISK **) &machine->LSystemDisk;
	disk_file[8] = ((machine->file_number / 10) % 10) + '0';
	disk_file[9] = (machine->file_number % 10) + '0';
#endif

	if (scan_init_flag == 0)
	{
		NWFSSet(&SystemDisk[0], 0, MAX_DISKS * 4);
		scan_init_flag = 1;
	}

	Sector = NWFSIOAlloc(IO_BLOCK_SIZE, DISKBUF_TAG);

	for (i=0; i<32; i++)
	{
		free_index = 0xFFFFFFFF;
		for (j=0; j<MAX_DISKS; j++)
		{
			if (SystemDisk[j] != 0)
			{
				if (SystemDisk[j]->DiskNumber == i)
					break;
			}
			else
			{
				if (free_index == 0xFFFFFFFF)
					free_index = j;
			}
		}
		if ((j == MAX_DISKS) && (free_index != 0xFFFFFFFF))
		{
			j = free_index;
#if (MULTI_MACHINE_EMULATION)
			disk_file[17] = ((i / 10) % 10) + '0';
			disk_file[18] = (i % 10) + '0';
#else
			disk_file[6] = ((i / 10) % 10) + '0';
			disk_file[7] = (i % 10) + '0';
#endif

			if (m2_fopen(&file_handle, &disk_file[0], M2READ_WRITE) == 0)
			{
				Sector[510] = 0;
				Sector[511] = 0;
				fread(Sector, 1, 512, (FILE *) file_handle);
				if ((Sector[510] != 0x55) || (Sector[511] != 0xAA))
				{
					if (quick_parse_disk_file(Sector, &capacity, &part_info[0]) == 0)
					{
						for (k=0; k<4; k++)
						{
							if (part_info[(k * 4) + 1] != 0)
								SetPartitionTableValues((PARTITION_INFO *) &Sector[(512-66) + (k*16)], part_info[(k*4)+1], part_info[(k*4)+2], part_info[(k*4)+3] + part_info[(k*4)+2] - 1, 0, 32, 32);
						}
						Sector[510] = 0x55;
						Sector[511] = 0xAA;
						m2_fwrite(file_handle, SECTOR_SIZE, 0, SECTORS_PER_BLOCK, (uint8 *) Sector);
						m2_fwrite(file_handle, SECTOR_SIZE, capacity - SECTORS_PER_BLOCK, SECTORS_PER_BLOCK, (uint8 *) Sector);
					}
				}


				   capacity = m2_fsize(file_handle) / 512;
				   SystemDisk[j] = (NWDISK *) NWFSAlloc(sizeof(NWDISK), NWDISK_TAG);
				   NWFSSet(SystemDisk[j], 0, sizeof(NWDISK));

				   SystemDisk[j]->DiskNumber = i;
				   SystemDisk[j]->PhysicalDiskHandle = (void *) file_handle;
				   SystemDisk[j]->Cylinders = capacity / (32 * 32);
				   SystemDisk[j]->TracksPerCylinder = 32;
				   SystemDisk[j]->SectorsPerTrack = 32;
				   SystemDisk[j]->BytesPerSector = 512;
				   SystemDisk[j]->driveSize = (LONGLONG)
					  (SystemDisk[j]->Cylinders *
					   SystemDisk[j]->TracksPerCylinder *
					   SystemDisk[j]->SectorsPerTrack *
					   SystemDisk[j]->BytesPerSector);
				TotalDisks ++;



				NWFSCopy(&SystemDisk[j]->partitionSignature, &Sector[0x01FE], 2);

				if (SystemDisk[j]->partitionSignature == 0xAA55)
				{
					NWFSCopy(&SystemDisk[j]->PartitionTable[0].fBootable,
						  &Sector[0x01BE], 64);

// scan for Netware Partitions and detect 3.x and 4.x/5.x partition types
					for (k=0; k < 4; k++)
					{
						if (SystemDisk[j]->PartitionTable[k].SysFlag == NETWARE_386_ID)
						{
							retCode = ReadDiskSectors(j,
								SystemDisk[j]->PartitionTable[k].StartLBA,
								Sector,
								IO_BLOCK_SIZE / SystemDisk[j]->BytesPerSector,
								IO_BLOCK_SIZE / SystemDisk[j]->BytesPerSector);
							if (!retCode)
							{
								   NWFSPrint("nwfs:  disk-%d read error in ScanDiskDevices (part)\n", (int)j);
							}

							if (!NWFSCompare(Sector, NwPartSignature, 16))
								SystemDisk[j]->PartitionVersion[k] = NW4X_PARTITION;
							else
								SystemDisk[j]->PartitionVersion[k] = NW3X_PARTITION;
						}
					}
				}
			}
		}
	}
	NWFSFree(Sector);
	return;
}

#else

#if (LINUX_20 | LINUX_22 | LINUX_24)

ULONG LocateDevice(kdev_t dev)
{
    register ULONG major = MAJOR(dev);
    register ULONG minor = MINOR(dev);
    register ULONG dev_number, end_minor;
    struct gendisk *search;

#if (LINUX_22 | LINUX_24)
    lock_kernel();
#endif
    search = gendisk_head;
    while (search)
    {
        if (search->major == major)
        {
           // get the minor device index
     	   dev_number = (minor >> search->minor_shift);

           // determine the last valid minor device number
#if (LINUX_20 | LINUX_22)
	   end_minor = search->max_nr * search->max_p;
#elif (LINUX_24)
	   end_minor = search->max_p;
#endif
           // check if this minor device is present
           // and that it begins at sector 0 (Physical HDD Device)
	   if ((dev_number < end_minor) && (search->part) && 
	       (search->part[dev_number].nr_sects))
           {
#if (VERBOSE)
	      NWFSPrint("dev-%s major-%d:%d ns-%d ss-%d t-%d\n",
	                search->major_name, (int)search->major,(int)minor,
			(int)search->part[dev_number].nr_sects,
			(int)search->part[dev_number].start_sect,
			(int)search->part[dev_number].type);
#endif

#if (LINUX_22 | LINUX_24)
              unlock_kernel();
#endif
              return TRUE;
           }
#if (LINUX_22 | LINUX_24)
           unlock_kernel();
#endif
           return 0;
        }
        search = search->next;
    }
#if (LINUX_22 | LINUX_24)
    unlock_kernel();
#endif
    return 0;
}

#endif

extern ULONG MaxDisksSupported;
extern ULONG MaxHandlesSupported;

void RemoveDiskDevices(void)
{
    register ULONG j;

    for (j=0; j < MAX_DISKS; j++)
    {
       if (SystemDisk[j])
       {
	  SyncDevice((ULONG)SystemDisk[j]->PhysicalDiskHandle);
          blkdev_put(SystemDisk[j]->inode.i_bdev, BDEV_FILE);
	  NWFSFree(SystemDisk[j]);
	  SystemDisk[j] = 0;
	  if (TotalDisks)
	     TotalDisks--;
       }
    }
}

extern struct blk_dev_struct blk_dev[MAX_BLKDEV];
extern int *blk_size[MAX_BLKDEV];
extern int *blksize_size[MAX_BLKDEV];
extern int *hardsect_size[MAX_BLKDEV];

void ScanDiskDevices(void)
{
    register kdev_t dev;
    register ULONG size;
    register ULONG j, i, retCode, len, major, minor;
    struct hd_geometry geometry;
    struct page *page;
    BYTE *Sector;

    page = alloc_page(GFP_ATOMIC);
    if (!page)
       return;

    Sector = (BYTE *)page_address(page);
    if (!Sector)
    {
       NWFSPrint("nwfs:  allocation error in AddDiskDevices\n");
       return;
    }

    for (FirstValidDisk = (ULONG)-1, j = 0; j < MAX_DISKS; j++)
    {
       if (!SystemDisk[j])
       {
          if (!DiskDevices[j])
             break;

	  dev = DiskDevices[j];
          major = MAJOR(dev);
          minor = MINOR(dev);

          // search the gendisk head and see if this device exists
          if (!LocateDevice(dev))
             continue;

	  SystemDisk[j] = (NWDISK *) NWFSAlloc(sizeof(NWDISK), NWDISK_TAG);
	  if (!SystemDisk[j])
	  {
	     NWFSPrint("nwfs: memory allocation failure in AddDiskDevices\n");
	     continue;
	  }
	  NWFSSet(SystemDisk[j], 0, sizeof(NWDISK));

          init_special_inode(&SystemDisk[j]->inode, S_IFBLK | S_IRUSR, dev);
          retCode = blkdev_open(&SystemDisk[j]->inode, &SystemDisk[j]->filp);
          if (retCode)
          {
             if (SystemDisk[j])
	        NWFSFree(SystemDisk[j]);
	     SystemDisk[j] = 0;
             continue;
          }

          // perform a get drive geometry call to determine if it's a hard
          // disk or CDROM

#if (LINUX_22 | LINUX_24)
          lock_kernel();
#endif
	  retCode = ioctl_by_bdev(SystemDisk[j]->inode.i_bdev, HDIO_GETGEO,
	                          (ULONG)&geometry);
#if (LINUX_22 | LINUX_24)
          unlock_kernel();
#endif

	  if (retCode)
          {
             if (SystemDisk[j])
	        NWFSFree(SystemDisk[j]);
	     SystemDisk[j] = 0;
	     continue;
          }

	  set_blocksize(dev, LogicalBlockSize);

          if (FirstValidDisk == (ULONG)-1)
             FirstValidDisk = j;

	  SystemDisk[j]->DiskNumber = j;
	  SystemDisk[j]->DeviceBlockSize = LogicalBlockSize;
          SystemDisk[j]->Cylinders = (LONGLONG) geometry.cylinders;
          SystemDisk[j]->TracksPerCylinder = geometry.heads;
          SystemDisk[j]->SectorsPerTrack = geometry.sectors;
	  SystemDisk[j]->PhysicalDiskHandle = (void *)((ULONG)dev);
	  SystemDisk[j]->BytesPerSector = hardsect_size[major] ? hardsect_size[major][minor] : 512;

	  size = (blk_size[major] ? (ULONG) blk_size[major][minor] : (ULONG) blk_size[major]);
	  SystemDisk[j]->driveSize = (LONGLONG)((LONGLONG) size *
				     (LONGLONG) SystemDisk[j]->BytesPerSector);

	  TotalDisks++;
	  if (TotalDisks > MaximumDisks)
	     MaximumDisks = TotalDisks;

       }

       retCode = ReadDiskSectors(j, 0, Sector,
				 IO_BLOCK_SIZE / SystemDisk[j]->BytesPerSector,
				 IO_BLOCK_SIZE / SystemDisk[j]->BytesPerSector);
       if (!retCode)
       {
	  NWFSPrint("nwfs:  disk-%d read error in ScanDiskDevices\n", (int)j);
	  continue;
       }

       len = SystemDisk[j]->BytesPerSector;

       NWFSCopy(&SystemDisk[j]->partitionSignature,
		   &Sector[0x01FE], 2);

       if (SystemDisk[j]->partitionSignature != 0xAA55)
       {
#if (VERBOSE)
	  NWFSPrint("nwfs:  partition signature 0xAA55 not found disk(%d)\n", j);
#endif
	  NWFSSet(&SystemDisk[j]->PartitionTable[0].fBootable, 0, 64);
	  continue;
       }
       else
       {
	  NWFSCopy(&SystemDisk[j]->PartitionTable[0].fBootable,
		   &Sector[0x01BE], 64);
       }

       // scan for Netware Partitions and detect 3.x and 4.x/5.x partition types
       for (i=0; i < 4; i++)
       {
	  if (SystemDisk[j]->PartitionTable[i].SysFlag == NETWARE_386_ID)
	  {
	     retCode = ReadDiskSectors(j,
			  SystemDisk[j]->PartitionTable[i].StartLBA,
			  Sector,
			  IO_BLOCK_SIZE / SystemDisk[j]->BytesPerSector,
			  IO_BLOCK_SIZE / SystemDisk[j]->BytesPerSector);
	     if (!retCode)
	     {
		NWFSPrint("nwfs:  disk-%d read error in ScanDiskDevices (part)\n", (int)j);
		continue;
	     }

	     if (!NWFSCompare(Sector, NwPartSignature, 16))
		SystemDisk[j]->PartitionVersion[i] = NW4X_PARTITION;
	     else
		SystemDisk[j]->PartitionVersion[i] = NW3X_PARTITION;
	  }
       }
    }
    __free_page(page);
    return;
}

#endif
