/*
 * init.c -- module initialization code
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * Written by Sos Pter <sp@osb.hu>, 2002-2003
 */


#include <linux/config.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/types.h>
#include <linux/init.h>
#include <linux/pm.h>
#include <linux/proc_fs.h>

#include <asm/system.h>
#include <asm/errno.h>
#include <asm/io.h>
#include <asm/uaccess.h>

#ifdef OMNIBOOK_STANDALONE
#include "omnibook.h"
#else
#include <linux/omnibook.h>
#endif

#include "dmi.h"
#include "init.h"

int omnibook_ectype;
int __init omnibook_userset;

struct proc_dir_entry *omnibook_proc_root __initdata = NULL;
static struct proc_dir_entry *proc_dmi;

/*
 * Module parameters
 *  0 disables the feature
 * >1 enables the feature
 * All feature enabled by default
 */

static int ectype __initdata = NONE;
static int ac __initdata = 1;
static int battery __initdata = 1;
static int blank __initdata = 1;
static int display __initdata = 1;
static int dmi __initdata = 1;
static int fan __initdata = 1;
static int fan_policy __initdata = 1;
static int onetouch __initdata = 1;
static int key_polling __initdata = 1;
static int lcd __initdata = 1;
static int temperature __initdata = 1;
static int touchpad __initdata = 1;

static int apmemu __initdata = 0;
static int dock __initdata = 0;
static int user __initdata = 0;

static int omnibook_proc_dmi(char *buffer, char **start, off_t off, int count, int *eof, void *data)
{
	int len;
	char *b = buffer;

	b += sprintf(b, "BIOS Vendor:   %s\n", omnibook_dmi_ident[OMNIBOOK_BIOS_VENDOR]);
	b += sprintf(b, "BIOS Version:  %s\n", omnibook_dmi_ident[OMNIBOOK_BIOS_VERSION]);
	b += sprintf(b, "BIOS Release:  %s\n", omnibook_dmi_ident[OMNIBOOK_BIOS_DATE]);
	b += sprintf(b, "System Vendor: %s\n", omnibook_dmi_ident[OMNIBOOK_SYS_VENDOR]);
	b += sprintf(b, "Product Name:  %s\n", omnibook_dmi_ident[OMNIBOOK_PRODUCT_NAME]);
	b += sprintf(b, "Version:       %s\n", omnibook_dmi_ident[OMNIBOOK_PRODUCT_VERSION]);
	b += sprintf(b, "Serial Number: %s\n", omnibook_dmi_ident[OMNIBOOK_SERIAL_NUMBER]);
	b += sprintf(b, "Board Vendor:  %s\n", omnibook_dmi_ident[OMNIBOOK_BOARD_VENDOR]);
	b += sprintf(b, "Board Name:    %s\n", omnibook_dmi_ident[OMNIBOOK_BOARD_NAME]);
	b += sprintf(b, "Board Version: %s\n", omnibook_dmi_ident[OMNIBOOK_BOARD_VERSION]);

	len = b - buffer;
	if (len < off + count)
		*eof = 1;
	*start = buffer + off;
	len -= off;
	if (len > count)
		len = count;
	if (len < 0)
		len = 0;
	return len;
}

#define HP_SIGNATURE            "Hewlett-Packard"

struct omnibook_tc_t {
	char	*tc;
	int	ectype;
};

/* HP technology codes */
static struct omnibook_tc_t omnibook_tc[] __initdata = {
  /* technology code	ectype */
	{ "CI",		OB4150 },
	{ "EA",		OB6000 },
	{ "EB",		OB6100 },
	{ "EG",		XE4500 },
	{ "FA",		OB500 },
	{ "FB",		OB510 },
	{ "GC",		XE3GC },
	{ "GD",		XE3GC },
	{ "GE",		XE3GC },
	{ "GF",		XE3GF },
	{ "IB",		XE3GF },
	{ "IC",		XE3GF },
	{ "ID",		XE3GF },
	{ "KA",		XE4500 },
	{ "KB",		XE4500 },
	{ "KC",		XE4500 },
	{ "KD",		XE4500 },
	{ "KE",		XE4500 },
	{ NULL,		NONE }
};

struct omnibook_models_t {
	/* DMI field matchers (table inputs) */
	char	*sys_vendor;
	char	*product_name;
	char	*product_version;
	char	*board_name;

	/* Table outputs */
	char	*syslog_name;		/* Name which will appear in the syslog */
	int	ectype;			/* Type of the embedded controller firmware, see omnibook.h and README */
};

static struct omnibook_models_t omnibook_models[] __initdata = {
  /* sys_vendor product_name                 product_version                   board_name syslog_name                  ectype */
  { NULL,       "HP OmniBook PC*",           "HP OmniBook XE3 GF*",            NULL,    NULL,                          XE3GF },
  { NULL,       "HP OmniBook PC*",           "HP OmniBook XT1000*",            NULL,    NULL,                          XE3GF },
  { NULL,       "HP OmniBook PC*",           "HP OmniBook XE3 GC*",            NULL,    NULL,                          XE3GC },
  /* HP Pavilion N5430 */
  { NULL,       "HP OmniBook PC*",           "HP OmniBook XE3 GD*",            NULL,    NULL,                          XE3GC },
  /* HP Pavilion N5415 */
  { NULL,       "HP OmniBook PC*",           "HP OmniBook XE3 GE*",            NULL,    NULL,                          XE3GC },
  { NULL,       "HP OmniBook PC*",           "HP OmniBook 500 FA*",            NULL,    NULL,                          OB500 },
  { NULL,       "HP OmniBook PC*",           "HP OmniBook 510 FB*",            NULL,    NULL,                          OB510 },
  { NULL,       "HP OmniBook PC*",           "HP OmniBook 4150*",              NULL,    NULL,                          OB4150 },
  { NULL,       "HP OmniBook PC*",           "HP OmniBook 6000 EA*",           NULL,    NULL,                          OB6000 },
  { NULL,       "HP OmniBook PC*",           "HP OmniBook 6100 EB*",           NULL,    NULL,                          OB6100 },
  /* HP OmniBook xe4100 */
  { NULL,       "HP OmniBook PC*",           "HP OmniBook xe4000*",            NULL,    NULL,                          XE4500 },
  { NULL,       "HP OmniBook PC*",           "HP OmniBook xe4400*",            NULL,    NULL,                          XE4500 },
  { NULL,       "HP OmniBook PC*",           "HP OmniBook xe4500*",            NULL,    NULL,                          XE4500 },
  /* HP OmniBook vt6200 and xt6200 */
  { NULL,       "HP OmniBook PC*",           "HP OmniBook 6200 EG*",           NULL,    NULL,                          XE4500 },
  /* HP Pavilion ze4125 */
  { NULL,       "HP NoteBook PC*",           "HP NoteBook ze4000*",            NULL,    NULL,                          XE4500 },
  /* There are no model specific strings of some HP Pavilion xt155 and some HP Pavilion ze4100 */
  { NULL,       "HP NoteBook PC*",           "HP NoteBook PC*",                NULL,    NULL,                          XE4500 },
  /* HP Pavilion N5290 */
  { NULL,       "HP Pavilion Notebook PC*",  "HP Pavilion Notebook XE3 GC*",   NULL,    NULL,                          XE3GC },
  /* HP Pavilion N5441 */
  { NULL,       "HP Pavilion Notebook PC*",  "HP Pavilion Notebook Model GD*", NULL,    NULL,                          XE3GC },
  /* HP Pavilion ZT1141 */
  { NULL,       "HP Pavilion Notebook PC*",  "HP Pavilion Notebook ZT1000*",   NULL,    NULL,                          XE3GF },
  /* There are no model specific strings of some HP Pavilion ZT1175 and ZT1195 notebooks */
  { NULL,       "HP Pavilion Notebook PC*",  "HP Pavilion Notebook*",          NULL,    NULL,                          XE3GF },
  { NULL,       "Pavilion ze4200*",          NULL,                             NULL,    "HP Pavilion ze4200 series",   XE4500 },
  { "TOSHIBA",  "S1000*",                    NULL,                             NULL,    "Toshiba Satellite 1000",      XE3GF },
  { "TOSHIBA",  "S1005*",                    NULL,                             NULL,    "Toshiba Satellite 1005",      XE3GF },
  { "TOSHIBA",  "S1110*",                    NULL,                             NULL,    "Toshiba Satellite 1110",      XE3GF },
  { "TOSHIBA",  "S1115*",                    NULL,                             NULL,    "Toshiba Satellite 1115",      XE3GF },
  { "TOSHIBA",  "S1900*",                    NULL,                             NULL,    "Toshiba Satellite 1900",      XE3GF },
  { "TOSHIBA",  "S1905*",                    NULL,                             NULL,    "Toshiba Satellite 1905",      XE3GF },
  { "TOSHIBA",  "S3000*",                    NULL,                             NULL,    "Toshiba Satellite 3000",      XE3GF },
  { "TOSHIBA",  "S3005*",                    NULL,                             NULL,    "Toshiba Satellite 3005",      XE3GF },
  { "TOSHIBA",  "Satellite 1000*",           NULL,                             NULL,    "Toshiba Satellite 1000",      XE3GF },
  { "TOSHIBA",  "Satellite 1005*",           NULL,                             NULL,    "Toshiba Satellite 1005",      XE3GF },
  { "TOSHIBA",  "Satellite 1110*",           NULL,                             NULL,    "Toshiba Satellite 1110",      XE3GF },
  { "TOSHIBA",  "Satellite 1115*",           NULL,                             NULL,    "Toshiba Satellite 1115",      XE3GF },
  { "TOSHIBA",  "Satellite 1900*",           NULL,                             NULL,    "Toshiba Satellite 1900",      XE3GF },
  { "TOSHIBA",  "Satellite 1905*",           NULL,                             NULL,    "Toshiba Satellite 1905",      XE3GF },
  { "TOSHIBA",  "Satellite 3000*",           NULL,                             NULL,    "Toshiba Satellite 3000",      XE3GF },
  { "TOSHIBA",  "Satellite 3005*",           NULL,                             NULL,    "Toshiba Satellite 3005",      XE3GF },
  { "COMPAL",   NULL,                        NULL,                             "ACL00", "Compal ACL00",                XE3GF },
  { "Acer",     "Aspire 1400 series*",       NULL,                             NULL,    "Acer Aspire 1400 series",     XE3GF },
  /* This sentinel at the end catches all unsupported models */
  { NULL, NULL, NULL, NULL, NONE }
};

/*
 * Compare the saved DMI info at "index" with a string.
 * A '*' at the end of the string will match anything.
 * Returns 0 for a match.
 * 
 * This preserves the semantics of the old omnibook_features[]
 * table.  I don't know if its generally useful or not.
 */
static int __init cmp_with_glob(int index, char *str)
{
	int retval = 0;
	char *glob;
	int len;

	if (str) {
		glob = strchr(str, '*');
		len = glob ? glob - str : strlen(str);
		retval = strncmp(omnibook_dmi_ident[index], str, len);
	}

	return retval;
}

static int __init omnibook_ident(void)
{
	struct omnibook_models_t *mp;

	for (mp = omnibook_models; mp->ectype != NONE; ++mp) {
		/* Check all fields for a match */
		if (cmp_with_glob(OMNIBOOK_SYS_VENDOR, mp->sys_vendor))
			continue;
		if (cmp_with_glob(OMNIBOOK_PRODUCT_NAME, mp->product_name))
			continue;
		if (cmp_with_glob(OMNIBOOK_PRODUCT_VERSION, mp->product_version))
			continue;
		if (cmp_with_glob(OMNIBOOK_BOARD_NAME, mp->board_name))
			continue;

		/* All required fields matched */
		break;
	}

	return (mp - omnibook_models);
}

static int __init omnibook_get_tc(void)
{	
	struct omnibook_tc_t *tc;

	for (tc = omnibook_tc; tc->ectype != NONE; ++tc) {
		/* Technology code appears in the first two chracters of BIOS version string */
		if (strncmp(omnibook_dmi_ident[OMNIBOOK_BIOS_VERSION], tc->tc, strlen(tc->tc)) == 0)
			break;
	}

	return (tc - omnibook_tc);
}

static int __init omnibook_init(void)
{
	int retval;
	int model = 0;
	int tc = 0;
	char *syslog_name;
	char *glob;
	mode_t pmode;

	omnibook_userset = user;

#ifdef MODULE
	printk(KERN_INFO "%s: module version %s.\n", MODULE_NAME, MODULE_VERSION);
#else
	printk(KERN_INFO "%s: driver version %s.\n", MODULE_NAME, MODULE_VERSION);
#endif

	/* saving DMI information */
	if (omnibook_dmi_scan_machine() != 0)
		return  -ENODEV;

	if (ectype != NONE)
		printk(KERN_WARNING "%s: Forced load with EC firmware type %d.\n", MODULE_NAME, ectype);
	else {
		model = omnibook_ident();
		if (omnibook_models[model].ectype != NONE) {
			ectype = omnibook_models[model].ectype;
			syslog_name = omnibook_models[model].syslog_name;
			if (!syslog_name) {
				syslog_name = omnibook_models[model].product_version;
				glob = strchr(syslog_name, '*');
				if (glob)
					*glob = '\0';
			}
			printk(KERN_INFO "%s: %s detected.\n", MODULE_NAME, syslog_name);
		} else {
			/* Without explicite informations try chechking for technology code of HP laptops */
			tc = omnibook_get_tc();
			if ((strncmp(omnibook_dmi_ident[OMNIBOOK_SYS_VENDOR], HP_SIGNATURE, strlen(HP_SIGNATURE)) == 0) &&
				(omnibook_tc[tc].ectype != NONE)) {
				ectype = omnibook_tc[tc].ectype;
				printk(KERN_INFO "%s: HP tecnology code %s detected.\n", MODULE_NAME, omnibook_tc[tc].tc);
			} else
				printk(KERN_INFO "%s: Unknown model detected.\n", MODULE_NAME);
		}
	}
	
	omnibook_ectype = ectype;

	switch (omnibook_ectype) {
	case XE3GF:
	case XE3GC:
	case OB500:
	case OB510:
	case OB6000:
	case OB6100:
	case XE4500:
	case OB4150:
		break;
	default:
		ac = 0;
		apmemu = 0;
		battery = 0;
		blank = 0;
		display = 0;
		dock = 0;
		fan = 0;
		fan_policy = 0;
		key_polling = 0;
		lcd = 0;
		onetouch = 0;
		temperature = 0;
		touchpad = 0;
	}

	omnibook_proc_root = proc_mkdir(MODULE_NAME, NULL);
	if (! omnibook_proc_root) {
		printk(KERN_ERR "%s: Unable to create /proc/%s.\n", MODULE_NAME, MODULE_NAME);
		return -ENOENT;
	}

	if (dmi) {
		pmode = S_IFREG | S_IRUGO;
		proc_dmi = create_proc_read_entry("dmi", pmode, omnibook_proc_root, omnibook_proc_dmi, NULL);
		if (! proc_dmi) {
			printk(KERN_ERR "%s: Unable to create /proc/%s/dmi.\n", MODULE_NAME, MODULE_NAME);
			return -ENOENT;
		}
	}

	if (ac)
		retval = omnibook_ac_init();
	else
		retval = 0;
	if (retval)
		return retval;
	ac = retval;

	if (battery)
		retval = omnibook_battery_init();
	else
		retval = 0;
	if (retval)
		return retval;
	battery = retval;

	/* /proc/apm emulation needs to read battery and AC adapter status */
	if ((ac == 0) && (battery == 0) && apmemu) {
		retval = omnibook_apmemu_init();
		if (retval)
			return retval;
	}

	if (blank)
		retval = omnibook_console_blank_init();
	else
		retval = 0;
	if (retval)
		return retval;

	if (display)
		retval = omnibook_display_init();
	else
		retval = 0;
	if (retval)
		return retval;

	if (dock)
		retval = omnibook_dock_init();
	else
		retval = 0;
	if (retval)
		return retval;

	if (fan)
		retval = omnibook_fan_init();
	else
		retval = 0;
	if (retval)
		return retval;

	if (fan_policy)
		retval = omnibook_fan_policy_init();
	else
		retval = 0;
	if (retval)
		return retval;

	if (key_polling)
		retval = omnibook_key_polling_init();
	else
		retval = 0;
	if (retval)
		return retval;

	if (lcd)
		retval = omnibook_brightness_init();
	else
		retval = 0;
	if (retval)
		return retval;

	if (onetouch)
		retval = omnibook_onetouch_init();
	else
		retval = 0;
	if (retval)
		return retval;

	if (temperature)
		retval = omnibook_temperature_init();
	else
		retval = 0;
	if (retval)
		return retval;

	if (touchpad)
		retval = omnibook_touchpad_init();
	else
		retval = 0;
	if (retval)
		return retval;

	return 0;
}

static void __exit omnibook_cleanup(void)
{
	omnibook_ac_cleanup();
	omnibook_apmemu_cleanup();
	omnibook_battery_cleanup();
	omnibook_brightness_cleanup();
	omnibook_console_blank_cleanup();
	omnibook_display_cleanup();
	omnibook_dock_cleanup();
	omnibook_fan_cleanup();
	omnibook_fan_policy_cleanup();
	omnibook_key_polling_cleanup();
	omnibook_onetouch_cleanup();
	omnibook_temperature_cleanup();
	omnibook_touchpad_cleanup();
	if (proc_dmi)
		remove_proc_entry("dmi", omnibook_proc_root);
	if (omnibook_proc_root)
		remove_proc_entry("omnibook", NULL);
	printk(KERN_INFO "%s: module is unloaded.\n", MODULE_NAME);
}

module_init(omnibook_init);
module_exit(omnibook_cleanup);

EXPORT_SYMBOL(omnibook_ectype);
EXPORT_SYMBOL(omnibook_proc_root);

MODULE_AUTHOR("Sos Pter <sp@osb.hu>");
MODULE_DESCRIPTION("Kernel interface for HP OmniBook, HP Pavilion, Toshiba Satellite and Compal ACL00 laptops");
MODULE_LICENSE("GPL");
MODULE_PARM(ectype, "i");
MODULE_PARM(ac, "i");
MODULE_PARM(apmemu, "i");
MODULE_PARM(battery, "i");
MODULE_PARM(blank, "i");
MODULE_PARM(display, "i");
MODULE_PARM(dock, "i");
MODULE_PARM(dmi, "i");
MODULE_PARM(fan, "i");
MODULE_PARM(fan_policy, "i");
MODULE_PARM(key_polling, "i");
MODULE_PARM(lcd, "i");
MODULE_PARM(onetouch, "i");
MODULE_PARM(temperature, "i");
MODULE_PARM(touchpad, "i");
MODULE_PARM(user, "i");
MODULE_PARM_DESC(ectype, "Type of embedded controller firmware");
MODULE_PARM_DESC(ac, "Use 0 to disable, 1 to enable AC adapter status monitoring");
MODULE_PARM_DESC(apmemu, "Use 0 to disable, 1 to enable /proc/apm emulation");
MODULE_PARM_DESC(battery, "Use 0 to disable, 1 to enable battery status monitoring");
MODULE_PARM_DESC(blank, "Use 0 to disable, 1 to enable lcd console blanking");
MODULE_PARM_DESC(display, "Use 0 to disable, 1 to enable display status handling");
MODULE_PARM_DESC(dock, "Use 0 to disable, 1 to enable docking station support");
MODULE_PARM_DESC(dmi, "Use 0 to disable, 1 to enable /proc/omnibook/dmi");
MODULE_PARM_DESC(fan, "Use 0 to disable, 1 to enable fan status monitor and control");
MODULE_PARM_DESC(fan_policy, "Use 0 to disable, 1 to enable fan control policy support");
MODULE_PARM_DESC(key_polling, "Use 0 to disable, 1 to enable to scancode emulation for volume keys");
MODULE_PARM_DESC(lcd, "Use 0 to disable, 1 to enable to LCD brightness support");
MODULE_PARM_DESC(onetouch, "Use 0 to disable, 1 to enable onetouch handling");
MODULE_PARM_DESC(temperature, "Use 0 to disable, 1 to enable thermal status and policy support");
MODULE_PARM_DESC(touchpad, "Use 0 to disable, 1 to enable touchpad handling");
MODULE_PARM_DESC(user, "Use 0 to disable, 1 to enable users to set parameters");

/* End of file */
