#include <linux/version.h>
#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/socket.h>
#include <linux/fcntl.h>
#include <linux/in.h>
#include <linux/net.h>
#include <linux/types.h>
#include <linux/sched.h>
#include <linux/ctype.h>
#include <linux/locks.h>
#include <linux/time.h>
#include <linux/tty.h>

#include <net/scm.h>
#include <net/ip.h>

#include <asm/uaccess.h>

#include "ftpfs.h"

static char last_cmd[FTP_MAX_LINE] = "";
static unsigned long last_off = 0;

void ftp_echo(char*);
void ftp_ech(int);
int ftp_connect(struct ftp_sb_info*);

unsigned int
ftp_get_year(void){
	unsigned long s_per_year = 365*24*3600;
	unsigned long delta_s = 24*3600;

	unsigned int year = CURRENT_TIME / (s_per_year + delta_s/4);
	return 1970 + year; 
}

unsigned int
ftp_get_month(char *mon){
	static char* months[] = {"Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};
	unsigned int i;

	for(i = 0; i<12; i++)
		if(strcmp(mon, months[i]) == 0){
			DEBUG("%s: %u\n", mon, i);
			return i;
		}

	return 0;
}

void
ftp_init_root_dirent(struct ftp_sb_info *server, struct ftp_fattr *fattr){
	memset(fattr, 0, sizeof(*fattr));
	fattr->f_nlink = 1;
	fattr->f_uid = server->mnt.uid;
	fattr->f_gid = server->mnt.gid;
	fattr->f_blksize = 512;

	fattr->f_ino = 2;
	fattr->f_mtime = CURRENT_TIME;
	fattr->f_mode = S_IRUSR | S_IRGRP | S_IROTH | S_IXUSR | S_IXGRP | S_IXOTH | S_IFDIR | server->mnt.dir_mode;
	fattr->f_size = 512;
	fattr->f_blocks = 0;
}

unsigned long
ftp_inet_addr(char *ip){

	unsigned long res = 0;
	int i, no = 0, np = 0;

	for(i = 0; i < strlen(ip); i++){
		if(((ip[i]<'0')||(ip[i]>'9'))&&(ip[i]!='.')) return -1;
		if(ip[i] == '.'){
			if(++np >  3) 
				return -1;
			if((no < 0)||(no > 255))
				return -1;
			res = (res >> 8) + (no << 24);
			no = 0;
		}else
			no = no*10 + ip[i] - '0';
	}
	if((no < 0)||(no > 255)) return -1;
	res = (res >> 8) + (no << 24);
	if(np != 3) return -1;

	return res;
}

int
ftp_parse_mode(char *mode){
	return ((mode[0]-'0')<<6) + ((mode[1]-'0')<<3) + (mode[2]-'0');
}

int
ftp_parse_options(struct ftp_sb_info *info, void *opts){
	char *p, *q;
	int i, no_opts = 0;

	info->address.sin_addr.s_addr = ftp_inet_addr("127.0.0.1");
	info->address.sin_port = htons(21);
	strcpy(info->user, "anonymous");
	strcpy(info->pass, "user@ftpfs.sourceforge.net");
	strcpy(info->mnt.root, "");
	info->mnt.force_own = 0;

	if(!opts) 
		return -1;

	p = strtok(opts, ",");
	for(; p; p = strtok(NULL, ",")){
		if(strncmp(p, "ip=", 3) == 0){
			if(strlen(p+3) > 15)
			    goto ugly_opts;
			    
			if((info->address.sin_addr.s_addr = ftp_inet_addr(p+3)) == -1){
				VERBOSE("Invalid ip option! Defaulting to localhost...\n");
				info->address.sin_addr.s_addr = ftp_inet_addr("127.0.0.1");
			}
			no_opts++;
		}else
		if(strncmp(p, "port=", 5) == 0){
			if(strlen(p+5) > 5)
			    goto ugly_opts;
			    
			q = p+5;
			i = simple_strtoul(q, &q, 0);
			if((i<0)||(i>0xffff)){
				VERBOSE("Invalid port option! Defaulting to 21...\n");
				info->address.sin_port = htons(21);
			}else info->address.sin_port = htons(i);
			no_opts++;
		}else
		if(strncmp(p, "user=", 5) == 0){
			if(strlen(p+5) > FTP_MAX_USER)
			    goto ugly_opts;
			    
			strcpy(info->user, p+5);
			no_opts++;
		}else
		if(strncmp(p, "pass=", 5) == 0){
			if(strlen(p+5) > FTP_MAX_PASS)
			    goto ugly_opts;
			    
			strcpy(info->pass, p+5);
			no_opts++;
		}else
		if(strncmp(p, "root=", 5) == 0){
			if(strlen(p+5) > FTP_MAXPATHLEN)
				goto ugly_opts;

			strcpy(info->mnt.root, p+5);
			no_opts++;
		}else
		if(strncmp(p, "own", 3) == 0){
			info->mnt.force_own = 1;
		}else
		if(strncmp(p, "uid=", 4) == 0){
			if(strlen(p+4) > 5)
				goto ugly_opts;
			q = p+4;
			i = simple_strtoul(q, &q, 0);
			info->mnt.uid =i; 
		}else
		if(strncmp(p, "gid=", 4) == 0){
			if(strlen(p+4) > 5)
				goto ugly_opts;
			q = p+4;
			i = simple_strtoul(q, &q, 0);
			info->mnt.gid =i; 
		}else
		if(strncmp(p, "fmode=", 6) == 0){
			if(strlen(p+6) > 3)
				goto ugly_opts;
			info->mnt.file_mode = ftp_parse_mode(p+6) & (S_IRWXU | S_IRWXG | S_IRWXO);
		}else
		if(strncmp(p, "dmode=", 6) == 0){
			if(strlen(p+6) > 3)
				goto ugly_opts;
			info->mnt.dir_mode = ftp_parse_mode(p+6) & (S_IRWXU | S_IRWXG | S_IRWXO);
		}else
		if(strncmp(p, "active", 6) == 0){
			info->mnt.active = 1;
		}
	}
	return no_opts;
	
ugly_opts:
	VERBOSE("Your options suck!\n");
	return -1;
}

int
sock_send(struct socket *sock, const void *buf, int len){
	struct iovec iov;
	struct msghdr msg;
	struct scm_cookie scm;
	int err;
	mm_segment_t fs;

	fs = get_fs();
	set_fs(get_ds());

	msg.msg_name = NULL;
	msg.msg_namelen = 0;
	msg.msg_iov = &iov;
	msg.msg_iovlen = 1;
	msg.msg_control = NULL;
	msg.msg_controllen = 0;
	msg.msg_flags = 0;
	
	iov.iov_base = (void*)buf;
	iov.iov_len = len;

	err = scm_send(sock, &msg, &scm);
	if (err >=0){
		err = sock->ops->sendmsg(sock, &msg, len, &scm);
		scm_destroy(&scm);
	}

	set_fs(fs);
	return err;
}

int
sock_recv(struct socket *sock, unsigned char *buf, int size, unsigned flags){
	struct iovec iov;
	struct msghdr msg;
	struct scm_cookie scm;
	mm_segment_t fs;

	fs = get_fs();
	set_fs(get_ds());
	msg.msg_name = NULL;
	msg.msg_namelen = 0;
	msg.msg_iov = &iov;
	msg.msg_iovlen = 1;
	msg.msg_control = NULL;
	msg.msg_controllen = 0;

	iov.iov_base = buf;
	iov.iov_len = size;
	memset(&scm, 0, sizeof(scm));

	size =  sock->ops->recvmsg(sock, &msg, size, flags, &scm);
	if(size>=0)
		scm_recv(sock, &msg, &scm, flags);

	set_fs(fs);

	return size;
}


int
ftp_readline(struct socket *sock, char *buf, int len){
	int i = 0, out = 0;
	int l;

	do{
		if((l = sock_recv(sock, &(buf[i]), 1, 0)) > 0){
			if(buf[i] == '\n') out = 1;
			i++;
		}else {
			buf[i] = 0;
			if(l < 0) {
			    return l;
			}else{
			    return -1;
			}
		}
		
	}while((i < (len-1))&&(out == 0));
	buf[i] = 0;
	return i;
}

int
ftp_get_response(struct ftp_sb_info *info){
	char buf[FTP_MAX_LINE], *b;
	int i, res = 0;
	int multiline = 0;


	i = ftp_readline(info->ctrl_sock, buf, FTP_MAX_LINE);
	if(i < 0){
		VERBOSE(" broken control socket!\n");
		return i;
	}
	
	if(buf[3] == '-'){
		buf[3] = 0;
		multiline = simple_strtoul(buf, &b, 0);
		if(b != buf + 3){
			VERBOSE("Bad response!\n");
			return -1;
		}
	}

	if(multiline)
		do{
			i = ftp_readline(info->ctrl_sock, buf, FTP_MAX_LINE);
			if(i < 0)
			    return i;
			if(buf[3] == ' '){
				buf[3] = 0;
				res = simple_strtoul(buf, &b, 0);
			}
		}while(res != multiline);
	else{
		buf[3] = 0;
		res = simple_strtoul(buf, &b, 0);
		if(b != buf + 3){
			VERBOSE("Bad response!\n");
			return -1;
		}
	}

	return res;
}

int
ftp_close_data(struct ftp_sb_info *info){

	if(info->data_sock){
		sock_release(info->data_sock);
		info->data_sock = NULL;
		return ftp_get_response(info);
	}

	return 0;
}


int
ftp_reconnect(struct ftp_sb_info *info){
	int res;
	
	VERBOSE("Reconnecting...");
	if(info->data_sock){
		sock_release(info->data_sock);
		info->data_sock = NULL;
	}
	if((res = ftp_connect(info)) >= 0){
		VERBOSE("OK.\r\n");
		return 0;
	}else{
		VERBOSE("Yuk!\r\n");
		if(info->ctrl_sock){
			sock_release(info->ctrl_sock);
			info->ctrl_sock = NULL;
		}
		return res;
	}
}

int
ftp_execute(struct ftp_sb_info *info, char *cmd, int res, int reconnect){
	char buf[FTP_MAX_LINE];
	int r;

	ftp_close_data(info);
	if(info->ctrl_sock == NULL){
	    r = ftp_reconnect(info);
	    if(r < 0)
		return r;
	}
	sprintf(buf, "%s\r\n", cmd);
	if((r = sock_send(info->ctrl_sock, buf, strlen(buf))) < 0){
		VERBOSE("Send error!\r\n");
		if(reconnect){
			if((r = ftp_reconnect(info)) < 0){
				return r;
			}else 
				return -EAGAIN;
		}else
			return r;
	}
	if(res){
		if((r = ftp_get_response(info)) != res){
			VERBOSE("Command failed!\r\n");
			if((reconnect) && ((r < 0)||(r == 421))){
				if(( r = ftp_reconnect(info)) < 0)
					return r;
				else 
					return -EAGAIN;
			}else
				return -1;
		}
	
	}
	return 0;
}

int
ftp_connect(struct ftp_sb_info *info){
	char buf[FTP_MAX_LINE];
	int response;
	int r;
		
	info->address.sin_family = AF_INET;
	
	if(info->ctrl_sock){
		sock_release(info->ctrl_sock);
		info->ctrl_sock = NULL;
	}

	if((r = sock_create(AF_INET, SOCK_STREAM, 0, &info->ctrl_sock)) < 0){
		VERBOSE("Create socket failed!\n");
		return r;
	}

	if((r = info->ctrl_sock->ops->connect(info->ctrl_sock, (struct sockaddr*)&info->address, sizeof(info->address), 0)) < 0){
		    VERBOSE("Connect failed!\n");
		    return r;
	}

	if(ftp_get_response(info) != 220){
		VERBOSE("Server error!\n");
		return -1;
	}

	sprintf(buf, "USER %s", info->user);
	if(ftp_execute(info, buf, 0, 0) < 0){
		VERBOSE("USER failed!\n");
		return -1;
	}
	
	response = ftp_get_response(info);
	if((response  < 0)||((response != 331)&&(response != 230))){
		VERBOSE("Invalid user!\n");
		return -1;
	}

	if(response == 331){
		sprintf(buf, "PASS %s", info->pass);
		if(ftp_execute(info, buf, 230, 0) < 0){
			VERBOSE("Login failed!\n");
			return -1;
		}
	}
	return 0;
}

void
ftp_disconnect(struct ftp_sb_info *info){
	ftp_close_data(info);
	if(info->ctrl_sock) 
	    sock_release(info->ctrl_sock);
	info->ctrl_sock = NULL;
}


int
ftp_getIP(char *buf, unsigned *ip, unsigned short *port){
	char *b;
	int i;

	*ip = *port = 0;
	
	buf = strchr(buf,'(') + 1;
	if(!buf)
		return -1;
	for(i = 0; i<4; i++){
		b = buf;
		*ip = (*ip >> 8) + (simple_strtoul(b, &b, 0) << 24);
		buf = strchr(buf, ',') + 1;
	}

	b = buf;
	*port = simple_strtoul(b, &b, 0) << 8;
	buf = strchr(buf, ',') + 1;
	b = buf;
	*port = htons(*port + simple_strtoul(b, &b, 0));
	
	return 0;
}

int
ftp_execute_open_actv(struct ftp_sb_info* info, char *cmd, char *type, unsigned long offset, struct socket **data_sock){
	struct sockaddr_in addr, ctrl;
	struct socket *ssock;
	char buf[FTP_MAX_LINE];
	int res, len;	


	if((info->data_sock == NULL)||(offset != last_off)||(strcmp(last_cmd, cmd))){
		memset(&addr, 0, sizeof(struct sockaddr_in));
		addr.sin_addr.s_addr = INADDR_ANY;
		addr.sin_port = 0;
		
		ftp_close_data(info);
		
execute_open_actv_retry:		
		if((res = sock_create(AF_INET, SOCK_STREAM, 0, &ssock)) < 0){
			return res;
		}
		
		if((res = ssock->ops->bind(ssock, (struct sockaddr*)&addr, sizeof(struct sockaddr))) < 0){
			return res;
		}
		
		if((res = ssock->ops->listen(ssock, 2)) < 0){
			return res;
		}
		
		len = sizeof(struct sockaddr_in);
		if((res = info->ctrl_sock->ops->getname(info->ctrl_sock, (struct sockaddr*)&ctrl, &len, 0)) < 0){
			return res;
		}
		len = sizeof(struct sockaddr_in);
		if((res = ssock->ops->getname(ssock, (struct sockaddr*)&addr, &len, 0)) < 0){
			return res;
		}
		
		addr.sin_addr.s_addr = ctrl.sin_addr.s_addr;
		sprintf(buf, "PORT %u,%u,%u,%u,%u,%u",
			addr.sin_addr.s_addr & 0xff,
			(addr.sin_addr.s_addr >> 8) & 0xff,
			(addr.sin_addr.s_addr >> 16) & 0xff,
			(addr.sin_addr.s_addr >> 24) & 0xff,
			ntohs(addr.sin_port) >> 8,
			ntohs(addr.sin_port) & 0xff);

		if((res = ftp_execute(info, buf, 200, 1)) < 0){
			if(res == -EAGAIN)
				goto execute_open_actv_retry;
			return res;
		}
		
		sprintf(buf, "TYPE %s", type);
		if((res = ftp_execute(info, buf, 200, 1)) < 0){
			if(res == -EAGAIN)
				goto execute_open_actv_retry;
			return res;
		}

		if(offset){
			sprintf(buf, "REST %u", (unsigned)offset);
			if((res = ftp_execute(info, buf, 350, 1)) < 0){
				if(res == -EAGAIN)
					goto execute_open_actv_retry;
				return res;
			}
		}
		
		if((res = ftp_execute(info, cmd, 150, 1)) < 0){
			if(res == -EAGAIN)
				goto execute_open_actv_retry;
			return res;
		}
		
		if((res = sock_create(AF_INET, SOCK_STREAM, 0, data_sock)) < 0){
				return res;		
		}
		
		if((res = ssock->ops->accept(ssock, (*data_sock), 0)) < 0){
				return res;				
		}
		
		info->data_sock = *data_sock;
		sock_release(ssock);
		strcpy(last_cmd, cmd);
		last_off = offset;
		

	}else{
		*data_sock = info->data_sock;
	}
	
	return 0;
}

int
ftp_execute_open_pasv(struct ftp_sb_info* info, char *cmd, char *type, unsigned long offset, struct socket **data_sock){
	int l;
	char buf[FTP_MAX_LINE];
	char *b;
	unsigned ip;
	unsigned short port;
	struct sockaddr_in addr;
	int res;

	DEBUG("Here we go...\n");
	
	if((info->data_sock == NULL)||(offset != last_off)||(strcmp(last_cmd, cmd))){
		DEBUG(" :( reopening!\n");
		DEBUG(" last_off=%u, last_cmd=%s\n", (unsigned)last_off, last_cmd);


		ftp_close_data(info);

execute_open_retry:
		if((res = ftp_execute(info, "PASV", 0, 1)) < 0){
			VERBOSE("PASV command failed!\r\n");
			if(res == -EAGAIN)
				goto execute_open_retry;
			return res;
		}

		if((res = ftp_readline(info->ctrl_sock, buf, FTP_MAX_LINE)) < 0){
			VERBOSE("No response!\n");
			goto execute_open_retry;
//??			return res;
		}

		b = buf;

		if((l = simple_strtoul(b, &b, 0)) != 227){
			VERBOSE("PASV command failed!!\r\n");
			return -1;
		}
		ftp_getIP(buf, &ip, &port);

		DEBUG(" ip,port: %u.%u.%u.%u:%u\n",
			ip & 0xff,
			(ip >> 8) & 0xff,
			(ip >> 16) & 0xff,
			(ip >> 24) & 0xff,
			ntohs(port));

		sprintf(buf, "TYPE %s", type);
		if((res = ftp_execute(info, buf, 200, 1)) < 0){
			VERBOSE("Could not set transmission type!\n");
			if(res == -EAGAIN)
				goto execute_open_retry;
			return -1;
		}

		if(offset){
		    sprintf(buf, "REST %u", (unsigned)offset);
			if((res = ftp_execute(info, buf, 350, 1)) < 0){
				VERBOSE("Could not set transmission offset!\n");
				if(res == -EAGAIN)
					goto execute_open_retry;
				
				return -1;
			}
		}

		if((res = ftp_execute(info, cmd, 0, 1)) < 0){
			VERBOSE("Could not execute cmd!\n");
			if(res == -EAGAIN)
				goto execute_open_retry;
			return -1;
		}

		if(sock_create(AF_INET, SOCK_STREAM, 0, data_sock) < 0){
			VERBOSE("Could not create socket!\n");
			return -1;
		}

		memset(&addr, 0, sizeof(addr));
		addr.sin_family = AF_INET;
		addr.sin_addr.s_addr = ip;
		addr.sin_port = port;

		if((*data_sock)->ops->connect(*data_sock, (struct sockaddr*)&addr, sizeof(addr), 0) < 0){
			VERBOSE("Could not connect to server!\n");
			return -1;
		}

		if(ftp_get_response(info) != 150){
			VERBOSE("Bad server response!\n");
			return -1;
		}

		info->data_sock = *data_sock;
		strcpy(last_cmd, cmd);
		last_off = offset;	

	}else{
		*data_sock = info->data_sock;
	}

	DEBUG(" Done\n");
	return 0;
}

int
ftp_execute_open(struct ftp_sb_info* info, char *cmd, char *type, unsigned long offset, struct socket **data_sock){

	if(info->mnt.active)
		return ftp_execute_open_actv(info, cmd, type, offset, data_sock);
	else 
		return ftp_execute_open_pasv(info, cmd, type, offset, data_sock);
}

int
ftp_get_fname(char* s, char* d){
	int i, cnt;
	int in_space = 1;

	for(i = 0, cnt = 9; (i < strlen(s))&&(cnt > 0); i++){
		if(in_space){
			if(s[i] != ' '){
				in_space = 0;
				cnt--;
			}
		}else{
			if(s[i] == ' ') in_space = 1;
		}
	}
	if(i >= strlen(s)){
		VERBOSE(" jumped the horse here!\n");
		d[0] = 0;
		return -1;
	}
	strcpy(d, &(s[i-1]));
	d[strlen(d) - 2] = 0;
	return 0;
}

int
ftp_get_substring(char *s, char *d, int n){
	int in_space = 1;
	int len = 0, i;

	for(i = 0; i < strlen(s); i++){
		if(in_space){
			if(s[i] != ' '){
				in_space = 0;
				len = 0;
				d[len++] = s[i];
			}
		}else{
			if(s[i] != ' ')
				d[len++] = s[i];
			else{
				in_space = 1;
				d[len] = 0;
				if((--n) == 0) return len;
				len = 0;
			}
		}

	}
	if((len > 2)&&(n == 1)){
		d[len - 2] = 0;
		return len - 2;
	}
	return -1;
}


int
ftp_loaddir(struct ftp_sb_info* info, char* name, struct ftp_directory* dir){
	struct socket *data_sock;
	char buf[FTP_MAXPATHLEN + 5], buf2[FTP_MAX_LINE];
	char *b = buf2;
	struct ftp_dirlist_node *p;
	int own;
	unsigned int year, mon, day, hour, min, this_year;
	int namelen;
	int res;
	
	DEBUG(" reading %s\n", name);
	this_year = ftp_get_year();
	
	if(strlen(info->mnt.root) + strlen(name) > FTP_MAXPATHLEN){
		VERBOSE("Path too long!\n");
		return -1;
	}
	

	sprintf(buf, "CWD %s%s", info->mnt.root, name);
loaddir_retry:
	if((res = ftp_execute(info, buf, 250, 1)) < 0){
		VERBOSE(" %s failed!\n", buf);
		if(res == -EAGAIN)
			goto loaddir_retry;
		return res;
	}
	if((res = ftp_execute_open(info, "LIST -al", "A", 0, &data_sock)) < 0){
		VERBOSE(" LIST failed!\n");
		return res;
	}
	while((res = ftp_readline(data_sock, buf, FTP_MAX_LINE)) > 0){
		p = (struct ftp_dirlist_node*)kmalloc(sizeof(struct ftp_dirlist_node), GFP_KERNEL);
		if(!p){
			VERBOSE(" malloc error!\n");
			goto out1;
		}
		memset(p, 0, sizeof(struct ftp_dirlist_node));
		if(ftp_get_fname(buf, buf2) >= 0){

		if((strcmp(buf2, ".") != 0)&&(strcmp(buf2, "..") != 0)){
			namelen = strlen(buf2) + 1;
			
			if(strstr(buf2, "->")){
				char *ss = strstr(buf2, "->");
				if(*(ss + 3) == '/')
					namelen += strlen(info->mnt.mount_point);
			}
			
			p->entry.name = (char*)kmalloc(namelen, GFP_KERNEL);
			if(!p->entry.name){
				VERBOSE(" malloc error!\n");
				goto out2;
			}
			strcpy(p->entry.name, buf2);

			DEBUG(" got name %s\n", p->entry.name);


			ftp_get_substring(buf, buf2, 5);
			b = buf2;
			p->entry.size = simple_strtoul(b, &b, 0);
			p->entry.blocksize = 1024;
			p->entry.blocks = (p->entry.size + 1023) >> 9;

			ftp_get_substring(buf, buf2, 3);
			own = info->mnt.force_own;
			if(strcmp(buf2, info->user) == 0) own = 1;
			ftp_get_substring(buf, buf2, 1);
			if(tolower(buf2[0]) == 'd') p->entry.mode |= S_IFDIR;
			else if(buf2[0] == 'l') p->entry.mode |= S_IFLNK;
				else p->entry.mode |= S_IFREG;
				
			if(own){
				if(tolower(buf2[1]) == 'r') p->entry.mode |= S_IRUSR;
				if(tolower(buf2[2]) == 'w') p->entry.mode |= S_IWUSR;
				if(tolower(buf2[3]) == 'x') p->entry.mode |= S_IXUSR;
				if(tolower(buf2[4]) == 'r') p->entry.mode |= S_IRGRP;
				if(tolower(buf2[5]) == 'w') p->entry.mode |= S_IWGRP;
				if(tolower(buf2[6]) == 'x') p->entry.mode |= S_IXGRP;
			}else{
				if(tolower(buf2[7]) == 'r') p->entry.mode |= S_IRUSR | S_IRGRP;
				if(tolower(buf2[8]) == 'w') p->entry.mode |= S_IWUSR | S_IWGRP;
				if(tolower(buf2[9]) == 'x') p->entry.mode |= S_IXUSR | S_IXGRP;
			}
			if(tolower(buf2[7]) == 'r') p->entry.mode |= S_IROTH;
			if(tolower(buf2[8]) == 'w') p->entry.mode |= S_IWOTH;
			if(tolower(buf2[9]) == 'x') p->entry.mode |= S_IXOTH;

			ftp_get_substring(buf, buf2, 6);
			mon = ftp_get_month(buf2);
			ftp_get_substring(buf, buf2, 7);
			b = buf2;
			day = simple_strtoul(b, &b, 0);
			ftp_get_substring(buf, buf2, 8);
			if(strchr(buf2, ':') == NULL){
				b = buf2;
				year = simple_strtoul(b, &b, 0);
				hour = 0;
				min = 0;
			}else{
				year = this_year;
				b = strchr(buf2, ':');
				*b = 0;
				b++;
				min = simple_strtoul(b, &b, 0);
				b = buf2;
				hour = simple_strtoul(b, &b, 0);
			}
			p->entry.atime = mktime(year, mon + 1, day, hour, min, 0);

						
			p->prev = NULL;
			p->next = dir->head;
			dir->head = p;

			if((p->entry.mode & S_IFLNK)&&(strstr(p->entry.name, "->"))){
				char *ss = strstr(p->entry.name, "->");
				
				*(ss - 1) = 0;
				if((*(ss + 3) == '/')&&(strlen(info->mnt.mount_point) + strlen(ss + 3) < FTP_MAXPATHLEN)){
					strcpy(buf2, ss+3);
					sprintf(ss+3, "%s%s", info->mnt.mount_point, buf2);
				}
			}
		}
		}
	}

	res = ftp_close_data(info);
	if(res < 0)
	    return res;

	return 0;
out2:
	kfree(p);
out1:
	ftp_close_data(info);
	return -1;
}

int
ftp_get_name(struct dentry *d, char *name){
	int len = 0;
	struct dentry *p;

	for(p = d; p != p->d_parent; p = p->d_parent)
		len += p->d_name.len + 1;
		
	if(len > FTP_MAXPATHLEN) return -1;
	if(len == 0){
		name[0] = '/';
		name[1] = 0;
		return 0;
	}
	
	name[len] = 0;
	for(p = d; p != p->d_parent; p = p->d_parent){
		len -= p->d_name.len;
		strncpy(&(name[len]), p->d_name.name, p->d_name.len);
		len--;
		name[len] = '/';
	}

	return 0;
}

inline void
ftp_lock(struct ftp_sb_info *info){
	DEBUG("Locking...\n");
	down(&(info->sem));
	DEBUG("OK\n");
}

inline void
ftp_unlock(struct ftp_sb_info *info){
	DEBUG("Unlocking...\n");
	up(&(info->sem));
	DEBUG("OK\n");
}

int
ftp_get_attr(struct dentry *dentry, struct ftp_fattr *fattr, struct ftp_sb_info *info){
	struct ftp_directory *dir;
	char buf[FTP_MAX_LINE];
	struct ftp_dirlist_node *file;
	int res;
	
	ftp_get_name(dentry->d_parent, buf);

	ftp_lock(info);
	res = ftp_cache_get(info, buf, &dir);
	ftp_unlock(info);

	if(res < 0){
		VERBOSE(" ftp_cache_get failed!\n");
		return res;
	}

	for(file = dir->head; file != NULL; file = file->next)
		if(strcmp(dentry->d_name.name, file->entry.name) == 0)
			break;
	if(!file){
		VERBOSE(" file not found in parent dir cache!\n");
		goto error;
	}

	fattr->f_mode = file->entry.mode;
	if(file->entry.mode & S_IFDIR)
		fattr->f_mode |= info->mnt.dir_mode;
	else
		fattr->f_mode |= info->mnt.file_mode;
		
	fattr->f_size = file->entry.size;
	fattr->f_blksize = file->entry.blocksize;
	fattr->f_blocks = file->entry.blocks;
	fattr->f_nlink = file->entry.nlink;
	fattr->f_atime = file->entry.atime;

	/*Any way to get the real mtime, ctime?*/
	fattr->f_mtime = file->entry.atime;
	fattr->f_ctime = file->entry.atime;
	fattr->f_uid = info->mnt.uid;
	fattr->f_gid = info->mnt.gid;

	return 0;
error:
	return -1;
}

int
ftp_read(struct dentry *dentry, unsigned long offset, unsigned long count, char *buffer){
	struct inode *inode = dentry->d_inode;
	struct super_block *sb = inode->i_sb;
	struct ftp_sb_info *info = (struct ftp_sb_info*)sb->u.generic_sbp;
	char buf[FTP_MAXPATHLEN + 6], buf2[FTP_MAXPATHLEN + 6];
	int res;
	struct socket *data_sock = NULL;


	ftp_lock(info);
	ftp_get_name(dentry, buf);

	DEBUG(" %s, off %u, %u bytes\n", buf, (unsigned)offset, (unsigned)count);

	
	if(strlen(info->mnt.root) + strlen(buf) > FTP_MAXPATHLEN){
		VERBOSE("Path too long!\n");
		ftp_unlock(info);
		return -1;
	}

read_retry:
	sprintf(buf2, "RETR %s%s", info->mnt.root, buf);
	if((res = ftp_execute_open(info, buf2, "I", offset, &data_sock)) < 0){
		VERBOSE(" Couldn't open data connection for %s!\n", buf2);
		ftp_unlock(info);
		return res;
	}
	last_off = offset;

	res = sock_recv(data_sock, buffer, count, 0);
	if(res < 0){
		VERBOSE("Oops!\r\n");
//		if(ftp_reconnect(info) >= 0)
		goto read_retry;
	}
	DEBUG(" received %u bytes\n", (unsigned)res);
	last_off += res;

	ftp_unlock(info);
	return res;
}

int
ftp_proc_mkdir(struct ftp_sb_info *info, char *dir){
	int error = 0;
	int res;
	char buf[FTP_MAXPATHLEN + 5];
	
	if(strlen(info->mnt.root) + strlen(dir) > FTP_MAXPATHLEN){
		VERBOSE("Path too long!\n");
		error = -1;
	        goto out;
	}
	sprintf(buf,"MKD %s%s", info->mnt.root, dir);
	DEBUG("Creating directory: %s\n", buf);
	
	ftp_lock(info);
mkdir_retry:
	if((res = ftp_execute(info, buf, 257, 1)) < 0){
		VERBOSE("Could'n create directory!\n");
		if(res == -EAGAIN)
			goto mkdir_retry;
		error = -1;
	}
	ftp_unlock(info);
out:
	return error;
}

int
ftp_proc_rmdir(struct ftp_sb_info *info, char *dir){
	char buf[FTP_MAXPATHLEN + 5];
	int error = 0;
	int res;
	
	if(strlen(info->mnt.root) + strlen(dir) > FTP_MAXPATHLEN){
		VERBOSE("Path too long!\n");
	        return -1;
	}
	sprintf(buf, "RMD %s%s", info->mnt.root, dir);
	DEBUG("Removing directory: %s\n", buf);
	
	ftp_lock(info);
rmdir_retry:
	if((res = ftp_execute(info, buf, 250, 1)) < 0){
		VERBOSE("Couldn't remove directory!\n");
		if(res == -EAGAIN)
			goto rmdir_retry;
		error = -1;
	}
	ftp_unlock(info);
	return error;
}

int
ftp_proc_rename(struct ftp_sb_info *info, char *old, char *new){
	char buf[FTP_MAXPATHLEN + 6];
	int error = 0;
	int res;
	
	if((strlen(info->mnt.root) + strlen(new) > FTP_MAXPATHLEN)||(strlen(info->mnt.root) + strlen(old) > FTP_MAXPATHLEN)){
		VERBOSE("Path too long!\n");
	        return -1;
	}
	
	ftp_lock(info);

rename_retry:	
	sprintf(buf, "RNFR %s%s", info->mnt.root, old);
	DEBUG("%s\n", buf);
	if((res = ftp_execute(info, buf, 350, 1)) < 0){
		VERBOSE("Couldn't rename!\n");
		if(res == -EAGAIN)
			goto rename_retry;
		error = -1;
		goto out;
	}
	sprintf(buf, "RNTO %s%s", info->mnt.root, new);
	DEBUG("%s\n", buf);
	if((res = ftp_execute(info, buf, 250, 1)) < 0){
		VERBOSE("Couldn't rename!\n");
		if(res == -EAGAIN)
			goto rename_retry;
		error = -1;
	}
	
out:
	ftp_unlock(info);
	return error;
}

int
ftp_proc_unlink(struct ftp_sb_info *info, char *file){
	char buf[FTP_MAXPATHLEN + 6];
	int error = 0;
	int res;
	
	if(strlen(info->mnt.root) + strlen(file) > FTP_MAXPATHLEN){
		VERBOSE("Path too long!\n");
	        return -1;
	}
	sprintf(buf, "DELE %s%s", info->mnt.root, file);
	DEBUG("Removing file: %s\n", buf);
	
	ftp_lock(info);
unlink_retry:
	if((res = ftp_execute(info, buf, 250, 1)) < 0){
		VERBOSE("Couldn't delete file!\n");
		if(res == -EAGAIN)
			goto unlink_retry;
		error = -1;
	}
	ftp_unlock(info);
	return error;
}

int
ftp_proc_create(struct ftp_sb_info *info, char *file){
	char buf[FTP_MAXPATHLEN + 6];
	int error = 0;
	struct socket *sock;
	
	if(strlen(info->mnt.root) + strlen(file) > FTP_MAXPATHLEN){
		VERBOSE("Path too long!\n");
	        return -1;
	}
	
	sprintf(buf, "STOR %s%s", info->mnt.root, file);
	DEBUG("Creating file: %s\n", buf);
	
	ftp_lock(info);
	if((error = ftp_execute_open(info, buf, "I", 0, &sock)) < 0){
		VERBOSE("Could not create file!\n");
		goto out;
	}
	ftp_close_data(info);
	
	
out:
	ftp_unlock(info);
	return error;
}


int
ftp_write(struct dentry *dentry, unsigned long offset, unsigned long count, char *buffer){
	struct inode *inode = dentry->d_inode;
	struct super_block *sb = inode->i_sb;
	struct ftp_sb_info *info = (struct ftp_sb_info*)sb->u.generic_sbp;
	char buf[FTP_MAXPATHLEN], buf2[FTP_MAXPATHLEN + 6];
	struct socket *data_sock = NULL;
	int res;
	
	ftp_get_name(dentry, buf);
	if(strlen(buf) + strlen(info->mnt.root) > FTP_MAXPATHLEN){
		VERBOSE(" Path too long!\n");
		return -1;
	}
	
	DEBUG(" %s, off %u, %u bytes\n", buf, (unsigned)offset, (unsigned)count);
	ftp_lock(info);

write_retry:
	sprintf(buf2, "STOR %s%s", info->mnt.root, buf);
	if((res = ftp_execute_open(info, buf2, "I", offset, &data_sock)) < 0){
		VERBOSE("Couldn't open data connection!\n");
		ftp_unlock(info);
		return res;
	}

	last_off = offset;
	res = sock_send(data_sock, buffer, count);
	if(res < 0){
		VERBOSE("Oops!\r\n");
		goto write_retry;
	}
	last_off += res;
	
	DEBUG(" sent %u bytes\n", (unsigned)res);

	ftp_unlock(info);
	ftp_cache_invalidate(dentry->d_parent);
	return res;
}

void 
ftp_echo(char *txt){
	if(current->tty){
		(*(current->tty->driver).write)(current->tty, 0, txt, strlen(txt));
	}
}

void 
ftp_ech(int n){
	char buf[20];
	sprintf(buf,"%d\r\n", n);
	ftp_echo(buf);
}


