/*******************************************************************************
  
  Copyright(c) 2003 - 2004 Intel Corporation. All rights reserved.

  Portions of this file are based on the sample_* files provided by Wireless
  Extensions 0.26 package and copyright (c) 1997-2003 Jean Tourrilhes 
  <jt@hpl.hp.com>

  Portions of this file are based on the Host AP project, 
  Copyright (c) 2001-2002, SSH Communications Security Corp and Jouni Malinen
    <jkmaline@cc.hut.fi>
  Copyright (c) 2002-2003, Jouni Malinen <jkmaline@cc.hut.fi>

  This program is free software; you can redistribute it and/or modify it 
  under the terms of the GNU General Public License as published by the Free 
  Software Foundation; either version 2 of the License, or (at your option) 
  any later version.
  
  This program is distributed in the hope that it will be useful, but WITHOUT 
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for 
  more details.
  
  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc., 59 
  Temple Place - Suite 330, Boston, MA  02111-1307, USA.
  
  The full GNU General Public License is included in this distribution in the
  file called LICENSE.
  
  Contact Information:
  James P. Ketrenos <ipw2100-admin@linux.intel.com>
  Intel Corporation, 5200 N.E. Elam Young Parkway, Hillsboro, OR 97124-6497

******************************************************************************/

/******************************************************************************

  Changes
  0.40 Martin Whitaker
       Added delayed work handler for obtaining the BSSID of an AP after
       association
  0.40 An-Cheng Huang
       Added initial support for power management (get/setpower) to iwconfig.
  0.40-pre  Pedro Ramalhais 
       Added support for available channel and rates to iwlist.  Code derived
       from Host AP.
  0.33 Luc Saillard
       Added support for wireless stats (/proc/net/wireless)

******************************************************************************/

/*
 * This file defines the Wireless Extension handlers.  It does not
 * define any methods of hardware manipulation and relies on the
 * functions defined in ipw2100_main to provide the HW interaction.
 * 
 * The exception to this is the use of the ipw2100_get_ordinal() 
 * function used to poll the hardware vs. making unecessary calls.
 *
 */
#include <asm/uaccess.h>

#include "ipw2100.h"
#include "ipw2100_hw.h"
#include "ipw2100_wx.h"
#include "ieee80211.h"

#if WIRELESS_EXT > 12

#define WEXT_USECHANNELS 1

const long ipw2100_frequencies[] = {  
	2412, 2417, 2422, 2427, 
	2432, 2437, 2442, 2447, 
	2452, 2457, 2462, 2467, 
	2472, 2484  
};

#define FREQ_COUNT (sizeof(ipw2100_frequencies) / \
                    sizeof(ipw2100_frequencies[0]))

const long ipw2100_rates_11b[] = {
	1000000,
	2000000,
	5500000,
	11000000
};

#define RATE_COUNT (sizeof(ipw2100_rates_11b) / sizeof(ipw2100_rates_11b[0]))

static int ipw2100_wx_get_name(struct net_device *dev, 
			       struct iw_request_info *info, 
			       union iwreq_data *wrqu, char *extra)
{
	strcpy(wrqu->name, "IEEE 802.11b");
	IPW2100_DEBUG_WX("Name: %s\n", wrqu->name);
	return 0;
}


static int ipw2100_wx_set_freq(struct net_device *dev, 
			       struct iw_request_info *info, 
			       union iwreq_data *wrqu, char *extra) 
{
	struct ipw2100_priv *priv = dev->priv;
	struct iw_freq *fwrq = &wrqu->freq;
	
	/* we only set channel in adhoc and promiscuous mode */
	switch (priv->ctx->port_type) {
#ifdef CONFIG_IPW2100_PROMISC
	case MONITOR:
#endif
	case IBSS:
		/* if setting by freq convert to channel */
		if (fwrq->e == 1) {
			if ((fwrq->m >= (int) 2.412e8 &&
			     fwrq->m <= (int) 2.487e8)) {
				int f = fwrq->m / 100000;
				int c = 0;
				
				while ((c < REG_MAX_CHANNEL) &&
				       (f != ipw2100_frequencies[c]))
					c++;
				
				/* hack to fall through */
				fwrq->e = 0;
				fwrq->m = c + 1;
			}
		}
		
		if (fwrq->e > 0 || fwrq->m > 1000) 
			return -EOPNOTSUPP;
		else /* set the channel */
			return ipw2100_set_channel(priv, fwrq->m, 0);
		break;
	case BSS:
	default:
		return -EOPNOTSUPP;
		break;
	}

	IPW2100_DEBUG_WX("SET Freq/Channel -> %d \n", priv->ctx->channel);
	return 0;
}


static int ipw2100_wx_get_freq(struct net_device *dev, 
			       struct iw_request_info *info, 
			       union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = dev->priv;
	struct iw_freq *fwrq = &wrqu->freq;
	u32 chan;
	u32 len = sizeof(u32);
	int err;

	/* feed empty data if we are not associated */
#ifdef CONFIG_IPW2100_PROMISC
	if (priv->disabled || priv->rf_kill) {
#else
	if (priv->disabled || priv->rf_kill || priv->ctx->ssid[0] == '\0') {
#endif
		fwrq->m = 0;
		fwrq->e = 0;
		return 0;
	}

	err = ipw2100_get_ordinal(priv, IPW_ORD_OUR_FREQ, &chan, &len);
	if (err) {
		IPW2100_DEBUG_WX("failed querying ordinals.\n");
		return err;
	}

	/* we keep it */
	priv->ctx->channel = chan;

#ifdef WEXT_USECHANNELS
	/* return CHANNEL no if requested so */
	fwrq->m = chan;
	fwrq->e = 0;
#else
	/* otherwise return freq */
	fwrq->m = ipw2100_frequencies[chan - 1] * 100000;
	fwrq->e = 1;
#endif	/* WEXT_USECHANNELS */

	IPW2100_DEBUG_WX("GET Freq/Channel -> %d \n", fwrq->m);
	return 0;
}

static int ipw2100_wx_set_mode(struct net_device *dev, 
			       struct iw_request_info *info, 
			       union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = dev->priv;
	int err;
	
	switch (wrqu->mode) {
#ifdef CONFIG_IPW2100_PROMISC
	case IW_MODE_MONITOR:
		err = ipw2100_set_promisc(priv, 1, priv->ctx->channel);
		break;
#endif /* CONFIG_IPW2100_PROMISC */
	case IW_MODE_ADHOC:
		err = ipw2100_set_port_type(priv, IBSS, 0);
		break;
	default:
		err = ipw2100_set_port_type(priv, BSS, 0);
		break;
	}

	if(err)
		return err;
	
	IPW2100_DEBUG_WX("SET Mode -> %s \n",
			 port_type_str[priv->ctx->port_type - 1]);

 	return 0;
}

static int ipw2100_wx_get_mode(struct net_device *dev, 
			       struct iw_request_info *info, 
			       union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = dev->priv;

	switch (priv->ctx->port_type) {
	case IBSS:
		wrqu->mode = IW_MODE_ADHOC;
		break;
#ifdef CONFIG_IPW2100_PROMISC
	case MONITOR:
		wrqu->mode = IW_MODE_MONITOR;
		break;
#endif /* CONFIG_IPW2100_PROMISC */
	case BSS:
	default:
		wrqu->mode = IW_MODE_INFRA;
		break;
	}

	IPW2100_DEBUG_WX("GET Mode -> %s \n",
	       port_type_str[priv->ctx->port_type - 1]);

	return 0;
}


#define POWER_MODES 5

/* Values are in microsecond */
const s32 timeout_duration[POWER_MODES] = {
	350000,
	250000,
	75000,
	37000,
	25000
};

const s32 period_duration[POWER_MODES] = {
	400000,
	700000,
	1000000,
	1000000,
	1000000
};



static int ipw2100_wx_get_range(struct net_device *dev, 
			       struct iw_request_info *info, 
			       union iwreq_data *wrqu, char *extra)
{
	struct iw_range *range = (struct iw_range *)extra;
	u16 val;
	int i;

	wrqu->data.length = sizeof(*range);
	memset(range, 0, sizeof(*range));

	/* Let's try to keep this struct in the same order as in
	 * linux/include/wireless.h
	 */
	
	/* TODO: See what values we can set, and remove the ones we can't
	 * set, or fill them with some default data.
	 */

	/* ~5 Mb/s real (802.11b) */
	range->throughput = 5 * 1000 * 1000;     

	// TODO: Not used in 802.11b?
//	range->min_nwid;	/* Minimal NWID we are able to set */
	// TOOD: Not used in 802.11b?
//	range->max_nwid;	/* Maximal NWID we are able to set */

//	range->old_num_channels;	/* Old Frequency (backward compat - moved lower ) */
//	range->old_num_frequency;	/* Old Frequency (backward compat - moved lower ) */
//	range->old_freq[6];		/* Filler to keep "version" at the same offset */

//	range->sensitivity;	/* signal level threshold range */
	
	range->max_qual.qual = 100;
	/* TODO: Find real max RSSI and stick here */
	range->max_qual.level = 0;
	range->max_qual.noise = IPW2100_RSSI_TO_DBM;
	range->max_qual.updated = 7; /* Updated all three */

	range->avg_qual.qual = 92; /* > 8% missed beacons is 'bad' */
	/* TODO: Find real 'good' to 'bad' threshol value for RSSI */
	range->avg_qual.level = 20 + IPW2100_RSSI_TO_DBM;
	range->avg_qual.noise = 0;
	range->avg_qual.updated = 7; /* Updated all three */

	range->num_bitrates = RATE_COUNT;

	for (i = 0; i < RATE_COUNT && i < IW_MAX_BITRATES; i++) {
		range->bitrate[i] = ipw2100_rates_11b[i];
	}
	
//	range->min_rts = 0;	// FIXME: What about min_rts? and max_rts=DEFAULT below?
	range->max_rts = DEFAULT_RTS_THRESHOLD;
	range->min_frag = MIN_FRAG_THRESHOLD;
	range->max_frag = MAX_FRAG_THRESHOLD;

	range->min_pmp = period_duration[0];	/* Minimal PM period */
	range->max_pmp = period_duration[POWER_MODES-1];/* Maximal PM period */
	range->min_pmt = timeout_duration[POWER_MODES-1];	/* Minimal PM timeout */
	range->max_pmt = timeout_duration[0];/* Maximal PM timeout */

        /* How to decode max/min PM period */
	range->pmp_flags = IW_POWER_PERIOD;
        /* How to decode max/min PM period */
	range->pmt_flags = IW_POWER_TIMEOUT;
	/* What PM options are supported */
	range->pm_capa = IW_POWER_TIMEOUT | IW_POWER_PERIOD;

	range->encoding_size[0] = 5;
	range->encoding_size[1] = 13;           /* Different token sizes */
	range->num_encoding_sizes = 2;		/* Number of entry in the list */
	range->max_encoding_tokens = WEP_KEYS;  /* Max number of tokens */
//	range->encoding_login_index;		/* token index for login token */

//	range->txpower_capa;		/* What options are supported */
//	range->num_txpower;		/* Number of entries in the list */
//	range->txpower[IW_MAX_TXPOWER];	/* list, in bps */
		
	/* Set the Wireless Extension versions */
	range->we_version_compiled = WIRELESS_EXT;
	range->we_version_source = 16;

//	range->retry_capa;	/* What retry options are supported */
//	range->retry_flags;	/* How to decode max/min retry limit */
//	range->r_time_flags;	/* How to decode max/min retry life */
//	range->min_retry;	/* Minimal number of retries */
//	range->max_retry;	/* Maximal number of retries */
//	range->min_r_time;	/* Minimal retry lifetime */
//	range->max_r_time;	/* Maximal retry lifetime */
											
        range->num_channels = FREQ_COUNT;

	val = 0;
	for (i = 0; i < FREQ_COUNT; i++) {
		// TODO: Include only legal frequencies for some countries
//		if (local->channel_mask & (1 << i)) {
			range->freq[val].i = i + 1;
			range->freq[val].m = ipw2100_frequencies[i] * 100000;
			range->freq[val].e = 1;
			val++;
//		}
		if (val == IW_MAX_FREQUENCIES)
		break;
	}
	range->num_frequency = val;

	IPW2100_DEBUG_WX("GET Range\n");

	return 0;
}

static int ipw2100_wx_set_wap(struct net_device *dev, 
			      struct iw_request_info *info, 
			      union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = dev->priv;
	int err;

	static const unsigned char any[] = {
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff
	};
	static const unsigned char off[] = {
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00
	};

	// sanity checks
	if (wrqu->ap_addr.sa_family != ARPHRD_ETHER) 
		return -EINVAL;

	if (!memcmp(any, wrqu->ap_addr.sa_data, ETH_ALEN) ||
	    !memcmp(off, wrqu->ap_addr.sa_data, ETH_ALEN)) {
		/* we disable mandatory BSSID association */
		IPW2100_DEBUG_WX("exit - disable mandatory BSSID\n");
		return ipw2100_set_mandatory_bssid(priv, NULL, 0);
	}

	err = ipw2100_set_mandatory_bssid(
		priv, wrqu->ap_addr.sa_data, 0);
	if (err)
		return err;

	IPW2100_DEBUG_WX("SET BSSID -> %02X:%02X:%02X:%02X:%02X:%02X\n",
	       wrqu->ap_addr.sa_data[0] & 0xff,
	       wrqu->ap_addr.sa_data[1] & 0xff,
	       wrqu->ap_addr.sa_data[2] & 0xff,
	       wrqu->ap_addr.sa_data[3] & 0xff,
	       wrqu->ap_addr.sa_data[4] & 0xff, 
	       wrqu->ap_addr.sa_data[5] & 0xff);

	return 0;
}

static int ipw2100_wx_get_wap(struct net_device *dev, 
			      struct iw_request_info *info, 
			      union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = dev->priv;
	int len = ETH_ALEN;

	wrqu->ap_addr.sa_family = ARPHRD_ETHER;

	/* Fetch from the hardware */
	if (priv->disabled || priv->rf_kill || 
	    ipw2100_get_ordinal(priv, IPW_ORD_STAT_ASSN_AP_BSSID, 
				&priv->ctx->bssid,  &len) ||
	    priv->ctx->ssid[0] == '\0') {
		memset(wrqu->ap_addr.sa_data, 0, ETH_ALEN);
		return 0;
	}

	memcpy(wrqu->ap_addr.sa_data, &priv->ctx->bssid, ETH_ALEN);

	IPW2100_DEBUG_WX("GET BSSID -> %02X:%02X:%02X:%02X:%02X:%02X\n",
			 (u8)wrqu->ap_addr.sa_data[0],
			 (u8)wrqu->ap_addr.sa_data[1],
			 (u8)wrqu->ap_addr.sa_data[2],
			 (u8)wrqu->ap_addr.sa_data[3],
			 (u8)wrqu->ap_addr.sa_data[4],
			 (u8)wrqu->ap_addr.sa_data[5]);

	return 0;
}

static int ipw2100_wx_set_essid(struct net_device *dev, 
				struct iw_request_info *info, 
				union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = dev->priv;
	char *essid = ""; /* ANY */
	int err;
  
	IPW2100_DEBUG_WX("enter\n");
	if (wrqu->essid.flags && wrqu->essid.length)
		essid = extra;
	
        if (priv->disabled) {
		IPW2100_DEBUG_WX("Caching ESSID: %s\n", essid);
		strncpy(priv->ctx->ssid, essid, IW_ESSID_MAX_SIZE);
                return 0;
        }
	
	err = ipw2100_set_essid(priv, essid, 0);
	if (err) {
		IPW2100_DEBUG_WX("SET SSID failed\n");
		return err;
	}
	
	IPW2100_DEBUG_WX("exit\n");
	
	return 0;
}

static int ipw2100_wx_get_essid(struct net_device *dev, 
				struct iw_request_info *info, 
				union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = dev->priv;

	wrqu->essid.length = strlen(priv->ctx->ssid) + 1;
	memcpy(extra, priv->ctx->ssid, wrqu->essid.length);
	wrqu->essid.flags = 1; /* active */

	IPW2100_DEBUG_WX("GET ESSID -> %s\n", extra);
	return 0;
}

static int ipw2100_wx_set_nick(struct net_device *dev, 
			       struct iw_request_info *info, 
			       union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = dev->priv;

	if (wrqu->data.length > IW_ESSID_MAX_SIZE)
		return -E2BIG;

	wrqu->data.length = min((size_t)wrqu->data.length, sizeof(priv->nick));
	memset(priv->nick, 0, sizeof(priv->nick));
	memcpy(priv->nick, extra,  wrqu->data.length);

	IPW2100_DEBUG_WX("SET Nickname -> %s \n", priv->nick);

	return 0;
}

static int ipw2100_wx_get_nick(struct net_device *dev, 
			       struct iw_request_info *info, 
			       union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = dev->priv;
	
	wrqu->data.length = strlen(priv->nick) + 1;
	memcpy(extra, priv->nick, wrqu->data.length);
	wrqu->data.flags = 1; /* active */
	
	IPW2100_DEBUG_WX("GET Nickname -> %s \n", extra);

	return 0;
}

static int ipw2100_wx_get_rate(struct net_device *dev, 
			       struct iw_request_info *info, 
			       union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = dev->priv;
	int val;
	int len = sizeof(val);
	int err;

	if (priv->disabled || priv->rf_kill || priv->ctx->ssid[0] == '\0') {
		wrqu->bitrate.value = 0;
		return 0;
	}

	err = ipw2100_get_ordinal(priv, IPW_ORD_CURRENT_TX_RATE, &val, &len);
	if (err) {
		IPW2100_DEBUG_WX("failed querying ordinals.\n");
		return err;
	}

	switch (val & TX_RATE_MASK) {
	case TX_RATE_1_MBIT:
		wrqu->bitrate.value = 1000000;
		break;
	case TX_RATE_2_MBIT:
		wrqu->bitrate.value = 2000000;
		break;
	case TX_RATE_5_5_MBIT:
		wrqu->bitrate.value = 5500000;
		break;
	case TX_RATE_11_MBIT:
		wrqu->bitrate.value = 11000000;
		break;
	default:
		wrqu->bitrate.value = 0;
	}

	IPW2100_DEBUG_WX("GET Rate -> %d \n", wrqu->bitrate.value);
	return 0;
}

static int ipw2100_wx_set_rts(struct net_device *dev, 
			      struct iw_request_info *info, 
			      union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = dev->priv;
	int err;

	if (wrqu->rts.value < 1 || 
	    wrqu->rts.value > 2304 || 
	    wrqu->rts.disabled)
		return -EINVAL;

	err = ipw2100_set_rts_threshold(priv, wrqu->rts.value);
	if (err)
		return err;

	IPW2100_DEBUG_WX("SET RTS Threshold -> %d \n", wrqu->rts.value);

	return 0;
}

static int ipw2100_wx_get_rts(struct net_device *dev, 
			      struct iw_request_info *info, 
			      union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = dev->priv;
	int len = sizeof(wrqu->rts.value);
	int err;

	if (priv->disabled || priv->rf_kill)
		wrqu->rts.value = priv->rts_threshold;
	else {
		err = ipw2100_get_ordinal(priv, IPW_ORD_RTS_THRESHOLD, 
					  &wrqu->rts.value, &len);
		if (err) {
			IPW2100_DEBUG_WX("query ordinal failed.\n");
			return err;
		}
	}

	wrqu->rts.fixed = 1; /* no auto select */
	wrqu->rts.disabled = 0; /* can't be disabled */

	IPW2100_DEBUG_WX("GET RTS Threshold -> %d \n", wrqu->rts.value);

	return 0;
}

static int ipw2100_wx_set_txpow(struct net_device *dev, 
				struct iw_request_info *info, 
				union iwreq_data *wrqu, char *extra)
{
	IPW2100_DEBUG_WX("TODO: Power management by wireless extension...\n");

	IPW2100_DEBUG_WX("SET TX Power -> %d \n", wrqu->rts.value);

	return 0;
}

static int ipw2100_wx_get_txpow(struct net_device *dev, 
				struct iw_request_info *info, 
				union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = dev->priv;

	wrqu->power.value = priv->tx_power;
	wrqu->power.fixed = 1;
	wrqu->power.flags = IW_TXPOW_DBM;
	wrqu->power.disabled = 0;

	IPW2100_DEBUG_WX("GET TX Power -> %d \n", wrqu->power.value);

	return 0;
}

static int ipw2100_wx_set_frag(struct net_device *dev, 
			       struct iw_request_info *info, 
			       union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = dev->priv;

	if (wrqu->frag.disabled)
		priv->ieee.fts = DEFAULT_FRAG_THRESHOLD;
	else {
		if (wrqu->frag.value < MIN_FRAG_THRESHOLD ||
		    wrqu->frag.value > MAX_FRAG_THRESHOLD)
			return -EINVAL;
		
		priv->ieee.fts = wrqu->frag.value & ~0x1;
	}

	IPW2100_DEBUG_WX("SET Frag Threshold -> %d \n", 
	       priv->ieee.fts);

	return 0;
}

static int ipw2100_wx_get_frag(struct net_device *dev, 
			       struct iw_request_info *info, 
			       union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = dev->priv;
	wrqu->frag.value = priv->ieee.fts;
	wrqu->frag.fixed = 0;	/* no auto select */
	wrqu->frag.disabled = 
		(wrqu->frag.value == DEFAULT_FRAG_THRESHOLD);

	IPW2100_DEBUG_WX("GET Frag Threshold -> %d \n", wrqu->frag.value);

	return 0;
}

static int ipw2100_wx_set_retry(struct net_device *dev, 
				struct iw_request_info *info, 
				union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = dev->priv;
	int err;

	if (wrqu->retry.flags & IW_RETRY_LIFETIME || 
	    wrqu->retry.disabled)
		return -EINVAL;

	if (!wrqu->retry.flags & IW_RETRY_LIMIT)
		return 0;

	if (wrqu->retry.flags & IW_RETRY_MIN) {
		err = ipw2100_set_short_retry(priv, wrqu->retry.value);
		if (err)
			return err;

		IPW2100_DEBUG_WX("SET Short Retry Limit -> %d \n",
		       wrqu->retry.value);

		return 0;
	} 

	if (wrqu->retry.flags & IW_RETRY_MAX) {
		err = ipw2100_set_long_retry(priv, wrqu->retry.value);
		if (err)
			return err;

		IPW2100_DEBUG_WX("SET Long Retry Limit -> %d \n",
		       wrqu->retry.value);

		return 0;
	}

	err = ipw2100_set_short_retry(priv, wrqu->retry.value);
	if (err)
		return err;
	err = ipw2100_set_long_retry(priv, wrqu->retry.value);
	if (err)
		return err;

	IPW2100_DEBUG_WX("SET Both Retry Limits -> %d \n", wrqu->retry.value);

	return 0;
}

static int ipw2100_wx_get_retry(struct net_device *dev, 
				struct iw_request_info *info, 
				union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = dev->priv;

	wrqu->retry.disabled = 0; /* can't be disabled */

	if ((wrqu->retry.flags & IW_RETRY_TYPE) == 
	    IW_RETRY_LIFETIME) 
		return -EINVAL;
	
	if (wrqu->retry.flags & IW_RETRY_MAX) {
		wrqu->retry.flags = IW_RETRY_LIMIT & IW_RETRY_MAX;
		wrqu->retry.value = priv->long_retry_limit;
	} else {
		wrqu->retry.flags =
		    (priv->short_retry_limit !=
		     priv->long_retry_limit) ?
		    IW_RETRY_LIMIT & IW_RETRY_MIN : IW_RETRY_LIMIT;

		wrqu->retry.value = priv->short_retry_limit;
	}

	IPW2100_DEBUG_WX("GET Retry -> %d \n", wrqu->retry.value);

	return 0;
}

static int ipw2100_wx_set_scan(struct net_device *dev, 
			       struct iw_request_info *info, 
			       union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = dev->priv;

	IPW2100_DEBUG_WX("Initiating scan...\n");
	if (ipw2100_start_scan(priv)) {
		IPW2100_DEBUG_WX("Start scan failed.\n");

		/* TODO: Mark a scan as pending so when hardware initialized
		 *       a scan starts */
	}

	return 0;
}

/*
 *
 * Following logic based on the code in drivers/net/wireless/airo.c from 2.6.1
 *
 */
static inline char *ipw2100_translate_scan(char *start, char *stop, 
					   struct ipw2100_beacon *beacon)
{
	struct iw_event iwe;
	int i;

	/* First entry *MUST* be the AP MAC address */
	iwe.cmd = SIOCGIWAP;
	iwe.u.ap_addr.sa_family = ARPHRD_ETHER; 
	memcpy(iwe.u.ap_addr.sa_data, beacon->bssid, ETH_ALEN);
	start = iwe_stream_add_event(start, stop, &iwe, IW_EV_ADDR_LEN);
	
	/* Remaining entries will be displayed in the order we provide them */

	/* Add the ESSID */
        iwe.u.data.length = beacon->ssid_len;
        if (iwe.u.data.length > 32)
		iwe.u.data.length = 32;
        iwe.cmd = SIOCGIWESSID;
        iwe.u.data.flags = 1;
        start = iwe_stream_add_point(start, stop, &iwe, beacon->ssid);

        /* Add mode */
        iwe.cmd = SIOCGIWMODE;
        if (beacon->capability & 
	    (WLAN_CAPABILITY_BSS | WLAN_CAPABILITY_IBSS)) {
		if (beacon->capability & WLAN_CAPABILITY_BSS)
			iwe.u.mode = IW_MODE_MASTER;
		else
			iwe.u.mode = IW_MODE_ADHOC;

		start = iwe_stream_add_event(start, stop, &iwe, 
					     IW_EV_UINT_LEN);
	}
	
        /* Add frequency */
	iwe.cmd = SIOCGIWFREQ;
	iwe.u.freq.m = ipw2100_frequencies[beacon->channel - 1] * 100000;
	iwe.u.freq.e = 1;
	start = iwe_stream_add_event(start, stop, &iwe, IW_EV_FREQ_LEN);

	/* Add rates */
	iwe.cmd = SIOCGIWRATE;
	iwe.u.bitrate.fixed = iwe.u.bitrate.disabled = 0;
	for (i = 0; i < beacon->rates_len; i++) {
		iwe.u.bitrate.value = (beacon->rates[i] & 0x7F) * 500000;
		start = iwe_stream_add_event(start, stop, &iwe, 
					     IW_EV_PARAM_LEN);
	}

	/* Add quality statistics */
	iwe.cmd = IWEVQUAL;
	iwe.u.qual.qual = beacon->rssi; /* For quality, we just use signal 
					 * strength since there is nothing
					 * else we can measure at this point */
	iwe.u.qual.level = beacon->rssi + IPW2100_RSSI_TO_DBM;
	iwe.u.qual.noise = IPW2100_RSSI_TO_DBM;
	iwe.u.qual.updated = 7;	

	start = iwe_stream_add_event(start, stop, &iwe, IW_EV_QUAL_LEN);

	/* Add encryption capability */
	iwe.cmd = SIOCGIWENCODE;
	if (beacon->capability & WLAN_CAPABILITY_PRIVACY)
		iwe.u.data.flags = IW_ENCODE_ENABLED | IW_ENCODE_NOKEY;
	else
		iwe.u.data.flags = IW_ENCODE_DISABLED;
	iwe.u.data.length = 0;
	start = iwe_stream_add_point(start, stop, &iwe, beacon->ssid);

	return start;
}

#define MAX_BEACON_AGE 4
static int ipw2100_wx_get_scan(struct net_device *dev, 
			       struct iw_request_info *info, 
			       union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = dev->priv;
	struct list_head *element;
	struct ipw2100_beacon *beacon;
	unsigned long flags;

	char *ev = extra;
	char *stop = ev + IW_SCAN_MAX_DATA;
	int i;

	IPW2100_DEBUG_WX("enter\n");
	
	spin_lock_irqsave(&priv->low_lock, flags);

	for (element = priv->beacon_list.next, i = 0;
	     element != &priv->beacon_list;
	     element = element->next, i++) {
		beacon = list_entry(element, struct ipw2100_beacon, list);
		if (beacon->last_scanned + MAX_BEACON_AGE >= priv->scans) 
			ev = ipw2100_translate_scan(ev, stop, beacon);
	}

	spin_unlock_irqrestore(&priv->low_lock, flags);

	wrqu->data.length = ev -  extra;
	wrqu->data.flags = 0;
	
	IPW2100_DEBUG_WX("exit: %d beacons returned.\n", i);

	return 0;
}


/*
 * Implementation based on code in hostap-driver v0.1.3 hostap_ioctl.c
 */
static int ipw2100_wx_set_encode(struct net_device *dev, 
				 struct iw_request_info *info, 
				 union iwreq_data *wrqu, char *key)
{
	struct ipw2100_priv *priv = dev->priv;
	return ieee80211_wx_set_encode(&priv->ieee, info, wrqu, key);
}

static int ipw2100_wx_get_encode(struct net_device *dev, 
				 struct iw_request_info *info, 
				 union iwreq_data *wrqu, char *key)
{
	struct ipw2100_priv *priv = dev->priv;
	return ieee80211_wx_get_encode(&priv->ieee, info, wrqu, key);
}

static int ipw2100_wx_set_power(struct net_device *dev, 
			        struct iw_request_info *info, 
			        union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = dev->priv;
	int err, i;
	s32 duration = wrqu->power.value;

	if (wrqu->power.disabled) {
		err = ipw2100_set_power_mode(priv, IPW_POWER_MODE_CAM);
		if (err) {
			IPW2100_DEBUG_WX("failed setting power mode.\n");
			return err;
		}

		IPW2100_DEBUG_WX("SET Power Management Mode -> off\n");

		return 0;
	} 

	switch (wrqu->power.flags & IW_POWER_MODE) {
	case IW_POWER_ON: break;
	/* not supported */
	case IW_POWER_UNICAST_R:
	case IW_POWER_MULTICAST_R:
	case IW_POWER_ALL_R:
	case IW_POWER_FORCE_S:
	case IW_POWER_REPEATER:
	default:
		return -EINVAL;
	}

	switch (wrqu->power.flags & IW_POWER_TYPE) {
	case IW_POWER_PERIOD:
		for (i = 0;
		     (i < POWER_MODES - 1) && 
			     (duration > period_duration[i]);
		     i++);
		i++;
		break;
	case IW_POWER_TIMEOUT:
		for (i = POWER_MODES - 1;
		     (i > 0) && (duration > timeout_duration[i]);
		     i--);
		i++;
		break;
	default: i = IPW_POWER_AUTOMATIC; break;
	}
	
	err = ipw2100_set_power_mode(priv, i);
	if (err) {
		IPW2100_DEBUG_WX("failed setting power mode.\n");
		return err;
	}

	IPW2100_DEBUG_WX("SET Power Management Mode -> %d \n", i);
	
	return 0;
}

static int ipw2100_wx_get_power(struct net_device *dev, 
			        struct iw_request_info *info, 
			        union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = dev->priv;

	int val;
	int len = sizeof(val);
	int err;

	err = ipw2100_get_ordinal(priv, IPW_ORD_POWER_MGMT_MODE, &val, &len);
	if (err) {
		IPW2100_DEBUG_WX("failed querying ordinals.\n");
		return err;
	}

	if (val == 0) {
		wrqu->power.disabled = 1;
	} else {
		wrqu->power.disabled = 0;

		if (val <= 5) {
			if ((wrqu->power.flags & IW_POWER_TYPE) == 
			    IW_POWER_TIMEOUT) {
				wrqu->power.flags = IW_POWER_TIMEOUT;
				wrqu->power.value = timeout_duration[val - 1];
			} else {
				wrqu->power.flags = IW_POWER_PERIOD;
				wrqu->power.value = period_duration[val - 1];
			}
		} else {
			wrqu->power.disabled = 0;
			wrqu->power.flags = IW_POWER_ON;
		}
	}

	IPW2100_DEBUG_WX("GET Power Management Mode -> %d \n",
			 val);
	
	return 0;
}

#ifdef CONFIG_IPW2100_PROMISC
static int ipw2100_wx_set_promisc(struct net_device *dev, 
				  struct iw_request_info *info, 
				  union iwreq_data *wrqu, char *extra)
{
	struct ipw2100_priv *priv = dev->priv;
	int *parms = (int *)extra;
	int enable = (parms[0] > 0);
	
	ipw2100_set_promisc(priv, enable, enable ? parms[1] : 0);

	return 0;
}

static int ipw2100_wx_reset(struct net_device *dev, 
			    struct iw_request_info *info, 
			    struct iwreq_data *wrqu, char *extra)
{
	ipw2100_reset_dapter(dev->priv);
	return 0;
}

#endif

static iw_handler ipw2100_wx_handlers[] =
{
        NULL,                     /* SIOCSIWCOMMIT */
        ipw2100_wx_get_name,      /* SIOCGIWNAME */
        NULL,                     /* SIOCSIWNWID */
        NULL,                     /* SIOCGIWNWID */
        ipw2100_wx_set_freq,      /* SIOCSIWFREQ */
        ipw2100_wx_get_freq,      /* SIOCGIWFREQ */
        ipw2100_wx_set_mode,      /* SIOCSIWMODE */
        ipw2100_wx_get_mode,      /* SIOCGIWMODE */
        NULL,                     /* SIOCSIWSENS */
        NULL,                     /* SIOCGIWSENS */
        NULL,                     /* SIOCSIWRANGE */
        ipw2100_wx_get_range,     /* SIOCGIWRANGE */
        NULL,                     /* SIOCSIWPRIV */
        NULL,                     /* SIOCGIWPRIV */
        NULL,                     /* SIOCSIWSTATS */
        NULL,                     /* SIOCGIWSTATS */
        NULL,                     /* SIOCSIWSPY */
        NULL,                     /* SIOCGIWSPY */
        NULL,                     /* SIOCGIWTHRSPY */
        NULL,                     /* SIOCWIWTHRSPY */
        ipw2100_wx_set_wap,       /* SIOCSIWAP */
        ipw2100_wx_get_wap,       /* SIOCGIWAP */
        NULL,                     /* -- hole -- */
        NULL,                     /* SIOCGIWAPLIST -- depricated */
        ipw2100_wx_set_scan,      /* SIOCSIWSCAN */
        ipw2100_wx_get_scan,      /* SIOCGIWSCAN */
        ipw2100_wx_set_essid,     /* SIOCSIWESSID */
        ipw2100_wx_get_essid,     /* SIOCGIWESSID */
        ipw2100_wx_set_nick,      /* SIOCSIWNICKN */
        ipw2100_wx_get_nick,      /* SIOCGIWNICKN */
        NULL,                     /* -- hole -- */
        NULL,                     /* -- hole -- */
        NULL,                     /* SIOCSIWRATE */
        ipw2100_wx_get_rate,      /* SIOCGIWRATE */
        ipw2100_wx_set_rts,       /* SIOCSIWRTS */
        ipw2100_wx_get_rts,       /* SIOCGIWRTS */
        ipw2100_wx_set_frag,      /* SIOCSIWFRAG */
        ipw2100_wx_get_frag,      /* SIOCGIWFRAG */
        ipw2100_wx_set_txpow,     /* SIOCSIWTXPOW */
        ipw2100_wx_get_txpow,     /* SIOCGIWTXPOW */
        ipw2100_wx_set_retry,     /* SIOCSIWRETRY */
        ipw2100_wx_get_retry,     /* SIOCGIWRETRY */
        ipw2100_wx_set_encode,    /* SIOCSIWENCODE */
        ipw2100_wx_get_encode,    /* SIOCGIWENCODE */
        ipw2100_wx_set_power,     /* SIOCSIWPOWER */
        ipw2100_wx_get_power,     /* SIOCGIWPOWER */
};

#ifdef CONFIG_IPW2100_PROMISC

static const struct iw_priv_args ipw2100_private_args[] = { 
	{
		SIOCIWFIRSTPRIV + 0x0, 
		IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 2, 0, "monitor" 
	}, {
		SIOCIWFIRSTPRIV + 0x1, 
		IW_PRIV_TYPE_INT | IW_PRIV_SIZE_FIXED | 0, 0, "reset" 
	},
};

const iw_handler ipw2100_private_handler[] = {
	ipw2100_wx_set_promisc,  /* SIOCIWFIRSTPRIV */
	ipw2100_wx_reset,        /* SIOCIWFIRSTPRIV + 1*/ 
};

#endif /* CONFIG_IPW2100_PROMISC */

struct iw_handler_def ipw2100_wx_handler_def = 
{
	.standard = ipw2100_wx_handlers,
	.num_standard = sizeof(ipw2100_wx_handlers) / sizeof(iw_handler),
#ifdef CONFIG_IPW2100_PROMISC
	.num_private = sizeof(ipw2100_private_handler) / sizeof(iw_handler),
 	.num_private_args = sizeof(ipw2100_private_args) /
	sizeof(struct iw_priv_args),
	.private = (iw_handler *)ipw2100_private_handler,
	.private_args = (struct iw_priv_args *)ipw2100_private_args,	
#else
	.num_private = 0,
	.num_private_args = 0,
	.private = NULL,
 	.private_args = NULL,
#endif /* CONFIG_IPW2100_PROMISC */
};

/*
 * Get wireless statistics.
 * Called by /proc/net/wireless
 */
struct iw_statistics *ipw2100_wx_wireless_stats(struct net_device * dev)
{
	enum {
		POOR = 30,
		FAIR = 60,
		GOOD = 80,
		VERY_GOOD = 90,
		EXCELLENT = 95,
		PERFECT = 100
	};
	struct ipw2100_priv *priv = dev->priv;
	struct iw_statistics *wstats;
	u32 rssi, quality, tx_retries, missed_beacons, tx_failures;
	u32 ord_len = sizeof(u32);

	if (!priv)
		return (struct iw_statistics *) NULL;

	/* if hw is disabled, then ipw2100_get_ordinal() can't be called.
	 * ipw2100_wx_wireless_stats seems to be called before fw is 
	 * initialized. */
	if (priv->disabled)
		return (struct iw_statistics *) NULL;

	wstats = &priv->wstats;

        /* If we don't have a connection the quality is 0*/
	if (!priv->connected) 
		wstats->qual.qual = 0; 
	else {
		if (ipw2100_get_ordinal(priv, IPW_ORD_RSSI_AVG_CURR, 
					&rssi, &ord_len))
			goto fail_get_ordinal;
		wstats->qual.level = rssi + IPW2100_RSSI_TO_DBM;
		if (rssi < 10)
			quality = POOR;
		else if (rssi < 15)
			quality = FAIR;
		else if (rssi < 20)
			quality = GOOD;
		else if (rssi < 30)
			quality = VERY_GOOD;
		else
			quality = EXCELLENT;
		
		if (ipw2100_get_ordinal(priv, IPW_ORD_STAT_PERCENT_RETRIES,
					&tx_retries, &ord_len))
			goto fail_get_ordinal;
		if (ipw2100_get_ordinal(priv, IPW_ORD_STAT_PERCENT_MISSED_BCNS,
					&missed_beacons, &ord_len))
			goto fail_get_ordinal;
		
		if (quality <= POOR || missed_beacons > 50 || tx_retries > 75) 
			quality = POOR;
		else if (quality <= FAIR || 
			 missed_beacons > 40 || tx_retries > 70) 
			quality = FAIR;
		else if (quality <= GOOD || 
			 missed_beacons > 32 || tx_retries > 65) 
			quality = GOOD;
		else if (quality <= VERY_GOOD ||
			 missed_beacons > 20 || tx_retries > 50) 
			quality = VERY_GOOD;
		else if (missed_beacons == 0 && tx_retries == 0)
			quality = PERFECT;
			
		wstats->qual.qual = quality;
	}
	
	wstats->qual.level = rssi + IPW2100_RSSI_TO_DBM;

	wstats->qual.noise = IPW2100_RSSI_TO_DBM;
	wstats->qual.updated = 7;	

        /* FIXME: this is percent and not a # */
	wstats->miss.beacon = missed_beacons; 

	if (ipw2100_get_ordinal(priv, IPW_ORD_STAT_TX_FAILURES, 
				&tx_failures, &ord_len))
		goto fail_get_ordinal;
	wstats->discard.retries = tx_failures;

	return wstats;

 fail_get_ordinal:
	IPW2100_DEBUG_WX("failed querying ordinals.\n");

	return (struct iw_statistics *) NULL;
}

void ipw2100_wx_event_work(struct ipw2100_priv *priv)
{
#if WIRELESS_EXT > 13
      union iwreq_data wrqu;
      int len = ETH_ALEN;

      IPW2100_DEBUG_WX("enter\n");
      if (priv->wx_ap_event_pending) {
              priv->wx_ap_event_pending = 0;

	      /* This is a temprorary fix to support this within
	       * 2.4; ideally we would just use schedule_delayed_work 
	       * instead of schedule_work and then we wouldn't have to
	       * sleep here */
	      set_current_state(TASK_UNINTERRUPTIBLE);
	      schedule_timeout(2);

              wrqu.ap_addr.sa_family = ARPHRD_ETHER;

              /* Fetch BSSID from the hardware */
              if (priv->disabled || priv->rf_kill ||
                  ipw2100_get_ordinal(priv, IPW_ORD_STAT_ASSN_AP_BSSID,
                                      &priv->ctx->bssid,  &len)) {
                      memset(wrqu.ap_addr.sa_data, 0, ETH_ALEN);
              } else {
                      memcpy(wrqu.ap_addr.sa_data, priv->ctx->bssid, ETH_ALEN);
              }
              wireless_send_event(priv->ndev, SIOCGIWAP, &wrqu, NULL);
      }
#endif /* WIRELESS_EXT > 13 */
}

#endif /* WIRELESS_EXT > 12 */ 
