/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the libgltf project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#ifndef RENDERSCENE_H
#define RENDERSCENE_H

#include "Camera.h"
#include "LoadScene.h"
#include "FPSCounter.h"

#define SSAA           2
#define LOAD_ONCE      1

#define FBOVERT "attribute vec4 vPosition;\n"\
"attribute vec2 texCoord;\n"\
"varying vec2 vTexCoord;\n"\
"void main()\n"\
"{\n"\
"    gl_Position =  vPosition;\n"\
"    vTexCoord = texCoord;\n"\
"}"
#define FBOFRAG "uniform sampler2D RenderTex;\n"\
"varying vec2 vTexCoord;\n"\
"void main()\n"\
"{\n"\
"    gl_FragColor = vec4(texture2D(RenderTex, vTexCoord).rgb, 1);\n"\
"}"
#define INVERSEVERT "attribute vec4 vPosition;\n"\
"attribute vec2 texCoord;\n"\
"varying vec2 vTexCoord;\n"\
"void main()\n"\
"{\n"\
"    gl_Position =  vPosition;\n"\
"    vTexCoord = texCoord;\n"\
"}"
#define INVERSEFRAG "uniform sampler2D RenderTex;\n"\
"varying vec2 vTexCoord;\n"\
"void main()\n"\
"{\n"\
"    gl_FragColor = vec4(texture2D(RenderTex, vTexCoord).rgb, 1.0);\n"\
"}"

namespace libgltf
{

class RenderWithFBO
{
public:
    RenderWithFBO();
    int createAndBindFbo(int width, int height, bool isUseMSAA);
    void releaseFbo();
    int renderFbo(int srcWidth, int srcHeight);

    void createTextureObj(int width, int height, GLuint& texId);
    void createRenderObj(int width, int height, GLuint& RboId);
    int createFrameBufferObj(GLuint& FboId, GLuint texId, GLuint RboId);
    int createMultiSampleTextureFrameBufObj(GLuint& fboId, GLuint& texId,
                                          GLuint& rboId, int width, int height);
    GLuint loadFboShader(const char* vShader, const char* fShader);
    void setBufferForFbo(GLuint& texCoordBuf, GLuint& vertexBuf,
                         GLfloat* pCoord, GLuint numCoord,
                         GLfloat* pSquare, GLuint numSquare);
    void createAndBindInverseFBO(int width, int height);
    void createBitmapTexture(int width, int height);
    void inverseBitMap(int width, int height);
    void inverseTexture(GLuint proId, GLuint texCoordBuf, GLuint vertexBuf);
    void releaseBitMapFBO();
    void releaseBitmapTexture();
    void releaseMSAAFBO();
    GLuint getMSAAFboId() const { return mMSAAFboId; }
    GLuint getFboId()  const { return mFboId; }
    int renderFboTexture();
private:

    GLuint mFboProId;
    GLuint mFboId;
    GLuint mRboId;
    GLuint mTexId;

    GLuint mShotTexId;
    GLuint mRenderTexId;

    GLuint mInverseFboId;
    GLuint mInverseRboId;
    GLuint mInverseTexId;

    GLuint mMSAAFboId;
    GLuint mMSAARboId;
    GLuint mMSAATexId;

    GLuint mVertexBuf;
    GLuint mTexCoordBuf;
};

class RenderPrimitive
{
public:
    const Node* getNode() const;
    void setNode(Node* node);
    const Material* getMaterial() const;
    void setMaterial(Material* pMaterial);

    unsigned int getVerterCount() const;
    void setVerterCount(unsigned int count);

    unsigned int getIndicesCount() const;
    void setIndicesCount(unsigned int count);

    unsigned int getVertexBuffer() const;
    void setVertexBuffer(unsigned int bufferId);

    unsigned int getNormalBuffer() const;
    void setNormalBuffer(unsigned int bufferId);

    unsigned int getTexCoordBuffer() const;
    void setTexCoordBuffer(unsigned int bufferId);

    unsigned int getJointBuffer() const;
    void setJointBuffer(unsigned int bufferId);

    unsigned int getWeightBuffer() const;
    void setWeightBuffer(unsigned int bufferId);

    unsigned int getIndicesBuffer() const;
    void setIndicesBuffer(unsigned int bufferId);

    DataType getIndicesDataType() const;
    void setIndicesDataType(DataType type);

    void setVertexBufferData(const char* srcBuf);

    void getPrimitiveBoundary(glm::vec3* vertexMax, glm::vec3* vertexMin);

    RenderPrimitive();
    ~RenderPrimitive();
private:
    RenderPrimitive(const RenderPrimitive&);
    RenderPrimitive& operator=(const RenderPrimitive&);

    DataType mIndicesDataType;
    Material* mpMaterial;
    Node* pNode;
    unsigned int mVerterCount;
    unsigned int mIndicesCount;
    unsigned int mVertexBuffer;
    unsigned int mNormalBuffer;
    unsigned int mTexCoordBuffer;
    unsigned int mJointBuffer;
    unsigned int mWeightBuffer;
    unsigned int mIndicesBuffer;
    const char* mVertexBufferData;
};

class RenderShader
{
public:
    Technique* getTechnique();
    void setTechnique(Technique* pTechnique);

    void pushRenderPrim(RenderPrimitive* p);
    RenderPrimitive* getRenderPrim(unsigned int i);
    unsigned int getRenderPrimSize() const;

    RenderShader();
    ~RenderShader();

private:
    RenderShader(const RenderShader&);
    RenderShader& operator=(const RenderShader&);

    std::vector<RenderPrimitive*> mPrimitiveVec;
    Technique* mpTechnique;
};

class RenderScene
{
    struct BindBufferInfo
    {
        unsigned int mBufferId;
        unsigned int mDataCount;
        const char * mSrcData;
        unsigned int mBufferLen;

        BindBufferInfo() : mBufferId(0), mDataCount(0), mSrcData(0), mBufferLen(0) {}
    };
public:
    // For camera
    glm::vec3* getModelCenterPos();
    void getCameraPos(glm::vec3* Eye, glm::vec3* view, glm::vec3* up);
    double getModelSize() const;
    bool renderFlyCamera(const glm::mat4& glPosViewMatrix, double time);
    CPhysicalCamera& getCamera();
    void getCameraIndex(Node* rootNode);

    // For render scene
    bool initScene(const std::string& jsonfile, std::vector<glTFFile>& o_glTFFiles);
    int initRender(const std::vector<glTFFile>& inputFiles);
    int prepareRender(glTFViewport* pViewport);
    void render();
    int completeRender();
    void releaseRender();

    // For bitmap
    int prepareRenderBitmap(glTFViewport* pViewport);
    void renderBitmap(double time);
    int completeRenderBitmap(glTFViewport* pViewport,
                              unsigned char* buffer, GLenum format);
    void setBitZoom(unsigned char* Dstbuffer,
                    unsigned char* Srcbuffer,
                    glTFViewport* pViewport,
                    int bufferDepth);

    // For animation
    void startAnimation();
    void stopAnimation();
    void stopAerialView();
    void startAerialView();
    bool isInAerialView() const;
    void setAnimTime(double time);
    double getAnimTime() const;
    void setAnimLoop(bool loop);
    bool getAnimLoop() const;
    double getAnimDuration() const;
    bool isAnimPlay() const;
    void resumeAnim();
    void setTimeForAnim();

    // For model rotation
    void enableRotation();
    void disableRotation();
    bool isRotationEnabled() const;

    void enableTransparency();
    void disableTransparency();
    void initFPS();
    void endFPS();
    void setViewMatrix(glm::mat4 viewMatrix);
    void resetViewMatrix();
    const glm::mat4& getViewMatrix() const;
    void startPatrol();
    void endPatrol();
    void enableMSAA();
    void disableMSAA();
    RenderScene();
    ~RenderScene();

private:
    RenderScene(const RenderScene&);
    RenderScene& operator=(const RenderScene&);

    void initOpengl();
    void initNodeTree(Node* pNode, const glm::mat4& matrix,
                      bool parentJointFlag, bool updateFlag);
    int loadScene(const std::vector<glTFFile>& inputFiles);
    void createDefaultCamera();
    void constructShader();
    void constructMesh(const std::string& meshName, Node* pNode);
    void constructPrimitive(const Primitives* pPrim, Node* pNode);
    unsigned int bindAttribute(const Attribute* pAttr);
    unsigned int bindIndices(const Attribute* pAttr);
    void bindAttributeBuffer(const Primitives* pPrimitive,
                             RenderPrimitive* pRP);
    void updateNodeMatrix(Node* pNode, const glm::mat4& matrix, bool flag);
    void updateFlyCamera();
    void updateAnimInfo(Node* pNode);
    bool upLoadTechPropertyTransparent();
    void upLoadTechPropertyOfJsonFile(Technique* pTech);
    void upLoadTechInfo(unsigned int progId, Technique* pTech);
    void upLoadMatrixInfo(unsigned int progId, RenderPrimitive* pPrimitive);
    void upLoadAnimation(unsigned int progId, RenderPrimitive* pPrimitive);
    void upLoadUniform(unsigned int progId, RenderPrimitive* pPrimitive);
    void upLoadAttribute(unsigned int progId, RenderPrimitive* pPrimitive);
    void drawTriangle(RenderPrimitive* pPrimitive);
    void renderPrimitive(RenderPrimitive* pPrimitive, unsigned int progId);
    void renderShader(RenderShader* pRenderShader);

    void realRender();
    Node* findNodeByName(Node* pPareNode, const std::string& nodeId);
    Node* findNodeByJoint(Node* pPareNode, const std::string& jointId);
    int initSSAAFrameBuf(glTFViewport* pViewport);
    void setModelBoundaryValue();

    CPhysicalCamera maCamera;
    ParseCamera* cCamera;
    std::vector<std::string> vCameraIndex;
    glm::mat4 mOrbitInitViewMatrix;
    glm::mat4 mWalkthroughInitViewMatrix;
    bool bAerialView;

    double  previousUpdateCameraTime;
    glm::mat4 flyInfo;
    double flyTime;
    bool bFlyCamera;
    bool bAnimation;
    Light* pLight;

    bool mAnimationPlay;
    bool mAnimationLoop;
    double mCurrentTime;
    double mLastPlaying;
    double mUpdateTimeOut;
    double mDuration;

    std::vector<RenderShader*> mShaderVec;
    Scene* pScene;
    Parser mLoadJson;
    std::map<std::string, BindBufferInfo> mBindBufferMap;

    ShaderProgram mShaderProgram;
    glTFViewport mCurrentViewport;
    RenderWithFBO fbo;

    bool mEnableTransparency;
    bool mEnableRotation;
    glm::mat4 mLastModelView;
    bool bIsTimeAvailable;
    bool loadFPSShader();
    void printFPS(glTFViewport* pViewpoit);
    glm::mat4 mViewMatrixBeforePatrol;
    bool mIsCameraAnimationRunning;
    bool mIsCameraAnimationRunningBeforePatrol;
    FPSCounter* pFPSCounter;
    bool mIsFPSCounterPrinted;
    bool mIsUseMSAA;
#if LOAD_ONCE
    std::string mCurrentImage;
    int mCurrentTextNumber;
#endif
};

} // namespace libgltf

#endif

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
