/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the libgltf project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#include "OpenGLContext.h"

#include <iostream>

#include <epoxy/gl.h>
#include <X11/Xlib.h>

#include<GL/gl.h>
#include<GL/glx.h>

/// Holds the information of our new child window
struct OpenGLContext_Impl
{
    Display*            dpy;
    Window              win;
    XVisualInfo*        vi;
    GLXContext          ctx;

    OpenGLContext_Impl():
        dpy(NULL),
        win(0),
        vi(NULL),
        ctx(0)
    {
    }
};

OpenGLContext::OpenGLContext()
    : mpImpl(new OpenGLContext_Impl)
    , mbInitialized(false)
{
}

OpenGLContext::~OpenGLContext()
{
    if(mpImpl->ctx)
    {
        glXMakeCurrent(mpImpl->dpy, None, NULL);
        /*
        if( glGetError() != GL_NO_ERROR )
        {
            std::cerr << "glError: " << glewGetErrorString(glGetError()) << std::endl;
        }
        */
        glXDestroyContext(mpImpl->dpy, mpImpl->ctx);
    }
    XFree(mpImpl->vi);
    delete mpImpl;
}

bool OpenGLContext::init()
{
    mpImpl->dpy = XOpenDisplay(NULL);

    if( mpImpl->dpy == NULL )
    {
        std::cerr << "Cannot connect to X server" << std::endl;
        return false;
    }

    Window root = DefaultRootWindow(mpImpl->dpy);

    GLint att[] = { GLX_RGBA, GLX_DEPTH_SIZE, 24, GLX_DOUBLEBUFFER, None };
    mpImpl->vi = glXChooseVisual(mpImpl->dpy, 0, att);

    if( mpImpl->vi == NULL )
    {
        std::cerr << "No appropriate visual found" << std::endl;
        return false;
    }

    Colormap cmap = XCreateColormap(mpImpl->dpy, root, mpImpl->vi->visual, AllocNone);

    XSetWindowAttributes swa;
    swa.colormap = cmap;
    swa.event_mask = ExposureMask | KeyPressMask;

    mpImpl->win = XCreateWindow(mpImpl->dpy, root, 0, 0, 600, 600, 0, mpImpl->vi->depth, InputOutput, mpImpl->vi->visual, CWColormap | CWEventMask, &swa);

    XMapWindow(mpImpl->dpy, mpImpl->win);
    XStoreName(mpImpl->dpy, mpImpl->win, "Test Window");

    mpImpl->ctx = glXCreateContext(mpImpl->dpy,
                                    mpImpl->vi,
                                    0,
                                    GL_TRUE);
    if( mpImpl->ctx == NULL )
    {
        std::cerr << "Unable to create GLX context" << std::endl;
        return false;
    }

    if( !glXMakeCurrent( mpImpl->dpy, mpImpl->win, mpImpl->ctx ) )
    {
        std::cerr << "Unable to select current GLX context" << std::endl;
        return false;
    }

    mbInitialized = true;
    return true;
}

void OpenGLContext::setWinPosAndSize(const int nX, const int nY, const int nWidth, const int nHeight)
{
    XMoveResizeWindow(mpImpl->dpy, mpImpl->win, nX, nY, nWidth, nHeight);
}

void OpenGLContext::setWinSize(const int nWidth, const int nHeight)
{
    XResizeWindow(mpImpl->dpy, mpImpl->win, nWidth, nHeight);
}

void OpenGLContext::makeCurrent()
{
    glXMakeCurrent( mpImpl->dpy, mpImpl->win, mpImpl->ctx );
}

void OpenGLContext::resetCurrent()
{
    glXMakeCurrent(mpImpl->dpy, None, NULL);
}

void OpenGLContext::swapBuffers()
{
    glXSwapBuffers(mpImpl->dpy, mpImpl->win);
}

void OpenGLContext::sync()
{
    glXWaitGL();
    XSync(mpImpl->dpy, false);
}

bool OpenGLContext::isInitialized() const
{
    return mbInitialized;
}

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
