/*
 * Copyright 2021-2022 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opentest4j.reporting.cli;

import org.opentest4j.reporting.tooling.validator.DefaultValidator;
import org.opentest4j.reporting.tooling.validator.ValidationResult;
import org.opentest4j.reporting.tooling.validator.Validator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import picocli.CommandLine.Command;
import picocli.CommandLine.Option;
import picocli.CommandLine.Parameters;

import java.net.URI;
import java.nio.file.Path;
import java.util.List;
import java.util.concurrent.Callable;

import static org.opentest4j.reporting.tooling.validator.Severity.ERROR;
import static org.opentest4j.reporting.tooling.validator.Severity.WARNING;

/**
 * Validates XML report files of both formats.
 */
@Command(name = "validate", description = "Validates XML report files of both formats")
public class ValidateCommand implements Callable<Integer> {

	private static final Logger LOG = LoggerFactory.getLogger(ValidateCommand.class);

	@Parameters(arity = "1..*")
	List<Path> xmlFiles;

	@Option(names = "--catalog")
	List<Path> catalogs = List.of();

	@Override
	public Integer call() {
		return validateAll() ? 0 : 1;
	}

	private boolean validateAll() {
		var validator = new DefaultValidator(catalogs.stream().map(Path::toUri).toArray(URI[]::new));
		boolean allValid = true;
		for (var xmlFile : xmlFiles) {
			allValid &= validate(validator, xmlFile);
		}
		return allValid;
	}

	private boolean validate(Validator validator, Path xmlFile) {
		try {
			var validationResult = validator.validate(xmlFile);
			logValidationMessages(validationResult);
			var errors = validationResult.count(ERROR);
			if (errors == 0) {
				LOG.info("Valid: {}", xmlFile);
				return true;
			}
			else {
				LOG.info("Invalid ({} {}): {}", errors, errors == 1 ? "error" : "errors", xmlFile);
				return false;
			}
		}
		catch (Exception e) {
			LOG.error("Failure during validation: {}", xmlFile, e);
			return false;
		}
	}

	private void logValidationMessages(ValidationResult validationResult) {
		validationResult.forEach(message -> {
			if (message.getSeverity() == WARNING) {
				LOG.warn("{}\n{}", message.getLocation(), message.getMessage());
			}
			else {
				LOG.error("{}\n{}", message.getLocation(), message.getMessage());
			}
		});
	}
}
