#!/usr/bin/env python
# Tamito KAJIYAMA <17 June 2001>
# $Id: main.py,v 1.68 2004/01/30 04:59:21 shy Exp $

import getopt
import os
import signal
import socket
import string
import sys
import time
import whrandom
import gettext

gettext.install('ninix')

if os.environ.has_key("DISPLAY"):
    if not sys.modules.has_key('gtk'):
        try:
            import pygtk
            pygtk.require("2.0")
        except ImportError:
            pass
    import gtk
    import pango
    import gobject

import locale

import ninix.home
import ninix.prefs
import ninix.sakura
import ninix.sstp
import ninix.dll

VERSION = "2.6"
VERSION_INFO = (r'\h\s[0]\w8ninix-aya %s\n' % VERSION +
                unicode(_('Are igai No Nanika with \"Nin\'i\" for X'), 'utf-8') + r'\n'
                r'\_q'
                r'Copyright (c) 2001, 2002 Tamito KAJIYAMA\n'
                r'Copyright (c) 2002, 2003 MATSUMURA Namihiko\n'
                r'Copyright (c) 2002-2004 Shyouzou Sugitani\n'
                r'Copyright (c) 2002, 2003 ABE Hideaki\n'
                r'Copyright (c) 2003, 2004 Shun-ichi TAHARA\e')

USAGE = """\
Usage: ninix [options]
Options:
  -H, --homedir DIR   ninix home directory (default: ~/.ninix)
  -U, --userdir DIR   user data directory (default: ninix home directory)
  -R, --raise NUM     raise surface windows NUM pixels
  --sstp-port NUM     additional port for listening SSTP requests
  --iscp-port NUM     alternative port for ISCP connection
  -h, --help          show this message
"""

locale_charset = locale.nl_langinfo(locale.CODESET)

def usage():
    sys.stderr.write(USAGE)
    sys.exit(1)

def layout_set_text(layout, text):
    try:
        layout.set_text(text)
    except:
        layout.set_text(text, -1) # pygtk <= 1.99.14

def main():
    # reopen stderr to avoid IOError
    sys.stderr = os.fdopen(sys.stderr.fileno(), 'w')
    # check if X is running
    if not os.environ.has_key("DISPLAY"):
        sys.stderr.write("Error: cannot open display (abort)\n")
        sys.exit(1)
    # parse command line arguments
    try:
        options, rest = getopt.getopt(sys.argv[1:], "H:U:R:ph",
            ["homedir=", "userdir=", "raise=", "sstp-port=", "iscp-port=",
             "debug=", "help"])
    except getopt.error, e:
        sys.stderr.write("Error: %s\n" % str(e))
        usage()
    if rest:
        usage()
    home_dir = os.environ.get("NINIX_HOME")
    user_dir = os.environ.get("NINIX_USER")
    bottom_margin = None
    sstp_port = [9801, 11000]
    iscp_port = 9821
    debug = 0
    # check environment variables
    val = os.environ.get("NINIX_SSTP_PORT")
    if val:
        port = []
        for num in string.split(val, ","):
            try:
                num = int(num)
                if num < 1024:
                    raise ValueError
                port.append(num)
            except ValueError:
                sys.stderr.write("Invalid NINIX_SSTP_PORT number (ignored)\n")
                break
        else:
            sstp_port.extend(port)
    val = os.environ.get("NINIX_ISCP_PORT")
    if val:
        try:
            num = int(val)
            if num < 1024:
                raise ValueError
        except ValueError:
            sys.stderr.write("Invalid NINIX_ISCP_PORT number (ignored)\n")
        else:
            iscp_port = num
    # parse command line options
    for opt, val in options:
        if opt in ["-H", "--homedir"]:
            home_dir = val
        elif opt in ["-U", "--userdir"]:
            user_dir = val
        elif opt in ["-R", "--raise"]:
            bottom_margin = int(val)
        elif opt == "--sstp-port":
            num = int(val)
            if num < 1024:
                sys.stderr.write("Invalid --sstp-port number (ignored)\n")
            else:
                sstp_port.append(num)
        elif opt == "--iscp-port":
            num = int(val)
            if num < 1024:
                sys.stderr.write("Invalid --iscp-port number (ignored)\n")
            else:
                iscp_port = num
        elif opt == "--debug":
            debug = int(val)
        else:
            usage()
    if iscp_port in sstp_port:
        sys.stderr.write(
            "Cannot use port %d for both SSTP and ISCP.\n" % iscp_port)
        sys.exit(1)
    if home_dir:
        if not os.path.isabs(os.path.expanduser(home_dir)):
            home_dir = os.path.join(os.getcwd(), home_dir)
        ninix.home.NINIX_HOME = home_dir
    if user_dir:
        if not os.path.isabs(os.path.expanduser(user_dir)):
            user_dir = os.path.join(os.getcwd(), user_dir)
        ninix.home.NINIX_USER = user_dir
    config = ninix.home.load_config()
    if config is None:
        sys.stderr.write("Home directory not found (abort)\n")
        sys.exit(1)
    # check if at least one set of ghost, surface, and balloon
    ghosts, shells, balloons, plugins = config
    balloons_all = create_balloon_list(balloons, ghosts, shells)
    if not ghosts:
        sys.stderr.write("Error: no ghost found\n")
        sys.exit(1)
    elif not shells and \
         not filter(lambda ghost: ghost[3], ghosts):
        sys.stderr.write("Error: no shell found\n")
        sys.exit(1)
    elif not balloons_all and \
         not filter(lambda shell: shell[2], shells) and \
         not filter(lambda ghost: ghost[4], ghosts):
        sys.stderr.write("Error: no balloon found\n")
        sys.exit(1)
    for desc, shiori_dir, use_makoto, surface_set, balloon, prefix, shiori_dll, shiori_name in ghosts:
        print "ghost:", desc.get("name", "<none>").encode(locale_charset, 'ignore')
        for name, surface_desc, surface_alias, surface in surface_set:
            if name:
                print "\tsurface:", name.encode(locale_charset, 'ignore'),
            if surface_alias is not None:
                print "(w/alias)",
            print
        if balloon:
            print "\tballoon:", balloon[0].get("name", "<none>").encode(locale_charset, 'ignore')
    for shell_name, surface_set, balloon in shells:
        print "shell:", (shell_name or "<none>").encode(locale_charset, 'ignore')
        for name, surface_desc, surface_alias, surface in surface_set:
            if name:
                print "\tsurface:", name.encode(locale_charset, 'ignore'),
            if surface_alias is not None:
                print "(w/alias)",
            print
        if balloon:
            print "\tballoon:", balloon[0].get("name", "<none>").encode(locale_charset, 'ignore')
    for desc, balloon in balloons_all:
        print "balloon:", desc.get("name", "<none>").encode(locale_charset, 'ignore')
    for plugin_name, plugin_dir, startup, menu_items in plugins:
        print "plugin:", plugin_name.encode(locale_charset, 'ignore')
    # read gtkrc
    gtk.rc_parse(ninix.home.get_gtkrc())
    # start
    sys.stdout.write("loading...")
    sys.stdout.flush()
    app = Application(config, bottom_margin,
                      sstp_port, iscp_port, debug)
    sys.stdout.write("done.\n")
    app.run()

def create_balloon_list(balloons, ghosts, shells):
    balloons_all = []
    balloons_all.extend(balloons)
    for desc, shiori_dir, use_makoto, surface_set, balloon, prefix, shiori_dll, shiori_name in ghosts:
        if balloon:
            balloons_all.append(balloon)
    for shell_name, surface_set, balloon in shells:
        if balloon:
            balloons_all.append(balloon)
    return balloons_all

class Application:
    PREFS_SAKURA_NAME      = "sakura_name"
    PREFS_SAKURA_SURFACE   = "sakura_surface"
    PREFS_DEFAULT_BALLOON  = "default_balloon"
    PREFS_IGNORE_DEFAULT   = "ignore_default"
    PREFS_SCRIPT_SPEED     = "script_speed"
    PREFS_SURFACE_SCALE    = "surface_scale"
    PREFS_SCALE_BALLOON    = "scale_balloon"
    PREFS_EVENT_KILL_LIST  = "event_kill_list"
    PREFS_MOUSE_BUTTON1    = "mouse_button1"
    PREFS_MOUSE_BUTTON3    = "mouse_button3"
    PREFS_BROWSER          = "browser"
    PREFS_HELPER_PATTERN   = "helper_%d_pattern"
    PREFS_HELPER_COMMAND   = "helper_%d_command"
    PREFS_TOP_MARGIN       = "top_margin"
    PREFS_BOTTOM_MARGIN    = "bottom_margin"
    def __init__(self, config, bottom_margin,
                 sstp_port=[9801, 11000], iscp_port=9821, debug=0):
        self.ghosts, self.shells, self.balloons, self.plugins = config
        self.balloons_all = create_balloon_list(self.balloons, self.ghosts, self.shells)
        self.sstp_port = sstp_port
        self.iscp_port = iscp_port
        self.directsstp_path = None
        self.debug = debug
        self.plugin_pids = []
        # create Sakura
        self.sakura  = ninix.sakura.Sakura(debug)
        default_path = os.path.join(os.environ['PYTHONPATH'], 'ninix/dll')
        self.saori_lib = ninix.dll.Library('saori', default_path, sakura=self.sakura)
        self.shiori_lib = ninix.dll.Library('shiori', default_path, saori_lib=self.saori_lib)
        self.ghost   = ninix.sakura.Ghost(self, self.sakura, self.shiori_lib, debug)
        self.surface = ninix.sakura.Surface(self.sakura, debug)
        self.balloon = ninix.sakura.Balloon(self.sakura, debug)
        self.sakura.set_ghost(self.ghost)
        self.sakura.set_surface(self.surface)
        self.sakura.set_balloon(self.balloon)
        # load user preferences
        self.prefs = ninix.prefs.load_prefs()
        #
        top_margin = self.prefs.getint(self.PREFS_TOP_MARGIN, 0)
        if bottom_margin == None:
            bottom_margin = self.prefs.getint(self.PREFS_BOTTOM_MARGIN, 0)
        self.set_top_margin(top_margin)
        self.set_bottom_margin(bottom_margin)
        ignore_default = self.prefs.getint(self.PREFS_IGNORE_DEFAULT, 0)
        self.set_ignore_default(ignore_default)
        # create preference dialog
        self.pref_dialog = PreferenceDialog(self)
        # create vanish dialog
        self.vanish_dialog = VanishDialog(self)
        # create usage dialog
        self.usage_dialog = UsageDialog()
        # load sakura and balloon
        self.load()
    def set_top_margin(self, margin):
        self.top_margin = margin
        self.sakura.set_top_margin(margin)
    def get_top_margin(self):
        return self.top_margin
    def set_bottom_margin(self, margin):
        self.bottom_margin = margin
        self.sakura.set_bottom_margin(margin)
    def get_bottom_margin(self):
        return self.bottom_margin
    def set_default_balloon(self, name):
        if self.find_balloon_by_name(name) != None:
            self.default_balloon = name
    def get_default_balloon(self):
        return self.default_balloon
    def set_ignore_default(self, flag):
        self.ignore_default = flag
    def get_ignore_default(self):
        return self.ignore_default
    def set_scale_menu(self, menuitem):
        if self.scale_menu.get_attach_widget():
            self.scale_menu.detach()
        menuitem.set_submenu(self.scale_menu)
    def create_scale_menu(self):
        self.scale_menu = gtk.Menu()
        item = gtk.CheckMenuItem(_('Scale Balloon'))
        if self.scale_balloon:
            item.set_active(gtk.TRUE)
        else:
            item.set_active(gtk.FALSE)
        item.connect("activate", self.toggle_scale_balloon)
        self.scale_menu.add(item)
        item.show()
        item = gtk.SeparatorMenuItem()
        self.scale_menu.add(item)
        item.show()
        group = None
        for label, value in [("100%", 100),
                             (" 90%",  90),
                             (" 80%",  80),
                             (" 70%",  70),
                             (" 60%",  60),
                             (" 50%",  50),
                             (" 40%",  40)]:
            item = group = gtk.RadioMenuItem(group, label)
            item.set_name("popup menu item")
            item.connect("activate", self.select_surface_scale, value)
            item.show()
            if value == self.surface_scale:
                item.set_active(gtk.TRUE)
            self.scale_menu.append(item)
    def set_speed_menu(self, menuitem):
        if self.speed_menu.get_attach_widget():
            self.speed_menu.detach()
        menuitem.set_submenu(self.speed_menu)
    def create_speed_menu(self):
        self.speed_menu = gtk.Menu()
        group = None
        for label, value in [(_("None"),    -1),
                             ("1 (" + _("Fast") + ")", 0),
                             ("2",        1),
                             ("3",        2),
                             ("4",        3),
                             ("5",        4),
                             ("6",        6),
                             ("7 (" + _("Slow") + ")", 8)]:
            item = group = gtk.RadioMenuItem(group, unicode(label, 'utf-8'))
            item.set_name("popup menu item")
            item.connect("activate", self.select_script_speed, value)
            item.show()
            if value == self.script_speed:
                item.set_active(gtk.TRUE)
            self.speed_menu.append(item)
    def set_ghost_menu(self, menuitem):
        if self.ghost_menu.get_attach_widget():
            self.ghost_menu.detach()
        menuitem.set_submenu(self.ghost_menu)
    def create_ghost_menu(self):
        # choose default ghost/shell
        name = self.prefs.get(self.PREFS_SAKURA_NAME)
        surface = self.prefs.getint(self.PREFS_SAKURA_SURFACE, 0)
        default_sakura = self.find_ghost_by_name(name, surface) or \
                         self.find_shell_by_name(name, surface) or \
                         self.choose_default_sakura()
        type = "g"
        self.ghost_menu = gtk.Menu()
        group = None  # radio button group for ghosts and shells
        for i in range(len(self.ghosts)):
            desc, shiori_dir, use_makoto, surface_set, balloon, prefix, shiori_dll, shiori_name = \
                  self.ghosts[i]
            icon = desc.get("icon", None)
            if icon != None:
                icon_path = os.path.join(shiori_dir, icon)
                if not os.path.exists(icon_path):
                    icon_path = None
            else:
                icon_path = None
            name = desc.get("name", unicode(_("Ghost"), 'utf-8') + "#%d" % (i + 1))
            if len(surface_set) <= 1:
                value = (type, i, 0)
                if icon_path != None:
                    item = gtk.ImageMenuItem(name)
                    pixbuf = ninix.pix.create_pixbuf_from_file(icon_path, 'ico')
                    pixbuf = pixbuf.scale_simple(16, 16, gtk.gdk.INTERP_BILINEAR)
                    image = gtk.Image()
                    image.set_from_pixbuf(pixbuf)
                    image.show()
                    item.set_image(image)
                else:
                    item = gtk.MenuItem(name)
                item.set_name("popup menu item")
                item.connect("activate", self.select_sakura, value)
                item.show()
                if value == default_sakura:
                    item.set_sensitive(gtk.FALSE)
                self.ghost_menu.append(item)
                self.sakura_menu[value] = item
            else:
                submenu = gtk.Menu()
                submenu.set_name("popup menu")
                if icon_path != None:
                    item = gtk.ImageMenuItem(name)
                    pixbuf = ninix.pix.create_pixbuf_from_file(icon_path, 'ico')
                    pixbuf = pixbuf.scale_simple(16, 16, gtk.gdk.INTERP_BILINEAR)
                    image = gtk.Image()
                    image.set_from_pixbuf(pixbuf)
                    image.show()
                    item.set_image(image)
                else:
                    item = gtk.MenuItem(name)
                item.set_name("popup menu item")
                item.set_submenu(submenu)
                item.show()
                self.ghost_menu.append(item)
                for j in range(len(surface_set)):
                    value = (type, i, j)
                    name = surface_set[j][0]
                    item = gtk.MenuItem(name)
                    item.set_name("popup menu item")
                    item.connect("activate", self.select_sakura, value)
                    item.show()
                    if value == default_sakura:
                        item.set_sensitive(gtk.FALSE)
                    submenu.append(item)
                    self.sakura_menu[value] = item
    def set_shell_menu(self, menuitem):
        if self.shell_menu.get_attach_widget():
            self.shell_menu.detach()
        menuitem.set_submenu(self.shell_menu)
    def create_shell_menu(self):
        type = "s"
        self.shell_menu = gtk.Menu()
        for i in range(len(self.shells)):
            name, surface_set, balloon = self.shells[i]
            if name is None:
                name = unicode(_("Shell") + "#%d" % (i + 1), 'utf-8')
            if len(surface_set) == 1:
                value = (type, i, 0)
                item = group = gtk.RadioMenuItem(group, name)
                item.set_name("popup menu item")
                item.connect("activate", self.select_sakura, value)
                item.show()
                if value == default_sakura:
                    item.set_active(gtk.TRUE)
                self.shell_menu.append(item)
                self.sakura_menu[value] = item
            else:
                submenu = gtk.Menu()
                submenu.set_name("popup menu")
                item = gtk.MenuItem(name)
                item.set_name("popup menu item")
                item.set_submenu(submenu)
                item.show()
                self.shell_menu.append(item)
                for j in range(len(surface_set)):
                    value = (type, i, j)
                    name = surface_set[j][0]
                    item = group = gtk.RadioMenuItem(group, name)
                    item.set_name("popup menu item")
                    item.connect("activate", self.select_sakura, value)
                    item.show()
                    if value == default_sakura:
                        item.set_active(gtk.TRUE)
                    submenu.append(item)
                    self.sakura_menu[value] = item
    def set_balloon_menu(self, menuitem):
        if self.balloons_menu.get_attach_widget():
            self.balloons_menu.detach()
        menuitem.set_submenu(self.balloons_menu)
    def create_balloon_menu(self):
        name = self.prefs.get(self.PREFS_DEFAULT_BALLOON)
        default_balloon = self.find_balloon_by_name(name) or 0
        self.balloon_menu = []
        self.balloons_menu = gtk.Menu()
        group = None
        for i in range(len(self.balloons_all)):
            desc, balloon = self.balloons_all[i]
            name = desc.get("name", unicode(_("Balloon"), 'utf-8') + "#%d" % (i + 1))
            item = group = gtk.RadioMenuItem(group, name)
            item.set_name("popup menu item")
            hid = item.connect("activate", self.select_balloon, i)
            item.show()
            if i == default_balloon:
                item.set_active(gtk.TRUE)
            self.balloons_menu.append(item)
            self.balloon_menu.append((item, hid))
    def set_plugin_menu(self, menuitem):
        if self.plugin_menu.get_attach_widget():
            self.plugin_menu.detach()
        menuitem.set_submenu(self.plugin_menu)
    def create_plugin_menu(self):
        self.plugin_menu = gtk.Menu() 
        for i in range(len(self.plugins)):
            plugin_name, plugin_dir, startup, menu_items = self.plugins[i]
            if not menu_items:
                continue
            elif len(menu_items) == 1:
                label, argv = menu_items[0]
                item = gtk.MenuItem(label)
                item.set_name("popup menu item")
                item.connect("activate", self.select_plugin, (i, 0))
                item.show()
                self.plugin_menu.append(item)
            else:
                submenu = gtk.Menu()
                submenu.set_name("popup menu")
                item = gtk.MenuItem(plugin_name)
                item.set_name("popup menu item")
                item.set_submenu(submenu)
                item.show()
                self.plugin_menu.append(item)
                for j in range(len(menu_items)):
                    label, argv = menu_items[j]
                    item = gtk.MenuItem(label)
                    item.set_name("popup menu item")
                    item.connect("activate", self.select_plugin, (i, j))
                    item.show()
                    submenu.append(item)
    def load(self):
        # choose default ghost/shell
        name = self.prefs.get(self.PREFS_SAKURA_NAME)
        surface = self.prefs.getint(self.PREFS_SAKURA_SURFACE, 0)
        default_sakura = self.find_ghost_by_name(name, surface) or \
                         self.find_shell_by_name(name, surface) or \
                         self.choose_default_sakura()
        self.current_sakura = None
        # choose default balloon
        self.default_balloon = self.prefs.get(self.PREFS_DEFAULT_BALLOON)
        self.current_balloon = self.find_balloon_by_name(self.default_balloon) or 0
        # event kill list
        list = self.prefs.get(self.PREFS_EVENT_KILL_LIST)
        if list:
            self.sakura.set_event_kill_list(string.split(list))
        # mouse buttons
        mouse_button1 = self.prefs.get(self.PREFS_MOUSE_BUTTON1)
        if mouse_button1:
            self.sakura.set_mouse_button1(mouse_button1)
        mouse_button3 = self.prefs.get(self.PREFS_MOUSE_BUTTON3)
        if mouse_button3:
            self.sakura.set_mouse_button3(mouse_button3)
        # browser
        command = self.prefs.get(self.PREFS_BROWSER)
        if command:
            self.sakura.set_browser(command)
        # helpers
        list = []
        while 1:
            pattern = self.prefs.get(self.PREFS_HELPER_PATTERN % len(list))
            command = self.prefs.get(self.PREFS_HELPER_COMMAND % len(list))
            if not pattern or not command:
                break
            list.append((pattern, command))
        if list:
            self.sakura.set_helpers(list)
        # surface scale
        self.surface_scale = self.prefs.getint(self.PREFS_SURFACE_SCALE)
        if self.surface_scale not in [40, 50, 60, 70, 80, 90, 100]:
            self.surface_scale = 100
        self.scale_balloon = self.prefs.getint(self.PREFS_SCALE_BALLOON)
        if self.scale_balloon != 1:
            self.scale_balloon = 0
        self.sakura.set_surface_scale(self.surface_scale, self.scale_balloon)
        # script speed
        self.script_speed = self.prefs.getint(self.PREFS_SCRIPT_SPEED)
        if self.script_speed not in [-1, 0, 1, 2, 3, 4, 6, 8]:
            self.script_speed = 3
        self.sakura.set_script_speed(self.script_speed)
        ### create popup menu
        self.sakura_menu = {}
        self.create_ghost_menu()
        self.create_shell_menu()
        self.balloon_menu = []
        self.create_balloon_menu()
        self.create_plugin_menu()
        self.create_scale_menu()
        self.create_speed_menu()
        # load ghost/shell
        self.current_sakura = default_sakura
        self.start_sakura(self.current_sakura, init=1)
    def find_ghost_by_name(self, name, surface):
        for i in range(len(self.ghosts)):
            desc, shiori_dir, use_makoto, surface_set, balloon, prefix, shiori_dll, shiori_name = \
                  self.ghosts[i]
            try:
                if desc.get("name") == name:
                    if not surface_set or surface < len(surface_set):
                        return ("g", i, surface)
            except: # old preferences(EUC-JP)
                pass
        return None
    def find_shell_by_name(self, name, surface):
        for i in range(len(self.shells)):
            shell_name, surface_set, balloon = self.shells[i]
            try:
                if shell_name == name and surface < len(surface_set):
                    return ("s", i, surface)
            except: # old preferences(EUC-JP)
                pass
        return None
    def choose_default_sakura(self):
        for i in range(len(self.ghosts)):
            desc, shiori_dir, use_makoto, surface_set, balloon, prefix, shiori_dll, shiori_name = \
                  self.ghosts[i]
            if surface_set:
                return ("g", i, 0)
        return ("s", 0, 0)
    def find_balloon_by_name(self, name):
        for i in range(len(self.balloons_all)):
            desc, balloon = self.balloons_all[i]
            try:
                if desc.get("name") == name:
                    return i
            except: # old preferences(EUC-JP)
                pass
        return None
    def run(self):
        # set SIGCHLD handler
        signal.signal(signal.SIGCHLD, self.terminate_plugin)
        # start SSTP server
        for port in self.sstp_port:
            try:
                server = ninix.sstp.SSTPServer(('', port), self.sakura)
            except socket.error, (code, message):
                sys.stderr.write("Port %d: %s (ignored)\n" % (port, message))
                continue
            self.sakura.add_sstp_server(server)
            print "Serving SSTP on port", port
        # start DirectSSTP Server
        home_dir = ninix.home.get_ninix_home()
        path = os.path.join(home_dir, 'socket')
        if not os.path.isdir(path):
            os.mkdir(path)
        path = os.path.join(path, str(os.getpid()))
        if os.path.exists(path):
            os.unlink(path)
        try:
            server = ninix.sstp.DirectSSTPServer(path, self.sakura)
        except socket.error, (code, message):
            sys.stderr.write("Port %s: %s (ignored)\n" % (path, message))
        else:
            self.sakura.add_directsstp_server(server, path)
            print "Serving DirectSSTP on", path
            self.directsstp_path = path
        # start plugins
        try:
            os.setpgid(0, 0)
        except OSError:
            pass
        for plugin_name, plugin_dir, startup, menu_items in self.plugins:
            if startup is not None:
                self.exec_plugin(plugin_dir, startup)
        # start Sakura
        gtk.timeout_add(10, self.ghost.start)
        gtk.mainloop()
    def flush_display(self):
        while gtk.events_pending():
            gtk.mainiteration()
    def close(self, event):
        if self.sakura.busy():
            gtk.gdk.beep()
            return
        self.sakura.notify_close(self.quit)
    def quit(self):
        self.save_preferences()
        # stop Sakura
        self.sakura.finalize()
        if os.path.exists(self.directsstp_path):
            os.unlink(self.directsstp_path)
        gtk.mainquit()
        # stop plugins
        signal.signal(signal.SIGHUP, signal.SIG_IGN)
        os.kill(0, signal.SIGHUP)
    def save_preferences(self):
        self.prefs = ninix.prefs.new_prefs()
        # last ghost/shell
        type, i, j = self.current_sakura
        if type == "g":
            desc, shiori_dir, use_makoto, surface_set, balloon, prefix, shiori_dll, shiori_name = \
                  self.ghosts[i]
            name = desc.get("name", "")
        else:
            name, surface_set, balloon = self.shells[i]
        self.prefs[self.PREFS_SAKURA_NAME] = name
        self.prefs[self.PREFS_SAKURA_SURFACE] = str(j)
        # default balloon
        if self.default_balloon == None:
            desc, balloon = self.balloons_all[0]
            self.prefs[self.PREFS_DEFAULT_BALLOON] = desc.get("name")
        else:
            self.prefs[self.PREFS_DEFAULT_BALLOON] = self.default_balloon
        self.prefs[self.PREFS_IGNORE_DEFAULT] = self.ignore_default
        # surface scale
        self.prefs[self.PREFS_SURFACE_SCALE] = str(self.surface_scale)
        self.prefs[self.PREFS_SCALE_BALLOON] = str(self.scale_balloon)
        # script speed
        self.prefs[self.PREFS_SCRIPT_SPEED] = str(self.script_speed)
        # event kill list
        list = self.sakura.get_event_kill_list()
        self.prefs[self.PREFS_EVENT_KILL_LIST] = string.join(list)
        # mouse button settings
        self.prefs[self.PREFS_MOUSE_BUTTON1] = self.sakura.get_mouse_button1()
        self.prefs[self.PREFS_MOUSE_BUTTON3] = self.sakura.get_mouse_button3()
        # browser
        self.prefs[self.PREFS_BROWSER] = self.sakura.get_browser()
        # helpers
        list = self.sakura.get_helpers()
        for i in range(len(list)):
            pattern, command = list[i]
            self.prefs[self.PREFS_HELPER_PATTERN % i] = pattern
            self.prefs[self.PREFS_HELPER_COMMAND % i] = command
        # display layout
        self.prefs[self.PREFS_TOP_MARGIN] = self.get_top_margin()
        self.prefs[self.PREFS_BOTTOM_MARGIN] = self.get_bottom_margin()
        # save preferences
        try:
            self.prefs.save()
        except IOError:
            sys.stderr.write("Cannot write preferences to file (ignored).\n")
    def select_ghost(self, sequential, event=1):
        if len(self.ghosts) < 2:
            return
        type, i, j = self.current_sakura
        if type == "s":
            i = 0
        if sequential:
            i = i + 1
            if i == len(self.ghosts):
                i = 0
        else:
            prev = i
            while 1:
                i = whrandom.randrange(0, len(self.ghosts))
                if i != prev:
                    break
        self.change_sakura(('g', i, 0), "automatic", event)
    def select_ghost_by_name(self, name, event=1):
        item = self.find_ghost_by_name(name, 0)
        if item is None:
            return
        self.change_sakura(item, "automatic", event)
    def select_sakura(self, event, item):
        ##print "select_sakura", item, self.current_sakura
        if self.current_sakura is None or item == self.current_sakura:
            return
        if self.sakura.busy():
            gtk.gdk.beep()
            return
        self.change_sakura(item, "manual")
    def change_sakura(self, item, method, event=1):
        type, i, j = item
        if type == "g":
            desc, shiori_dir, use_makoto, surface_set, balloon, prefix, shiori_dll, shiori_name = \
                  self.ghosts[i]
            if surface_set:
                name, surface_desc, surface_alias, surface = surface_set[j]
            else:
                shell_name, surface_set, balloon = self.shells[0]
                name, surface_desc, surface_alias, surface = surface_set[0]
        elif type == "s":
            desc, shiori_dir, use_makoto, surface_set, balloon, prefix, shiori_dll, shiori_name = \
                  self.ghosts[0]
            shell_name, surface_set, shell_balloon = self.shells[i]
            name, surface_desc, surface_alias, surface = surface_set[j]
        def proc(self=self, item=item):
            self.stop_sakura(self.start_sakura, item)
        if not event:
            proc()
        elif self.current_sakura[0] == type and self.current_sakura[1] == i:
            self.sakura.notify_shell_changing(name, proc)
        else:
            name = surface_desc.get("sakura.name", desc.get("sakura.name"))
            self.sakura.notify_ghost_changing(name, method, proc)
    def stop_sakura(self, *starter):
        self.usage_dialog.close()
        self.sakura.stop()
        menuitem = self.sakura_menu[self.current_sakura]
        menuitem.set_sensitive(gtk.TRUE)
        for menu in [self.ghost_menu, self.shell_menu, self.balloons_menu,
                     self.plugin_menu, self.scale_menu, self.speed_menu]:
            if menu.get_attach_widget():
                menu.detach()
        apply(gtk.idle_add, starter)
    def start_sakura(self, item, init=0):
        type, i, j = item
        if self.current_sakura[0] == type and self.current_sakura[1] == i:
            ghost_changed = 0
        else:
            ghost_changed = 1
        if type == "g":
            print "ghost", item
            desc, shiori_dir, use_makoto, surface_set, balloon, prefix, shiori_dll, shiori_name = \
                  self.ghosts[i]
            self.prefs[self.PREFS_SAKURA_NAME] = desc.get("name", "")
            if surface_set:
                name, surface_desc, surface_alias, surface = surface_set[j]
            else:
                shell_name, surface_set, balloon = self.shells[0]
                name, surface_desc, surface_alias, surface = surface_set[0]
                surface_set = None
        elif type == "s":
            print "shell", item
            desc, shiori_dir, use_makoto, surface_set, balloon, prefix, shiori_dll, shiori_name = \
                  self.ghosts[0]
            shell_name, surface_set, shell_balloon = self.shells[i]
            name, surface_desc, surface_alias, surface = surface_set[j]
            if shell_balloon:
                balloon = shell_balloon
        if not surface_set:
            surface_name = None
        elif len(surface_set) == 1:
            surface_name = unicode(_("Master"), 'utf-8')
        else:
            surface_name = name
        self.current_sakura = (type, i, j)
        if init:
            self.ghost.new(desc, shiori_dir, use_makoto, prefix, shiori_dll, shiori_name)
        elif ghost_changed:
            name = self.sakura.get_selfname()
            self.ghost.finalize()
            self.ghost.new(desc, shiori_dir, use_makoto, prefix, shiori_dll, shiori_name)
        self.surface.new(surface_desc, surface_alias, surface, surface_name)
        if self.ignore_default or not balloon:
            balloon_name = desc.get("balloon", "")
            number = None
            if balloon_name:
                number = self.find_balloon_by_name(balloon_name)
            if number != None:
                balloon = self.balloons_all[number]
            else:
                number = self.find_balloon_by_name(self.default_balloon) or 0
                balloon = self.balloons_all[number]
        if init or balloon != self.balloons_all[self.current_balloon]:
            self.current_balloon = self.balloons_all.index(balloon)
            desc, balloon = self.balloons_all[self.current_balloon]
            self.balloon.new(desc, balloon)
            menuitem, hid = self.balloon_menu[self.current_balloon]
            menuitem.handler_block(hid)
            menuitem.activate()
            menuitem.handler_unblock(hid)
        self.sakura.restart()
        if init:
            pass
        elif ghost_changed:
            self.sakura.notify_ghost_changed(name)
        else:
            self.sakura.notify_shell_changed(name)
        menuitem = self.sakura_menu[self.current_sakura]
        menuitem.set_sensitive(gtk.FALSE)
    def select_balloon(self, widget, item):
        ##print "select_balloon", item, self.current_sakura
        if not widget.active:
            return
        if self.current_balloon is None or item == self.current_balloon:
            return
        if self.sakura.busy():
            menuitem, hid = self.balloon_menu[self.current_balloon]
            menuitem.activate()
            gtk.gdk.beep()
            return
        self.current_balloon = item
        print "balloon", item
        desc, balloon = self.balloons_all[item]
        self.sakura.balloon.hide(0)
        self.sakura.balloon.hide(1)
        self.balloon.new(desc, balloon)
        self.sakura.set_balloon(self.balloon)
        self.sakura.balloon.set_balloon(0, 0)
        self.sakura.balloon.set_balloon(1, 0)
    def reload(self, event=None):
        if self.sakura.busy():
            gtk.gdk.beep()
            return
        def proc(self=self):
            self.stop_sakura(self.reload_sakura)
        self.sakura.notify_ninix_reloading(proc)
    def reload_current_sakura(self):
        self.save_preferences()
        type, i, j = self.current_sakura
        if type == "s":
            i = 0
        home_dir = ninix.home.get_ninix_home()
        ghost_dir = self.ghost.get_prefix()
        ghost, shell = ninix.home.search_ghosts(home_dir, [ghost_dir])
        if ghost:
            self.ghosts[i] = ghost[0]
        else:
            del self.ghosts[i]
            self.shells.append(shell[0])
        self.balloons_all = create_balloon_list(self.balloons, self.ghosts, self.shells)
        self.load()        
    def reload_config(self):
        self.save_preferences()
        sys.stdout.write("reloading...")
        sys.stdout.flush()
        config = ninix.home.load_config()
        if config is None:
            sys.stdout.write("failed! (abort)\n")
            sys.exit(1)
        self.ghosts, self.shells, self.balloons, self.plugins = config
        self.balloons_all = create_balloon_list(self.balloons, self.ghosts, self.shells)
        self.load()
        sys.stdout.write("done.\n")
    def reload_sakura(self):
        self.ghost.finalize()
        self.reload_config()
        self.sakura.restart()
        self.sakura.notify_ninix_reloaded()
    def toggle_scale_balloon(self, widget, value=None):
        if widget.get_active():
            self.scale_balloon = 1
        else:
            self.scale_balloon = 0
        self.sakura.set_surface_scale(self.surface_scale, self.scale_balloon)
    def select_surface_scale(self, widget, value):
        ##print "select_surface_scale", value
        if self.surface_scale == value:
            return
        self.surface_scale = value
        self.sakura.set_surface_scale(value, self.scale_balloon)
    def select_script_speed(self, widget, value):
        ##print "select_script_speed", value
        if self.script_speed == value:
            return
        self.script_speed = value
        self.sakura.set_script_speed(value)
    def vanish(self, event):
        if self.sakura.busy():
            gtk.gdk.beep()
            return
        self.sakura.notify_vanish_selecting()
        self.vanish_dialog.show()
    def notify_vanish_confirmation(self, confirmed):
        if not confirmed:
            self.sakura.notify_vanish_cancel()
            return
        def proc(self=self):
            count = self.sakura.get_ghost_vanished_count()
            self.sakura.set_ghost_vanished_count(count + 1)
            self.sakura.set_ghost_time(0)
            self.stop_sakura(self.vanish_sakura)
        self.sakura.notify_vanish_selected(proc)
    def vanish_sakura(self):
        name = self.sakura.get_selfname()
        # remove ghost
        prefix = self.ghost.get_prefix()
        for file in os.listdir(prefix):
            if file != 'HISTORY':
                command = "rm -rf " + os.path.join(prefix, file)
                print command
                if os.system(command):
                    print "***FAILED***"
        # select another ghost
        type, i, j = self.current_sakura
        if type == "s":
            i = 0
        list = range(len(self.ghosts))
        list.remove(i)
        next_i = whrandom.choice(list)
        self.current_sakura = ('g', next_i, 0)
        # reload and start the new ghost
        self.save_preferences()
        del self.ghosts[i]
        self.balloons_all = create_balloon_list(self.balloons, self.ghosts, self.shells)
        self.load()
        self.sakura.restart()
        self.sakura.notify_ghost_changed(name, vanished=1)
    def toggle_bind(self, event, args):
        side, id = args
        self.surface.toggle_bind(side, id)
    def select_plugin(self, event, item):
        i, j = item
        plugin_name, plugin_dir, startup, menu_items = self.plugins[i]
        label, argv = menu_items[j]
        self.exec_plugin(plugin_dir, argv)
    def terminate_plugin(self, signum, frame):
        for pid in self.plugin_pids[:]:
            try:
                (pid, status) = os.waitpid(pid, os.WNOHANG)
            except OSError:
                ##print "Process %d not found." % pid
                self.plugin_pids.remove(pid)
            if pid > 0:
                ##print "Process %d terminated." % pid
                self.plugin_pids.remove(pid)
        ##print "Running subprocesses:", self.plugin_pids
    def exec_plugin(self, plugin_dir, argv):
        ##print "exec_plugin:", string.join(argv)
        if not os.path.exists(argv[0]):
            return
        port = self.sakura.get_sstp_port()
        if port is None:
            port = "none"
        environ = os.environ.copy()
        environ["NINIX_PID"] = str(os.getpid())
        environ["NINIX_SSTP_PORT"] = str(port)
        environ["NINIX_PLUGIN_DIR"] = plugin_dir
        try:
            pid = os.fork()
        except OSError:
            sys.stderr.write("Error: %s failed (ignored)\n" % argv[0])
            return
        if pid == 0:
            os.chdir(plugin_dir)
            try:
                os.execve(argv[0], argv, environ)
            except OSError:
                sys.stderr.write("Error: %s failed (abort)\n" % argv[0])
                os._exit(1)
        self.plugin_pids.append(pid)
    def edit_preferences(self, widget):
        self.saved_prefs = [
            self.sakura.get_event_kill_list(),
            self.balloon.get_balloon_fonts(),
            self.sakura.get_mouse_button1(),
            self.sakura.get_mouse_button3(),
            self.sakura.get_browser(),
            self.sakura.get_helpers(),
            self.get_top_margin(),
            self.get_bottom_margin(),
            self.get_default_balloon(),
            self.get_ignore_default(),
            ]
        self.pref_dialog.set_event_kill_list(self.saved_prefs[0])
        self.pref_dialog.set_balloon_fonts(self.saved_prefs[1])
        self.pref_dialog.set_mouse_button1(self.saved_prefs[2])
        self.pref_dialog.set_mouse_button3(self.saved_prefs[3])
        self.pref_dialog.set_browser(self.saved_prefs[4])
        self.pref_dialog.set_helpers(self.saved_prefs[5])
        self.pref_dialog.set_top_margin(self.saved_prefs[6])
        self.pref_dialog.set_bottom_margin(self.saved_prefs[7])
        self.pref_dialog.set_default_balloon(self.saved_prefs[8])
        self.pref_dialog.set_ignore_default(self.saved_prefs[9])
        self.pref_dialog.show()
    def notify_preferences_changed(self, done):
        self.sakura.set_event_kill_list(self.pref_dialog.get_event_kill_list())
        self.balloon.set_balloon_fonts(self.pref_dialog.get_balloon_fonts())
        self.sakura.set_mouse_button1(self.pref_dialog.get_mouse_button1())
        self.sakura.set_mouse_button3(self.pref_dialog.get_mouse_button3())
        self.sakura.set_browser(self.pref_dialog.get_browser())
        self.sakura.set_helpers(self.pref_dialog.get_helpers())
        self.set_top_margin(self.pref_dialog.get_top_margin())
        self.set_bottom_margin(self.pref_dialog.get_bottom_margin())
        self.set_default_balloon(self.pref_dialog.get_default_balloon())
        self.set_ignore_default(self.pref_dialog.get_ignore_default())
    def notify_preferences_reverted(self):
        self.sakura.set_event_kill_list(self.saved_prefs[0])
        self.balloon.set_balloon_fonts(self.saved_prefs[1])
        self.sakura.set_mouse_button1(self.saved_prefs[2])
        self.sakura.set_mouse_button3(self.saved_prefs[3])
        self.sakura.set_browser(self.saved_prefs[4])
        self.sakura.set_helpers(self.saved_prefs[5])
        self.set_top_margin(self.saved_prefs[6])
        self.set_bottom_margin(self.saved_prefs[7])
        self.set_default_balloon(self.saved_prefs[8])
        self.set_ignore_default(self.saved_prefs[9])
    def show_usage(self, widget):
        self.sakura.save_history()
        history = {}
        for i in range(len(self.ghosts)):
            desc = self.ghosts[i][0]
            name = desc.get("name", unicode(_("Ghost"), 'utf-8') + "#%d" % (i + 1))
            ghost_time = 0
            prefix = self.ghosts[i][5]
            path = os.path.join(prefix, 'HISTORY')
            if os.path.exists(path):
                try:
                    file = open(path, "r")
                except IOError, (code, message):
                    sys.stderr.write("cannot read %s\n" % path)
                else:
                    while 1:
                        line = file.readline()
                        if not line:
                            break # EOF
                        comma = string.find(line, ',')
                        if comma >= 0:
                            key = string.strip(line[:comma])
                            value = string.strip(line[comma+1:])
                        if key == 'time':
                            try:
                                ghost_time = int(value)
                            except:
                                pass
                file.close()
            history[name] = ghost_time
        self.usage_dialog.open(history)

class PreferenceDialog:
    def __init__(self, app):
        self.app = app
        self.window = gtk.Dialog()
        self.window.set_title("Preferences")
        self.window.connect("delete_event", self.cancel)
        self.notebook = gtk.Notebook()
        self.notebook.set_tab_pos(gtk.POS_TOP)
        self.window.vbox.pack_start(self.notebook)
        self.notebook.show()
        for name, constructor in [
            (_("Event"),   self.make_page_events),
            (_("Font"),   self.make_page_fonts),
            (_("Mouse"),     self.make_page_mouse),
            (_("Browser"),   self.make_page_browser),
            (_("Helper"),   self.make_page_helper),
            (_("Misc"),   self.make_page_misc),
            ]:
            self.notebook.append_page(constructor(), gtk.Label(unicode(name, 'utf-8')))
        box = gtk.HButtonBox()
        box.set_layout(gtk.BUTTONBOX_END)
        self.window.action_area.pack_start(box)
        box.show()
        button = gtk.Button("OK")
        button.connect("clicked", self.ok)
        box.add(button)
        button.show()
        button = gtk.Button("Apply")
        button.connect("clicked", self.apply)
        box.add(button)
        button.show()
        button = gtk.Button("Cancel")
        button.connect("clicked", self.cancel)
        box.add(button)
        button.show()
        self.rule_editor = RuleEditor(self.window)
    def ok(self, widget):
        self.hide()
        self.app.notify_preferences_changed(1)
    def apply(self, widget):
        self.app.notify_preferences_changed(0)
    def cancel(self, widget, event=None):
        self.app.notify_preferences_reverted()
        self.hide()
        return gtk.TRUE
    def show(self):
        self.window.show()
    def hide(self):
        self.window.hide()
    def make_page_events(self):
        frame = gtk.Frame(unicode(_("Event(s) to be ignored"), 'utf-8'))
        frame.set_size_request(450, 250)
        frame.set_border_width(5)
        frame.show()
        box = gtk.VBox()
        frame.add(box)
        box.show()
        table = gtk.Table(8, 2, gtk.TRUE)
        table.set_row_spacings(3)
        table.set_col_spacings(5)
        table.set_border_width(5)
        box.pack_start(table, gtk.FALSE)
        table.show()
        events = [
            "OnBoot",          "OnClose",
            "OnGhostChanging", "OnGhostChanged",
            "OnShellChanging", "OnShellChanged",
            "OnSurfaceChange", "OnSurfaceRestore",
            "OnMinuteChange",  "OnSecondChange",
            "OnMouseClick",    "OnMouseDoubleClick",
            "OnMouseMove",     "OnMouseWheel",
            "OnKeyPress",
            ]
        self.event_kill_list = {}
        for i in range(len(events)):
            y, x = divmod(i, 2)
            button = gtk.CheckButton(events[i])
            table.attach(button, x, x+1, y, y+1)
            button.show()
            self.event_kill_list[events[i]] = button
        return frame
    def make_page_fonts(self):
        page = gtk.VBox(spacing=5)
        page.set_border_width(5)
        # font
        frame = gtk.Frame(unicode(_("Font(s) for balloons"), 'utf-8'))
        frame.set_size_request(450, -1)
        page.pack_start(frame)
        frame.show()
        self.fontsel = gtk.FontSelection()
        self.fontsel.show()
        frame.add(self.fontsel)
        page.show()
        return page
    def make_page_mouse(self):
        page = gtk.VBox(spacing=5)
        page.set_border_width(5)
        frame = gtk.Frame(unicode(_("Left button"), 'utf-8'))
        frame.set_size_request(400, 125)
        frame.show()
        page.pack_start(frame)
        box = gtk.VBox(spacing=5)
        frame.add(box)
        box.show()
        box.set_border_width(5)
        button1 = gtk.RadioButton(None, unicode(_("delete balloon(s)"), 'utf-8'))
        box.pack_start(button1, gtk.FALSE)
        button1.show()
        button2 = gtk.RadioButton(button1, unicode(_("raise all windows"), 'utf-8'))
        box.pack_start(button2, gtk.FALSE)
        button2.show()
        button3 = gtk.RadioButton(button1, unicode(_("lower all windows"), 'utf-8'))
        box.pack_start(button3, gtk.FALSE)
        button3.show()
        self.mouse_button1 = {
            ninix.sakura.BUTTON1_CLOSE: button1,
            ninix.sakura.BUTTON1_RAISE: button2,
            ninix.sakura.BUTTON1_LOWER: button3,
            }
        frame = gtk.Frame(unicode(_("Right button"), 'utf-8'))
        frame.set_size_request(400, 125)
        frame.show()
        page.pack_start(frame)
        box = gtk.VBox(spacing=5)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        button1 = gtk.RadioButton(None, unicode(_("delete balloon(s)"), 'utf-8'))
        box.pack_start(button1, gtk.FALSE)
        button1.show()
        button2 = gtk.RadioButton(button1, unicode(_("raise all windows"), 'utf-8'))
        box.pack_start(button2, gtk.FALSE)
        button2.show()
        button3 = gtk.RadioButton(button1, unicode(_("lower all windows"), 'utf-8'))
        box.pack_start(button3, gtk.FALSE)
        button3.show()
        self.mouse_button3 = {
            ninix.sakura.BUTTON3_CLOSE: button1,
            ninix.sakura.BUTTON3_RAISE: button2,
            ninix.sakura.BUTTON3_LOWER: button3,
            }
        page.show()
        return page
    def make_page_browser(self):
        page = gtk.VBox(spacing=5)
        page.set_border_width(5)
        # browser
        frame = gtk.Frame(unicode(_("Browser"), 'utf-8'))
        frame.set_size_request(450, -1)
        page.pack_start(frame)
        frame.show()
        box = gtk.VBox(spacing=2)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        self.browser = gtk.Entry()
        box.pack_start(self.browser, gtk.FALSE)
        self.browser.show()
        # help messages
        frame = gtk.Frame()
        frame.set_size_request(450, -1)
        page.pack_start(frame, gtk.FALSE)
        frame.show()
        box = gtk.VBox(spacing=2)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        label = gtk.Label(unicode(_("- %s in this command line will be replaced with the URL"), 'utf-8'))
        label.set_alignment(0, -1)
        box.pack_start(label, gtk.FALSE)
        label.show()
        label = gtk.Label(unicode(_("- trailing & is not required.(automagically added)"), 'utf-8'))
        label.set_alignment(0, -1)
        box.pack_start(label, gtk.FALSE)
        label.show()
        page.show()
        return page
    def make_page_helper(self):
        page = gtk.VBox(spacing=5)
        page.set_border_width(5)
        frame = gtk.Frame(unicode(_("Application"), 'utf-8'))
        frame.set_size_request(450, -1)
        page.pack_start(frame)
        frame.show()
        box = gtk.VBox(spacing=5)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        swin = gtk.ScrolledWindow()
        swin.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
        box.pack_start(swin)
        swin.show()
        self.helpers = gtk.CList(2, ["Pattern", "Command"])
        self.helpers.set_column_width(0, 100)
        self.helpers.set_column_auto_resize(1, gtk.TRUE)
        swin.add(self.helpers)
        self.helpers.show()
        bbox = gtk.HButtonBox()
        bbox.set_spacing(10)
        bbox.set_layout(gtk.BUTTONBOX_EDGE)
        ##bbox.set_child_size_default(60, -1) # FIXME
        bbox.set_border_width(5)
        box.pack_start(bbox, gtk.FALSE)
        bbox.show()
        button = gtk.Button("New")
        button.connect("clicked", self.rule_new)
        bbox.pack_start(button)
        button.show()
        button = gtk.Button("Edit")
        button.connect("clicked", self.rule_edit)
        bbox.pack_start(button)
        button.show()
        button = gtk.Button("Delete")
        button.connect("clicked", self.rule_delete)
        bbox.pack_start(button)
        button.show()
        button = gtk.Button("Up")
        button.connect("clicked", self.rule_up)
        bbox.pack_start(button)
        button.show()
        button = gtk.Button("Down")
        button.connect("clicked", self.rule_down)
        bbox.pack_start(button)
        button.show()
        frame = gtk.Frame()
        frame.set_size_request(450, -1)
        page.pack_start(frame, gtk.FALSE)
        frame.show()
        box = gtk.VBox(spacing=2)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        label = gtk.Label(unicode(_("- %s in this command line will be replaced with the filename"), 'utf-8'))
        label.set_alignment(0, -1)
        box.pack_start(label, gtk.FALSE)
        label.show()
        label = gtk.Label(unicode(_("- trailing & is not required.(automagically added)"), 'utf-8'))
        label.set_alignment(0, -1)
        box.pack_start(label, gtk.FALSE)
        label.show()
        page.show()
        return page
    def make_page_misc(self):
        scrn_h = gtk.gdk.screen_height()
        page = gtk.VBox(spacing=5)
        page.set_border_width(5)
        page.show()
        frame = gtk.Frame(unicode(_("Top & Bottom Margin"), 'utf-8'))
        frame.set_size_request(450, -1)
        page.pack_start(frame, gtk.FALSE)
        frame.show()
        box = gtk.VBox(spacing=5)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        hbox = gtk.HBox(spacing=5)
        box.add(hbox)
        hbox.show()
        label = gtk.Label(unicode(_("Top Margin"), 'utf-8'))
        hbox.pack_start(label, gtk.FALSE)
        label.show()
        self.top_adjustment = gtk.Adjustment(0, 0, scrn_h/4, 1)
        button = gtk.SpinButton(self.top_adjustment)
        hbox.pack_start(button, gtk.FALSE)
        button.show()
        hbox = gtk.HBox(spacing=5)
        box.add(hbox)
        hbox.show()
        label = gtk.Label(unicode(_("Bottom Margin"), 'utf-8'))
        hbox.pack_start(label, gtk.FALSE)
        label.show()
        self.bottom_adjustment = gtk.Adjustment(0, 0, scrn_h/4, 1)
        button = gtk.SpinButton(self.bottom_adjustment)
        hbox.pack_start(button, gtk.FALSE)
        button.show()
        frame = gtk.Frame(unicode(_("Default Balloon"), 'utf-8'))
        frame.set_size_request(450, -1)
        page.pack_start(frame)
        frame.show()
        box = gtk.VBox(spacing=5)
        box.set_border_width(5)
        frame.add(box)
        box.show()
        scrolled = gtk.ScrolledWindow()
        scrolled.set_policy(gtk.POLICY_NEVER, gtk.POLICY_ALWAYS)
        scrolled.set_shadow_type(gtk.SHADOW_ETCHED_IN)
        box.pack_start(scrolled, gtk.TRUE)
        scrolled.show()
        model = gtk.ListStore(gobject.TYPE_STRING)
        for desc, balloon in self.app.balloons_all:
            name = desc.get("name", "")
            iter = model.append()
            model.set_value(iter, 0, name)
        treeview = gtk.TreeView(model)
        column = gtk.TreeViewColumn(_("Balloon Name"), gtk.CellRendererText(), text=0)
        treeview.append_column(column)
        treeview.get_selection().set_mode(gtk.SELECTION_SINGLE)
        self.balloon_treeview = treeview
        scrolled.add(treeview)
        treeview.show()
        button = gtk.CheckButton(unicode(_("Always Use This Balloon"), 'utf-8'))
        self.ignore_button = button
        box.pack_start(button, gtk.FALSE)
        button.show()
        return page
    def rule_new(self, widget):
        self.rule_editor.set_pattern("")
        self.rule_editor.set_command("")
        if self.rule_editor.run("New rule..."):
            pattern = self.rule_editor.get_pattern()
            command = self.rule_editor.get_command()
            self.helpers.freeze()
            self.helpers.append((pattern, command))
            self.helpers.thaw()
    def rule_edit(self, widget):
        if not self.helpers.selection:
            return
        row = self.helpers.selection[0]
        self.rule_editor.set_pattern(self.helpers.get_text(row, 0))
        self.rule_editor.set_command(self.helpers.get_text(row, 1))
        if self.rule_editor.run("Edit rule..."):
            self.helpers.freeze()
            self.helpers.set_text(row, 0, self.rule_editor.get_pattern())
            self.helpers.set_text(row, 1, self.rule_editor.get_command())
            self.helpers.thaw()
    def rule_delete(self, widget):
        if not self.helpers.selection:
            return
        row = self.helpers.selection[0]
        self.helpers.remove(row)
    def rule_up(self, widget):
        if not self.helpers.selection:
            return
        row = self.helpers.selection[0]
        if row > 0:
            self.helpers.swap_rows(row, row - 1)
    def rule_down(self, widget):
        if not self.helpers.selection:
            return
        row = self.helpers.selection[0]
        if row < self.helpers.rows - 1:
            self.helpers.swap_rows(row, row + 1)
    def set_event_kill_list(self, kill_list):
        for name in self.event_kill_list.keys():
            if name == "OnFirstBoot":
                continue
            if name in kill_list:
                self.event_kill_list[name].set_active(gtk.TRUE)
            else:
                self.event_kill_list[name].set_active(gtk.FALSE)
    def get_event_kill_list(self):
        buffer = []
        for name in self.event_kill_list.keys():
            if self.event_kill_list[name].get_active():
                buffer.append(name)
                if name == "OnBoot":
                    buffer.append("OnFirstBoot")
        return buffer
    def set_balloon_fonts(self, name):
        self.fontsel.set_font_name(name)
    def get_balloon_fonts(self):
        name = self.fontsel.get_font_name()
        return name
    def set_mouse_button1(self, name):
        self.mouse_button1[name].set_active(gtk.TRUE)
    def get_mouse_button1(self):
        for name in self.mouse_button1.keys():
            if self.mouse_button1[name].get_active():
                return name
        return None # should not reach here
    def set_mouse_button3(self, name):
        self.mouse_button3[name].set_active(gtk.TRUE)
    def get_mouse_button3(self):
        for name in self.mouse_button3.keys():
            if self.mouse_button3[name].get_active():
                return name
        return None # should not reach here
    def set_browser(self, command):
        self.browser.set_text(command)
    def get_browser(self):
        return self.browser.get_text()
    def set_helpers(self, list):
        self.helpers.freeze()
        self.helpers.clear()
        for pattern, command in list:
            self.helpers.append((pattern, command))
        self.helpers.thaw()
    def get_helpers(self):
        list = []
        for row in range(self.helpers.rows):
            pattern = self.helpers.get_text(row, 0)
            command = self.helpers.get_text(row, 1)
            list.append((pattern, command))
        return list
    def set_top_margin(self, margin):
        self.top_adjustment.set_value(margin)
    def get_top_margin(self):
        margin = self.top_adjustment.get_value()
        return int(margin)
    def set_bottom_margin(self, margin):
        self.bottom_adjustment.set_value(margin)
    def get_bottom_margin(self):
        margin = self.bottom_adjustment.get_value()
        return int(margin)
    def set_default_balloon(self, name):
        model = self.balloon_treeview.get_model()
        iter = model.get_iter_first()
        while iter != None:
            value = model.get_value(iter, 0)
            if value == name or name == None:
                self.balloon_treeview.get_selection().select_iter(iter)
                break
            iter = model.iter_next(iter) 
    def get_default_balloon(self):
        name = None
        selected = self.balloon_treeview.get_selection().get_selected()
        if selected:
            model, iter = selected
            name = model.get_value(iter, 0)
        return name
    def set_ignore_default(self, flag):
        if flag:
            self.ignore_button.set_active(gtk.TRUE)
        else:
            self.ignore_button.set_active(gtk.FALSE)
    def get_ignore_default(self):
        if self.ignore_button.get_active():
            return 1
        else:
            return 0

class RuleEditor:
    def __init__(self, master=None):
        self.dialog = gtk.Dialog()
        self.dialog.connect("delete_event", self.cancel)
        self.dialog.set_modal(gtk.TRUE)
        self.dialog.set_position(gtk.WIN_POS_CENTER)
        if master is not None:
            self.dialog.set_transient_for(master)
        # entries
        table = gtk.Table(2, 2)
        table.set_row_spacings(5)
        table.set_col_spacings(5)
        table.set_border_width(10)
        self.dialog.vbox.pack_start(table)
        label = gtk.Label("Pattern")
        table.attach(label, 0, 1, 0, 1, xoptions=gtk.FILL)
        self.pattern_entry = gtk.Entry()
        self.pattern_entry.set_size_request(300, -1)
        self.pattern_entry.connect("changed", self.changed)
        table.attach(self.pattern_entry, 1, 2, 0, 1)
        label = gtk.Label("Command")
        table.attach(label, 0, 1, 1, 2, xoptions=gtk.FILL)
        self.command_entry = gtk.Entry()
        self.command_entry.set_size_request(300, -1)
        self.command_entry.connect("changed", self.changed)
        table.attach(self.command_entry, 1, 2, 1, 2)
        self.dialog.vbox.show_all()
        # buttons
        self.ok_button = gtk.Button("OK")
        self.ok_button.connect("clicked", self.ok)
        self.dialog.action_area.pack_start(self.ok_button)
        button = gtk.Button("Cancel")
        button.connect("clicked", self.cancel)
        self.dialog.action_area.pack_start(button)
        self.dialog.action_area.show_all()
    def set_pattern(self, text):
        self.pattern_entry.set_text(text)
    def get_pattern(self):
        return self.pattern_entry.get_text()
    def set_command(self, text):
        self.command_entry.set_text(text)
    def get_command(self):
        return self.command_entry.get_text()
    def changed(self, widget, event=None):
        if self.pattern_entry.get_text() and \
           self.command_entry.get_text():
            self.ok_button.set_sensitive(gtk.TRUE)
        else:
            self.ok_button.set_sensitive(gtk.FALSE)
    def run(self, title):
        self.dialog.set_title(title)
        self.dialog.show()
        gtk.mainloop()
        return self.done
    def hide(self):
        self.dialog.hide()
        gtk.mainquit()
    def ok(self, widget, event=None):
        self.done = 1
        self.hide()
        return gtk.TRUE
    def cancel(self, widget, event=None):
        self.done = 0
        self.hide()
        return gtk.TRUE

class UsageDialog:
    def __init__(self):
        self.window = gtk.Dialog()
        self.window.set_title("Usage")
        self.window.connect("delete_event", self.close)
        self.darea = gtk.DrawingArea()
        self.darea.set_events(gtk.gdk.EXPOSURE_MASK)
        self.size = (450, 320)
        apply(self.darea.set_size_request, self.size)
        self.darea.connect("configure_event", self.configure)
        self.darea.connect("expose_event", self.redraw)
        self.window.vbox.pack_start(self.darea)
        self.darea.show()
        box = gtk.HButtonBox()
        box.set_layout(gtk.BUTTONBOX_END)
        self.window.action_area.pack_start(box)
        box.show()
        button = gtk.Button("Close")
        button.connect("clicked", self.close)
        box.add(button)
        button.show()
        self.opened = 0
    def open(self, history):
        if self.opened:
            return
        self.history = history
        self.window.show()
        self.opened = 1
    def close(self, widget=None, event=None):
        self.window.hide()
        self.opened = 0
        return gtk.TRUE
    def configure(self, darea, event):
        x, y, w, h = darea.get_allocation()
        self.size = (w, h)
    def redraw(self, darea, event):
        items = []
        for name in self.history.keys():
            items.append((name, self.history[name]))
        items.sort(lambda x, y: cmp(y[1], x[1]))
        if not items:
            return # should not reach here
        total = float(0)
        for name, clock in items:
            total = total + clock
        # prepare gc
        layout = pango.Layout(darea.get_pango_context())
        font_desc = pango.FontDescription()
        font_desc.set_size(9 * 1024) # size * PANGO_SCALE
        font_desc.set_family('Sans') # FIXME
        layout.set_font_description(font_desc)
        white_gc = darea.get_style().white_gc
        black_gc = darea.get_style().black_gc
        cmap = darea.get_colormap()
        gray_gc = darea.window.new_gc()
        gray_gc.foreground = cmap.alloc_color("#cfcfcf")
        # redraw graph
        w, h = self.size
        darea.window.draw_rectangle(white_gc, gtk.TRUE, 0, 0, w, h)
        w3 = w4 = 0
        rows = []
        for name, clock in items[:16]:
            layout_set_text(layout, name)
            name_w, name_h = layout.get_pixel_size()
            rate = "%.1f%%" % (clock / total * 100)
            layout_set_text(layout, rate)
            rate_w, rate_h = layout.get_pixel_size()
            w3 = max(rate_w, w3)
            time = "%d:%02d" % divmod(clock / 60, 60)
            layout_set_text(layout, time)
            time_w, time_h = layout.get_pixel_size()
            w4 = max(time_w, w4)
            rows.append((clock, name, name_w, name_h, rate, rate_w, rate_h, time, time_w, time_h))
        w1 = w2 = (w - w3 - w4 - 70) / 2
        x = 20
        y = 15
        x = x + w1 + 10
        label = "name"
        layout_set_text(layout, label)
        label_name_w, label_name_h = layout.get_pixel_size()
        darea.window.draw_layout(gray_gc, x, y, layout)
        x = x + w2 + 10
        label = "rate"
        layout_set_text(layout, label)
        label_rate_w, label_rate_h = layout.get_pixel_size()
        darea.window.draw_layout(gray_gc, x + w3 - label_rate_w, y, layout)
        x = x + w3 + 10
        label = "time"
        layout_set_text(layout, label)
        label_time_w, label_time_h = layout.get_pixel_size()
        darea.window.draw_layout(gray_gc, x + w4 - label_time_w, y, layout)
        y = y + max([label_name_h, label_rate_h, label_time_h]) + 4
        for clock, name, name_w, name_h, rate, rate_w, rate_h, time, time_w, time_h  in rows:
            x = 20
            bw = int(clock / total * w1)
            bh = max([name_h, rate_h, time_h]) - 1
            darea.window.draw_rectangle(gray_gc,  gtk.FALSE, x+1, y+1, bw, bh)
            darea.window.draw_rectangle(white_gc, gtk.TRUE,  x,   y,   bw, bh)
            darea.window.draw_rectangle(black_gc, gtk.FALSE, x,   y,   bw, bh)
            x = x + w1 + 10
            layout_set_text(layout, name)
            end = len(name)
            while end > 0:
                w, h = layout.get_pixel_size()
                if w > 168:
                    end = end - 1
                    layout_set_text(layout, name[:end] + u'...')
                else:
                    break
            darea.window.draw_layout(black_gc, x, y, layout)
            x = x + w2 + 10
            layout_set_text(layout, rate)
            darea.window.draw_layout(black_gc, x + w3 - rate_w, y, layout)
            x = x + w3 + 10
            layout_set_text(layout, time)
            darea.window.draw_layout(black_gc, x + w4 - time_w, y, layout)
            y = y + max([name_h, rate_h, time_h]) + 4

class VanishDialog:
    def __init__(self, app):
        self.app = app
        self.window = gtk.Dialog()
        self.window.connect("delete_event", self.cancel)
        self.window.set_title("Vanish")
        self.window.set_modal(gtk.TRUE)
        self.window.set_position(gtk.WIN_POS_CENTER)
        self.label = gtk.Label(unicode(_("Vanish"), 'utf-8'))
        self.window.vbox.pack_start(self.label, padding=10)
        self.label.show()
        box = gtk.HButtonBox()
        box.set_layout(gtk.BUTTONBOX_END)
        self.window.action_area.pack_start(box)
        box.show()
        button = gtk.Button(unicode(_("Yes"), 'utf-8'))
        button.connect("clicked", self.ok)
        box.add(button)
        button.show()
        button = gtk.Button(unicode(_("No"), 'utf-8'))
        button.connect("clicked", self.cancel)
        box.add(button)
        button.show()
    def set_message(self, message):
        self.label.set_text(message)
    def show(self):
        self.window.show()
    def ok(self, widget, event=None):
        self.window.hide()
        self.app.notify_vanish_confirmation(1)
        return gtk.TRUE
    def cancel(self, widget, event=None):
        self.window.hide()
        self.app.notify_vanish_confirmation(0)
        return gtk.TRUE

if __name__ == "__main__":
    main()
