/***************************************************************************
                          qgsmaplayer.h  -  description
                             -------------------
    begin                : Fri Jun 28 2002
    copyright            : (C) 2002 by Gary E.Sherman
    email                : sherman at mrcc.com
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
/* $Id: qgsmaplayer.h 6265 2006-12-15 23:22:01Z g_j_m $ */

#ifndef QGSMAPLAYER_H
#define QGSMAPLAYER_H

#include <vector>
#include <map>

#include <QObject>
#include <QPixmap>
#include <QMenu> // not forward declared because other .h files
                 // inherit from us and we're being nice to them.

#include "qgis.h"
#include "qgsrect.h"
#include "qgsfield.h"
#include "qgscoordinatetransform.h"

class QAction;
class QgisApp;
class QgsMapToPixel;
class QgsFeature;
class QgsLegend;
class QgsLegendLayerFile;

class QDomNode;
class QDomDocument;
class QKeyEvent;

/** \class QgsMapLayer
 * \brief Base class for all map layer types.
 * This class is the base class for all map layer types (shapefile,
 * raster, database).
 */
class QgsMapLayer : public QObject
{
    Q_OBJECT;

public:

    /*! Constructor
     * @param type Type of layer as defined in LAYERS enum
     * @param lyrname Display Name of the layer
     */
    QgsMapLayer(int type = 0, QString lyrname = QString::null, QString source = QString::null);

    //! Destructor
    virtual ~ QgsMapLayer();

    /*! Get the type of the layer
     * @return Integer matching a value in the LAYERS enum
     */
    int type() const;

    /*! Get this layer's unique ID */
    QString const & getLayerID() const;

    /*! Set the display name of the layer
       # @param name New name for the layer
       # @param updateLegend false if legend should not be updated
       (to avoid infinite recursion when this function is called from the legend itself)
     */
    void setLayerName(const QString & name, bool updateLegend=true);

    /*! Get the display name of the layer
     * @return the layer name
     */
    QString const & name() const;

    /*! Get the internal name of the layer. This is the name used to created the
     * layer from the data source
     * @return internal datasource name of the layer
     */
    QString const & sourceName() const;

    /*! Virtual function to calculate the extent of the current layer.
     * This function must be overridden in all child classes and implemented
     * based on the layer type
     */
    virtual QgsRect calculateExtent();

    /*! Accesor for mShowInOverviewFlag */
    bool showInOverviewStatus()
    {
        return mShowInOverview;
    }


    virtual void draw(QPainter *, QgsRect *, int);

    //! Returns FALSE if an error occurred during drawing
    virtual bool draw(QPainter *, QgsRect *, QgsMapToPixel *, bool);
    virtual void drawLabels(QPainter *, QgsRect *, QgsMapToPixel *);

    /*!Select features on the map canvas by dragging a rectangle */
    virtual void select(QgsRect *, bool )
    {}
    ;

    //! Display the attribute table for the layer
    /**
       \param qgisApp   This should be the QgisApp that spawned this table.
     */
    virtual void table(QgisApp * qgisApp)
    {};

    /*! Return the extent of the layer as a QRect
     */
    const QgsRect extent();

    /*! Returns the status of the layer. An invalid layer is one which has a bad datasource
     * or other problem. Child classes set this flag when intialized
     *@return True if the layer is valid and can be accessed
     */
    bool isValid();

    /** Write property of QString labelField. */
    virtual void setLabelField(const QString & _newVal);

    /** Read property of QString labelField. */
    virtual const QString & labelField();

    //! Visibility of the layer
    bool visible();

    /*! Gets a version of the internal layer definition that has sensitive
     *  bits removed (for example, the password). This function should
     * be used when displaying the source name for general viewing.
    */
    QString publicSource() const;

    //! Returns the source for the layer
    QString const & source() const;

    /** Write property of int featureType. */
    virtual void setFeatureType(const int &_newVal);

    /** Read property of int featureType. */
    virtual const int &featureType();

    /** Return the context menu for the layer */
    virtual QMenu* contextMenu();

    /**
     * Returns the sublayers of this layer
     *
     * (Useful for providers that manage their own layers, such as WMS)
     *
     */
    virtual QStringList subLayers()
    {
      return QStringList();  // Empty
    }
    
    /**
     * Reorders the *previously selected* sublayers of this layer from bottom to top
     *
     * (Useful for providers that manage their own layers, such as WMS)
     *
     */
    virtual void setLayerOrder(QStringList layers)
    {
      // NOOP
    }
    
    /**
     * Set the visibility of the given sublayer name
     */
    virtual void setSubLayerVisibility(QString name, bool vis)
    {
      // NOOP
    }

    //! Layers enum defining the types of layers that can be added to a map
    enum LAYERS
    {
        VECTOR,
        RASTER,
        DATABASE
    };

    /**Shows the properties dialog for the map layer*/
    virtual void showLayerProperties() = 0;

    /**
        All inherited layers must be able to display a conext menu if requested

        @note Calls initContextMenu_()
    */
    void initContextMenu(QgisApp * app);

    /**
        Allows children to tailor context menu

        @note Calls initContextMenu_()
    */
    virtual void initContextMenu_(QgisApp * app)
    {
        // NOP; children can optionally over-ride
    }

    void setLegendLayerFile(QgsLegendLayerFile* llf) {mLegendLayerFile = llf;}

    /**True if the layer can be edited*/
    virtual bool isEditable() const =0;

    /** sets state from DOM document

       @param layer_node is DOM node corresponding to ``maplayer'' tag

       @note

       The DOM node corresponds to a DOM document project file XML element read
       by QgsProject.

       This, in turn, calls readXML_(), which is over-rideable by sub-classes so
       that they can read their own specific state from the given DOM node.

       Invoked by QgsProject::read().

       @returns true if successful

     */
    bool readXML( QDomNode & layer_node );


    /** stores state in DOM node

       @param layer_node is DOM node corresponding to ``projectlayers'' tag

       @note

       The DOM node corresponds to a DOM document project file XML element to be
       written by QgsProject.

       This, in turn, calls writeXML_(), which is over-rideable by sub-classes so
       that they can write their own specific state to the given DOM node.

       Invoked by QgsProject::write().

       @returns true if successful

    */
    bool writeXML( QDomNode & layer_node, QDomDocument & document );

    /** Accessor for the coordinate transformation object */
    QgsCoordinateTransform * coordinateTransform();

    /** A simple helper method to find out if on the fly projections 
        are enabled or not */
    bool projectionsEnabled() const;
    
    // Convenience function to project an extent into the layer source
    // SRS, but also split it into two extents if it crosses
    // the +/- 180 degree line. Modifies the given extent to be in the
    // source SRS coordinates, and if it was split, returns true, and
    // also sets the contents of the r2 parameter
    bool projectExtent(QgsRect& extent, QgsRect& r2);

    /**Returns the path to an icon which characterises the type of layer*/
    virtual QString layerTypeIconPath() = 0;

    void setLegend(QgsLegend* legend) {mLegend = legend;}
    const QgsLegend* legend() {return mLegend;}

    /**Refresh the symbology part of the legend. Specific implementations have to be provided by subclasses*/
    virtual void refreshLegend() = 0;

    /**Copies the symbology settings from another layer. Returns true in case of success*/
    virtual bool copySymbologySettings(const QgsMapLayer& other) = 0;

    /**Returns true if this layer can be in the same symbology group with another layer*/
    virtual bool isSymbologyCompatible(const QgsMapLayer& other) const = 0;

    virtual std::vector < QgsField > const &fields() const;

    /** \brief accessor for transparency level.  */
    virtual unsigned int getTransparency()=0;


    /**
     * If an operation returns 0 (e.g. draw()), this function
     * returns the text of the error associated with the failure.
     * Interactive users of this provider can then, for example,
     * call a QMessageBox to display the contents.
     */
    virtual QString errorCaptionString();
  
    /**
     * If an operation returns 0 (e.g. draw()), this function
     * returns the text of the error associated with the failure.
     * Interactive users of this provider can then, for example,
     * call a QMessageBox to display the contents.
     */
    virtual QString errorString();

public  slots:
    /** \brief Mutator for transparency level. Should be between 0 and 255 */
    virtual void setTransparency(unsigned int)=0;
   //! event handler for when a coordinate transofrm fails due to bad vertex error
   virtual void invalidTransformInput();


    //! keyPress event so we can check if cancel was pressed
    void keyPressed ( QKeyEvent * e );

    //! set visibility
    void setVisible(bool vis);

    /*! Slot connected to popup menus of derived classes. Used to indicate whether this layer
     * should be shown or hidden in the map overview. */
    //virtual void toggleShowInOverview();

    /**Copies the legend pixmap of this layer to the legenditem and adds an overview glasses if necessary*/
    void updateItemPixmap();

    /**Ensures that the overview item in the popup menu is checked/ unchecked correctly*/
    void updateOverviewPopupItem();

    /** set whether this layer is in the overview or not

    @note this will hopefully eventually supercede toggleOverviewStatus() since
    this makes explicit the state change

    */
    virtual void inOverview( bool );

    /** Accessor and mutator for the minimum scale member */
    void setMinScale(float theMinScale)
    {
        mMinScale=theMinScale;
    };
    float minScale()
    {
        return mMinScale;
    };

    /** Accessor and mutator for the maximum scale member */
    void setMaxScale(float theMaxScale)
    {
        mMaxScale=theMaxScale;
    };
    float maxScale()
    {
        return mMaxScale;
    };

    /** Accessor and mutator for the scale based visilibility flag */
    void setScaleBasedVisibility( bool theVisibilityFlag)
    {
        mScaleBasedVisibility=theVisibilityFlag;
    };
    bool scaleBasedVisibility()
    {
        return mScaleBasedVisibility;
    };

    /** Used to ask the layer for its projection as a WKT string. Must be reimplemented by each provider. */
    virtual QString getProjectionWKT()  = 0 ;
signals:

    void visibilityChanged(void);

    /** \brief emit a signal to notify of a progress event */
    void drawingProgress(int theProgress, int theTotalSteps);

    /** \brief emit a signal to be caught by gisapp and display a msg on status bar */
    void setStatus(QString theStatusQString);

    /** This signal should be connected with the slot QgsMapCanvas::refresh() */
    void repaintRequested();

    /** This is used to notify the application whether this layer should be shown in overview or not. */
    //@{
    //void showInOverview(QString theLayerId, bool);
    void showInOverview(QgsMapLayer * maplayer, bool);
    //@}

    /** This is used to send a request that any mapcanvas using this layer update its extents */
    void recalculateExtents();

protected:

    /** \brief Transparency level for this layer should be 0-255 (255 being opaque)  */
    unsigned int transparencyLevelInt;
  
    /** called by readXML(), used by children to read state specific to them from
        project files.
    */
    virtual bool readXML_( QDomNode & layer_node );

    /** called by writeXML(), used by children to write state specific to them to
        project files.
    */
    virtual bool writeXML_( QDomNode & layer_node, QDomDocument & document );


    //! Extent of the layer
    QgsRect layerExtent;

    //! Indicates if the layer is valid and can be drawn
    bool valid;

    //! data source description string, varies by layer type
    QString dataSource;

    //! Geometry type as defined in enum WKBTYPE (qgis.h)
    int geometryType;

    /** Pixmap to show a bogus vertex was encoutnered in this layer (applies to vector layers only) */
    QPixmap mProjectionErrorPixmap;

    //! A little pixmap to show if this layer is represented in overview or now
    QPixmap mInOverviewPixmap;

    QPixmap mEditablePixmap;

    /** Internal name of the layer. Derived from the datasource */
    QString internalName;

    //! context menu
    QMenu* popMenu;

    //! the action in popmenu that sets overview status
    QAction* mShowInOverviewAction;

    /** Whether this layer is to be shown in the overview map or not */
    bool mShowInOverview;

    //   /** action item for pop-up menu

    //       @note obviously should be in synch with mShowInOverview

    //       Is set in context menu.

    //       @todo this is a GUI element and should not be here
    //   */
    //   QAction* mActionInOverview;
    /** A flag to let the draw() render loop know if the user has requested drawing be cancelled */
    bool mDrawingCancelled;

    //! A QgsCoordinateTransform is used for on the fly reprojection of map layers
    QgsCoordinateTransform * mCoordinateTransform; 

    /**Pointer to the legend layer item of the legend*/
    QgsLegend* mLegend;

    /**Pointer to the legend layer file of the legend. It is used to modify the pixmap with overview
     glasses, editing or pyramid symbols*/
    QgsLegendLayerFile* mLegendLayerFile;

private:                       // Private attributes

    /// QgsMapLayer not copyable
    QgsMapLayer( QgsMapLayer const & );

    /// QgsMapLayer not copyable
    QgsMapLayer & operator=( QgsMapLayer const & );

    /// A convenience function to capitalise the layer name
    static QString capitaliseLayerName(const QString name);

    /** Unique ID of this layer - used to refer to this layer  in QGIS code */
    QString ID;

    /** Type of the layer (eg. vector, raster, database  */
    int layerType;

    //! Tag for embedding additional information
    QString tag;

    /** Name of the layer - used for display  */
    QString layerName;
    
    /**  true if visible ? */
    bool m_visible;

    /** debugging member
        invoked when a connect() is made to this object
    */
    void connectNotify( const char * signal );

    // Calculates the bounding box of the given extent in the inverse
    // projected spatial reference system.
    QgsRect calcProjectedBoundingBox(QgsRect& extent);

    /** Minimum scale at which this layer should be displayed */
    float mMinScale;
    /** Maximum scale at which this layer should be displayed */
    float mMaxScale;
    /** A flag that tells us whether to use the above vars to restrict layer visibility */
    bool mScaleBasedVisibility;

public:                        // Public attributes

    /** map label ? */
    QString m_labelField;

};

#endif
