# -*- python -*-
#
# Copyright (C) 2006 Red Hat, Inc.
# Copyright (C) 2006 Daniel P. Berrange <berrange@redhat.com>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
# MA 02110-1301 USA.
#

import os
import os.path
import sys

import libvirt

import locale
import gettext
import logging
import logging.handlers
import threading
import traceback
import signal
from optparse import OptionParser, OptionValueError

try:
    # Make sure we have a default '_' implementation, in case something
    # fails before gettext is set up
    __builtins__._ = lambda msg: msg
except:
    pass

# These are substituted into code based on --prefix given to configure
appname = "virt-manager"
appversion = "0.8.5"
gettext_app = "virt-manager"
gettext_dir = "/usr/share/locale"
virtinst_str = "0.500.4"
virtinst_version = tuple([ int(num) for num in virtinst_str.split('.')])

gconf_dir = "/apps/" + appname
asset_dir = "/usr/share/virt-manager"
glade_dir = asset_dir
icon_dir = asset_dir + "/pixmaps"
pylib_dir = "/usr/share/virt-manager"
pyarchlib_dir = "/usr/lib/virt-manager"
data_dir = "/usr/share"

def _show_startup_error(msg, details):
    import gtk
    from virtManager.error import vmmErrorDialog
    err = vmmErrorDialog(None, 0, gtk.MESSAGE_ERROR, gtk.BUTTONS_CLOSE, "", "")
    title = _("Error starting Virtual Machine Manager")
    err.show_err(title + ": " +  msg, details, title, async=False)

def setup_i18n():
    locale.setlocale(locale.LC_ALL, '')
    gettext.install(gettext_app, gettext_dir)
    gettext.bindtextdomain(gettext_app, gettext_dir)

def setup_glade_i18n():
    import gtk.glade
    gtk.glade.bindtextdomain(gettext_app, gettext_dir)
    gtk.glade.textdomain(gettext_app)

def setup_pypath():
    global glade_dir, icon_dir, data_dir
    # Hacks for find assets in local dir for dev purposes
    if os.path.exists(os.getcwd() + "/src/vmm-about.glade"):
        glade_dir = os.getcwd() + "/src"
    if os.path.exists(os.getcwd() + "/pixmaps/state_running.png"):
        icon_dir = os.getcwd() + "/pixmaps"
    if os.path.exists(os.getcwd() + "../gnome/help/virt-manager/C/virt-manager.xml"):
        data_dir = os.getcwd() + "../"

    # First 2 hacks are to point python to local dir for source files in dev,
    # the third is the main path if you have  normal install
    if os.path.exists(os.getcwd() + "/src/virt-manager.py"):
        pass
    elif os.path.exists(os.getcwd() + "/build/src/virt-manager.py"):
        sys.path.insert(0, os.getcwd() + "/src")
    else:
        sys.path.insert(0, pylib_dir)
        sys.path.insert(0, pyarchlib_dir)

def drop_tty():
    # We fork and setsid so that we drop the controlling
    # tty. This prevents libvirt's SSH tunnels from prompting
    # for user input if SSH keys/agent aren't configured.
    if os.fork() != 0:
        os._exit(0)

    os.setsid()

def drop_stdio():
    # We close STDIN/OUT/ERR since they're generally spewing
    # junk to console when domains are in process of shutting
    # down. Real errors will (hopefully) all be logged to the
    # main log file. This is also again to stop SSH prompting
    # for input
    for fd in range(0, 2):
        try:
            os.close(fd)
        except OSError:
            pass

    os.open(os.devnull, os.O_RDWR)
    os.dup2(0, 1)
    os.dup2(0, 2)

def setup_logging(debug_stdout):
    # Configure python logging to capture all logs we generate
    # to $HOME/.virt-manager/virt-manager.log This file has
    # proved invaluable for debugging
    MAX_LOGSIZE   = 1024 * 1024  # 1MB
    ROTATE_NUM    = 5
    DIR_NAME      = ".virt-manager"
    FILE_NAME     = "virt-manager.log"
    FILE_MODE     = 'a'
    FILE_FORMAT   = "[%(asctime)s virt-manager %(process)d] %(levelname)s (%(module)s:%(lineno)d) %(message)s"
    DATEFMT       = "%a, %d %b %Y %H:%M:%S"

    # set up logging
    vm_dir = os.path.expanduser("~/%s" % DIR_NAME)
    if not os.access(vm_dir, os.W_OK):
        if os.path.exists(vm_dir):
            raise RuntimeError("No write access to %s" % vm_dir)

        try:
            os.mkdir(vm_dir, 0751)
        except IOError, e:
            raise RuntimeError("Could not create directory %s: %s" %
                               (vm_dir, e))

    # XXX should we get logging level from gconf, or command line args ?
    filename = "%s/%s" % (vm_dir, FILE_NAME)
    rootLogger = logging.getLogger()
    rootLogger.setLevel(logging.DEBUG)
    fileHandler = logging.handlers.RotatingFileHandler(filename, FILE_MODE, MAX_LOGSIZE, ROTATE_NUM)
    fileHandler.setFormatter(logging.Formatter(FILE_FORMAT, DATEFMT))
    rootLogger.addHandler(fileHandler)

    if debug_stdout:
        streamHandler = logging.StreamHandler(sys.stderr)
        streamHandler.setLevel(logging.DEBUG)
        streamHandler.setFormatter(logging.Formatter("%(asctime)s (%(module)s:%(lineno)d): %(message)s"))
        rootLogger.addHandler(streamHandler)


    logging.info("Application startup")

    # Register libvirt handler
    def libvirt_callback(ctx, err):
        if err[3] != libvirt.VIR_ERR_ERROR:
            # Don't log libvirt errors: global error handler will do that
            logging.warn("Non-error from libvirt: '%s'" % err[2])
    libvirt.registerErrorHandler(f=libvirt_callback, ctx=None)

    # Log uncaught exceptions
    def exception_log(typ, val, tb):
        if not traceback:
            return
        s = traceback.format_exception(typ, val, tb)
        logging.exception("".join(s))
        sys.__excepthook__(typ, val, tb)
    sys.excepthook = exception_log

def parse_commandline():
    optParser = OptionParser(version=appversion)
    optParser.add_option("--profile", dest="profile", help="Generate runtime performance profile stats", metavar="FILE")
    optParser.set_defaults(uuid=None)
    optParser.add_option("-c", "--connect", dest="uri",
        help="Connect to hypervisor at URI", metavar="URI")
    optParser.add_option("--debug", action="store_true", dest="debug",
        help="Print debug output to stdout (implies --no-fork)",
        default=False)
    optParser.add_option("--no-dbus", action="store_true", dest="nodbus",
        help="Disable DBus service for controlling UI")
    optParser.add_option("--no-fork", action="store_true", dest="nofork",
        help="Don't fork into background on startup")
    optParser.add_option("--no-conn-autostart", action="store_true",
                         dest="no_conn_auto",
                         help="Do not autostart connections")
    optParser.add_option("--show-domain-creator", action="callback",
        callback=opt_show_cb, dest="show", help="Create a new virtual machine")
    optParser.add_option("--show-domain-editor",  type="string", metavar="UUID",
        action="callback", callback=opt_show_cb, help="Edit a domain configuration")
    optParser.add_option("--show-domain-performance", type="string", metavar="UUID",
        action="callback", callback=opt_show_cb, help="Show a domain performance")
    optParser.add_option("--show-domain-console", type="string", metavar="UUID",
        action="callback", callback=opt_show_cb, help="Show a domain console")
    optParser.add_option("--show-host-summary", action="callback",
       callback=opt_show_cb, help="Show a host summary")

    return optParser.parse_args()

def launch_specific_window(engine, show, uri, uuid):
    if not engine.wait_for_open(uri):
        # Connection failed, don't attempt to continue
        return
    if show=='creator':
        engine.show_create(uri)
    elif show=='editor':
        engine.show_details_config(uri, uuid)
    elif show=='performance':
        engine.show_details_performance(uri, uuid)
    elif show=='console':
        engine.show_console(uri, uuid)

# maps --show-* to engine (ie local instance) methods
def show_engine(engine, show, uri, uuid, no_conn_auto):
    if show=='creator' or show=='editor' \
       or show=='performance' or show=='console':
        # Create a thread so we can wait for connection to _fully_ start,
        # to prevent races accessing uninit'd vars for ex. the create wizard
        # which expects the connection to be active.
        thread = threading.Thread(target=launch_specific_window,
                                  args=(engine, show, uri, uuid),
                                  name="Launching '%s' window" % show)
        thread.start()

    elif show=='summary' or uri:
        engine.connect_to_uri(uri)
    else:
        engine.show_manager()

    if not no_conn_auto:
        engine.autostart_connections()


# maps --show-* to remote manager (ie dbus call) methods
def show_remote(managerObj, show, uri, uuid):
    if show=='creator' or show=='editor' \
       or show=='performance' or show=='console':
        # Create a thread so we can wait for connection to _fully_ start,
        # to prevent races accessing uninit'd vars for ex. the create wizard
        # which expects the connection to be active.
        thread = threading.Thread(target=launch_specific_window,
                                  args=(managerObj, show, uri, uuid),
                                  name="Launching '%s' window" % show)
        thread.start()
    elif show=='summary' or uri:
        managerObj.show_host_summary(uri)
    else:
        managerObj.show_manager()

# Generic OptionParser callback for all --show-* options
# This routine stores UUID to options.uuid for all --show-* options
# where is metavar="UUID" and also sets options.show
def opt_show_cb(option, opt_str, value, parser):
    if option.metavar=="UUID":
        setattr(parser.values, "uuid", value)
    s = str(option)
    show = s[s.rindex('-')+1:]
    setattr(parser.values, "show", show)

# Run me!
def main():
    setup_i18n()
    setup_pypath()

    # Need to do this before GTK strips args like --sync
    origargs = " ".join(sys.argv[:])

    # Urgh, pygtk merely logs a warning when failing to open
    # the X11 display connection, and lets everything carry
    # on as if all were fine. Ultimately bad stuff happens,
    # so lets catch it here & get the hell out...
    import warnings
    warnings.filterwarnings('error', module='gtk')
    try:
        import gobject

        # Set program name for gnome shell (before importing gtk, which
        # seems to call set_prgname on its own)
        if hasattr(gobject, "set_prgname"):
            gobject.set_prgname(appname)

        import gtk
    except Warning, e:
        # ...the risk is we catch too much though
        # Damned if we do, damned if we dont :-)(
        raise RuntimeError(_("Unable to initialize GTK: %s") % str(e))
    warnings.resetwarnings()

    (options, ignore) = parse_commandline()
    setup_logging(options.debug)

    logging.debug("Launched as: %s" % origargs)

    # Make sure we have a sufficiently new virtinst version, since we are
    # very closely tied to the lib
    msg = ("virt-manager requires the python-virtinst library version " +
            virtinst_str + " or greater. This can be downloaded at:"
            "\n\nhttp://virt-manager.org/download.html")
    try:
        import virtinst
        ignore = virtinst.__version__
        ignore = virtinst.__version_info__
    except Exception, e:
        logging.exception("Error import virtinst")
        raise RuntimeError(str(e) + "\n\n" + msg)

    if virtinst.__version_info__ < virtinst_version:
        raise RuntimeError("virtinst version %s is too old." %
                            (virtinst.__version__) +
                           "\n\n" + msg)

    # Add our icon dir to icon theme
    icon_theme = gtk.icon_theme_get_default()
    icon_theme.prepend_search_path(icon_dir)

    gtk.gdk.threads_init()
    setup_glade_i18n()

    import dbus
    import dbus.glib
    dbus.glib.threads_init()
    import dbus.service

    # Specifically init config/gconf before the fork, so that pam
    # doesn't think we closed the app, therefor robbing us of
    # display access
    from virtManager.config import vmmConfig
    config = vmmConfig(appname, appversion, gconf_dir, glade_dir,
                       icon_dir, data_dir)

    # Now we've got basic environment up & running we can fork
    if not options.nofork and not options.debug:
        drop_tty()
        drop_stdio()

        # Ignore SIGHUP, otherwise a serial console closing drops the whole app
        signal.signal(signal.SIGHUP, signal.SIG_IGN)

    from virtManager.engine import vmmEngine
    from virtManager.remote import vmmRemote

    gtk.window_set_default_icon_from_file(icon_dir + "/" +
                                          appname + "-icon.svg")

    if options.show and options.uri==None:
        raise OptionValueError("can't use --show-* options without --connect")

    engine = vmmEngine(config)

    if (not (options.nodbus) and
        not ((os.getenv("DBUS_SESSION_BUS_ADDRESS") is None) and
             (os.getenv("DBUS_STARTER_ADDRESS") is None))):

        try:
            bus = None
            if os.getenv("DBUS_STARTER_ADDRESS") is None:
                bus = dbus.SessionBus()
            else:
                bus = dbus.StarterBus()

            dbusProxy = bus.get_object("org.freedesktop.DBus",
                                       "/org/freedesktop/DBus")
            dbusObj = dbus.Interface(dbusProxy, "org.freedesktop.DBus")

            if dbusObj.NameHasOwner("com.redhat.virt.manager"):
                # We're already running, so just talk to existing process

                managerProxy = bus.get_object("com.redhat.virt.manager",
                                              "/com/redhat/virt/manager")
                managerObj = dbus.Interface(managerProxy,
                                            "com.redhat.virt.manager")

                show_remote(managerObj, options.show,
                            options.uri, options.uuid)

                # yes, we exit completely now - remote service is in charge
                logging.debug("Connected to already running instance.")
                return

            else:
                # Grab the service to allow others to talk to us later
                name = dbus.service.BusName("com.redhat.virt.manager", bus=bus)
                vmmRemote(engine, name)

        except:
            # Something went wrong doing dbus setup, just ignore & carry on
            logging.warning("Could not get connection to session bus, "
                            "disabling DBus service " +
                            str(sys.exc_info()[0]) + " " +
                            str(sys.exc_info()[1]))

    # At this point we're either starting a brand new controlling instance,
    # or the dbus comms to existing instance has failed

    # Finally start the app for real
    gtk.gdk.threads_enter()
    try:
        show_engine(engine, options.show, options.uri, options.uuid,
                    options.no_conn_auto)
        if options.profile != None:
            import hotshot
            prof = hotshot.Profile(options.profile)
            prof.runcall(gtk.main)
            prof.close()
        else:
            gtk.main()
    finally:
        gtk.gdk.threads_leave()

if __name__ == "__main__":
    try:
        main()
    except KeyboardInterrupt:
        logging.debug("Received KeyboardInterrupt. Exiting application.")
    except SystemExit:
        raise
    except Exception, run_e:
        logging.exception(run_e)
        _show_startup_error(str(run_e), "".join(traceback.format_exc()))
