/*
 *
 * This software is subject to the terms of the IBM Public License.
 * You must accept the terms of this license to use this software.
 *
 *  Copyright (C) 2000, International Business Machines Corporation
 * and others.  All Rights Reserved.
 *
 *  Version : 1.0 (2000-07-19)
 *
 *  Author  :   Tadayuki Yoshida <tadayuki@jp.ibm.com>
 *
 */

#include "glibctest.h"
#include "charmap.h"

/*
 * void charmap_data_display ( const void *datap )
 *
 *  See "indexedlist.h"
 */

static void
charmap_data_display ( void *datap )
{
  printf ("sym[%s] code[%s]", ((charmapdata *) datap)->symbol,
	  ((charmapdata *) datap)->code);
}

/*
 * void charmap_data_free ( const void *datap )
 *
 *  See "indexedlist.h"
 */

static void
charmap_data_free ( void *datap )
{
  if (datap)
    {
      free ((void *) ((charmapdata *) datap)->symbol);
      free ((void *) ((charmapdata *) datap)->code);
    }
}

/*
 * void extract_charcode ( const charmap *cmp, const char *buf, char *code )
 *
 *  Extract charcode from buf into code.
 */

void
extract_charcode (const charmap * cmp, const char *buf, char *code)
{
  int idx = 0, j, k;
  char tmp1[2] = "\0";
  char tmp2[3] = "\0";
  char tmpdigit[4] = "\0";

  while (iswhitespace(buf[idx]))
    {
      idx++;
    }

  j = 0;
  *code = (char) NULL;
  while (!iswhitespace(buf[idx]))
    {
      if ((buf[idx] == cmp->escape_char) && (buf[idx + 1] == 'x'))
	{ /* hexadecimal digits */
	  idx += 2;
	  while (ishexdigit (buf[idx]))
	    {
	      code[j++] = buf[idx++];
	    }
	  code[j] = (char) NULL;
	}
      else if ((buf[idx] == cmp->escape_char) && (buf[idx + 1] == 'd'))
	{ /* decimal digits */
	  idx += 2;
	  k = 0;
	  while (isnormaldigit (buf[idx]))
	    {
	      tmpdigit[k++] = buf[idx++];
	    }
	  tmpdigit[k] = (char) NULL;
	  tmp1[0] = (unsigned int) strtoul (tmpdigit, NULL, 10);
	  tmp1[1] = (char) NULL;
	  ul2xstr (tmp1, tmp2);
	  stringcat (code, tmp2);
	  j += stringlen (tmp2);
	}
      else if (isoctaldigit (buf[idx]))
	{ /* octal digits */
	  idx++;
	  k = 0;
	  while (isnormaldigit (buf[idx]))
	    {
	      tmpdigit[k++] = buf[idx++];
	    }
	  tmpdigit[k] = (char) NULL;
	  tmp1[0] = (unsigned int) strtoul (tmpdigit, NULL, 8);
	  tmp1[1] = (char) NULL;
	  ul2xstr (tmp1, tmp2);
	  stringcat (code, tmp2);
	  j += stringlen (tmp2);
	}
      else
        {
          return;
        }
    }
}

/*
void
extract_charcode (const charmap * cmp, const char *buf, char *code)
{
  int i = 0, j = 0;

  while (iswhitespace(buf[i]))
    {
      i++;
    }

  while ((buf[i] == cmp->escape_char) && (buf[i + 1] == 'x'))
    {
      i += 2;
      while (ishexdigit (buf[i]))
	{
	  code[j++] = buf[i++];
	}
    }
  code[j] = (char) NULL;
}
*/

/*
 * int charmap_get_mb_cur_max ( const charmap *cmp )
 *
 *  Get charmap data from buf.
 *  return : true or false
 */

int
charmap_get_mb_cur_max ( const charmap *cmp )
{
  return (cmp->mb_cur_max);
}

/*
 * char charmap_get_escape_char ( const charmap *cmp )
 *
 *  Get charmap data from buf.
 *  return : true or false
 */

char
charmap_get_escape_char ( const charmap *cmp )
{
  return (cmp->escape_char);
}

/*
 * char charmap_get_comment_char ( const charmap *cmp )
 *
 *  Get charmap data from buf.
 *  return : true or false
 */

char
charmap_get_comment_char ( const charmap *cmp )
{
  return (cmp->comment_char);
}

/*
 * int charmap_get_total_num ( const charmap *cmp )
 *
 *  Get charmap data from buf.
 *  return : true or false
 */

int
charmap_get_total_num ( const charmap *cmp )
{
  return (indexedlist_get_total_num (cmp->listp));
}

/*
 * char * charmap_get_data ( charmap *cmp, int idx, char *symbol, char *code )
 *
 *  Get charmap data from buf.
 *  return : true or false
 */

int
charmap_get_data ( const charmap *cmp, const int idx, char **symbolp, char **codep )
{
  charmapdata *ptr;
  int ret;

  if ( idx < indexedlist_get_total_num (cmp->listp) )
    {
      ptr = (charmapdata *) indexedlist_search (cmp->listp, idx);
      stringpcpy (symbolp, ptr->symbol);
      stringpcpy (codep, ptr->code);
      ret = 0;
    }
  else
    {
      fprintf (stderr, "error: index (%d) exceeded total_num.\n");
      *symbolp = NULL;
      *codep = NULL;
      ret = CHARMAP_IDX_ERROR;
    }

  return (ret);
}

/*
 * int get_charmap_data_from_buf ( charmap *cmp, char *buffline, char *tmpsym, char *tmpcode )
 *
 *  Get charmap data from buf.
 *  return : true or false
 */

static boolean
get_charmap_data_from_buf (const charmap * cmp, const char *bufp, char **tmpsymp, char *tmpcode)
{
  int i, j, start, end;
  i = 0;
  j = 0;

  while (bufp[i] != '<')
    {
      i++;
    }

  /* set symbol */
  start = i;
  do
    {
      if (bufp[i] == cmp->escape_char)
	{
	  i += 2;
	}
      else
	{
	  i++;
	}
    }
  while (bufp[i] != '>');
  end = i;

  if (end == (start + 1)) /* means "<>" */
    {
      return (false);
    }

  *tmpsymp = (char *) calloc (end - start + 2, sizeof (char));

  for (i=start, j=0; i <= end; i++, j++)
    {
      *((*tmpsymp)+j) = bufp[i];
    }
  *((*tmpsymp)+j) = (char) NULL;

  extract_charcode (cmp, (bufp + end + 1), tmpcode);

  return (true);
}

static void
add_charmap (const charmap * cmp, const int idx, char *sym, char *code)
{
  charmapdata *ptr;

  ptr = (charmapdata *) malloc (sizeof (charmapdata));

  ptr->symbol = (char *) calloc (stringlen (sym) + 1, sizeof (char));
  ptr->code = (char *) calloc (stringlen (code) + 1, sizeof (char));
  stringcpy (ptr->symbol, sym);
  stringcpy (ptr->code, code);

  indexedlist_add (cmp->listp, idx, ptr);
}

static int
process_ellipsis_charmap (const charmap * cmp, const int idx, const char *bufp)
{
  int i, j, sidx, eidx, len;
  char *prefix;
  int start, end;
  char *tmpsym;
  char *tmpulv;
  char *tmpcode;

  i = 0;
  while (*(bufp+i) != '<')
    {
      i++;
    }
  sidx = ++i;
  while (*(bufp+i) != '>')
    {
      i++;
    }
  eidx = i - 1;

  tmpsym = (char *) calloc (eidx - sidx + 2, sizeof (char));
  for (i=sidx, j=0; i <= eidx; i++, j++)
    {
      tmpsym[j] = bufp[i];
    }
  tmpsym[j] = (char) NULL;
  j = 0;
  while (!isnormaldigit (tmpsym[j]))
    {
      j++;
    }
  eidx = j - 1;
  prefix = (char *) calloc (eidx + 1, sizeof (char));
  for (j=0; j <= eidx; j++)
    {
      prefix[j] = tmpsym[j];
    }
  prefix[eidx+1] = (char) NULL;
  start = atoi (tmpsym + eidx+1);
  len = eidx+1;

  while (*(bufp+i) != '<')
    {
      i++;
    }
  i++;
  j=0;
  while (*(bufp+i) != '>')
    {
      tmpsym[j++] = *(bufp+i++);
    }
  tmpsym[j] = (char) NULL;
  end = atoi (tmpsym + len);

  while (*(bufp+i) != cmp->escape_char)
    {
      i++;
    }

  len = stringlen (tmpsym) - 1;

  tmpulv =(char *) calloc (cmp->mb_cur_max + 1, sizeof (char));
  tmpcode = (char *) calloc (cmp->mb_cur_max * 2 + 1, sizeof (char));
  extract_charcode (cmp, bufp+i, tmpcode);
  for (j = start; j<= end; j++)
    {
      sprintf (tmpsym, "<%s%0*d>", prefix, len, j);
      add_charmap (cmp, idx + (j- start), tmpsym, tmpcode);
      xstr2ul (tmpcode, tmpulv);
      (tmpulv[stringlen (tmpulv) - 1])++;
      ul2xstr (tmpulv, tmpcode);
    }

  free ((void *) tmpsym);
  free ((void *) prefix);
  free ((void *) tmpcode);
  free ((void *) tmpulv);

  return (idx + end - start + 1);
}

/*
 * void init_charmap ( charmap *cmp )
 *
 *  Initialize charmap (pointed to cmp)
 */

charmap *
charmap_init ( FILE *fp )
{
  int i, j, total_num, tmp;
  int start, end;
  char tmpnum[10];
  fpos_t pos;
  char *bufp;
  charmap *cmp;
  char *tmpsym;
  char *tmpcode;
  int idx;

  cmp = (charmap *) malloc (sizeof (charmap));
  cmp->fp = fp;
  cmp->escape_char = '\\';
  cmp->comment_char = '#';
  cmp->mb_cur_min = 1;
  cmp->mb_cur_max = 1;
  cmp->code_set_name = (char *) NULL;

  while (NULL != get_oneline (cmp->fp, &bufp))
    {
      i=0;
      if (iscommentline (bufp, cmp->comment_char))
	{
	  free ((void *) bufp);
	  continue;
	}

      if (stringeqn (bufp, "CHARMAP", 7))
	{
	  free ((void *) bufp);
	  break;
	}

      if (stringeqn (bufp, "<code_set_name>", 15))
	{
	  while (!iswhitespace(*(bufp+i)))
	    { /* skip "<code_set_name>" */
	      i++;
	    }
	  while (iswhitespace(*(bufp+i)))
	    {
	      i++;
	    }
	  tmp = i;
	  while (*(bufp+i) != (char) NULL)
	    {
	      i++;
	    }
	  cmp->code_set_name = (char *) calloc (i-tmp, sizeof (char));
	  stringcpy (cmp->code_set_name, bufp+tmp);
	  free ((void *) bufp);
	}
      else if (stringeqn (bufp, "<comment_char>", 14))
	{
	  while (!iswhitespace(*(bufp+i)))
	    { /* skip "<comment_char>" */
	      i++;
	    }
	  while (iswhitespace(*(bufp+i)))
	    {
	      i++;
	    }
	  cmp->comment_char = *(bufp+i);
	  free ((void *) bufp);
	}
      else if (stringeqn (bufp, "<escape_char>", 13))
	{
	  while (!iswhitespace(*(bufp+i)))
	    { /* skip "<escape_char>" */
	      i++;
	    }
	  while (iswhitespace(*(bufp+i)))
	    {
	      i++;
	    }
	  cmp->escape_char = *(bufp+i);
	  free ((void *) bufp);
	}
      else if (stringeqn (bufp, "<mb_cur_min>", 12))
	{
	  while (!iswhitespace(*(bufp+i)))
	    { /* skip "<mb_cur_min>" */
	      i++;
	    }
	  while (iswhitespace(*(bufp+i)))
	    {
	      i++;
	    }
	  cmp->mb_cur_min = atoi (bufp+i);
	  free ((void *) bufp);
	}
      else if (stringeqn (bufp, "<mb_cur_max>", 12))
	{
	  while (!iswhitespace(*(bufp+i)))
	    { /* skip "<mb_cur_max>" */
	      i++;
	    }
	  while (iswhitespace(*(bufp+i)))
	    {
	      i++;
	    }
	  cmp->mb_cur_max = atoi (bufp+i);
	  free ((void *) bufp);
	}
    }

  fgetpos (cmp->fp, &pos);

  /*
   * Counting Number of Entries
   */

  total_num = 0;
  while (NULL != get_oneline (cmp->fp, &bufp))
    {
      i = 0;
      if (iscommentline (bufp, cmp->comment_char))
	{
	  free ((void *) bufp);
	  continue;
	}

      if (stringeqn (bufp, "END CHARMAP", 11))
	{
	  free ((void *) bufp);
	  break;
	}

      if (substring (bufp, ">.") && substring (bufp, ".<"))
	{
	  j = 0;
	  while (!isnormaldigit (*(bufp+i)))
	    {
	      i++;
	    }
	  while (isnormaldigit (*(bufp+i)))
	    {
	      tmpnum[j++] = *(bufp+i++);
	    }
	  tmpnum[j] = (char) NULL;
	  start = atoi (tmpnum);
	  j = 0;
	  while (!isnormaldigit (*(bufp+i)))
	    {
	      i++;
	    }
	  while (isnormaldigit (*(bufp+i)))
	    {
	      tmpnum[j++] = *(bufp+i++);
	    }
	  tmpnum[j] = (char) NULL;
	  end = atoi (tmpnum);

	  total_num += end - start + 1;
	  free ((void *) bufp);
	}
      else
	{
	  total_num++;
	  free ((void *) bufp);
	}
    }

  cmp->listp = indexedlist_init (total_num, charmap_data_display, charmap_data_free);

  fsetpos (cmp->fp, &pos);

  /*
   *  Create charmap
   */
  idx = 0;
  while (NULL != get_oneline (cmp->fp, &bufp))
    {
      if (iscommentline (bufp, cmp->comment_char))
	{
	  free ((void *) bufp);
	  continue;
	}

      if (stringeqn (bufp, "END CHARMAP", 11))
	{
	  free ((void *) bufp);
	  break;
	}

      if (substring (bufp, ">.") && substring (bufp, ".<"))
	{
	  idx = process_ellipsis_charmap (cmp, idx, bufp);
	}
      else
	{
	  tmpcode = (char *) calloc (cmp->mb_cur_max * 2 + 1, sizeof (char));
	  if (get_charmap_data_from_buf (cmp, bufp, &tmpsym, tmpcode))
	    {
	      add_charmap (cmp, idx, tmpsym, tmpcode);
	      idx++;
	      free ((void *) tmpsym);
	    }
	  else
	    {
	      fprintf (stderr, "Error: Invalid line %s\n", bufp);
	    }
	  free ((void *) tmpcode);
	}
      free ((void *) bufp);
    }

  fclose (cmp->fp);

  return (cmp);
}

void
charmap_display (charmap * cmp)
{
  indexedlist_display (cmp->listp);
  printf ("\n");
}

int
end_of_charmap (charmap * cmp, const int idx)
{
  return (end_of_indexedlist (cmp->listp, idx));
}

void
charmap_free ( charmap *cmp )
{
  if (cmp)
    {
      indexedlist_free (cmp->listp);
      free ((void *)cmp->code_set_name);
      free ((void *)cmp);
    }
}
