/*
 *
 * This software is subject to the terms of the IBM Public License.
 * You must accept the terms of this license to use this software.
 *
 *  Copyright (C) 2000, International Business Machines Corporation
 * and others.  All Rights Reserved.
 *
 *  Version : 1.0 (2000-07-19)
 *
 *  Author  :   Tadayuki Yoshida <tadayuki@jp.ibm.com>
 *
 */

#include "glibctest.h"
#include "test_lang.h"

static boolean nl_langinfo_test ( answer *, wc_answer *, boolean, boolean );
static boolean localeconv_test ( answer *, wc_answer *, boolean, boolean );
static boolean strftime_test ( answer *, wc_answer *, boolean, boolean );
#ifdef HAVE_WCSFTIME
static boolean wcsftime_test ( answer *, wc_answer *, boolean, boolean );
#endif
static boolean strptime_test ( answer *, wc_answer *, boolean, boolean );
static boolean strfmon_test ( answer *, wc_answer *, boolean, boolean );

void
api_table_create ( const api_table *tp )
{
  api_table_add (tp, "nl_langinfo", nl_langinfo_test);
  api_table_add (tp, "localeconv", localeconv_test);
  api_table_add (tp, "strftime", strftime_test);
#ifdef HAVE_WCSFTIME
  api_table_add (tp, "wcsftime", wcsftime_test);
#endif
  api_table_add (tp, "strptime", strptime_test);
  api_table_add (tp, "strfmon", strfmon_test);
}

/**********************************************************************
*                                                                     *
*  static void construct_tm(struct tm *ptm)                           *
*                                                                     *
*                                                                     *
***********************************************************************/

static void
construct_tm (struct tm **ptm)
{
  /*
    extern int daylight;
    extern long int timezone;
    extern char *tzname[];
  */
   
  time_t tmbin;
    
  time (&tmbin);
  *ptm = localtime (&tmbin);

  /*    12/31/1999 14:36:56 (Fri) */

  (*ptm)->tm_sec = 56;
  (*ptm)->tm_min = 36;
  (*ptm)->tm_hour = 14;
  (*ptm)->tm_mday = 31;
  (*ptm)->tm_mon = 11;
  (*ptm)->tm_year = 99;
  (*ptm)->tm_wday = 5;
  (*ptm)->tm_yday = 364;
  (*ptm)->tm_isdst = 0;
}

/**********************************************************************
*                                                                     *
*  static void get_nl_item ( char *target, nl_item *itemp )           *
*                                                                     *
*                                                                     *
***********************************************************************/

static nl_item
get_nl_item (char *target)
{
  register int i;

  struct
  {
    char *name;
    nl_item item;
  }
  ansitem[] =
  {
    {
    "D_T_FMT", D_T_FMT}
    ,				/* string for formatting date and time */
    {
    "D_FMT", D_FMT}
    ,				/* string for formatting date */
    {
    "T_FMT", T_FMT}
    ,				/* string for formatting time */
    {
    "AM_STR", AM_STR}
    ,				/* string for a.m. */
    {
    "PM_STR", PM_STR}
    ,				/* string for p.m. */
    {
    "ABDAY_1", ABDAY_1}
    ,				/* abbreviated first day of the week (Sun) */
    {
    "ABDAY_2", ABDAY_2}
    ,				/* abbreviated second day of the week (Mon) */
    {
    "ABDAY_3", ABDAY_3}
    ,				/* abbreviated third day of the week (Tue) */
    {
    "ABDAY_4", ABDAY_4}
    ,				/* abbreviated fourth day of the week (Wed) */
    {
    "ABDAY_5", ABDAY_5}
    ,				/* abbreviated fifth day of the week (Thu) */
    {
    "ABDAY_6", ABDAY_6}
    ,				/* abbreviated sixth day of the week (Fri) */
    {
    "ABDAY_7", ABDAY_7}
    ,				/* abbreviated seventh day of the week (Sat) */
    {
    "DAY_1", DAY_1}
    ,				/* name of the first day of the week (Sunday) */
    {
    "DAY_2", DAY_2}
    ,				/* name of the second day of the week (Monday) */
    {
    "DAY_3", DAY_3}
    ,				/* name of the third day of the week (Tuesday) */
    {
    "DAY_4", DAY_4}
    ,				/* name of the fourth day of the week (Wednesday) */
    {
    "DAY_5", DAY_5}
    ,				/* name of the fifth day of the week (Thursday) */
    {
    "DAY_6", DAY_6}
    ,				/* name of the sixth day of the week (Friday) */
    {
    "DAY_7", DAY_7}
    ,				/* name of the seventh day of the week (Saturday) */
    {
    "ABMON_1", ABMON_1}
    ,				/* abbreviated first month (Jan) */
    {
    "ABMON_2", ABMON_2}
    ,				/* abbreviated second month (Feb) */
    {
    "ABMON_3", ABMON_3}
    ,				/* abbreviated third month (Mar) */
    {
    "ABMON_4", ABMON_4}
    ,				/* abbreviated fourth month (Apr) */
    {
    "ABMON_5", ABMON_5}
    ,				/* abbreviated fifth month (May) */
    {
    "ABMON_6", ABMON_6}
    ,				/* abbreviated sixth month (Jun) */
    {
    "ABMON_7", ABMON_7}
    ,				/* abbreviated seventh month (Jul) */
    {
    "ABMON_8", ABMON_8}
    ,				/* abbreviated eighth month (Aug) */
    {
    "ABMON_9", ABMON_9}
    ,				/* abbreviated ninth month (Sep) */
    {
    "ABMON_10", ABMON_10}
    ,				/* abbreviated tenth month (Oct) */
    {
    "ABMON_11", ABMON_11}
    ,				/* abbreviated eleventh month (Nov) */
    {
    "ABMON_12", ABMON_12}
    ,				/* abbreviated twelveth month (Dec) */
    {
    "MON_1", MON_1}
    ,				/* name of the first month (January) */
    {
    "MON_2", MON_2}
    ,				/* name of the second month (February) */
    {
    "MON_3", MON_3}
    ,				/* name of the third month (March) */
    {
    "MON_4", MON_4}
    ,				/* name of the fourth month (April) */
    {
    "MON_5", MON_5}
    ,				/* name of the fifth month (May) */
    {
    "MON_6", MON_6}
    ,				/* name of the sixth month (June) */
    {
    "MON_7", MON_7}
    ,				/* name of the seventh month (July) */
    {
    "MON_8", MON_8}
    ,				/* name of the eighth month (August) */
    {
    "MON_9", MON_9}
    ,				/* name of the ninth month (September) */
    {
    "MON_10", MON_10}
    ,				/* name of the tenth month (October) */
    {
    "MON_11", MON_11}
    ,				/* name of the eleventh month (November) */
    {
    "MON_12", MON_12}
    ,				/* name of the twelveth month (December) */
    {
    "RADIXCHAR", RADIXCHAR}
    ,				/* radix character */
    {
    "THOUSEP", THOUSEP}
    ,				/* separator for thousands */
    {
    "YESSTR", YESSTR}
    ,				/* affiramitive response for yes/no queries */
    {
    "NOSTR", NOSTR}
    ,				/* negative response for yes/no queries */
    {
    "CRNCYSTR", CRNCYSTR}
    ,                           /* currency symbol; - leading, + trailing */
    {
    "CODESET", CODESET}
    ,				/* codeset name *//*2000-04-13 */
    {
    "T_FMT_AMPM", T_FMT_AMPM}
    ,				/* am/pm time formating string */
    {
    "ERA", ERA}
    ,				/* era description segments */
    {
    "ERA_D_FMT", ERA_D_FMT}
    ,				/* era date format string */
    {
    "ERA_D_T_FMT", ERA_D_T_FMT}
    ,				/* era date and time format string */
    {
    "ERA_T_FMT", ERA_T_FMT}
    ,				/* era time format string */
    {
    "ALT_DIGITS", ALT_DIGITS}
    ,				/* alternative symbols for digits */
    {
    "YESEXPR", YESEXPR}
    ,				/* affirmative response expression */
    {
    "NOEXPR", NOEXPR}
    ,				/* negative response expression */
    {
    (char *) NULL, (nl_item) 0}
  };

  for (i = 0; ansitem[i].name; i++)
    {
      if (stringeq (target, ansitem[i].name))
	{
	  return (ansitem[i].item);
	}
    }

  fprintf (stderr, OOPSSTR);
  fprintf (stderr, "nl_item : %s No such item.\n", target);
  return (nl_item) NULL;
}

static void
copy_lconv_item_d ( char **item, int lconv )
{
  *item = (char *) mem_calloc (count_digit (lconv) + 1, sizeof (char));
  sprintf (*item, "%d", lconv);
}

/**********************************************************************
*                                                                     *
*  static void   get_lconv_item ( char *target )                      *
*                                                                     *
*                                                                     *
***********************************************************************/

static void
get_lconv_item (char *target, char **item)
{
  struct lconv *curloc;
  size_t tmpsize;

  curloc = localeconv ();

  if (stringeq (target, "decimal_point"))
    {
      stringpcpy (item, curloc->decimal_point);
    }
  else if (stringeq (target, "thousands_sep"))
    {
      stringpcpy (item, curloc->thousands_sep);
    }
  else if (stringeq (target, "grouping"))
    {
      copy_lconv_item_d (item, (int) *curloc->grouping);
    }
  else if (stringeq (target, "int_curr_symbol"))
    {
      stringpcpy (item, curloc->int_curr_symbol);
    }
  else if (stringeq (target, "currency_symbol"))
    {
      stringpcpy (item, curloc->currency_symbol);
    }
  else if (stringeq (target, "mon_decimal_point"))
    {
      stringpcpy (item, curloc->mon_decimal_point);
    }
  else if (stringeq (target, "mon_thousands_sep"))
    {
      stringpcpy (item, curloc->mon_thousands_sep);
    }
  else if (stringeq (target, "mon_grouping"))
    {
      copy_lconv_item_d (item, (int) *curloc->mon_grouping);
    }
  else if (stringeq (target, "positive_sign"))
    {
      stringpcpy (item, curloc->positive_sign);
    }
  else if (stringeq (target, "negative_sign"))
    {
      stringpcpy (item, curloc->negative_sign);
    }
  else if (stringeq (target, "int_frac_digits"))
    {
      copy_lconv_item_d (item, curloc->int_frac_digits);
    }
  else if (stringeq (target, "frac_digits"))
    {
      copy_lconv_item_d (item, curloc->frac_digits);
    }
  else if (stringeq (target, "p_cs_precedes"))
    {
      copy_lconv_item_d (item, curloc->p_cs_precedes);
    }
  else if (stringeq (target, "p_sep_by_space"))
    {
      copy_lconv_item_d (item, curloc->p_sep_by_space);
    }
  else if (stringeq (target, "n_cs_precedes"))
    {
      copy_lconv_item_d (item, curloc->n_cs_precedes);
    }
  else if (stringeq (target, "n_sep_by_space"))
    {
      copy_lconv_item_d (item, curloc->n_sep_by_space);
    }
  else if (stringeq (target, "p_sign_posn"))
    {
      copy_lconv_item_d (item, curloc->p_sign_posn);
    }
  else if (stringeq (target, "n_sign_posn"))
    {
      copy_lconv_item_d (item, curloc->n_sign_posn);
    }
  else
    {
      fprintf (stderr, OOPSSTR);
      fprintf (stderr, "lconv_item : %s Not Found.\n", target);
      stringpcpy (item, "\0");
    }
}

/*****************************************************************/

static boolean
nl_langinfo_test ( answer * ansp,
		   wc_answer * wc_ansp,
		   boolean genans_flag,
		   boolean verbose_flag )
{
  char *test_case;
  char *result_string;
  size_t tmpsize;
  size_t padding = 15;
  boolean result;

  tmpsize = stringlen (ansp->api_name) + stringlen (ansp->param[0]) + stringlen (ansp->param[1]) + stringlen (ansp->param[2]) + padding;
  test_case = (char *) mem_calloc (tmpsize, sizeof (char));

  sprintf (test_case, "%s(%s)", ansp->api_name, ansp->param[0]);
  stringpcpy (&result_string, nl_langinfo (get_nl_item (ansp->param[0])));

  result = stringeq (result_string, ansp->result);
  display_result (ansp, test_case, result_string, ansp->result,
		  result, genans_flag, verbose_flag);

  mem_free ((void *) result_string);
  mem_free ((void *) test_case);

  return (result);
}

static boolean
localeconv_test ( answer * ansp,
		  wc_answer * wc_ansp,
		  boolean genans_flag,
		  boolean verbose_flag )
{
  char *test_case;
  char *result_string;
  size_t tmpsize;
  size_t padding = 15;
  boolean result;

  tmpsize = stringlen (ansp->api_name) + stringlen (ansp->param[0]) + stringlen (ansp->param[1]) + stringlen (ansp->param[2]) + padding;
  test_case = (char *) mem_calloc (tmpsize, sizeof (char));

  sprintf (test_case, "%s:%s", ansp->api_name, ansp->param[0]);
  get_lconv_item (ansp->param[0], &result_string);

  result = stringeq (result_string, ansp->result);
  display_result (ansp, test_case, result_string, ansp->result,
		      result, genans_flag, verbose_flag);

  mem_free ((void *) result_string);
  mem_free ((void *) test_case);

  return (result);
}


static boolean
strfmon_test ( answer * ansp,
	       wc_answer * wc_ansp,
	       boolean genans_flag,
	       boolean verbose_flag )
{
  char *test_case;
  char result_string[1024];
  char *endp;
  double money;
  size_t result_size = 1024;
  size_t padding = 15;
  size_t tmpsize;
  ssize_t strfmon_ret;
  boolean result;

  tmpsize = stringlen (ansp->api_name) + stringlen (ansp->param[0]) + stringlen (ansp->param[1]) + stringlen (ansp->param[2]) + padding;
  test_case = (char *) mem_calloc (tmpsize, sizeof (char));

  money = strtod (ansp->param[1], &endp);
  sprintf (test_case, "%s(...,%s,%f)", ansp->api_name, ansp->param[0], money);

  strfmon_ret = strfmon (result_string, result_size, ansp->param[0], money);
  if (strfmon_ret == -1)
    {
      fprintf (stderr, "%s : error\n", test_case);
    }
  result = stringeq (result_string, ansp->result);
  display_result (ansp, test_case, result_string, ansp->result,
		  result, genans_flag, verbose_flag);

  mem_free ((void *) test_case);

  return (result);
}


static boolean
strftime_test ( answer * ansp,
		wc_answer * wc_ansp,
		boolean genans_flag,
		boolean verbose_flag )
{
  char *test_case;
  char result_string[1024];
  struct tm tmobj;
  struct tm *tmvalue;
  size_t strftime_ret;
  size_t result_size = 1024;
  size_t tmpsize;
  size_t padding = 15;
  boolean result;

  tmpsize = stringlen (ansp->api_name) + stringlen (ansp->param[0]) + stringlen (ansp->param[1]) + stringlen (ansp->param[2]) + padding;
  test_case = (char *) mem_calloc (tmpsize, sizeof (char));

  tmvalue = &tmobj;
  construct_tm (&tmvalue);

  sprintf (test_case, "%s(...,%s,...)", ansp->api_name, ansp->param[0]);
  strftime_ret = strftime (result_string, result_size, ansp->param[0], tmvalue);
  if (strftime_ret == 0)
    {
      fprintf (stderr, "%s : error\n", test_case);
    }
  result = stringeq (result_string, ansp->result);
  display_result (ansp, test_case, result_string, ansp->result,
		  result, genans_flag, verbose_flag);

  mem_free ((void *) test_case);

  return (result);
}

#ifdef HAVE_WCSFTIME
static boolean
wcsftime_test ( answer * ansp,
		wc_answer * wc_ansp,
		boolean genans_flag,
		boolean verbose_flag )
{
  char *test_case;
  char *result_string;
  wchar_t result_wstring[1024];
  struct tm tmobj;
  struct tm *tmvalue;
  size_t strftime_ret;
  size_t result_size = 1024;
  size_t tmpsize;
  size_t padding = 15;
  boolean result;

  tmpsize = stringlen (ansp->api_name) + stringlen (ansp->param[0]) + stringlen (ansp->param[1]) + stringlen (ansp->param[2]) + padding;
  test_case = (char *) mem_calloc (tmpsize, sizeof (char));

  tmvalue = &tmobj;
  construct_tm (&tmvalue);

  sprintf (test_case, "%s(...,%s,...)", ansp->api_name, ansp->param[0]);
  strftime_ret = wcsftime (result_wstring, result_size, wc_ansp->param[0], tmvalue);
  if (strftime_ret == 0)
    {
      fprintf (stderr, "%s : error\n", test_case);
    }
  convert_wcsmbs (&result_string, result_wstring);

  result = wstringeq (result_wstring, wc_ansp->result);
  display_result (ansp, test_case, result_string, ansp->result,
		  result, genans_flag, verbose_flag);

  mem_free ((void *) result_string);
  mem_free ((void *) test_case);

  return (result);
}
#endif

static boolean
strptime_test ( answer * ansp,
		wc_answer * wc_ansp,
		boolean genans_flag,
		boolean verbose_flag )
{
  char *test_case;
  char result_string[1024];
  char temp_string[1024];
  struct tm tmobj;
  struct tm *tmvalue;
  struct tm tmtarget;
  size_t strftime_ret;
  size_t result_size = 1024;
  size_t tmpsize;
  size_t padding = 15;
  boolean result;

  tmpsize = stringlen (ansp->api_name) + stringlen (ansp->param[0]) + stringlen (ansp->param[1]) + stringlen (ansp->param[2]) + padding;
  test_case = (char *) mem_calloc (tmpsize, sizeof (char));

  tmvalue = &tmobj;
  construct_tm (&tmvalue);

  tmtarget.tm_sec = 0;
  tmtarget.tm_min = 0;
  tmtarget.tm_hour = 0;
  tmtarget.tm_mday = 0;
  tmtarget.tm_mon = 0;
  tmtarget.tm_year = 0;
  tmtarget.tm_wday = 0;
  tmtarget.tm_yday = 0;
  tmtarget.tm_isdst = 0;

  strftime_ret = strftime (temp_string, result_size, ansp->param[0], tmvalue);

  if (strftime_ret == 0)
    {
      fprintf (stderr, "%s : error\n", test_case);
    }
  strptime (temp_string, ansp->param[0], &tmtarget);
  sprintf (test_case, "%s(...,%s,...)", ansp->api_name, ansp->param[0]);
  /*
    sprintf (temp_string, "%d",
    strftime (result_string2, tmpsize, ansp->param[0], tmvalue));
  */
  sprintf (result_string,
	   "[%d][%d][%d][%d][%d][%d][%d][%d][%d]",
	   tmtarget.tm_sec, tmtarget.tm_min, tmtarget.tm_hour,
	   tmtarget.tm_mday, tmtarget.tm_mon, tmtarget.tm_year,
	   tmtarget.tm_wday, tmtarget.tm_yday, tmtarget.tm_isdst);

  result = stringeq (result_string, ansp->result);
  display_result (ansp, test_case, result_string, ansp->result,
		  result, genans_flag, verbose_flag);

  mem_free ((void *) test_case);

  return (result);
}
