/*
 *
 * This software is subject to the terms of the IBM Public License.
 * You must accept the terms of this license to use this software.
 *
 *  Copyright (C) 2000, International Business Machines Corporation
 * and others.  All Rights Reserved.
 *
 *  Version : 1.0 (2000-07-19)
 *
 *  Author  :   Tadayuki Yoshida <tadayuki@jp.ibm.com>
 *
 */
#include "glibctest.h"
#include "lcctype.h"
#include "charmap.h"
#include "hash.h"

static void
lc_ctype_char_display ( void *ptr )
{
  printf (" sym1[%s], sym2[%s]",
	  ((lc_ctype_char *) ptr)->symbol1,
	  ((lc_ctype_char *) ptr)->symbol2);
}

int
lc_ctype_get_total_num ( lc_ctype *lcp )
{
  return (lcp->total_num);
}

static void
lc_ctype_char_free ( void *ptr )
{
  if (ptr)
    {
      mem_free ((void *)(((lc_ctype_char *)ptr)->symbol1));
      mem_free ((void *)(((lc_ctype_char *)ptr)->symbol2));
    }
}

boolean
lc_ctype_defined_class ( lc_ctype *lcp, char *classname )
{
  int i;
  boolean ret = false;

  for (i =0 ; i < lcp->total_num; i++)
    {
      if (stringeq (classname, lcp->classp[i].classname))
	{
	  ret = true;
	  break;
	}
    }

  return (ret);
}

static void
define_user_class (lc_ctype *lcp, char *bufp, char **ptr)
{
  int i, j, k, idx, tmp;
  char *tmpsym;

  idx = 0;
  while (iswhitespace(*(bufp+idx)))
    {
      idx++;
    }

  k = 0;
  while (true)
    {
      tmp = idx;
      do
	{
	  idx += (*(bufp+idx) == lcp->escape_char) ? 2 : 1;
	}
      while (*(bufp+idx) != ';' && *(bufp+idx) != (char) NULL);
      tmpsym = (char *) mem_calloc (idx - tmp + 1, sizeof (char));
      for (i=tmp, j=0; i < idx; i++, j++)
	{
	  tmpsym[j] = *(bufp+i);
	}
      tmpsym[j] = (char) NULL;

      stringpcpy (&(ptr[k]), tmpsym);
      mem_free ((void *) tmpsym);

      if (*(bufp+idx) == (char) NULL)
	{
	  break;
	}
      if (*(bufp+idx) == ';')
	{
	  idx++;
	}
      k++;
    }
}

static char *
scan_one_angle_token (lc_ctype *lcp, char *bufp, char **tokenp)
{
  int i, j, start, end, idx;

  idx = 0;
  while (*(bufp+idx) != '<')
    {
      idx++;
    }
  start = idx;
  do
    {
      idx += (*(bufp+idx) == lcp->escape_char) ? 2 : 1;
    }
  while (*(bufp+idx) != '>');
  end = idx;
  *tokenp = (char *) mem_calloc (end - start + 2, sizeof (char));
  for (i=start, j=0; i <= end; i++, j++)
    {
      *((*tokenp)+j) = bufp[i];
    }
  *((*tokenp)+end+1) = (char) NULL;

  return (bufp+end+1);
}

static char *
scan_one_token (lc_ctype *lcp, char *buf, char **tokenp)
{
  int i, j;
  char *tmptokenp;
  char *tmptoken1, *tmptoken2;
  char *bufp = buf;
  char *oldp;
  size_t tmpsize;
  int tmp;

  i = 0;
  j = 0;
  while (iswhitespace(*bufp))
    {
      bufp++;
    }
  if (*bufp == '(')
    {
      bufp = scan_one_angle_token (lcp, bufp + 1, &tmptoken1);
      bufp = scan_one_angle_token (lcp, bufp + 1, &tmptoken2);
      tmpsize = stringlen (tmptoken1) + stringlen (tmptoken2) + 10;
      tmptokenp = (char *) mem_alloc (tmpsize * sizeof (char));
      stringcpy (tmptokenp, "(");
      stringcat (tmptokenp, tmptoken1);
      stringcat (tmptokenp, ",");
      stringcat (tmptokenp, tmptoken2);
      stringcat (tmptokenp, ")");
      mem_free ((void *) tmptoken1);
      mem_free ((void *) tmptoken2);
      *tokenp = tmptokenp;
      return (bufp + 1);
    }
  else if (*bufp == '<')
    {
      do
	{
	  bufp = scan_one_angle_token (lcp, bufp, &tmptoken1);
	  tmpsize = stringlen (tmptoken1) + 1;
	  tmptokenp = (char *) mem_alloc (tmpsize * sizeof (char));
	  stringcat (tmptokenp, tmptoken1);
	  if (*bufp == '.') /* coutinue <foo>....<bar> */
	    {
	      tmpsize = 0;
	      do
		{
		  tmpsize++;
		}
	      while (*(++bufp) == '.');
	      mem_free ((void *) tmptoken1);
	      tmptoken1 = (char *) mem_calloc (tmpsize + 1, sizeof (char));
	      for (j=0; j<tmpsize; j++)
		{
		  tmptoken1[j] = '.';
		}
	      tmptoken1[tmpsize] = (char) NULL;
	      bufp = scan_one_angle_token (lcp, bufp, &tmptoken2);
	      tmpsize = stringlen (tmptokenp) + stringlen (tmptoken1) + stringlen (tmptoken2) + 2;
	      oldp = tmptokenp;
	      tmptokenp = (char *) mem_calloc (tmpsize, sizeof (char));
	      stringcpy (tmptokenp, oldp);
	      stringcat (tmptokenp, tmptoken1);
	      stringcat (tmptokenp, tmptoken2);
	      mem_free ((void *) oldp);
	      mem_free ((void *) tmptoken2);
	    }
	  mem_free ((void *) tmptoken1);
	}
      while (*bufp != ';' && *bufp != (char) NULL);
      *tokenp = tmptokenp;
      return (bufp);
    }
  else
    {
      j = 0;
      do
	{
	  j += (*(bufp+j) == lcp->escape_char) ? 2 : 1;
	}
      while (*(bufp+j) != ';' && *(bufp+j) != (char) NULL);
      tmpsize = j;
      tmptokenp = (char *) mem_calloc (tmpsize + 1, sizeof (char));
      for (j=0; j < tmpsize; j++)
	{
	  tmptokenp[j] = *(bufp+j);
	}
      tmptokenp[tmpsize] = (char) NULL;
      *tokenp = tmptokenp;
      return (bufp+tmpsize);
    }
}

static int
count_entry (lc_ctype *lcp, char *buf)
{
  int i = 0, j, k;
  int count = 0, start, end;
  char *tmpsym;
  char tmpnum[10];
  char *bufp = buf;

  while (true)
    {
      bufp = scan_one_token (lcp, bufp, &tmpsym);
      if (substring (tmpsym, ".."))
	{
	  j = 0;
	  k = 0;
	  while (!isnormaldigit (tmpsym[j]))
	    j++;
	  while (isnormaldigit (tmpsym[j]))
	    {
	      tmpnum[k++] = tmpsym[j++];
	    }
	  tmpnum[k] = (char) NULL;
	  start = atoi (tmpnum);
	  k = 0;
	  while (!isnormaldigit (tmpsym[j]))
	    j++;
	  while (isnormaldigit (tmpsym[j]))
	    {
	      tmpnum[k++] = tmpsym[j++];
	    }
	  tmpnum[k] = (char) NULL;
	  end = atoi (tmpnum);
	  count += end - start + 1;
	}
      else
	{
	  count++;
	}
      mem_free ((void *) tmpsym);
      if (*bufp == (char) NULL)
	{
	  break;
	}
      if (*bufp == ';')
	{
	  bufp++;
	}
    }

  return (count);
}

static char *
get_one_entry_buf (lc_ctype *lcp, char **bufp)
{
  char tmpline[1024];
  char *tmpbufp, *tmpbufpold;
  int i;
  int currentlen, tmplen, maxlen, len;
  int unit = 1024;

  maxlen = unit;
  tmpbufp = (char *) mem_calloc (maxlen, sizeof (char));

  while (true)
    {
      if (NULL != fgets (tmpline, 1024, lcp->fp))
	{
	  len = stringlen (tmpline);
	  tmpline[len - 1] = (char) NULL;
	  i = 0;
	  while (iswhitespace(tmpline[i]))
	    i++;
	  tmplen = stringlen (tmpline + i);
	  currentlen = stringlen (tmpbufp);
	  if (maxlen - currentlen < tmplen)
	    {
	      tmpbufpold = tmpbufp;
	      maxlen += unit;
	      tmpbufp = (char *) mem_calloc (maxlen, sizeof (char));
	      memcpy (tmpbufp, tmpbufpold, maxlen - unit);
	      mem_free ((void *) tmpbufpold);
	    }
	  stringcat (tmpbufp, tmpline + i);
	  if (tmpline[len - 2] == lcp->escape_char)
	    {
	      tmpbufp[stringlen (tmpbufp) - 1] = (char) NULL;
	    }
	  else
	    {
	      break;
	    }
	}
    }

  *bufp = tmpbufp;

  return (tmpbufp);
}

static void
insert_lcctype_hash (lc_ctype *lcp, lc_ctype_data * classp, char *symbol)
{
  lc_ctype_char *ptr;
  char *key;
  size_t tmpsize;

  ptr = (lc_ctype_char *) mem_alloc (sizeof (lc_ctype_char));

  stringpcpy (&(ptr->symbol1), symbol);
  ptr->symbol2 = (char *) NULL;

  extract_realkey (&key, symbol, lcp->escape_char);
  hashtable_insert (classp->hashtable, key, ptr);

  mem_free ((void *) key);
}

static void
insert_lcctype_hash_pair (lc_ctype *lcp, lc_ctype_data * classp, char *symbol)
{
  char *leftsym;
  char *key;
  char *rightsym;
  int i, j = 0, idx;
  lc_ctype_char *ptr;
  size_t tmpsize;

  idx = 1;
  while (symbol[idx] != ',')
    {
      idx++;
    }
  tmpsize = idx;
  leftsym = (char *) mem_calloc (tmpsize, sizeof (char));
  for (i=1, j=0; i < idx ; i++, j++)
    {
      leftsym[j] = symbol[i];
    }
  leftsym[j] = (char) NULL;
  idx++;

  j = idx;
  while (symbol[idx] != (char) NULL)
    {
      idx++;
    }
  tmpsize = idx - j;
  rightsym = (char *) mem_calloc (tmpsize, sizeof (char));
  for (i=j, j=0; i < idx-1 ; i++, j++)
    {
      rightsym[j] = symbol[i];
    }
  rightsym[j] = (char) NULL;

  ptr = (lc_ctype_char *) mem_alloc (sizeof (lc_ctype_char));

  ptr->symbol1 = leftsym;
  ptr->symbol2 = rightsym;

  extract_realkey (&key, leftsym, lcp->escape_char);
  hashtable_insert (classp->hashtable, key, ptr);

  mem_free ((void *) key);
}

static int
insert_lcctype_hash_list (lc_ctype *lcp, lc_ctype_data * classp, char *symbol)
{
  int i, j = 0, k, len, idx, tmp;
  char *prefix;
  int start, end;
  char *tmpsym;
  size_t tmpsize;

  idx = 0;
  while (symbol[idx] != '<') /* skip to char after '<' */
    {
      idx++;
    }
  tmp = ++idx;
  while (symbol[idx] != '>')
    {
      idx++;
    }
  tmpsize = idx - tmp + 1;
  tmpsym = (char *) mem_calloc (tmpsize, sizeof (char));
  for (i=tmp, j=0; i < idx ; i++, j++)
    {
      tmpsym[j] = symbol[i];
    }
  tmpsym[j] = (char) NULL;

  j = 0;
  while (!isnormaldigit (tmpsym[j]))
    {
      j++;
    }
  tmpsize = j +1;
  prefix = (char *) mem_calloc (tmpsize, sizeof (char));
  for (j=0; j<tmpsize-1; j++)
    {
      prefix[j] = tmpsym[j];
    }
  prefix[j] = (char) NULL;

  start = atoi (tmpsym + j);
  len = j;

  j = 0;
  do
    {
      idx++;
    }
  while (symbol[idx] != '<');
  tmp = ++idx;
  while (symbol[idx] != '>')
    {
      idx++;
    }
  tmpsize = stringlen (tmpsym) + 1;
  memset ((void *)tmpsym, (char) NULL, tmpsize);
  for (i=tmp, j=0; i < idx ; i++, j++)
    {
      tmpsym[j] = symbol[i];
    }
  tmpsym[j] = (char) NULL;
  end = atoi (tmpsym + len);

  len = stringlen (tmpsym) - 1;

  memset ((void *)tmpsym, (char) NULL, tmpsize);
  for (k = start; k <= end; k++)
    {
      sprintf (tmpsym, "<%s%0*d>", prefix, len, k);
      insert_lcctype_hash (lcp, classp, tmpsym);
    }

  mem_free ((void *) tmpsym);
  mem_free ((void *) prefix);

  return 0;
}

static void
create_one_entry (lc_ctype *lcp, char *classname, lc_ctype_data * ptr,
		  char *buf)
{
  int i, j, e;
  int num;
  char *tmpsym;
  char *bufp;
  size_t tmpsize;

  bufp = buf;
  ptr->conv_flag = false;
  if ((stringeq (classname, "toupper")) || (stringeq (classname, "tolower")))
    {
      ptr->conv_flag = true;
    }
  else
    {
      for (j = 0; j < lcp->userdefined_conv_num; j++)
	{
	  if (stringeq (classname, lcp->userdefined_conv[j]))
	    {
	      ptr->conv_flag = true;
	    }
	}
    }

  stringpcpy (&(ptr->classname), classname);

  num = count_entry (lcp, bufp);
  ptr->hashtable = hashtable_init (num, lc_ctype_char_display, lc_ctype_char_free);

  while (true)
    {
      bufp = scan_one_token (lcp, bufp, &tmpsym);
      if (substring (tmpsym, ".."))
	{
	  /* <foo>..<bar> */
	  insert_lcctype_hash_list (lcp, ptr, tmpsym);
	}
      else if (tmpsym[0] == '(' && tmpsym[stringlen (tmpsym) - 1] == ')')
	{
	  /* (<foo>,<bar>) */
	  insert_lcctype_hash_pair (lcp, ptr, tmpsym);
	}
      else
	{
	  /* <foo> */
	  insert_lcctype_hash (lcp, ptr, tmpsym);
	}
      mem_free ((void *) tmpsym);

      if (*bufp == (char) NULL)
	{
	  break;
	}
      if (*bufp == ';')
	{
	  bufp++;
	}
    }

  return;
}

boolean
ispredefinedclass ( char *classname, lc_ctype *lcp )
{
  int i;

  for (i=0; i < lcp->userdefined_class_num ; i++)
    {
      if (stringeq (classname, lcp->userdefined_class[i]))
	{
	  return true;
	}
    }

  for (i=0; i < lcp->userdefined_conv_num ; i++)
    {
      if (stringeq (classname, lcp->userdefined_conv[i]))
	{
	  return true;
	}
    }

  return false;
}

boolean
isuserdefinedclass ( char *classname, lc_ctype *lcp )
{
  int i;

  for (i=0; i < lcp->userdefined_class_num ; i++)
    {
      if (stringeq (classname, lcp->userdefined_class[i]))
	{
	  return true;
	}
    }

  for (i=0; i < lcp->userdefined_conv_num ; i++)
    {
      if (stringeq (classname, lcp->userdefined_conv[i]))
	{
	  return true;
	}
    }

  return false;
}

lc_ctype *
lc_ctype_init ( FILE *fp )
{
  int i, j, tmp, end, idx;
  int num;
  int num_of_entry;
  char *bufp, *save;
  fpos_t pos;
  char *tmpclass;
  lc_ctype *lcp;
  int count;
  lc_ctype_data *classp;

  file_process (fp);
  fclose (fp);
  lcp = (lc_ctype *) mem_alloc (sizeof (lc_ctype));
  lcp->classp = (lc_ctype_data *) NULL;
  lcp->escape_char = '\\';
  lcp->comment_char = '#';
  lcp->repertoiremap = (char *)NULL;
  lcp->classp = (lc_ctype_data *) NULL;
  lcp->userdefined_class = (char **) NULL;
  lcp->userdefined_class_num = 0;
  lcp->userdefined_conv = (char **) NULL;
  lcp->userdefined_conv_num = 0;
  lcp->total_num = 0;

  if (NULL == (lcp->fp = fopen (TMPFILE, "r")))
    {
      fprintf (stderr, "file ");
      fprintf (stderr, TMPFILE);
      fprintf (stderr, " can't open.\n");
      exit (1);
    }

  /* initialize elements */
  while (NULL != get_oneline (lcp->fp, &bufp))
    {
      if (iscommentline (bufp, lcp->comment_char))
	{
	  mem_free ((void *)bufp);
	  continue;
	}

      if (stringeqn (bufp, "LC_CTYPE", 8))
	{
	  mem_free ((void *)bufp);
	  break;
	}

      i = 0;
      if (stringeqn (bufp, "comment_char", 12))
	{
	  while (!iswhitespace(*(bufp+i))) /* skip "comment_char" */
	    {
	      i++;
	    }
	  while (iswhitespace(*(bufp+i)))
	    {
	      i++;
	    }
	  lcp->comment_char = *(bufp+i);
	}
      else if (stringeqn (bufp, "escape_char", 11))
	{
	  while (!iswhitespace(*(bufp+i))) /* skip "escape_char" */
	    {
	      i++;
	    }
	  while (iswhitespace(*(bufp+i)))
	    {
	      i++;
	    }
	  lcp->escape_char = *(bufp+i);
	}
      else if (stringeqn (bufp, "repertoiremap", 13))
	{
	  while (!iswhitespace(*(bufp+i))) /* skip "repertoiremap" */
	    {
	      i++;
	    }
	  while (iswhitespace(*(bufp+i)))
	    {
	      i++;
	    }
	  tmp = i;
          while (*(bufp+i) != (char) NULL)
            {
              i++;
            }
          lcp->repertoiremap = (char *) mem_calloc (i - tmp, sizeof (char));
	  stringcpy (lcp->repertoiremap, bufp+tmp);
	}
      mem_free ((void *) bufp);
    }

  /* memorize 'LC_CTYPE' position in file */
  fgetpos (lcp->fp, &pos);

  /*
   * Counting Number of Entries
   */

  /* create user defined class */

  num_of_entry = 0;
  while (NULL != get_one_entry_buf (lcp, &bufp))
    {
      if (stringeqn (bufp, "END", 3))
	{
	  mem_free ((void *) bufp);
	  break;
	}

      if (iscommentline (bufp, lcp->comment_char))
	{
	  mem_free ((void *) bufp);
	  continue;
	}

      i = 0;
      /* scan first entry */
      while (iswhitespace(bufp[i]))
	{
	  i++;
	}
      tmp = i;
      while (!iswhitespace(bufp[i]))
	{
	  i++;
	}
      end = i;

      tmpclass = (char *) mem_calloc (end - tmp + 1, sizeof (char));
      for (i=tmp, j=0; i < end; i++, j++)
	{
	  tmpclass[j] = bufp[i];
	}
      tmpclass[end] = (char) NULL;

      if (stringeq (tmpclass, "charclass"))
	{
	  num = count_entry (lcp, bufp + end);
	  lcp->userdefined_class = (char **) mem_calloc (num, sizeof (char *));
	  define_user_class (lcp, bufp + end, lcp->userdefined_class);
	  lcp->userdefined_class_num = num;
	}
      else if (stringeq (tmpclass, "charconv"))
	{
	  num = count_entry (lcp, bufp + end);
	  lcp->userdefined_conv = (char **) mem_calloc (num, sizeof (char *));
	  define_user_class (lcp, bufp + end, lcp->userdefined_conv);
	  lcp->userdefined_conv_num = num;
	}
      else
	{
	  num_of_entry++;
	}
      mem_free ((void *) bufp);
      mem_free ((void *) tmpclass);
    }

  lcp->classp =
    (lc_ctype_data *) mem_alloc (num_of_entry * sizeof (lc_ctype_data));
  lcp->total_num = num_of_entry;
  /* seek to 'LC_CTYPE' */
  fsetpos (lcp->fp, &pos);

  /***********************************
   *        START SCANNING           *
   ***********************************/

  fprintf (stderr, "\n");
  fprintf (stderr, "  # classname  entries\n");
  fprintf (stderr, "----------------------\n");

  count = 0;
  while (NULL != get_one_entry_buf (lcp, &bufp))
    {
      if (iscommentline (bufp, lcp->comment_char))
	{
	  mem_free ((void *) bufp);
	  continue;
	}

      idx = 0;
      /* scan first entry */
      while (iswhitespace(bufp[idx]))
	{
	  idx++;
	}
      tmp = idx;
      while (!iswhitespace(bufp[idx]))
	{
	  idx++;
	}
      end = idx;
      tmpclass = (char *) mem_calloc (end - tmp + 1, sizeof (char));
      for (i=tmp, j=0; i < end; i++, j++)
	{
	  tmpclass[j] = bufp[i];
	}
      tmpclass[end] = (char) NULL;

      if ((stringeq (tmpclass, "charclass")) ||
	  (stringeq (tmpclass, "charconv")))
	{
	  mem_free ((void *) bufp);
	  mem_free ((void *) tmpclass);
	  continue;
	}
      else if (stringeq (tmpclass, "END"))
	{
	  mem_free ((void *) bufp);
	  mem_free ((void *) tmpclass);
	  break;
	}
      else
	{
	  create_one_entry (lcp, tmpclass, &(lcp->classp[count]), (bufp + end));

	  fprintf (stderr, "%3d %9s", count, lcp->classp[count].classname);

	  if (isuserdefinedclass(lcp->classp[count].classname, lcp))
	    {
	      fprintf (stderr, "*");
	    }
	  else
	    {
	      fprintf (stderr, " ");
	    }

	  fprintf (stderr, " %7d\n", lcp->classp[count].hashtable->total_num);

	  mem_free ((void *) bufp);
	  mem_free ((void *) tmpclass);
	  count++;
	}
    }

  fclose (lcp->fp);
  remove (TMPFILE);

  fprintf (stderr, "---------------------------------------\n");

  return (lcp);
}

void
lc_ctype_display ( lc_ctype *lcp )
{
  int i;

  for (i = 0; i < lcp->total_num; i++)
    {
      printf ("class %s\n", lcp->classp[i].classname);
      hashtable_display (lcp->classp[i].hashtable);
    }
}

boolean
lc_ctype_search_wctype ( lc_ctype *lcp,
			 const char *classname,
			 const char *symbol )
{
  boolean ret;
  int idx;
  lc_ctype_char *ptr;
  char *key;

  for (idx = 0; idx < lcp->total_num; idx++)
    {
      if (stringeq (lcp->classp[idx].classname, classname))
	{
	  break;
	}
    }

  if ( idx < lcp->total_num )
    {
      extract_realkey (&key, symbol, lcp->escape_char);
      ptr = (lc_ctype_char *) hashtable_search (lcp->classp[idx].hashtable, key);
      ret = (ptr != NULL) ? true : false;
      mem_free ((void *) key);
    }
  else
    {
      /* fprintf (stderr, "error %s: No such class name\n", classname); */
      ret = false;
    }

  return (ret);
}

char *
lc_ctype_search_toxxx ( const lc_ctype *lcp,
			const char *classname,
			const char *symbol )
{
  lc_ctype_char *ptr;
  char *key;
  int idx;
  char *ret;

  for (idx = 0; idx < lcp->total_num; idx++)
    {
      if (stringeq (lcp->classp[idx].classname, classname))
	{
	  break;
	}
    }

  if ( idx < lcp->total_num )
    {
      extract_realkey (&key, symbol, lcp->escape_char);
      ptr = (lc_ctype_char *) hashtable_search (lcp->classp[idx].hashtable, key);
      ret = (char *) ((ptr) ? ptr->symbol2 : symbol);
      mem_free ((void *) key);
    }
  else
    {
      fprintf (stderr, "error %s: No such class name\n");
      ret = (char *)NULL;
    }

  return (ret);
}

void
lc_ctype_free ( lc_ctype *lcp )
{
  int i;

  mem_free ((void *)lcp->repertoiremap);

  for (i=0; i < lcp->userdefined_class_num ; i++)
    {
      mem_free ((void *)(lcp->userdefined_class[i]));
    }
  mem_free ((void *)lcp->userdefined_class);

  for (i=0; i < lcp->userdefined_conv_num ; i++)
    {
      mem_free ((void *)(lcp->userdefined_conv[i]));
    }
  mem_free ((void *)lcp->userdefined_conv);

  for (i = 0; i < lcp->total_num; i++)
    {
      mem_free ((void *)(lcp->classp[i].classname));
      hashtable_free (lcp->classp[i].hashtable);
    }

  mem_free ((void *)lcp->classp);
  mem_free ((void *)lcp);
}
