/*
 *	Ported to Linux's Second Extended File System as part of the
 *	dump and restore backup suit
 *	Remy Card <card@Linux.EU.Org>, 1994-1997
 *	Stelian Pop <pop@noos.fr>, 1999-2000
 *	Stelian Pop <pop@noos.fr> - Alcve <www.alcove.fr>, 2000
 */

/*
 * Copyright (c) 1983, 1993
 *	The Regents of the University of California.  All rights reserved.
 * (c) UNIX System Laboratories, Inc.
 * All or some portions of this file are derived from material licensed
 * to the University of California by American Telephone and Telegraph
 * Co. or Unix System Laboratories, Inc. and are reproduced herein with
 * the permission of UNIX System Laboratories, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static const char rcsid[] =
	"$Id: tape.c,v 1.23 2000/12/21 15:01:54 stelian Exp $";
#endif /* not lint */

#include <config.h>
#include <sys/param.h>
#include <sys/file.h>
#include <sys/mtio.h>
#include <sys/stat.h>

#ifdef	__linux__
#include <sys/time.h>
#include <linux/ext2_fs.h>
#include <bsdcompat.h>
#else	/* __linux__ */
#include <ufs/ufs/dinode.h>
#endif	/* __linux__ */
#include <protocols/dumprestore.h>

#include <errno.h>
#include <compaterr.h>
#include <setjmp.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#ifdef	__linux__
#include <ext2fs/ext2fs.h>
#endif

#include "restore.h"
#include "extern.h"
#include "pathnames.h"

static long	fssize = MAXBSIZE;
static int	mt = -1;
static int	pipein = 0;
static char	magtape[MAXPATHLEN];
static char	magtapeprefix[MAXPATHLEN];
static int	blkcnt;
static int	numtrec;
static char	*tapebuf;
static union	u_spcl endoftapemark;
static long	blksread;		/* blocks read since last header */
static long	tpblksread = 0;		/* TP_BSIZE blocks read */
static long	tapesread;
static sigjmp_buf	restart;
static int	gettingfile = 0;	/* restart has a valid frame */
static char	*host = NULL;

static int	ofile;
static char	*map;
static char	lnkbuf[MAXPATHLEN + 1];
static int	pathlen;

int		oldinofmt;	/* old inode format conversion required */
int		Bcvt;		/* Swap Bytes (for CCI or sun) */
static int	Qcvt;		/* Swap quads (for sun) */

#define	FLUSHTAPEBUF()	blkcnt = ntrec + 1

static void	 accthdr __P((struct s_spcl *));
static int	 checksum __P((int *));
static void	 findinode __P((struct s_spcl *));
static void	 findtapeblksize __P((void));
static int	 gethead __P((struct s_spcl *));
static void	 readtape __P((char *));
static void	 setdumpnum __P((void));
static u_int	 swabi __P((u_int));
static u_long	 swabl __P((u_long));
static u_char	*swab64 __P((u_char *, int));
static u_char	*swab32 __P((u_char *, int));
static u_char	*swab16 __P((u_char *, int));
static void	 terminateinput __P((void));
static void	 xtrfile __P((char *, size_t));
static void	 xtrlnkfile __P((char *, size_t));
static void	 xtrlnkskip __P((char *, size_t));
static void	 xtrmap __P((char *, size_t));
static void	 xtrmapskip __P((char *, size_t));
static void	 xtrskip __P((char *, size_t));

#define COMPARE_ONTHEFLY 1

#if COMPARE_ONTHEFLY
static int	ifile;		/* input file for compare */
static int	cmperror;	/* compare error */
static void	xtrcmpfile __P((char *, size_t));
static void	xtrcmpskip __P((char *, size_t));
#endif

static int readmapflag;

/*
 * Set up an input source
 */
void
setinput(char *source)
{
	FLUSHTAPEBUF();
	if (bflag)
		newtapebuf(ntrec);
	else
		newtapebuf(NTREC > HIGHDENSITYTREC ? NTREC : HIGHDENSITYTREC);
	terminal = stdin;

#ifdef RRESTORE
	if (strchr(source, ':')) {
		host = source;
		source = strchr(host, ':');
		*source++ = '\0';
		if (rmthost(host) == 0)
			exit(1);
	} else
#endif
	if (strcmp(source, "-") == 0) {
		/*
		 * Since input is coming from a pipe we must establish
		 * our own connection to the terminal.
		 */
		terminal = fopen(_PATH_TTY, "r");
		if (terminal == NULL) {
			warn("cannot open %s", _PATH_TTY);
			terminal = fopen(_PATH_DEVNULL, "r");
			if (terminal == NULL)
				err(1, "cannot open %s", _PATH_DEVNULL);
		}
		pipein++;
	}
	setuid(getuid());	/* no longer need or want root privileges */
	if (Mflag) {
		strncpy(magtapeprefix, source, MAXPATHLEN);
		magtapeprefix[MAXPATHLEN-1] = '\0';
		snprintf(magtape, MAXPATHLEN, "%s%03d", source, 1);
	}
	else
		strncpy(magtape, source, MAXPATHLEN);
	magtape[MAXPATHLEN - 1] = '\0';
}

void
newtapebuf(long size)
{
	static int tapebufsize = -1;

	ntrec = size;
	if (size <= tapebufsize)
		return;
	if (tapebuf != NULL)
		free(tapebuf);
	tapebuf = malloc(size * TP_BSIZE);
	if (tapebuf == NULL)
		errx(1, "Cannot allocate space for tape buffer");
	tapebufsize = size;
}

/*
 * Verify that the tape drive can be accessed and
 * that it actually is a dump tape.
 */
void
setup(void)
{
	int i, j, *ip;
	struct stat stbuf;

	Vprintf(stdout, "Verify tape and initialize maps\n");
#ifdef RRESTORE
	if (host)
		mt = rmtopen(magtape, 0);
	else
#endif
	if (pipein)
		mt = 0;
	else
		mt = open(magtape, O_RDONLY, 0);
	if (mt < 0)
		err(1, "%s", magtape);
	volno = 1;
	setdumpnum();
	FLUSHTAPEBUF();
	if (!pipein && !bflag)
		findtapeblksize();
	if (gethead(&spcl) == FAIL) {
		blkcnt--; /* push back this block */
		blksread--;
		tpblksread--;
		cvtflag++;
		if (gethead(&spcl) == FAIL)
			errx(1, "Tape is not a dump tape");
		fprintf(stderr, "Converting to new file system format.\n");
	}
	if (pipein) {
		endoftapemark.s_spcl.c_magic = cvtflag ? OFS_MAGIC : NFS_MAGIC;
		endoftapemark.s_spcl.c_type = TS_END;
		ip = (int *)&endoftapemark;
		j = sizeof(union u_spcl) / sizeof(int);
		i = 0;
		do
			i += *ip++;
		while (--j);
		endoftapemark.s_spcl.c_checksum = CHECKSUM - i;
	}
	if (vflag || command == 't' || command == 'C')
		printdumpinfo();
	if (filesys[0] == '\0') {
		char *dirptr;
		strncpy(filesys, spcl.c_filesys, NAMELEN);
		filesys[NAMELEN - 1] = '\0';
		dirptr = strstr(filesys, " (dir");
		if (dirptr != NULL)
			*dirptr = '\0';
	}
	dumptime = spcl.c_ddate;
	dumpdate = spcl.c_date;
	if (stat(".", &stbuf) < 0)
		err(1, "cannot stat .");
	if (stbuf.st_blksize > 0 && stbuf.st_blksize < TP_BSIZE )
		fssize = TP_BSIZE;
	if (stbuf.st_blksize >= TP_BSIZE && stbuf.st_blksize <= MAXBSIZE)
		fssize = stbuf.st_blksize;
	if (((fssize - 1) & fssize) != 0)
		errx(1, "bad block size %ld", fssize);
	if (spcl.c_volume != 1)
		errx(1, "Tape is not volume 1 of the dump");
	if (gethead(&spcl) == FAIL) {
		Dprintf(stdout, "header read failed at %ld blocks\n", (long)blksread);
		panic("no header after volume mark!\n");
	}
	findinode(&spcl);
	if (spcl.c_type != TS_CLRI)
		errx(1, "Cannot find file removal list");
	maxino = (spcl.c_count * TP_BSIZE * NBBY) + 1;
	Dprintf(stdout, "maxino = %ld\n", (long)maxino);
	map = calloc((unsigned)1, (unsigned)howmany(maxino, NBBY));
	if (map == NULL)
		errx(1, "no memory for active inode map");
	usedinomap = map;
	curfile.action = USING;
	getfile(xtrmap, xtrmapskip);
	if (spcl.c_type != TS_BITS)
		errx(1, "Cannot find file dump list");
	map = calloc((unsigned)1, (unsigned)howmany(maxino, NBBY));
	if (map == (char *)NULL)
		errx(1, "no memory for file dump list");
	dumpmap = map;
	curfile.action = USING;
	getfile(xtrmap, xtrmapskip);
	/*
	 * If there may be whiteout entries on the tape, pretend that the
	 * whiteout inode exists, so that the whiteout entries can be
	 * extracted.
	 */
	if (oldinofmt == 0)
		SETINO(WINO, dumpmap);
}

/*
 * Prompt user to load a new dump volume.
 * "Nextvol" is the next suggested volume to use.
 * This suggested volume is enforced when doing full
 * or incremental restores, but can be overridden by
 * the user when only extracting a subset of the files.
 */
void
getvol(long nextvol)
{
	long newvol = 0, savecnt = 0, wantnext = 0, i;
	union u_spcl tmpspcl;
#	define tmpbuf tmpspcl.s_spcl
	char buf[TP_BSIZE];
	int haderror = 0;

	if (nextvol == 1) {
		tapesread = 0;
		gettingfile = 0;
	}
	if (pipein) {
		if (nextvol != 1)
			panic("Changing volumes on pipe input?\n");
		if (volno == 1)
			return;
		goto gethdr;
	}
	savecnt = blksread;
again:
	if (pipein)
		exit(1); /* pipes do not get a second chance */
	if (command == 'R' || command == 'r' || curfile.action != SKIP) {
		newvol = nextvol;
		wantnext = 1;
	} else {
		newvol = 0;
		wantnext = 0;
	}
	while (newvol <= 0) {
		if (tapesread == 0) {
			fprintf(stderr, "%s%s%s%s%s",
			    "You have not read any tapes yet.\n",
			    "Unless you know which volume your",
			    " file(s) are on you should start\n",
			    "with the last volume and work",
			    " towards the first.\n");
		} else {
			fprintf(stderr, "You have read volumes");
			strcpy(buf, ": ");
			for (i = 1; i < 32; i++)
				if (tapesread & (1 << i)) {
					fprintf(stderr, "%s%ld", buf, (long)i);
					strcpy(buf, ", ");
				}
			fprintf(stderr, "\n");
		}
		do	{
			fprintf(stderr, "Specify next volume #: ");
			(void) fflush(stderr);
			(void) fgets(buf, TP_BSIZE, terminal);
		} while (!feof(terminal) && buf[0] == '\n');
		if (feof(terminal))
			exit(1);
		newvol = atoi(buf);
		if (newvol <= 0) {
			fprintf(stderr,
			    "Volume numbers are positive numerics\n");
		}
	}
	if (newvol == volno) {
		tapesread |= 1 << volno;
		return;
	}
	closemt();
	if (Mflag) {
		snprintf(magtape, MAXPATHLEN, "%s%03ld", magtapeprefix, newvol);
		magtape[MAXPATHLEN - 1] = '\0';
	}
	if (!Mflag || haderror) {
		haderror = 0;
		fprintf(stderr, "Mount tape volume %ld\n", (long)newvol);
		fprintf(stderr, "Enter ``none'' if there are no more tapes\n");
		fprintf(stderr, "otherwise enter tape name (default: %s) ", magtape);
		(void) fflush(stderr);
		(void) fgets(buf, TP_BSIZE, terminal);
		if (feof(terminal))
			exit(1);
		if (!strcmp(buf, "none\n")) {
			terminateinput();
			return;
		}
		if (buf[0] != '\n') {
			(void) strcpy(magtape, buf);
			magtape[strlen(magtape) - 1] = '\0';
		}
	}
#ifdef RRESTORE
	if (host)
		mt = rmtopen(magtape, 0);
	else
#endif
		mt = open(magtape, O_RDONLY, 0);

	if (mt == -1) {
		fprintf(stderr, "Cannot open %s\n", magtape);
		volno = -1;
		haderror = 1;
		goto again;
	}
gethdr:
	volno = newvol;
	setdumpnum();
	FLUSHTAPEBUF();
	if (gethead(&tmpbuf) == FAIL) {
		Dprintf(stdout, "header read failed at %ld blocks\n", (long)blksread);
		fprintf(stderr, "tape is not dump tape\n");
		volno = 0;
		haderror = 1;
		goto again;
	}
	if (tmpbuf.c_volume != volno) {
		fprintf(stderr, "Wrong volume (%d)\n", tmpbuf.c_volume);
		volno = 0;
		haderror = 1;
		goto again;
	}
	if (tmpbuf.c_date != dumpdate || tmpbuf.c_ddate != dumptime) {
#ifdef	__linux__
		fprintf(stderr, "Wrong dump date\n\tgot: %s",
			ctime4(&tmpbuf.c_date));
		fprintf(stderr, "\twanted: %s", ctime4(&dumpdate));
#else
		fprintf(stderr, "Wrong dump date\n\tgot: %s",
			ctime(&tmpbuf.c_date));
		fprintf(stderr, "\twanted: %s", ctime(&dumpdate));
#endif
		volno = 0;
		haderror = 1;
		goto again;
	}
	tapesread |= 1 << volno;
	blksread = savecnt;
 	/*
 	 * If continuing from the previous volume, skip over any
 	 * blocks read already at the end of the previous volume.
 	 *
 	 * If coming to this volume at random, skip to the beginning
 	 * of the next record.
 	 */
	Dprintf(stdout, "read %ld recs, tape starts with %ld\n",
		tpblksread, (long)tmpbuf.c_firstrec);
 	if (tmpbuf.c_type == TS_TAPE && (tmpbuf.c_flags & DR_NEWHEADER)) {
 		if (!wantnext) {
 			tpblksread = tmpbuf.c_firstrec;
 			for (i = tmpbuf.c_count; i > 0; i--)
 				readtape(buf);
 		} else if (tmpbuf.c_firstrec > 0 &&
			   tmpbuf.c_firstrec < tpblksread - 1) {
			/*
			 * -1 since we've read the volume header
			 */
 			i = tpblksread - tmpbuf.c_firstrec - 1;
			Dprintf(stderr, "Skipping %ld duplicate record%s.\n",
				(long)i, i > 1 ? "s" : "");
 			while (--i >= 0)
 				readtape(buf);
 		}
 	}
	if (curfile.action == USING) {
		if (volno == 1)
			panic("active file into volume 1\n");
		return;
	}
	/*
	 * Skip up to the beginning of the next record
	 */
	if (tmpbuf.c_type == TS_TAPE && (tmpbuf.c_flags & DR_NEWHEADER))
		for (i = tmpbuf.c_count; i > 0; i--)
			readtape(buf);
	(void) gethead(&spcl);
	findinode(&spcl);
	if (gettingfile) {
		gettingfile = 0;
		siglongjmp(restart, 1);
	}
}

/*
 * Handle unexpected EOF.
 */
static void
terminateinput(void)
{

	if (gettingfile && curfile.action == USING) {
		printf("Warning: %s %s\n",
		    "End-of-input encountered while extracting", curfile.name);
	}
	curfile.name = "<name unknown>";
	curfile.action = UNKNOWN;
	curfile.dip = NULL;
	curfile.ino = maxino;
	if (gettingfile) {
		gettingfile = 0;
		siglongjmp(restart, 1);
	}
}

/*
 * handle multiple dumps per tape by skipping forward to the
 * appropriate one.
 */
static void
setdumpnum(void)
{
	struct mtop tcom;

	if (dumpnum == 1 || volno != 1)
		return;
	if (pipein)
		errx(1, "Cannot have multiple dumps on pipe input");
	tcom.mt_op = MTFSF;
	tcom.mt_count = dumpnum - 1;
#ifdef RRESTORE
	if (host)
		rmtioctl(MTFSF, dumpnum - 1);
	else
#endif
		if (ioctl(mt, (int)MTIOCTOP, (char *)&tcom) < 0)
			warn("ioctl MTFSF");
}

void
printdumpinfo(void)
{
#ifdef	__linux__
	fprintf(stdout, "Dump   date: %s", ctime4(&spcl.c_date));
	fprintf(stdout, "Dumped from: %s",
	    (spcl.c_ddate == 0) ? "the epoch\n" : ctime4(&spcl.c_ddate));
#else
	fprintf(stdout, "Dump   date: %s", ctime(&spcl.c_date));
	fprintf(stdout, "Dumped from: %s",
	    (spcl.c_ddate == 0) ? "the epoch\n" : ctime(&spcl.c_ddate));
#endif
	if (spcl.c_host[0] == '\0')
		return;
	fprintf(stdout, "Level %d dump of %s on %s:%s\n",
		spcl.c_level, spcl.c_filesys, spcl.c_host, spcl.c_dev);
	fprintf(stdout, "Label: %s\n", spcl.c_label);
}

int
extractfile(char *name)
{
	unsigned int flags;
	mode_t mode;
	struct timeval timep[2];
	struct entry *ep;

	curfile.name = name;
	curfile.action = USING;
#ifdef	__linux__
	timep[0].tv_sec = curfile.dip->di_atime.tv_sec;
	timep[0].tv_usec = curfile.dip->di_atime.tv_usec;
	timep[1].tv_sec = curfile.dip->di_mtime.tv_sec;
	timep[1].tv_usec = curfile.dip->di_mtime.tv_usec;
#else	/* __linux__ */
	timep[0].tv_sec = curfile.dip->di_atime;
	timep[0].tv_usec = curfile.dip->di_atimensec / 1000;
	timep[1].tv_sec = curfile.dip->di_mtime;
	timep[1].tv_usec = curfile.dip->di_mtimensec / 1000;
#endif	/* __linux__ */
	mode = curfile.dip->di_mode;
	flags = curfile.dip->di_flags;
	switch (mode & IFMT) {

	default:
		fprintf(stderr, "%s: unknown file mode 0%o\n", name, mode);
		skipfile();
		return (FAIL);

	case IFSOCK:
		Vprintf(stdout, "skipped socket %s\n", name);
		skipfile();
		return (GOOD);

	case IFDIR:
		if (mflag) {
			ep = lookupname(name);
			if (ep == NULL || ep->e_flags & EXTRACT)
				panic("unextracted directory %s\n", name);
			skipfile();
			return (GOOD);
		}
		Vprintf(stdout, "extract file %s\n", name);
		return (genliteraldir(name, curfile.ino));

	case IFLNK:
	{
#ifdef HAVE_LCHOWN
		uid_t luid = curfile.dip->di_uid;
		gid_t lgid = curfile.dip->di_gid;
#endif
		lnkbuf[0] = '\0';
		pathlen = 0;
		getfile(xtrlnkfile, xtrlnkskip);
		if (pathlen == 0) {
			Vprintf(stdout,
			    "%s: zero length symbolic link (ignored)\n", name);
			return (GOOD);
		}
		if (linkit(lnkbuf, name, SYMLINK) == FAIL)
			return (FAIL);
#ifdef HAVE_LCHOWN
		(void) lchown(name, luid, lgid);
#endif
		return (GOOD);
	}

	case IFIFO:
		Vprintf(stdout, "extract fifo %s\n", name);
		if (Nflag) {
			skipfile();
			return (GOOD);
		}
		if (uflag && !Nflag)
			(void)unlink(name);
		if (mkfifo(name, mode) < 0) {
			warn("%s: cannot create fifo", name);
			skipfile();
			return (FAIL);
		}
		(void) chown(name, curfile.dip->di_uid, curfile.dip->di_gid);
		(void) chmod(name, mode);
		if (flags)
#ifdef  __linux__
			(void) fsetflags(name, flags);
#else
			(void) chflags(name, flags);
#endif
		skipfile();
		utimes(name, timep);
		return (GOOD);

	case IFCHR:
	case IFBLK:
		Vprintf(stdout, "extract special file %s\n", name);
		if (Nflag) {
			skipfile();
			return (GOOD);
		}
		if (uflag)
			(void)unlink(name);
		if (mknod(name, mode, (int)curfile.dip->di_rdev) < 0) {
			warn("%s: cannot create special file", name);
			skipfile();
			return (FAIL);
		}
		(void) chown(name, curfile.dip->di_uid, curfile.dip->di_gid);
		(void) chmod(name, mode);
		if (flags)
#ifdef	__linux__
			{
			warn("%s: fsetflags called on a special file", name);
			(void) fsetflags(name, flags);
			}
#else
			(void) chflags(name, flags);
#endif
		skipfile();
		utimes(name, timep);
		return (GOOD);

	case IFREG:
		Vprintf(stdout, "extract file %s\n", name);
		if (Nflag) {
			skipfile();
			return (GOOD);
		}
		if (uflag)
			(void)unlink(name);
		if ((ofile = open(name, O_WRONLY | O_CREAT | O_TRUNC,
		    0666)) < 0) {
			warn("%s: cannot create file", name);
			skipfile();
			return (FAIL);
		}
		(void) fchown(ofile, curfile.dip->di_uid, curfile.dip->di_gid);
		(void) fchmod(ofile, mode);
		if (flags)
#ifdef	__linux__
			(void) setflags(ofile, flags);
#else
			(void) fchflags(ofile, flags);
#endif
		getfile(xtrfile, xtrskip);
		(void) close(ofile);
		utimes(name, timep);
		return (GOOD);
	}
	/* NOTREACHED */
}

/*
 * skip over bit maps on the tape
 */
void
skipmaps(void)
{

	while (spcl.c_type == TS_BITS || spcl.c_type == TS_CLRI)
		skipfile();
}

/*
 * skip over a file on the tape
 */
void
skipfile(void)
{

	curfile.action = SKIP;
	getfile(xtrnull, xtrnull);
}

/*
 * Extract a file from the tape.
 * When an allocated block is found it is passed to the fill function;
 * when an unallocated block (hole) is found, a zeroed buffer is passed
 * to the skip function.
 */
void
getfile(void (*fill) __P((char *, size_t)), void (*skip) __P((char *, size_t)))
{
	register int i;
	volatile int curblk = 0;
	volatile quad_t size = spcl.c_dinode.di_size;
	volatile int last_write_was_hole = 0;
	quad_t origsize = size;
	static char clearedbuf[MAXBSIZE];
	char buf[MAXBSIZE / TP_BSIZE][TP_BSIZE];
	char junk[TP_BSIZE];

	if (spcl.c_type == TS_END)
		panic("ran off end of tape\n");
	if (spcl.c_magic != NFS_MAGIC)
		panic("not at beginning of a file\n");
	if (!gettingfile && setjmp(restart) != 0)
		return;
	gettingfile++;
loop:
	for (i = 0; i < spcl.c_count; i++) {
		if (readmapflag || spcl.c_addr[i]) {
			readtape(&buf[curblk++][0]);
			if (curblk == fssize / TP_BSIZE) {
				(*fill)((char *)buf, (size_t)(size > TP_BSIZE ?
				     fssize : (curblk - 1) * TP_BSIZE + size));
				curblk = 0;
				last_write_was_hole = 0;
			}
		} else {
			if (curblk > 0) {
				(*fill)((char *)buf, (size_t)(size > TP_BSIZE ?
				     curblk * TP_BSIZE :
				     (curblk - 1) * TP_BSIZE + size));
				curblk = 0;
			}
			(*skip)(clearedbuf, (long)(size > TP_BSIZE ?
				TP_BSIZE : size));
			last_write_was_hole = 1;
		}
		if ((size -= TP_BSIZE) <= 0) {
			for (i++; i < spcl.c_count; i++)
				if (readmapflag || spcl.c_addr[i])
					readtape(junk);
			break;
		}
	}
	if (gethead(&spcl) == GOOD && size > 0) {
		if (spcl.c_type == TS_ADDR)
			goto loop;
		Dprintf(stdout,
			"Missing address (header) block for %s at %ld blocks\n",
			curfile.name, (long)blksread);
	}
	if (curblk > 0) {
		(*fill)((char *)buf, (size_t)(curblk * TP_BSIZE) + size);
		last_write_was_hole = 0;
	}
	if (last_write_was_hole) {
		ftruncate(ofile, origsize);
	}
	findinode(&spcl);
	gettingfile = 0;
}

/*
 * Write out the next block of a file.
 */
static void
xtrfile(char *buf, size_t size)
{

	if (Nflag)
		return;
	if (write(ofile, buf, (int) size) == -1)
		err(1, "write error extracting inode %lu, name %s\nwrite",
			(unsigned long)curfile.ino, curfile.name);
}

/*
 * Skip over a hole in a file.
 */
/* ARGSUSED */
static void
xtrskip(char *buf, size_t size)
{

	if (lseek(ofile, (off_t)size, SEEK_CUR) == -1)
		err(1, "seek error extracting inode %lu, name %s\nlseek",
			(unsigned long)curfile.ino, curfile.name);
}

/*
 * Collect the next block of a symbolic link.
 */
static void
xtrlnkfile(char *buf, size_t size)
{

	pathlen += size;
	if (pathlen > MAXPATHLEN)
		errx(1, "symbolic link name: %s->%s%s; too long %d",
		    curfile.name, lnkbuf, buf, pathlen);
	(void) strcat(lnkbuf, buf);
}

/*
 * Skip over a hole in a symbolic link (should never happen).
 */
/* ARGSUSED */
static void
xtrlnkskip(char *buf, size_t size)
{

	errx(1, "unallocated block in symbolic link %s", curfile.name);
}

/*
 * Collect the next block of a bit map.
 */
static void
xtrmap(char *buf, size_t size)
{

	memmove(map, buf, size);
	map += size;
}

/*
 * Skip over a hole in a bit map (should never happen).
 */
/* ARGSUSED */
static void
xtrmapskip(char *buf, size_t size)
{

	panic("hole in map\n");
	map += size;
}

/*
 * Noop, when an extraction function is not needed.
 */
/* ARGSUSED */
void
xtrnull(char *buf, size_t size)
{

	return;
}

#if COMPARE_ONTHEFLY
/*
 * Compare the next block of a file.
 */
static void
xtrcmpfile(char *buf, size_t size)
{
	static char cmpbuf[MAXBSIZE];

	if (cmperror)
		return;
	
	if (read(ifile, cmpbuf, size) != size) {
		fprintf(stderr, "%s: size has changed.\n", 
			curfile.name);
		cmperror = 1;
		return;
	}
	
	if (memcmp(buf, cmpbuf, size) != 0) {
		fprintf(stderr, "%s: tape and disk copies are different\n",
			curfile.name);
		cmperror = 1;
		return;
	}
}

/*
 * Skip over a hole in a file.
 */
static void
xtrcmpskip(char *buf, size_t size)
{
	static char cmpbuf[MAXBSIZE];
	int i;

	if (cmperror)
		return;
	
	if (read(ifile, cmpbuf, size) != size) {
		fprintf(stderr, "%s: size has changed.\n", 
			curfile.name);
		cmperror = 1;
		return;
	}

	for (i = 0; i < size; ++i)
		if (cmpbuf[i] != '\0') {
			fprintf(stderr, "%s: tape and disk copies are different\n",
				curfile.name);
			cmperror = 1;
			return;
		}
}
#endif /* COMPARE_ONTHEFLY */

#if !COMPARE_ONTHEFLY
static int
do_cmpfiles(int fd_tape, int fd_disk, long size)
{
	static char buf_tape[BUFSIZ];
	static char buf_disk[BUFSIZ];
	ssize_t n_tape;
	ssize_t n_disk;

	while (size > 0) {
		if ((n_tape = read(fd_tape, buf_tape, sizeof(buf_tape))) < 1) {
			close(fd_tape), close(fd_disk);
			panic("do_cmpfiles: unexpected EOF[1]");
		}
		if ((n_disk = read(fd_disk, buf_disk, sizeof(buf_tape))) < 1) {
			close(fd_tape), close(fd_disk);
			panic("do_cmpfiles: unexpected EOF[2]");
		}
		if (n_tape != n_disk) {
			close(fd_tape), close(fd_disk);
			panic("do_cmpfiles: sizes different!");
		}
		if (memcmp(buf_tape, buf_disk, (size_t)n_tape) != 0) return (1);
		size -= n_tape;
	}
	return (0);
}

/* for debugging compare problems */
#undef COMPARE_FAIL_KEEP_FILE

static
#ifdef COMPARE_FAIL_KEEP_FILE
/* return true if tapefile should be unlinked after compare */
int
#else
void
#endif
cmpfiles(char *tapefile, char *diskfile, struct stat *sbuf_disk)
{
	struct stat sbuf_tape;
	int fd_tape, fd_disk;

	if (stat(tapefile, &sbuf_tape) != 0) {
		panic("Can't lstat tmp file %s: %s\n", tapefile,
		      strerror(errno));
		compare_errors = 1;
	}

	if (sbuf_disk->st_size != sbuf_tape.st_size) {
		fprintf(stderr,
			"%s: size changed from %ld to %ld.\n",
			diskfile, (long)sbuf_tape.st_size, (long)sbuf_disk->st_size);
		compare_errors = 1;
#ifdef COMPARE_FAIL_KEEP_FILE
		return (0);
#else
		return;
#endif
	}

	if ((fd_tape = open(tapefile, O_RDONLY)) < 0) {
		panic("Can't open %s: %s\n", tapefile, strerror(errno));
		compare_errors = 1;
	}
	if ((fd_disk = open(diskfile, O_RDONLY)) < 0) {
		close(fd_tape);
		panic("Can't open %s: %s\n", diskfile, strerror(errno));
		compare_errors = 1;
	}

	if (do_cmpfiles(fd_tape, fd_disk, sbuf_tape.st_size)) {
		fprintf(stderr, "%s: tape and disk copies are different\n",
			diskfile);
		close(fd_tape);
		close(fd_disk);
		compare_errors = 1;
#ifdef COMPARE_FAIL_KEEP_FILE
		/* rename the file to live in /tmp */
		/* rename `tapefile' to /tmp/<basename of diskfile> */
		{
			char *p = strrchr(diskfile, '/');
			char newname[MAXPATHLEN];
			if (!p) {
				panic("can't find / in %s\n", diskfile);
			}
			snprintf(newname, sizeof(newname), "%s/debug/%s", tmpdir, p + 1);
			if (rename(tapefile, newname)) {
				panic("rename from %s to %s failed: %s\n",
				      tapefile, newname,
				      strerror(errno));
			} else {
				fprintf(stderr, "*** %s saved to %s\n",
					tapefile, newname);
			}
		}
		
		/* don't unlink the file (it's not there anymore */
		/* anyway) */
		return (0);
#else
		return;
#endif
	}
	close(fd_tape);
	close(fd_disk);
#ifdef COMPARE_FAIL_KEEP_FILE
	return (1);
#endif
}
#endif /* !COMPARE_ONTHEFLY */

#if !COMPARE_ONTHEFLY
static char tmpfilename[MAXPATHLEN];
#endif

void
comparefile(char *name)
{
	int mode;
	struct stat sb;
	int r;
#if !COMPARE_ONTHEFLY
	static char *tmpfile = NULL;
	struct stat stemp;
#endif

	if ((r = lstat(name, &sb)) != 0) {
		warn("%s: does not exist (%d)", name, r);
		compare_errors = 1;
		skipfile();
		return;
	}

	curfile.name = name;
	curfile.action = USING;
	mode = curfile.dip->di_mode;

	Vprintf(stdout, "comparing %s (size: %ld, mode: 0%o)\n", name,
		(long)sb.st_size, mode);

	if (sb.st_mode != mode) {
		fprintf(stderr, "%s: mode changed from 0%o to 0%o.\n",
			name, mode & 07777, sb.st_mode & 07777);
		compare_errors = 1;
	}
	switch (mode & IFMT) {
	default:
		skipfile();
		return;

	case IFSOCK:
		skipfile();
		return;

	case IFDIR:
		skipfile();
		return;

	case IFLNK: {
		char lbuf[MAXPATHLEN + 1];
		int lsize;

		if (!(sb.st_mode & S_IFLNK)) {
			fprintf(stderr, "%s: is no longer a symbolic link\n",
				name);
			compare_errors = 1;
			return;
		}
		lnkbuf[0] = '\0';
		pathlen = 0;
		getfile(xtrlnkfile, xtrlnkskip);
		if (pathlen == 0) {
			fprintf(stderr,
				"%s: zero length symbolic link (ignored)\n",
				name);
			compare_errors = 1;
			return;
		}
		if ((lsize = readlink(name, lbuf, MAXPATHLEN)) < 0) {
			panic("readlink of %s failed: %s", name,
			      strerror(errno));
			compare_errors = 1;
		}
		lbuf[lsize] = 0;
		if (strcmp(lbuf, lnkbuf) != 0) {
			fprintf(stderr,
				"%s: symbolic link changed from %s to %s.\n",
				name, lnkbuf, lbuf);
			compare_errors = 1;
			return;
		}
		return;
	}

	case IFCHR:
	case IFBLK:
		if (!(sb.st_mode & (S_IFCHR|S_IFBLK))) {
			fprintf(stderr, "%s: no longer a special file\n",
				name);
			compare_errors = 1;
			skipfile();
			return;
		}

		if (sb.st_rdev != (int)curfile.dip->di_rdev) {
			fprintf(stderr,
				"%s: device changed from %d,%d to %d,%d.\n",
				name,
				((int)curfile.dip->di_rdev >> 8) & 0xff,
				(int)curfile.dip->di_rdev & 0xff,
				((int)sb.st_rdev >> 8) & 0xff,
				(int)sb.st_rdev & 0xff);
			compare_errors = 1;
		}
		skipfile();
		return;

	case IFREG:
#if COMPARE_ONTHEFLY
		if ((ifile = open(name, O_RDONLY)) < 0) {
			panic("Can't open %s: %s\n", name, strerror(errno));
			skipfile();
			compare_errors = 1;
		}
		else {
			cmperror = 0;
			getfile(xtrcmpfile, xtrcmpskip);
			if (!cmperror) {
				char c;
				if (read(ifile, &c, 1) != 0) {
					fprintf(stderr, "%s: size has changed.\n", 
						name);
					cmperror = 1;
				}
			}
			if (cmperror)
				compare_errors = 1;
			close(ifile);
		}
#else
		if (tmpfile == NULL) {
			/* argument to mktemp() must not be in RO space: */
			snprintf(tmpfilename, sizeof(tmpfilename), "%s/restoreCXXXXXX", tmpdir);
			tmpfile = mktemp(&tmpfilename[0]);
		}
		if ((stat(tmpfile, &stemp) == 0) && (unlink(tmpfile) != 0)) {
			panic("cannot delete tmp file %s: %s\n",
			      tmpfile, strerror(errno));
		}
		if ((ofile = creat(tmpfile, 0600)) < 0) {
			panic("cannot create file temp file %s: %s\n",
			      name, strerror(errno));
		}
		getfile(xtrfile, xtrskip);
		(void) close(ofile);
#ifdef COMPARE_FAIL_KEEP_FILE
		if (cmpfiles(tmpfile, name, &sb))
			unlink(tmpfile);
#else
		cmpfiles(tmpfile, name, &sb);
		unlink(tmpfile);
#endif
#endif /* COMPARE_ONTHEFLY */
		return;
	}
	/* NOTREACHED */
}

/*
 * Read TP_BSIZE blocks from the input.
 * Handle read errors, and end of media.
 */
static void
readtape(char *buf)
{
	ssize_t rd, newvol, i;
	int cnt, seek_failed;

	if (blkcnt < numtrec) {
		memmove(buf, &tapebuf[(blkcnt++ * TP_BSIZE)], TP_BSIZE);
		blksread++;
		tpblksread++;
		return;
	}
	for (i = 0; i < ntrec; i++)
		((struct s_spcl *)&tapebuf[i * TP_BSIZE])->c_magic = 0;
	if (numtrec == 0)
		numtrec = ntrec;
	cnt = ntrec * TP_BSIZE;
	rd = 0;
getmore:
#ifdef RRESTORE
	if (host)
		i = rmtread(&tapebuf[rd], cnt);
	else
#endif
		i = read(mt, &tapebuf[rd], cnt);
	/*
	 * Check for mid-tape short read error.
	 * If found, skip rest of buffer and start with the next.
	 */
	if (!pipein && numtrec < ntrec && i > 0) {
		Dprintf(stdout, "mid-media short read error.\n");
		numtrec = ntrec;
	}
	/*
	 * Handle partial block read.
	 */
	if (pipein && i == 0 && rd > 0)
		i = rd;
	else if (i > 0 && i != ntrec * TP_BSIZE) {
		if (pipein) {
			rd += i;
			cnt -= i;
			if (cnt > 0)
				goto getmore;
			i = rd;
		} else {
			/*
			 * Short read. Process the blocks read.
			 */
			if (i % TP_BSIZE != 0)
				Vprintf(stdout,
				    "partial block read: %ld should be %ld\n",
				    (long)i, ntrec * TP_BSIZE);
			numtrec = i / TP_BSIZE;
		}
	}
	/*
	 * Handle read error.
	 */
	if (i < 0) {
		fprintf(stderr, "Tape read error while ");
		switch (curfile.action) {
		default:
			fprintf(stderr, "trying to set up tape\n");
			break;
		case UNKNOWN:
			fprintf(stderr, "trying to resynchronize\n");
			break;
		case USING:
			fprintf(stderr, "restoring %s\n", curfile.name);
			break;
		case SKIP:
			fprintf(stderr, "skipping over inode %lu\n",
				(unsigned long)curfile.ino);
			break;
		}
		if (!yflag && !reply("continue"))
			exit(1);
		i = ntrec * TP_BSIZE;
		memset(tapebuf, 0, (size_t)i);
#ifdef RRESTORE
		if (host)
			seek_failed = (rmtseek(i, 1) < 0);
		else
#endif
			seek_failed = (lseek(mt, i, SEEK_CUR) == (off_t)-1);

		if (seek_failed) {
			warn("continuation failed");
			if (!yflag && !reply("assume end-of-tape and continue"))
				exit(1);
			i = 0;
		}
	}
	/*
	 * Handle end of tape.
	 */
	if (i == 0) {
		Vprintf(stdout, "End-of-tape encountered\n");
		if (!pipein) {
			newvol = volno + 1;
			volno = 0;
			numtrec = 0;
			getvol(newvol);
			readtape(buf);
			return;
		}
		if (rd % TP_BSIZE != 0)
			panic("partial block read: %d should be %d\n",
				rd, ntrec * TP_BSIZE);
		terminateinput();
		memmove(&tapebuf[rd], &endoftapemark, TP_BSIZE);
	}
	blkcnt = 0;
	memmove(buf, &tapebuf[(blkcnt++ * TP_BSIZE)], TP_BSIZE);
	blksread++;
	tpblksread++;
}

static void
findtapeblksize(void)
{
	register long i;

	for (i = 0; i < ntrec; i++)
		((struct s_spcl *)&tapebuf[i * TP_BSIZE])->c_magic = 0;
	blkcnt = 0;
#ifdef RRESTORE
	if (host)
		i = rmtread(tapebuf, (size_t)(ntrec * TP_BSIZE));
	else
#endif
		i = read(mt, tapebuf, (size_t)(ntrec * TP_BSIZE));

	if (i <= 0)
		err(1, "tape read error");
	if (i % TP_BSIZE != 0)
		errx(1, "Tape block size (%ld) is not a multiple of dump block size (%d)", 
				(long)i, TP_BSIZE);
	ntrec = i / TP_BSIZE;
	numtrec = ntrec;
	Vprintf(stdout, "Tape block size is %ld\n", ntrec);
}

void
closemt(void)
{

	if (mt < 0)
		return;
#ifdef RRESTORE
	if (host)
		rmtclose();
	else
#endif
		(void) close(mt);
}

/*
 * Read the next block from the tape.
 * Check to see if it is one of several vintage headers.
 * If it is an old style header, convert it to a new style header.
 * If it is not any valid header, return an error.
 */
static int
gethead(struct s_spcl *buf)
{
	int32_t i;
	union {
		quad_t	qval;
		int32_t	val[2];
	} qcvt;
	union u_ospcl {
		char dummy[TP_BSIZE];
		struct	s_ospcl {
			int32_t	c_type;
			int32_t	c_date;
			int32_t	c_ddate;
			int32_t	c_volume;
			int32_t	c_tapea;
			u_int16_t c_inumber;
			int32_t	c_magic;
			int32_t	c_checksum;
			struct odinode {
				u_int16_t odi_mode;
				u_int16_t odi_nlink;
				u_int16_t odi_uid;
				u_int16_t odi_gid;
				int32_t	odi_size;
				int32_t	odi_rdev;
				char	odi_addr[36];
				int32_t	odi_atime;
				int32_t	odi_mtime;
				int32_t	odi_ctime;
			} c_dinode;
			int32_t	c_count;
			char	c_addr[256];
		} s_ospcl;
	} u_ospcl;

	if (!cvtflag) {
		readtape((char *)buf);
		if (buf->c_magic != NFS_MAGIC) {
			if (swabi(buf->c_magic) != NFS_MAGIC)
				return (FAIL);
			if (!Bcvt) {
				Vprintf(stdout, "Note: Doing Byte swapping\n");
				Bcvt = 1;
			}
		}
		if (checksum((int *)buf) == FAIL)
			return (FAIL);
		if (Bcvt)
			swabst((u_char *)"8i4s31i528bi192b2i", (u_char *)buf);
		goto good;
	}
	readtape((char *)(&u_ospcl.s_ospcl));
	memset((char *)buf, 0, (long)TP_BSIZE);
	buf->c_type = u_ospcl.s_ospcl.c_type;
	buf->c_date = u_ospcl.s_ospcl.c_date;
	buf->c_ddate = u_ospcl.s_ospcl.c_ddate;
	buf->c_volume = u_ospcl.s_ospcl.c_volume;
	buf->c_tapea = u_ospcl.s_ospcl.c_tapea;
	buf->c_inumber = u_ospcl.s_ospcl.c_inumber;
	buf->c_checksum = u_ospcl.s_ospcl.c_checksum;
	buf->c_magic = u_ospcl.s_ospcl.c_magic;
	buf->c_dinode.di_mode = u_ospcl.s_ospcl.c_dinode.odi_mode;
	buf->c_dinode.di_nlink = u_ospcl.s_ospcl.c_dinode.odi_nlink;
	buf->c_dinode.di_uid = u_ospcl.s_ospcl.c_dinode.odi_uid;
	buf->c_dinode.di_gid = u_ospcl.s_ospcl.c_dinode.odi_gid;
	buf->c_dinode.di_size = u_ospcl.s_ospcl.c_dinode.odi_size;
	buf->c_dinode.di_rdev = u_ospcl.s_ospcl.c_dinode.odi_rdev;
#ifdef	__linux__
	buf->c_dinode.di_atime.tv_sec = u_ospcl.s_ospcl.c_dinode.odi_atime;
	buf->c_dinode.di_mtime.tv_sec = u_ospcl.s_ospcl.c_dinode.odi_mtime;
	buf->c_dinode.di_ctime.tv_sec = u_ospcl.s_ospcl.c_dinode.odi_ctime;
#else	/* __linux__ */
	buf->c_dinode.di_atime = u_ospcl.s_ospcl.c_dinode.odi_atime;
	buf->c_dinode.di_mtime = u_ospcl.s_ospcl.c_dinode.odi_mtime;
	buf->c_dinode.di_ctime = u_ospcl.s_ospcl.c_dinode.odi_ctime;
#endif	/* __linux__ */
	buf->c_count = u_ospcl.s_ospcl.c_count;
	memmove(buf->c_addr, u_ospcl.s_ospcl.c_addr, (long)256);
	if (u_ospcl.s_ospcl.c_magic != OFS_MAGIC ||
	    checksum((int *)(&u_ospcl.s_ospcl)) == FAIL)
		return(FAIL);
	buf->c_magic = NFS_MAGIC;

good:
	if ((buf->c_dinode.di_size == 0 || buf->c_dinode.di_size > 0xfffffff) &&
	    (buf->c_dinode.di_mode & IFMT) == IFDIR && Qcvt == 0) {
		qcvt.qval = buf->c_dinode.di_size;
		if (qcvt.val[0] || qcvt.val[1]) {
			printf("Note: Doing Quad swapping\n");
			Qcvt = 1;
		}
	}
	if (Qcvt) {
		qcvt.qval = buf->c_dinode.di_size;
		i = qcvt.val[1];
		qcvt.val[1] = qcvt.val[0];
		qcvt.val[0] = i;
		buf->c_dinode.di_size = qcvt.qval;
	}
	readmapflag = 0;

	switch (buf->c_type) {

	case TS_CLRI:
	case TS_BITS:
		/*
		 * Have to patch up missing information in bit map headers
		 */
		buf->c_inumber = 0;
		buf->c_dinode.di_size = buf->c_count * TP_BSIZE;
		if (buf->c_count > TP_NINDIR)
			readmapflag = 1;
		else 
			for (i = 0; i < buf->c_count; i++)
				buf->c_addr[i]++;
		break;

	case TS_TAPE:
		if ((buf->c_flags & DR_NEWINODEFMT) == 0)
			oldinofmt = 1;
		/* fall through */
	case TS_END:
		buf->c_inumber = 0;
		break;

	case TS_INODE:
	case TS_ADDR:
		break;

	default:
		panic("gethead: unknown inode type %d\n", buf->c_type);
		break;
	}
	/*
	 * If we are restoring a filesystem with old format inodes,
	 * copy the uid/gid to the new location.
	 */
	if (oldinofmt) {
		buf->c_dinode.di_uid = buf->c_dinode.di_ouid;
		buf->c_dinode.di_gid = buf->c_dinode.di_ogid;
	}
	if (dflag)
		accthdr(buf);
	return(GOOD);
}

/*
 * Check that a header is where it belongs and predict the next header
 */
static void
accthdr(struct s_spcl *header)
{
	static ino_t previno = 0x7fffffff;
	static int prevtype;
	static long predict;
	long blks, i;

	if (header->c_type == TS_TAPE) {
		fprintf(stderr, "Volume header (%s inode format) ",
		    oldinofmt ? "old" : "new");
 		if (header->c_firstrec)
 			fprintf(stderr, "begins with record %d",
 				header->c_firstrec);
 		fprintf(stderr, "\n");
		previno = 0x7fffffff;
		return;
	}
	if (previno == 0x7fffffff)
		goto newcalc;
	switch (prevtype) {
	case TS_BITS:
		fprintf(stderr, "Dumped inodes map header");
		break;
	case TS_CLRI:
		fprintf(stderr, "Used inodes map header");
		break;
	case TS_INODE:
		fprintf(stderr, "File header, ino %lu", (unsigned long)previno);
		break;
	case TS_ADDR:
		fprintf(stderr, "File continuation header, ino %ld", (long)previno);
		break;
	case TS_END:
		fprintf(stderr, "End of tape header");
		break;
	}
	if (predict != blksread - 1)
		fprintf(stderr, "; predicted %ld blocks, got %ld blocks",
			predict, blksread - 1);
	fprintf(stderr, "\n");
newcalc:
	blks = 0;
	if (header->c_type != TS_END)
		for (i = 0; i < header->c_count; i++)
			if (readmapflag || header->c_addr[i] != 0)
				blks++;
	predict = blks;
	blksread = 0;
	prevtype = header->c_type;
	previno = header->c_inumber;
}

/*
 * Find an inode header.
 * Complain if had to skip, and complain is set.
 */
static void
findinode(struct s_spcl *header)
{
	static long skipcnt = 0;
	long i;
	char buf[TP_BSIZE];

	curfile.name = "<name unknown>";
	curfile.action = UNKNOWN;
	curfile.dip = NULL;
	curfile.ino = 0;
	do {
		if (header->c_magic != NFS_MAGIC) {
			skipcnt++;
			while (gethead(header) == FAIL ||
			    header->c_date != dumpdate)
				skipcnt++;
		}
		switch (header->c_type) {

		case TS_ADDR:
			/*
			 * Skip up to the beginning of the next record
			 */
			for (i = 0; i < header->c_count; i++)
				if (header->c_addr[i])
					readtape(buf);
			while (gethead(header) == FAIL ||
			    header->c_date != dumpdate)
				skipcnt++;
			break;

		case TS_INODE:
			curfile.dip = &header->c_dinode;
			curfile.ino = header->c_inumber;
			break;

		case TS_END:
			curfile.ino = maxino;
			break;

		case TS_CLRI:
			curfile.name = "<file removal list>";
			break;

		case TS_BITS:
			curfile.name = "<file dump list>";
			break;

		case TS_TAPE:
			panic("unexpected tape header\n");
			/* NOTREACHED */

		default:
			panic("unknown tape header type %d\n", spcl.c_type);
			/* NOTREACHED */

		}
	} while (header->c_type == TS_ADDR);
	if (skipcnt > 0)
		fprintf(stderr, "resync restore, skipped %ld blocks\n",
		    skipcnt);
	skipcnt = 0;
}

static int
checksum(int *buf)
{
	register int i, j;

	j = sizeof(union u_spcl) / sizeof(int);
	i = 0;
	if(!Bcvt) {
		do
			i += *buf++;
		while (--j);
	} else {
		/* What happens if we want to read restore tapes
			for a 16bit int machine??? */
		do
			i += swabi(*buf++);
		while (--j);
	}

	if (i != CHECKSUM) {
		fprintf(stderr, "Checksum error %o, inode %lu file %s\n", i,
			(unsigned long)curfile.ino, curfile.name);
		return(FAIL);
	}
	return(GOOD);
}

#ifdef RRESTORE
#ifdef __STDC__
#include <stdarg.h>
#else
#include <varargs.h>
#endif

void
#ifdef __STDC__
msg(const char *fmt, ...)
#else
msg(fmt, va_alist)
	char *fmt;
	va_dcl
#endif
{
	va_list ap;
#ifdef __STDC__
	va_start(ap, fmt);
#else
	va_start(ap);
#endif
	(void)vfprintf(stderr, fmt, ap);
	va_end(ap);
}
#endif /* RRESTORE */

static u_char *
swab16(u_char *sp, int n)
{
	char c;

	while (--n >= 0) {
		c = sp[0]; sp[0] = sp[1]; sp[1] = c;
		sp += 2;
	}
	return (sp);
}

static u_char *
swab32(u_char *sp, int n)
{
	char c;

	while (--n >= 0) {
		c = sp[0]; sp[0] = sp[3]; sp[3] = c;
		c = sp[1]; sp[1] = sp[2]; sp[2] = c;
		sp += 4;
	}
	return (sp);
}

static u_char *
swab64(u_char *sp, int n)
{
	char c;

	while (--n >= 0) {
		c = sp[0]; sp[0] = sp[7]; sp[7] = c;
		c = sp[1]; sp[1] = sp[6]; sp[6] = c;
		c = sp[2]; sp[2] = sp[5]; sp[5] = c;
		c = sp[3]; sp[3] = sp[4]; sp[4] = c;
		sp += 8;
	}
	return (sp);
}

void
swabst(u_char *cp, u_char *sp)
{
	int n = 0;

	while (*cp) {
		switch (*cp) {
		case '0': case '1': case '2': case '3': case '4':
		case '5': case '6': case '7': case '8': case '9':
			n = (n * 10) + (*cp++ - '0');
			continue;

		case 's': case 'w': case 'h':
			if (n == 0)
				n = 1;
			sp = swab16(sp, n);
			break;

		case 'i':
			if (n == 0)
				n = 1;
			sp = swab32(sp, n);
			break;

		case 'l':
			if (n == 0)
				n = 1;
			sp = swab64(sp, n);
			break;

		default: /* Any other character, like 'b' counts as byte. */
			if (n == 0)
				n = 1;
			sp += n;
			break;
		}
		cp++;
		n = 0;
	}
}

static u_int
swabi(u_int x)
{
	swabst((u_char *)"i", (u_char *)&x);
	return (x);
}

static u_long
swabl(u_long x)
{
	swabst((u_char *)"l", (u_char *)&x);
	return (x);
}
