/*
 * Copyright (C) 2002 Edscott Wilson Garcia
 * EMail: edscott@imp.mx
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>

#include <dirent.h>
#include <errno.h>
#include <limits.h>
#include <memory.h>
#include <regex.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <dbh.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>


#include "glade_support.h"

#include "constants.h"
#include "types.h"

#include "modules.h"
#include "basic_row.h"

#include "trash.h"
#include "tubo.h"


#include "callbacks.h"
#include "entry.h"
#include "icons.h"
#include "ls.h"
#include "misc.h"
#include "monitor.h"
#include "refresh.h"
#include "widgets.h"

/*#define DEBUG*/
#define GNOME_TRASH 	g_get_home_dir(),".Trash"
/* XXX this is kde2. I dunno about kde3*/
#define KDE_TRASH 	g_get_home_dir(),"Desktop","Trash"

static xftrash_functions *xftrash_fun=NULL;

static GtkTreeIter *target_iter;
static GtkTreeView *target_treeview;
static unsigned int trashcount;
static unsigned int smallcount, countbyte;
static xfdir_t trash_xfdir;
static int target_type;
static gboolean just_count;


static DBHashTable *trashbin = NULL;
static DBHashTable *newtrashbin = NULL;


static 
const char *trash_path(void){
    gchar *trashfile=NULL;
    if (!trashfile) {
	gchar *xdg_dir=xfce_resource_save_location (XFCE_RESOURCE_CACHE,"/",TRUE);
	trashfile=g_build_filename(xdg_dir,CURRENT_TRASHBIN,NULL);
	g_free(xdg_dir);
    }
    return (const char *)trashfile;
}

/*FIXME these  functions should be shared 
(fork_stderr, fork_stdout, fork_wait, fork_done)
and duplicates in other modules removed...
 see smb routines to begin with. then find.c
 cpy.c, and wherever Tubo( appears...
*/
static void *local_fork_object=NULL;
static int fork_stderr(int n, void *data)
{
    char *line;
    if(n)
	return TRUE;		/* this would mean binary data */
    line = (char *)data;
    print_diagnostics("xfce/error", line, NULL);
    return TRUE;
}

static int fork_stdout(int n, void *data)
{
    char *line;
    if(n)
	return TRUE;		/* this would mean binary data */
    line = (char *)data;
    print_diagnostics(NULL, line, NULL);
    /*print_diagnostics("xfce/info", line, NULL);*/
    return TRUE;
}

static void fork_done(pid_t pid){
  local_fork_object = NULL;    
}

static
int fork_wait(gboolean animate){
	
  while (local_fork_object){
     if (animate){
        GtkWidget *progress;
        progress = WIDGET("progressbar1");
	gtk_progress_bar_pulse((GtkProgressBar *) progress);
     } 
     while (gtk_events_pending()) gtk_main_iteration();
     usleep(5000);
  }
  return TRUE;
}

static
void rm_directory (gchar *fullpath){
	char *argument[4];
	argument[0]="rm";
	argument[1]="-rfv";
	argument[2]=fullpath;
	argument[3]=(char *)0;
		    
	print_diagnostics(NULL,_("Deleting...")," ",fullpath,"\n",NULL);
	local_fork_object = Tubo (fork_function, 
	    (void *)argument, 
	    fork_done, 
	    NULL, 
	    fork_stdout, fork_stderr,0,FALSE);
	 set_load_wait();
	 fork_wait(FALSE);
	 print_diagnostics(NULL,_("Deleted")," ",fullpath,"\n",NULL);
	 unset_load_wait();
}

static void delete_trash(DBHashTable * dbh)
{
    struct stat st;
    gchar *fullpath = (gchar *)DBH_DATA(dbh);
    if(lstat(fullpath, &st) < 0)
	return;
    if(S_ISDIR(st.st_mode)) rm_directory(fullpath);
    else g_warning("non-directory found in trashbin: %s",fullpath);
    return;
}

static 
DBHashTable *open_trash_dbh(gboolean create){
    DBHashTable *trash_dbh=NULL;

    if (!g_file_test(trash_path(),G_FILE_TEST_EXISTS) && !create){
	/*print_diagnostics("xfce/info",strerror(ENOENT)," :",trash_path(),"\n",NULL);*/
	/* only with verbose */
	print_diagnostics(NULL,strerror(ENOENT)," :",trash_path(),"\n",NULL);
    }
    else if ((trash_dbh = DBH_open((char *)trash_path())) == NULL ){
	if (create) {
	    trash_dbh = DBH_create((char *)trash_path(), 11);
	} 
	if (!trash_dbh) {
	    print_diagnostics("xfce/warning",strerror(EIO),"\n",NULL);
	}	    
    }   
    return  trash_dbh;
}

G_MODULE_EXPORT
int 
delete_all_trash(GtkTreeView *treeview)
{
    GtkTreeModel *treemodel = gtk_tree_view_get_model(treeview);
    GtkTreeIter iter;
    tree_entry_t *en;

    if ((trashbin=open_trash_dbh(FALSE))==NULL) return -1;
    if(!set_load_wait()) {
	D(g_warning("!set_load_wait()");)
	print_diagnostics("xfce/error",strerror(ECHILD),"\n",NULL);
	return -1;
    }
    cursor_wait();

    DBH_foreach_sweep(trashbin, delete_trash);
    DBH_close(trashbin);

    unlink(trash_path());

    if(!get_trash_root(treeview, &iter, &en))	return 1;

    if(IS_LOADED(en->type))
    {
	GtkTreePath *treepath;
	prune_row(treemodel, &iter,NULL,en);
	insert_dummy_row(treemodel, &iter,NULL,en,NULL,NULL);
	if(en->tag){
	    g_free(en->tag);
	    en->tag=NULL;
	}
	treepath = gtk_tree_model_get_path(treemodel, &iter);
	gtk_tree_view_collapse_row(treeview, treepath);
	gtk_tree_path_free(treepath);
	set_icon(treemodel, &iter);
    }
/* zap gnome and kde trash too */
    {
	gchar *gnome=g_build_filename(GNOME_TRASH,NULL);
	gchar *kde=g_build_filename(KDE_TRASH,NULL);
	if (kde && g_file_test(kde,G_FILE_TEST_IS_DIR)) rm_directory(kde);
	if (gnome && g_file_test(gnome,G_FILE_TEST_IS_DIR)) rm_directory(gnome);
	g_free(kde);
	g_free(gnome);
    }

    
    cursor_reset();
    print_status("xfce/info", _("Trash has been deleted"), NULL);
    unset_load_wait();
    local_monitor(TRUE);
    return 0;

}


static void count_check(DBHashTable * dbh)
{
    char *p;
    char *fullpath = (char *)DBH_DATA(dbh);

    if (!fullpath) return;
    p = strrchr(fullpath, '/');
    /*printf("DBG:%s -> %s\n",fullpath,p); */
    if(!p || strlen(p) <= 1)
	return;
    p++;
    if (!g_file_test(fullpath,G_FILE_TEST_EXISTS)){
	return;
    }
    trashcount++;
    return;
}

G_MODULE_EXPORT
int count_trash(void)
{
    trashcount = 0;		/* for count step */
    if ((trashbin=open_trash_dbh(FALSE))==NULL) return -1;
    DBH_foreach_sweep(trashbin, count_check);
    DBH_close(trashbin);
    return (trashcount);
}



static void add_bin(DBHashTable * dbh)
{
    gchar *basename;
    gchar *fullpath = (char *)DBH_DATA(dbh);
    if (!fullpath || !strlen(fullpath)) return;
    basename = g_path_get_basename(fullpath);

#ifdef DEBUG
    printf("DBG:%s -> %s\n",fullpath,basename); 
#endif

    if(!basename || strlen(basename) <= 1) {
	g_free(basename);
	return;
    }
    if (!g_file_test(fullpath,G_FILE_TEST_EXISTS)) {
	return;
    }
    if(!just_count)
    {
	trash_xfdir.gl[trash_xfdir.pathc].en = stat_entry(fullpath, target_type);
	if(!trash_xfdir.gl[trash_xfdir.pathc].en)
	{
	    printf("xffm: aarrgg! could not stat %s!!\n", fullpath);
	    return;
	}
	if (strstr(fullpath,"..Wastebasket")){
	    gchar *dirname=g_path_get_dirname(fullpath);
	    g_free(basename);
	    basename=g_path_get_basename(dirname);
	    g_free(dirname);
	}
       	
	trash_xfdir.gl[trash_xfdir.pathc].pathv = g_strconcat("[",basename,"]",NULL);
	trash_xfdir.pathc++;
    }
    else {
	trashcount++;
    }
    g_free(basename);
    return;
}

G_MODULE_EXPORT
int open_trash(GtkTreeView * treeview, GtkTreeIter * iter, GtkTreePath * treepath, gpointer user_data)
{
    GtkTreeIter child_iter;
    tree_entry_t *child_en;    
    tree_entry_t *en;
    gboolean xfce_trash;
    gchar *gnome=NULL,*kde=NULL;
    GtkTreeModel *treemodel = gtk_tree_view_get_model(treeview);
    gtk_tree_model_get(treemodel, iter, ENTRY_COLUMN, &en, -1);
    if (!en) return -1;   
    target_iter = iter;
    target_treeview = treeview;
    target_type = en->type;
    UNSET_ERASED_SPACE(en->type);

    smallcount = 0, countbyte = 0x10;	/* for dummy tag */
    trashcount = 0;		/* for count step */
    trash_xfdir.pathc = 0;	/* for read step */
    if ((trashbin=open_trash_dbh(FALSE))==NULL) {
	print_diagnostics("xfce/info",_("No trash has been collected."),NULL);
	xfce_trash=FALSE;
    }
    else xfce_trash=TRUE;

    cursor_wait();
    /* other trash */
    gnome=g_build_filename(GNOME_TRASH,NULL);
    kde=g_build_filename(KDE_TRASH,NULL);
	
    if (g_file_test(gnome,G_FILE_TEST_IS_DIR)){
	    child_en=stat_entry(gnome, target_type);
	    add_row (treemodel,iter, NULL, &child_iter, child_en, "[GNOME]");
	    erase_dummy_row(treemodel, iter, NULL);
    } else { g_free(gnome); gnome=NULL; }
    
    if (g_file_test(kde,G_FILE_TEST_IS_DIR)){
	    child_en=stat_entry(kde, target_type);
	    add_row (treemodel,iter, NULL, &child_iter, child_en, "[KDE]");
	   erase_dummy_row(treemodel, iter, NULL);
    } else { g_free(kde); kde=NULL; }
    
    if (xfce_trash) {
	    erase_dummy_row(treemodel, iter, NULL);
    }
    else if (!kde && !gnome) {
	reset_dummy_row(treemodel, iter,NULL,en,NULL,NULL);
    }
    if (kde || gnome) {
	    g_free(gnome); gnome=NULL;
	    g_free(kde); kde=NULL;
    } 
    
    if (!trashbin) {
	SET_LOADED(en->type);
	cursor_reset();
	return 0;
    }
    
    /*child_en = mk_entry(en->type);
    add_row (treemodel,iter, NULL, &child_iter, child_en, "[Xfce]");*/
    
    just_count = TRUE;
    DBH_foreach_sweep(trashbin, add_bin);
#ifdef DEBUG
    /*printf("DBG:pathc=%d\n",trash_xfdir.pathc); */
#endif

    if(trashcount)
    {
	trash_xfdir.gl = (dir_t *) malloc(trashcount * sizeof(dir_t));
	if(!trash_xfdir.gl)
	{
	    if (trashbin) DBH_close(trashbin);
	    g_warning("malloc(trashcount * sizeof(dir_t)) == NULL");
	    return -1;
	}
	just_count = FALSE;
	DBH_foreach_sweep(trashbin, add_bin);
        D(printf("DBG:pathc=%d\n",trash_xfdir.pathc); )
	D(printf("DBG:count=%d\n",trashcount); )
	if(trashcount != trash_xfdir.pathc)
	{
	    SET_ERASED_SPACE(en->type);
	    D(printf ("DBG:listed=%d, counted=%d\n", trash_xfdir.pathc, trashcount); )
	}


	add_contents_row(treemodel, iter, &trash_xfdir);
	xfdirfree(&trash_xfdir);
    } 
    /*else { remove_row(treemodel,&child_iter, NULL,child_en);}*/
    
    SET_LOADED(en->type);
    /*if (trashcount) erase_dummy_row(treemodel, iter, NULL);*/
    /*else reset_dummy_row(treemodel, iter,NULL,en,NULL,NULL);*/
    
    if(DBH_ERASED_SPACE(trashbin))
    {
	D(printf("DBG: erased space=%d\n",DBH_ERASED_SPACE(trashbin)); )
	SET_ERASED_SPACE(en->type);
    }
    /*(else UNSET_ERASED_SPACE(en->type);*/
    DBH_close(trashbin);
    if (en->tag){
	    g_free(en->tag);
	    en->tag=NULL;
    }
    if(tree_details->stop)
    {
	tree_details->stop = FALSE;
	en->tag = g_strdup_printf("%s : %s", FILENAME(en), _("load aborted."));
	cursor_reset();
	return -2;

    }
    else
    {
	hide_stop();
	en->tag = g_strdup_printf("%s : %d %s", _("Trash"), trash_xfdir.pathc, _("items"));
    }
    cursor_reset();
    return 0;
}



static int read_trash_locations(char *path)
{
    DIR *directory;
    int count = 0;
    struct dirent *d;
    regex_t shortpreg;
    gchar *newpath=NULL;
    gboolean shortpreg_compiled;

    if(tree_details->stop) return 0;
    set_progress(-1, 0);

    directory = opendir(path);
    if(!directory)
	return 0;
    
    if (!regcomp(&shortpreg, "^\\.\\.Wastebasket$", REG_EXTENDED | REG_ICASE | REG_NOSUB)) 
	shortpreg_compiled=TRUE;
    else 
	shortpreg_compiled=FALSE;

    
    while((d = readdir(directory)) != NULL)
    {
	if(tree_details->stop)
	{
	    closedir(directory);
	    return 0;
	}
	if(strcmp(d->d_name, ".") == 0 || strcmp(d->d_name, "..") == 0)
	    continue;
	set_progress(-1, 0);

	newpath=g_build_filename(path, d->d_name, NULL);
	if(shortpreg_compiled && !regexec(&shortpreg, d->d_name, 0, NULL, 0) )
	{
	    GString *gs;
#ifdef DEBUG
	    printf("DBG:read(1) trash from %s\n",newpath); 
#endif
	    gs = g_string_new(newpath);
	    sprintf((char *)DBH_KEY(trashbin), "%10u", g_string_hash(gs));
	    /* overwrite record if exists might be faster by probability.*/
	    if(!DBH_load(trashbin))
	    {
	      memcpy(DBH_DATA(trashbin), (void *)newpath, strlen(newpath) + 1);
	      DBH_set_recordsize(trashbin, strlen(newpath) + 1);
	      DBH_update(trashbin);
	    }
	    g_string_free(gs, TRUE);
	    count++; 
	    /* message at nonverbose level: */
	    print_diagnostics("xfce/waste_basket_full",newpath,"\n",NULL);
	} else {
	    struct stat st;
	    lstat(newpath,&st);
	    if (S_ISDIR(st.st_mode)){
		/* recurse */
		count += read_trash_locations(newpath);
	    }
	}
	g_free(newpath);
    }
    closedir(directory);
    if (shortpreg_compiled)regfree(&shortpreg);
    return (count);
}

static 
void 
purge_trash(DBHashTable * dbh)
{
    char *p;
    char *fullpath = (char *)DBH_DATA(dbh);
    struct stat st;

    if(!newtrashbin)
	assert_not_reached();
    p = strrchr(fullpath, '/');
    if(p)
    {
	p++;
	if(lstat(fullpath, &st) < 0)
	{
	    return;
	}
    }
    /* copy the key from the old record to the new record */
    memcpy(DBH_KEY(newtrashbin), DBH_KEY(trashbin), DBH_KEYLENGTH(trashbin));
    /* copy the data from the old record to the new record */
    memcpy(newtrashbin->data, trashbin->data, DBH_RECORD_SIZE(trashbin));
    /* I always forget this instruction, and it is the most important
     * (since DBH records have a variable size): */
    DBH_set_recordsize(newtrashbin, DBH_RECORD_SIZE(trashbin));
    /* write the record to the new file; */
    if(!DBH_update(newtrashbin))
	assert_not_reached();
    /*printf("DBG:clean record: %s\n",fullpath); */
    return;
}

/* returns -1 on failure, otherwise count of elements gathered */
G_MODULE_EXPORT
int collect_trash(GtkTreeView * treeview, char *path)
{
    int count=0;
    gchar *xdg_dir=xfce_resource_save_location (XFCE_RESOURCE_CACHE,"/",TRUE); 
    gchar *work_directory=g_build_filename(xdg_dir,"xfce4","xffm","histories",NULL);

    g_free(xdg_dir);

    /* so the DBH_regen tmp file is generated here */
    if (chdir(work_directory)<0){
	print_diagnostics("xfce/error",strerror(errno),"\n",work_directory,"\n",NULL);
	g_free(work_directory);
	return -1;
    }
    g_free(work_directory);
    

    if ((trashbin=open_trash_dbh(TRUE))==NULL) return -1;
 
    {
      gchar *m=g_strdup_printf(_("Collecting trash from %s"),path);
      print_diagnostics("xfce/info", m, "...", "\n",NULL);
      g_free(m);
    }
    
    cursor_wait();
    /* step one: add collected entries: */
    show_stop();
    
#if 0
    if (strrchr(path,'/')&&strcmp(strrchr(path,'/')+1,"..Wastebasket")==0){
#ifdef DEBUG
		printf("DBG:read(3) trash from %s\n",path); 
#endif
	    count = read_trash(treeview, path);
    }
#endif
    
    count += read_trash_locations(path);
    if(tree_details->stop)
    {
	tree_details->stop = FALSE;
	print_status("xfce/info",
		strerror(ETIMEDOUT),
		NULL);
	DBH_close(trashbin);
    }
    else
    {
	/* step two:   make dbh logical and physical structures the same 
	 *             eliminating stale entries while you're at it.*/
	gchar *s;
	gchar *fname;
	hide_stop();
	s = g_strdup_printf(_("%d trash items collected."), count);
	print_status("xfce/info", s, NULL);
	g_free(s);s=NULL;
	fname = g_strdup("trashbin.XXXXXX");
	close(mkstemp(fname));
	newtrashbin = DBH_create(fname, DBH_KEYLENGTH(trashbin));
	DBH_foreach_sweep(trashbin, purge_trash);
	DBH_close(trashbin);
	DBH_close(newtrashbin);
	rename(fname, trash_path());
	g_free(fname);

    }
    chdir(GETWD);
    cursor_reset();
    return count;
}

/*FIXME (maybe done already) remove.c should be fixed so that this function is only called
once for each directory, not for each file in directory...
*/

/* returns -1 on failure, otherwise count of elements gathered */
G_MODULE_EXPORT
int add2trash(char *filepath)
{
    GString *gs;
    gchar *path=g_path_get_dirname(filepath);
    if (!path || !strlen(path)) return -1;
#ifdef DEBUG
   /*printf ("DBG: adding to trash -> %s\n",path);*/
#endif

    if ((trashbin=open_trash_dbh(TRUE))==NULL) return -1;
    {
      gchar *m=g_strdup_printf(_("Adding to trash: %s"),filepath);	
      print_diagnostics("xfce/waste_basket_full", m, "\n", NULL);
      g_free(m);
    }

    gs = g_string_new(path);
    sprintf((char *)DBH_KEY(trashbin), "%10u", g_string_hash(gs));
    if(!DBH_load(trashbin))
    {
	memcpy(DBH_DATA(trashbin), (void *)path, strlen(path) + 1);
	DBH_set_recordsize(trashbin, strlen(path) + 1);
	DBH_update(trashbin);
    }
    g_string_free(gs, TRUE);

    DBH_close(trashbin);
    return 1;
}

/* FIXME XXX: put in a function like this one to do finds over all
 wastebaskets in the trashbin
*/ 
G_MODULE_EXPORT
int bookmarks_collect_trash(GtkTreeView * treeview)
{
    GList *tmp, *list;
    int r, t = 0;
    gchar *s;
    list = BOOK_get_bookmark_dirlist();
    for(tmp = list; tmp; tmp = tmp->next)
    {
	r = collect_trash(treeview, (char *)tmp->data);
	g_free((char *)tmp->data);
	tmp->data=NULL;
	if(r > 0)
	    t += r;
    }
    g_list_free(list);
    s=g_strdup_printf(_("%d trash items collected."), t);
    print_status("xfce/trash_closed", s, NULL);
    return t;
}

/****************** callbacks *****************************/
G_MODULE_EXPORT
void collect_trash_callback(GtkMenuItem * menuitem, gpointer user_data)
{
    gint tree_id = get_active_tree_id();
    GtkTreeView *treeview = tree_details->treestuff[tree_id].treeview;
    GtkTreeIter iter;
    tree_entry_t *en;
    int caso = 0;

    if(!get_selectpath_iter(&iter, &en))
	return;
    if(IS_DIR(en->type))
	caso = 1;
    else if(IS_BOOKMARK_TYPE(en->type) && IS_ROOT_TYPE(en->type))
	caso = 2;
    else
    {
	print_status("xfce/error", strerror(EINVAL), NULL);
	return;
    }

    if(caso == 1)
	collect_trash(treeview, en->path);
    else if(caso == 2)
	bookmarks_collect_trash(treeview);

}
G_MODULE_EXPORT
void uncollect_trash_callback(GtkMenuItem * menuitem, gpointer user_data)
{
    gint tree_id = get_active_tree_id();
    GtkTreeView *treeview = tree_details->treestuff[tree_id].treeview;
    GtkTreeModel *treemodel = tree_details->treestuff[tree_id].treemodel;
    tree_entry_t *en;
    GtkTreeIter iter;
    GtkTreePath *treepath;


    if(unlink(trash_path()))
    {
	print_diagnostics("xfce/error", strerror(errno), ":\n",trash_path(),"\n", NULL);
	return;
    }
    else
    {
	print_status("xfce/info", _("Trash has been uncollected"), NULL);
    }


    if(!gtk_tree_model_get_iter_first(treemodel, &iter)) return;
    gtk_tree_model_get(treemodel, &iter, ENTRY_COLUMN, &en, -1);
    while(en && !IS_TRASH_TYPE(en->type))
    {
	if(!gtk_tree_model_iter_next(treemodel, &iter)) return;
	gtk_tree_model_get(treemodel, &iter, ENTRY_COLUMN, &en, -1);
    }
    if(IS_LOADED(en->type))
    {
	prune_row(treemodel, &iter,NULL,en);
	insert_dummy_row(treemodel, &iter,NULL,en,NULL,NULL);
	if(en->tag){
	    g_free(en->tag);
	    en->tag=NULL;
	}
	treepath = gtk_tree_model_get_path(treemodel, &iter);
	gtk_tree_view_collapse_row(treeview, treepath);
	gtk_tree_path_free(treepath);
	set_icon(treemodel, &iter);
    }
    print_status("xfce/info", _("Trash has been uncollected"), NULL);

}
G_MODULE_EXPORT
void uncollect_from_trash_callback(GtkMenuItem * menuitem, gpointer user_data)
{
    GtkTreeIter iter, parent;
    tree_entry_t *en;
    gint tree_id = get_active_tree_id();
    GtkTreeView *treeview = tree_details->treestuff[tree_id].treeview;
    GtkTreeModel *treemodel = tree_details->treestuff[tree_id].treemodel;
    en = get_selected_entry(&iter);
    if(!en)	assert_not_reached();
     
    if ((trashbin=open_trash_dbh(FALSE))!=NULL) 
    {
	GString *gs;
	gs = g_string_new(en->path);
	sprintf((char *)DBH_KEY(trashbin), "%10u", g_string_hash(gs));
	g_string_free(gs, TRUE);
	DBH_erase(trashbin);
	DBH_close(trashbin);
    }
    
    remove_row(treemodel, &iter,NULL,en);
    
    /* if the removed element was the last, insert a dummy.
     * insert_dummy_row will only insert dummy on leaf nodes */
    get_trash_root(treeview, &parent, &en);
    insert_dummy_row(treemodel, &parent,NULL,en,"xfce/info",_("No trash has been collected."));
    SET_ERASED_SPACE(en->type);

}


G_MODULE_EXPORT
xftrash_functions *module_init(void){
    xftrash_fun = g_new0 (xftrash_functions,1);
    if (!xftrash_fun) g_assert_not_reached();
    xftrash_fun->open_trash=open_trash;
    xftrash_fun->collect_trash=collect_trash;
    xftrash_fun->add2trash=add2trash;
    xftrash_fun->count_trash=count_trash;
    xftrash_fun->collect_trash_callback=collect_trash_callback;
    xftrash_fun->uncollect_trash_callback=uncollect_trash_callback;
    xftrash_fun->uncollect_from_trash_callback=uncollect_from_trash_callback;
    xftrash_fun->delete_all_trash=delete_all_trash;
    return xftrash_fun;
}

