/*
 * Copyright (C) 2001,  Espen Skoglund <esk@ira.uka.de>
 * Copyright (C) 2001,  Haavard Kvaalen <havardk@xmms.org>
 * Copyright (C) 2004,  Matti Hmlinen <ccr@tnsp.org>
 *                
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 *                
 */

#if defined(HAVE_CONFIG_H)
#include "config.h"
#endif

#include <glib.h>
#include <stdio.h>
#include <string.h>

#include "titlestring.h"
#include "../xmms/i18n.h"


#define	VOUTSIZE	(256)
#define	VMAXLEVELS	(64)


#define VPUTCH(x) {				\
if (*outpos >= VOUTSIZE)			\
	{					\
	*outsize += *outpos;			\
	if (!*result)				\
		*result = g_malloc0(*outsize);	\
		else				\
		*result = g_realloc(*result, *outsize);	\
	if (!*result) return;			\
	outbuf[*outpos] = 0;			\
	strcat(*result, outbuf);		\
	*outpos = 0;				\
	}					\
outbuf[(*outpos)++] = (x);			\
}

#define VPUTCH2(x) {				\
if (outpos >= VOUTSIZE)				\
	{					\
	outsize += outpos;			\
	if (!result)				\
		result = g_malloc0(outsize);	\
		else				\
		result = g_realloc(result, outsize);	\
	if (!result) return NULL;		\
	outbuf[outpos] = 0;			\
	strcat(result, outbuf);			\
	outpos = 0;				\
	}					\
outbuf[outpos++] = (x);				\
}

void xmms_vputstr(gchar *outbuf, gchar **result, guint *outsize, guint *outpos, gchar *pstr,
	gboolean f_left, gboolean f_space, gboolean f_zero,
	gboolean f_cond, gint v_width, gint v_precision, gboolean *f_output)
{
	gint n, cnt;
	gchar c;
	
	if (!pstr) return;
	*f_output = TRUE;

	/* Calculate printed size */
	n = 0;
	if (v_width > 0)
	{
 		n = strlen(pstr);
 		if ((v_precision >= 0) && (v_precision < n))
 			n = v_precision;

		/* Do left-padding */
		cnt = (v_width - n);
		if (!f_left && (cnt > 0))
			while (cnt-- > 0)
			{
				VPUTCH(f_zero ? '0' : ' ');
			}
	}

	/* Insert the string */
	if (v_precision >= 0)
	{
		while ((v_precision-- > 0) &&
			((c = *pstr++) != 0))
			VPUTCH(c);
	}
	else
	{
		while ((c = *pstr++) != 0)
			VPUTCH(c);
	}
	
	/* Do right-padding */
	cnt = (v_width - n);
	if (f_left && (cnt > 0))
		while (cnt-- > 0)
		{
			VPUTCH(' ');
		}
}


void xmms_vputnum(gchar *outbuf, gchar **result, guint *outsize, guint *outpos, gint ival,
	gboolean f_left, gboolean f_space, gboolean f_zero,
	gboolean f_cond, gint v_width, gint v_precision, gboolean *f_output)
{
	gint i, n, ndigits, cnt;
	gchar pstr[32], vdigits[] = "0123456789";
	
	if (ival == 0) return;
	*f_output = TRUE;
	
	/* Create string */
	ndigits = 0;
	do
	{
		pstr[ndigits++] = vdigits[ival % 10];
		ival /= 10;
	} while ((ival > 0) && (ndigits < sizeof(pstr)));

	n = (ndigits > v_precision) ? ndigits : v_precision;

	/* Do left-padding */
	if (!f_left && (v_width < n))
	{
		if (f_zero)
			n = v_width;
		else
			for (i = (v_width - n); i-- > 0;)
				VPUTCH(' ');
	}

	/* Do zero-padding */
	for (i = (n - ndigits); i-- > 0;)
		VPUTCH('0');
	
	/* Add the value */
	while (ndigits > 0)
		VPUTCH(pstr[--ndigits]);

	/* Do right-padding */
	cnt = (v_width - n);
	if (f_left && (cnt > 0))
		while (cnt-- > 0)
		{
			VPUTCH(' ');
		}
}


#define VISDIGIT(c) ((c >= '0') && (c <= '9'))

#define VCHECK(input, field) \
	(((char *) &input->field - (char *) input) < input->__size)

#define VS(field) (VCHECK(input, field) ? input->field : NULL)
#define VI(field) (VCHECK(input, field) ? input->field : 0)

#define VCHKS(field)			\
	if (!VS(field) && f_check)	\
		f_ok = FALSE; else	\
	if (VS(field) && !f_check)	\
		f_ok = FALSE;

#define VCHKN(field)			\
	if (!VI(field) && f_check)	\
		f_ok = FALSE; else	\
	if (VI(field) && !f_check)	\
		f_ok = FALSE;

#define VPUTSTR(pval) xmms_vputstr(outbuf, &result, &outsize,	\
	&outpos, pval, f_left, f_space, f_zero, f_cond,		\
	v_width, v_precision, &f_output);

#define VPUTNUM(pval) xmms_vputnum(outbuf, &result, &outsize,	\
	&outpos, pval, f_left, f_space, f_zero, f_cond,		\
	v_width, v_precision, &f_output);


gchar *xmms_get_titlestring(gchar *fmt, TitleInput *input)
{
	gchar outbuf[VOUTSIZE + 1], *result;
	gboolean f_left, f_space, f_zero, f_parse, f_cond, f_check, f_ok, f_output;
	guint outpos, outsize;
	gint v_width, v_precision, c_level, c_set[VMAXLEVELS];

	if (!fmt)
		return NULL;

	f_output = FALSE;
	result = NULL;
	c_level = outsize = outpos = 0;
	f_ok = c_set[c_level] = TRUE;

	while (*fmt)
	switch (*fmt)
	{
		case ')':
			/* Condition block end */
			if (c_level > 0)
				c_level--;
			fmt++;
			break;

		case '(':
			/* Normal block start */
			if (c_level < VMAXLEVELS)
				c_set[c_level + 1] = c_set[c_level];
			c_level++;
			fmt++;
			break;

		case '%':
			/* Check if we are skipping */
			fmt++;
			if (!c_set[c_level]) break;
			
			/* Parse flags */
			f_left = f_space = f_zero = f_parse = f_cond = f_check = FALSE;
			f_parse = TRUE;

			while (f_parse)
			switch (*fmt)
			{
				case '?':
					f_cond = TRUE;
					f_check = TRUE;
					fmt++;
					break;
						
				case '!':
					f_check = FALSE;
					f_cond = TRUE;
					fmt++;
					break;
					
				case '-':
					f_left = TRUE;
					fmt++;
					break;
					
				case ' ':
					f_space = TRUE;
					fmt++;
					break;
					
				case '0':
					f_zero = TRUE;
					fmt++;
					break;
						
				default:
					f_parse = FALSE;
					break;
			}
			
			/* Parse field width */
			if (VISDIGIT(*fmt))
			{
				v_width = 0;
				while (*fmt && VISDIGIT(*fmt))
				{
					v_width *= 10;
					v_width += (gint) (*fmt - '0');
					fmt++;
				}
			} else
				v_width = -1;
			
			
			/* Parse precision */
			if (*fmt == '.')
			{
				fmt++;
				if (VISDIGIT(*fmt))
				{
					v_precision = 0;
					while (*fmt && VISDIGIT(*fmt))
					{
						v_precision *= 10;
						v_precision += (gint) (*fmt - '0');
						fmt++;
					}
				} else
					v_precision = -1;
				
			} else
				v_precision = -1;
			
			/* Check if we need to check fields */
			if (f_cond)
			{
				f_ok = f_parse = TRUE;
				while (f_parse)
				switch (*fmt++) {
					case 'a':
						VCHKS(album_name);
						break;
				
					case 'c':
						VCHKS(comment);
						break;
				
					case 'd':
						VCHKS(date);
						break;
				
					case 'e':
						VCHKS(file_ext);
						break;
					
					case 'f':
						VCHKS(file_name);
						break;
				
					case 'F':
						VCHKS(file_path);
						break;
				
					case 'g':
						VCHKS(genre);
						break;
					
					case 'n':
						VCHKN(track_number);
						break;
				
					case 'p':
						VCHKS(performer);
						break;
					
					case 't':
						VCHKS(track_name);
						break;
					
					case 'y':
						VCHKN(year);
						break;

					case '(':
						/* Condition block start */
						if (c_level < VMAXLEVELS)
							c_set[++c_level] = f_ok;

					default:
						f_parse = FALSE;
						break;
				}
			}

			
			/* Parse format conversion */
			if (!f_cond)
			{
				switch (*fmt++)
				{
					case 'a':
						VPUTSTR(VS(album_name));
						break;
					
					case 'c':
						VPUTSTR(VS(comment));
						break;
				
					case 'd':
						VPUTSTR(VS(date));
						break;
				
					case 'e':
						VPUTSTR(VS(file_ext));
						break;
					
					case 'f':
						VPUTSTR(VS(file_name));
						break;
				
					case 'F':
						VPUTSTR(VS(file_path));
						break;
				
					case 'g':
						VPUTSTR(VS(genre));
						break;
					
					case 'n':
						VPUTNUM(VI(track_number));
						break;
				
					case 'p':
						VPUTSTR(VS(performer));
						break;
					
					case 't':
						VPUTSTR(VS(track_name));
						break;
					
					case 'y':
						VPUTNUM(VI(year));
						break;
				
					case '%':
						VPUTCH2('%');
						break;
				
					default:
						VPUTCH2('%');
						VPUTCH2(*fmt);
						break;
				}
			}
			break;

		default:
			if (c_set[c_level])
			{
				/* Normal character */
				VPUTCH2(*fmt++);
			} else
				fmt++;
			break;
	}

	/* Terminate with NUL */
	outbuf[outpos++] = 0;

	/* Check if we actually output anything */
	if (!f_output)
	{
		g_free(result);
		return NULL;
	}

	/* Return the result */
	if (result)
	{
		/* We have data in dynamic buffer, combine with current */
		result = g_realloc(result, outsize + strlen(outbuf) + 1);
		if (!result)
			return NULL;
		
		strncat(result, outbuf, outpos);
		return result;
	}
	else
		return g_strdup(outbuf);
}



static struct tagdescr {
	char tag;
	char *description;
} descriptions[] = {
	{'p', N_("Performer/Artist")},
	{'a', N_("Album")},
	{'g', N_("Genre")},
	{'f', N_("File name")},
	{'F', N_("File path")},
	{'e', N_("File extension")},
	{'t', N_("Track name")},
	{'n', N_("Track number")},
	{'d', N_("Date")},
	{'y', N_("Year")},
	{'c', N_("Comment")},
};

GtkWidget* xmms_titlestring_descriptions(char* tags, int columns)
{
	GtkWidget *table, *label;
	gchar tagstr[5];
	gint num = strlen(tags);
	int r, c, i;

	g_return_val_if_fail(tags != NULL, NULL);
	g_return_val_if_fail(columns <= num, NULL);

	table = gtk_table_new((num + columns - 1) / columns, columns * 2, FALSE);
	gtk_table_set_row_spacings(GTK_TABLE(table), 2);
	gtk_table_set_col_spacings(GTK_TABLE(table), 5);
	
	for (c = 0; c < columns; c++)
	{
		for (r = 0; r < (num + columns - 1 - c) / columns; r++)
		{
			sprintf(tagstr, "%%%c:", *tags);
			label = gtk_label_new(tagstr);
			gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
			gtk_table_attach(GTK_TABLE(table), label, 2 * c, 2 * c + 1, r, r + 1,
					 GTK_FILL, GTK_FILL, 0, 0);
			gtk_widget_show(label);
				
			for (i = 0; i <= sizeof(descriptions) / sizeof(struct tagdescr); i++)
			{
				if (i == sizeof(descriptions) / sizeof(struct tagdescr))
					g_warning("xmms_titlestring_descriptions(): Invalid tag: %c", *tags);
				else if (*tags == descriptions[i].tag)
				{
					label = gtk_label_new(gettext(descriptions[i].description));
					gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
					gtk_table_attach(GTK_TABLE(table), label, 2 * c + 1, 2 * c + 2, r, r + 1,
							 GTK_EXPAND | GTK_FILL,  GTK_EXPAND | GTK_FILL, 0 ,0);
					gtk_widget_show(label);
					break;
				}
			}
			tags++;
		}
			
	}

	return table;
}

