/*
    journal.c -- reiserfs journal code
    Copyright (C) 2001, 2002 Yury Umanets <torque@ukrpost.net>, see COPYING for 
    licensing and copyright details.

    Some parts of this code are from the original reiserfs code, as found in 
    reiserfsprogs and the linux kernel.
    Copyright (C) 1996-2002 Hans Reiser, see COPYING.NAMESYS for licensing and 
    copyright details.
*/

#ifdef HAVE_CONFIG_H
#	include <config.h>
#endif

#include <string.h>
#include <reiserfs/reiserfs.h>
#include <reiserfs/debug.h>

#include <dal/dal.h>

#define N_(String) (String)
#if ENABLE_NLS
#	include <libintl.h>
#	define _(String) dgettext (PACKAGE, String)
#else
#	define _(String) (String)
#endif

#define JOURNAL_DESC_SIGN	"ReIsErLB"
#define JOURNAL_DESC_COMM	1
#define JOURNAL_DESC_NEXT 	2

uint32_t reiserfs_jr_max_trans(blk_t max_trans, blk_t len, 
	size_t blocksize) 
{
    uint32_t ratio = 1;

    if (blocksize < 4096)
		ratio = 4096 / blocksize;
	
    if (!max_trans)
		max_trans = JOURNAL_MAX_TRANS / ratio;
    
    if (len / max_trans < JOURNAL_MIN_RATIO)
		max_trans = len / JOURNAL_MIN_RATIO;
    
    if (max_trans > JOURNAL_MAX_TRANS / ratio)
		max_trans = JOURNAL_MAX_TRANS / ratio;
    
    if (max_trans < JOURNAL_MIN_TRANS / ratio)
		max_trans = JOURNAL_MIN_TRANS / ratio;
    
    return max_trans;
}

blk_t reiserfs_jr_max_len(dal_t *dal, blk_t start, int relocated) {
    if (relocated)
		return dal_len(dal) - start - 1;
    
    return (dal_block_size(dal) * 8) - start - 1;
}

int reiserfs_jr_params_check(dal_t *dal, blk_t start, blk_t len, int relocated) {
    blk_t max_len;
    
    ASSERT(dal != NULL, return 0);

	if (!relocated) {
		blk_t super_blk = (DEFAULT_SUPER_OFFSET / dal_block_size(dal));
		
		if (start && start != super_blk + 2) {
			libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
				_("Invalid journal start (%lu) for journal on host device."), start);
			return 0;	
		}
	}
	
   	max_len = reiserfs_jr_max_len(dal, start, relocated);
    
    if (len > max_len) {
    	libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
    	    _("Journal size is too big (%lu). It must be smaller or equal "
			"%lu blocks for block size %d."), len, max_len, dal_block_size(dal));
    	return 0;
    }
    
    if (len && len < JOURNAL_MIN_SIZE) {
        libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
    	    _("Journal size (%lu) is smaller minimal recomended (%lu)."), 
			len, JOURNAL_MIN_SIZE);
    	return 0;
    }
    
    return 1;
}    

void reiserfs_jr_params_update(reiserfs_journal_params_t *params, blk_t start, 
    blk_t len, blk_t max_trans, uint32_t dev, size_t blocksize)
{
    
    ASSERT(params != NULL, return);
    
    set_jp_start(params, start);
    set_jp_len(params, len);
    
    set_jp_max_trans_len(params, 
		reiserfs_jr_max_trans(max_trans, len, blocksize));
    
    set_jp_magic(params, reiserfs_tools_random());
    set_jp_max_batch(params, max_trans * JOURNAL_MAX_BATCH/JOURNAL_MAX_TRANS);
    
    set_jp_max_commit_age(params, JOURNAL_MAX_COMMIT_AGE);
    set_jp_max_trans_age(params, JOURNAL_MAX_TRANS_AGE);
    
    set_jp_dev(params, dev);
}

reiserfs_journal_t *reiserfs_jr_open(dal_t *dal, blk_t start, blk_t len, int relocated) {
    struct stat stat;
    reiserfs_journal_t *jr;
    reiserfs_block_t *block;
    reiserfs_journal_params_t *params;
    
    ASSERT(dal != NULL, return NULL);
    
    if (!(block = reiserfs_block_read(dal, start + len)))
		reiserfs_block_reading_failed(start + len, goto error);
    
    /* Checking for journal validness */
    params = &((reiserfs_journal_t *)block->data)->jh_params;
    libreiserfs_exception_fetch_all();
    if (!reiserfs_jr_params_check(dal, get_jp_start(params), 
		get_jp_len(params), relocated))
    {	    
		libreiserfs_exception_leave_all();
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
			_("Invalid journal parameters detected."));
		goto error_free_block;
    }	
    libreiserfs_exception_leave_all();
    
    if (!(jr = (reiserfs_journal_t *)libreiserfs_calloc(sizeof(*jr), 0)))
		goto error_free_block;
	
    memcpy(jr, block->data, sizeof(*jr));
    
    if (!dal_stat(dal, &stat)) {
		libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
			_("Can't stat journal device."));
		goto error_free_jr;    
    }

    set_jp_dev(&jr->jh_params, (uint32_t)stat.st_rdev);
    
    reiserfs_block_free(block);
    
    return jr;

error_free_jr:
    libreiserfs_free(jr);    
error_free_block:
    reiserfs_block_free(block);
error:
    return NULL;    
}

reiserfs_journal_t *reiserfs_jr_create(dal_t *dal, blk_t start, blk_t len,
    blk_t max_trans, int relocated, reiserfs_gauge_t *gauge)
{
	uint32_t dev;
	reiserfs_geom_t geom;
    reiserfs_journal_t *jr;
    reiserfs_block_t *block;
    reiserfs_journal_params_t *params;
    
    ASSERT(dal != NULL, return NULL);
    
    if (!reiserfs_jr_params_check(dal, start, len, relocated))
		return NULL;
    
	if (!reiserfs_geom_init(&geom, dal, start, start + len))
		return NULL;
	
    if (gauge) {
		libreiserfs_gauge_reset(gauge);
		libreiserfs_gauge_set_name(gauge, _("initializing journal"));
    }	
    
	if (!reiserfs_geom_fill(&geom, 0, gauge))
		return NULL;
			
    /* Journal parameters */
    if (!(jr = (reiserfs_journal_t *)libreiserfs_calloc(sizeof(*jr), 0)))
		goto error;
   
	dev = 0;
    if (relocated) {
		struct stat stat;
		
		if (!dal_stat(dal, &stat)) {
			libreiserfs_exception_throw(EXCEPTION_ERROR, EXCEPTION_CANCEL, 
				_("Can't stat journal device."));
			goto error_free_jr;	
		}
		dev = (uint32_t)stat.st_rdev;
    }
	reiserfs_jr_params_update(&jr->jh_params, start, len, max_trans, dev, 
		dal_block_size(dal));
    
    if (!(block = reiserfs_block_alloc_with_copy(dal, start + len, (void *)jr)))
		goto error_free_jr;
	
    /* Writing journal parameters onto device */
	if (!reiserfs_block_write(dal, block)) {
		reiserfs_block_writing_failed(reiserfs_block_location(block), 
			goto error_free_block);
    }
    reiserfs_block_free(block);
    
    return jr;
    
error_free_block:
    reiserfs_block_free(block);    
error_free_jr:
    libreiserfs_free(jr);
error:
    return NULL;    
}

void reiserfs_jr_close(reiserfs_journal_t *jr) {
    ASSERT(jr != NULL, return);
    libreiserfs_free(jr);
}

int reiserfs_jr_sync(dal_t *dal, reiserfs_journal_t *jr) {
    reiserfs_block_t *block;

    ASSERT(jr != NULL, return 0);
    ASSERT(dal != NULL, return 0);
    
    if (!(block = reiserfs_block_alloc_with_copy(dal, get_jp_start(&jr->jh_params) + 
			get_jp_len(&jr->jh_params), (void *)jr)))
		goto error;
    
    if (!reiserfs_block_write(dal, block)) {
		reiserfs_block_writing_failed(reiserfs_block_location(block),
			goto error_free_block);
    }
    
    reiserfs_block_free(block);
    return 1;
    
error_free_block:
    reiserfs_block_free(block);
error:
    return 0;    
}

/* Transaction functions */
static int reiserfs_jr_desc_match_comm(reiserfs_block_t *desc, reiserfs_block_t *comm) {
	return (get_jc_commit_trans_id(comm) == get_jd_desc_trans_id(desc) &&
		get_jc_commit_trans_len(comm) == get_jd_desc_trans_len(desc));
}

static blk_t reiserfs_jr_desc_prop(reiserfs_journal_t *jr, reiserfs_block_t *desc, 
	int prop) 
{
	blk_t offset = reiserfs_block_location(desc) - get_jp_start(&jr->jh_params);
	return get_jp_start(&jr->jh_params) + ((offset + get_jd_desc_trans_len(desc) + prop) 
		% get_jp_len(&jr->jh_params));
}

static blk_t reiserfs_jr_desc_comm(reiserfs_journal_t *jr, reiserfs_block_t *desc) {
	return reiserfs_jr_desc_prop(jr, desc, JOURNAL_DESC_COMM);
}

static blk_t reiserfs_jr_desc_next(reiserfs_journal_t *jr, reiserfs_block_t *desc) {
	return reiserfs_jr_desc_prop(jr, desc, JOURNAL_DESC_NEXT);
}

static int reiserfs_jr_desc_block(reiserfs_block_t *desc) {
    if (!memcmp(desc->data + dal_block_size(desc->dal) - 12, JOURNAL_DESC_SIGN, 8) && 
			LE32_TO_CPU(*((uint32_t *)(desc->data + 4))) > 0)
		return 1;
	
	return 0;
}

/* 
	Checking whether given transaction is valid. Transaction is 
	valid if his description block is valid, commit block is valid 
	and commit block matches description block.
	
	Transaction looks like this:
	desc_block + [ trans_blocks ] + comm_block
*/
static int reiserfs_jr_desc_valid(dal_t *dal, reiserfs_journal_t *jr, 
	reiserfs_block_t *desc)
{
	reiserfs_block_t *comm;
    
	if (!desc || !reiserfs_jr_desc_block(desc))
		return 0;

	if (!(comm = reiserfs_block_read(dal, reiserfs_jr_desc_comm(jr, desc))))
		return 0;
    
    if (!reiserfs_jr_desc_match_comm(desc, comm))
		goto error_free_comm;
	
    reiserfs_block_free(comm);
    return 1;
	
error_free_comm:
	reiserfs_block_free(comm);
error:
	return 0;
}

static void reiserfs_jr_desc_desc2trans(reiserfs_journal_t *jr, reiserfs_block_t *desc, 
	reiserfs_journal_trans_t *trans) 
{
	trans->jt_mount_id = get_jd_desc_mount_id(desc);
	trans->jt_trans_id = get_jd_desc_trans_id(desc);
	trans->jt_desc_blocknr = reiserfs_block_location(desc);
	trans->jt_trans_len = get_jd_desc_trans_len(desc);
	trans->jt_commit_blocknr = reiserfs_jr_desc_comm(jr, desc);
	trans->jt_next_trans_offset = reiserfs_jr_desc_next(jr, desc) - 
		get_jp_len(&jr->jh_params);
}

blk_t reiserfs_jr_boundary_transactions(dal_t *dal, reiserfs_journal_t *jr,
    reiserfs_journal_trans_t *oldest, reiserfs_journal_trans_t *newest, 
	reiserfs_gauge_t *gauge)
{
    reiserfs_block_t *desc;
    blk_t curr_nr, trans_nr;
    uint32_t newest_id, oldest_id;

    oldest_id = 0; newest_id = 0xffffffff;
    
    if (gauge) {
		libreiserfs_gauge_reset(gauge);
		libreiserfs_gauge_set_name(gauge, _("replaying journal"));
    }	
    
    trans_nr = 0;
    for (curr_nr = 0; curr_nr < get_jp_len(&jr->jh_params); curr_nr++) {
		
		if (!(desc = reiserfs_block_read(dal, get_jp_start(&jr->jh_params) + curr_nr)))
				reiserfs_block_reading_failed(get_jp_start(&jr->jh_params) + curr_nr, 
					return 0);
	
		if (gauge) {
			libreiserfs_gauge_set_value(gauge, 
				(unsigned int)((curr_nr * 100) / get_jp_len(&jr->jh_params)) + 1);
		}	

		if (!reiserfs_jr_desc_valid(dal, jr, desc)) {
			reiserfs_block_free(desc);
			continue;
		}

		trans_nr++;

		if (get_jd_desc_trans_id(desc) < oldest_id) {
			oldest_id = get_jd_desc_trans_id(desc);
			reiserfs_jr_desc_desc2trans(jr, desc, oldest);
		}

		if (get_jd_desc_trans_id(desc) > newest_id) {
			newest_id = get_jd_desc_trans_id(desc);
			reiserfs_jr_desc_desc2trans(jr, desc, newest);
		}

		curr_nr += get_jd_desc_trans_len(desc) + 1;
		reiserfs_block_free(desc);
    }
	
    if (gauge)
		libreiserfs_gauge_done(gauge);
    
    return trans_nr;
}

