/*
 *  freetypemodule.c - a FreeType module for Python
 *                     Version 1.0 <20 April 2000>
 *
 *  Copyright (c) Tamito KAJIYAMA <kajiyama@grad.sccs.chukyo-u.ac.jp>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of the
 *  License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "Python.h"

#include "freetype.h"

static PyObject *ErrorObject;

#define onError(message) \
      { PyErr_SetString(ErrorObject, message); return NULL; }

/* Type definitions */

typedef struct {
  PyObject_HEAD
  TT_Engine engine;
  int closed;
} FreeTypeObject;

typedef struct {
  PyObject_HEAD
  TT_Face face;
  TT_Face_Properties properties;
  FreeTypeObject *parent;
  int closed;
} FaceObject;

typedef struct {
  PyObject_HEAD
  TT_Instance instance;
  TT_Instance_Metrics metrics;
  FaceObject *parent;
  int closed;
} InstanceObject;

typedef struct {
  PyObject_HEAD
  TT_CharMap charmap;
} CharMapObject;

typedef struct {
  PyObject_HEAD
  TT_Glyph glyph;
  TT_Glyph_Metrics metrics;
  PyObject *outline;
  InstanceObject *parent;
  int closed;
} GlyphObject;

/* Prototype declarations */

static FreeTypeObject *
newFreeTypeObject();

static FaceObject *
newFaceObject(FreeTypeObject *, char *, int);

static InstanceObject *
newInstanceObject(FaceObject *);

static CharMapObject *
newCharMapObject(FaceObject *, int);

static GlyphObject *
newGlyphObject(InstanceObject *, int, int);

/*----------------------------------------------------------------------
 *  FreeType object
 */

staticforward PyTypeObject FreeType_Type;

#define FreeTypeObject_Check(v) ((v)->ob_type == &FreeType_Type)

static FreeTypeObject *
newFreeTypeObject()
{
  FreeTypeObject *self;

  self = PyObject_NEW(FreeTypeObject, &FreeType_Type);
  if (self == NULL)
    return NULL;
  if (TT_Init_FreeType(&self->engine))
    onError("cannot open FreeType engine")
  self->closed = 0;
  return self;
}

static PyObject *
FreeType_open_face(self, args)
     FreeTypeObject *self;
     PyObject *args;
{
  char *fontfile;

  if (!PyArg_ParseTuple(args, "s:open_face", &fontfile))
    return NULL;
  return (PyObject *)newFaceObject(self, fontfile, -1);
}

static PyObject *
FreeType_open_collection(self, args)
     FreeTypeObject *self;
     PyObject *args;
{
  char *fontfile;
  int faceindex;

  if (!PyArg_ParseTuple(args, "si:open_collection", &fontfile, &faceindex))
    return NULL;
  if (faceindex < 0)
    onError("face index out of range")
  return (PyObject *)newFaceObject(self, fontfile, faceindex);
}

static PyObject *
FreeType_close(self, args)
     FreeTypeObject *self;
     PyObject *args;
{
  if (!PyArg_ParseTuple(args, ":close"))
    return NULL;
  if (!self->closed) {
    if (TT_Done_FreeType(self->engine))
      onError("cannot close FreeType engine")
    self->closed = 1;
  }
  Py_INCREF(Py_None);
  return Py_None;
}

static PyMethodDef FreeType_methods[] = {
  {"open_face",		(PyCFunction)FreeType_open_face, 	1},
  {"open_collection",	(PyCFunction)FreeType_open_collection, 	1},
  {"close",             (PyCFunction)FreeType_close,		1},
  {NULL, NULL} /* sentinel */
};

static void
FreeType_dealloc(self)
     FreeTypeObject *self;
{
  if (!self->closed && TT_Done_FreeType(self->engine))
    PyErr_SetString(ErrorObject, "cannot close FreeType engine");
  PyMem_DEL(self);
}

static PyObject *
FreeType_getattr(self, name)
     FreeTypeObject *self;
     char *name;
{
  return Py_FindMethod(FreeType_methods, (PyObject *)self, name);
}

statichere PyTypeObject FreeType_Type = {
  PyObject_HEAD_INIT(NULL)
  0,					/*ob_size*/
  "FreeType",				/*tp_name*/
  sizeof(FreeTypeObject),		/*tp_basicsize*/
  0,					/*tp_itemsize*/
  /* methods */
  (destructor)FreeType_dealloc,		/*tp_dealloc*/
  0,					/*tp_print*/
  (getattrfunc)FreeType_getattr,	/*tp_getattr*/
  0,					/*tp_setattr*/
  0,					/*tp_compare*/
  0,					/*tp_repr*/
  0,					/*tp_as_number*/
  0,					/*tp_as_sequence*/
  0,					/*tp_as_mapping*/
  0,					/*tp_hash*/
};

/*----------------------------------------------------------------------
 *  Face object
 */

staticforward PyTypeObject Face_Type;

#define FaceObject_Check(v) ((v)->ob_type == &Face_Type)

static FaceObject *
newFaceObject(freetype, fontfile, faceindex)
     FreeTypeObject *freetype;
     char *fontfile;
     int faceindex;
{
  FaceObject *self;

  self = PyObject_NEW(FaceObject, &Face_Type);
  if (self == NULL)
    return NULL;
  if (TT_Open_Face(freetype->engine, fontfile, &self->face))
    onError("cannot open face")
  if (TT_Get_Face_Properties(self->face, &self->properties))
    onError("cannot get face properties")
  if (faceindex > -1) {
    if (TT_Close_Face(self->face))
      onError("cannot close face")
    if (faceindex >= self->properties.num_Faces)
      onError("face index out of range")
    if (TT_Open_Collection(freetype->engine, fontfile, faceindex, &self->face))
      onError("cannot open collection")
  }
  Py_INCREF(freetype);
  self->parent = freetype;
  self->closed = 0;
  return self;
}

static PyObject *
Face_new_instance(self, args)
     FaceObject *self;
     PyObject *args;
{
  if (!PyArg_ParseTuple(args, ":new_instance"))
    return NULL;
  return (PyObject *)newInstanceObject(self);
}

static PyObject *
Face_get_charmap_id(self, args)
     FaceObject *self;
     PyObject *args;
{
  int index;
  TT_UShort platformID, encodingID;

  if (!PyArg_ParseTuple(args, "i:get_charmap_id", &index))
    return NULL;
  if (TT_Get_CharMap_ID(self->face, index, &platformID, &encodingID))
    onError("cannot get charmap ID")
  return Py_BuildValue("ii", platformID, encodingID);
}

static PyObject *
Face_get_charmap(self, args)
     FaceObject *self;
     PyObject *args;
{
  int index;

  if (!PyArg_ParseTuple(args, "i:get_charmap", &index))
    return NULL;
  return (PyObject *)newCharMapObject(self, index);
}

static PyObject *
Face_get_name_id(self, args)
     FaceObject *self;
     PyObject *args;
{
  int index;
  TT_UShort platformID, encodingID, languageID, nameID;

  if (!PyArg_ParseTuple(args, "i:get_name_id", &index))
    return NULL;
  if (TT_Get_Name_ID(self->face, index,
                     &platformID, &encodingID, &languageID, &nameID))
    onError("cannot get name ID")
  return Py_BuildValue("iiii", platformID, encodingID, languageID, nameID);
}

static PyObject *
Face_get_name_string(self, args)
     FaceObject *self;
     PyObject *args;
{
  int nameID;
  TT_String *string;
  TT_UShort length;

  if (!PyArg_ParseTuple(args, "i:get_name_string", &nameID))
    return NULL;
  if (TT_Get_Name_String(self->face, nameID, &string, &length))
    onError("cannot get name string")
  return PyString_FromStringAndSize((const char *)string, length);
}

static PyObject *
Face_close(self, args)
     FaceObject *self;
     PyObject *args;
{
  if (!PyArg_ParseTuple(args, ":close"))
    return NULL;
  if (!self->closed &&
      !self->parent->closed) {
    if (TT_Close_Face(self->face))
      onError("cannot close face")
    self->closed = 1;
  }
  Py_INCREF(Py_None);
  return Py_None;
}

static PyMethodDef Face_methods[] = {
  {"new_instance",	(PyCFunction)Face_new_instance, 	1},
  {"get_charmap_id",	(PyCFunction)Face_get_charmap_id,	1},
  {"get_charmap",	(PyCFunction)Face_get_charmap,		1},
  {"get_name_id",	(PyCFunction)Face_get_name_id,		1},
  {"get_name_string",	(PyCFunction)Face_get_name_string,	1},
  {"close",		(PyCFunction)Face_close,		1},
  {NULL, NULL} /* sentinel */
};

static void
Face_dealloc(self)
     FaceObject *self;
{
  Py_DECREF(self->parent);
  PyMem_DEL(self);
}

static PyObject *
Face_getattr(self, name)
     FaceObject *self;
     char *name;
{
  if (strcmp(name, "num_faces") == 0)
    return PyInt_FromLong(self->properties.num_Faces);
  else if (strcmp(name, "num_charmaps") == 0)
    return PyInt_FromLong(self->properties.num_CharMaps);
  else if (strcmp(name, "num_glyphs") == 0)
    return PyInt_FromLong(self->properties.num_Glyphs);
  else if (strcmp(name, "max_points") == 0)
    return PyInt_FromLong(self->properties.max_Points);
  else if (strcmp(name, "max_contours") == 0)
    return PyInt_FromLong(self->properties.max_Contours);
  else if (strcmp(name, "num_names") == 0)
    return PyInt_FromLong(self->properties.num_Names);
  else if (strcmp(name, "table_version") == 0)
    return PyFloat_FromDouble(self->properties.header->Table_Version/65536.0);
  else if (strcmp(name, "font_revision") == 0)
    return PyFloat_FromDouble(self->properties.header->Font_Revision/65536.0);
  else if (strcmp(name, "units_per_em") == 0)
    return PyInt_FromLong(self->properties.header->Units_Per_EM);
  else
    return Py_FindMethod(Face_methods, (PyObject *)self, name);
}

statichere PyTypeObject Face_Type = {
  PyObject_HEAD_INIT(NULL)
  0,				/*ob_size*/
  "Face",			/*tp_name*/
  sizeof(FaceObject),		/*tp_basicsize*/
  0,				/*tp_itemsize*/
  /* methods */
  (destructor)Face_dealloc,	/*tp_dealloc*/
  0,				/*tp_print*/
  (getattrfunc)Face_getattr,	/*tp_getattr*/
  0,				/*tp_setattr*/
  0,				/*tp_compare*/
  0,				/*tp_repr*/
  0,				/*tp_as_number*/
  0,				/*tp_as_sequence*/
  0,				/*tp_as_mapping*/
  0,				/*tp_hash*/
};

/*----------------------------------------------------------------------
 *  Instance object
 */

staticforward PyTypeObject Instance_Type;

#define InstanceObject_Check(v) ((v)->ob_type == &Instance_Type)

static InstanceObject *
newInstanceObject(face)
     FaceObject *face;
{
  InstanceObject *self;

  self = PyObject_NEW(InstanceObject, &Instance_Type);
  if (self == NULL)
    return NULL;
  if (TT_New_Instance(face->face, &self->instance))
    onError("cannot create new instance")
  if (TT_Set_Instance_Resolutions(self->instance, 96, 96))
    onError("cannot initialize instance resolutions")
  if (TT_Set_Instance_CharSize(self->instance, 10 * 64))
    onError("cannot initialize instance char size")
  if (TT_Get_Instance_Metrics(self->instance, &self->metrics))
    onError("cannot get instance metrics")
  Py_INCREF(face);
  self->parent = face;
  self->closed = 0;
  return self;
}

static PyObject *
Instance_set_resolutions(self, args)
     InstanceObject *self;
     PyObject *args;
{
  int xres, yres;

  if (!PyArg_ParseTuple(args, "ii:set_resolutions", &xres, &yres))
    return NULL;
  if (TT_Set_Instance_Resolutions(self->instance, xres, yres))
    onError("cannot set instance resolutions")
  if (TT_Get_Instance_Metrics(self->instance, &self->metrics))
    onError("cannot get instance metrics")
  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
Instance_set_charsize(self, args)
     InstanceObject *self;
     PyObject *args;
{
  float size;

  if (!PyArg_ParseTuple(args, "f:set_charsize", &size))
    return NULL;
  if (TT_Set_Instance_CharSize(self->instance, size * 64))
    onError("cannot set instance char size")
  if (TT_Get_Instance_Metrics(self->instance, &self->metrics))
    onError("cannot get instance metrics")
  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
Instance_set_charsizes(self, args)
     InstanceObject *self;
     PyObject *args;
{
  float width, height;

  if (!PyArg_ParseTuple(args, "ff:set_charsizes", &width, &height))
    return NULL;
  if (TT_Set_Instance_CharSizes(self->instance, width * 64, height * 64))
    onError("cannot set instance char sizes")
  if (TT_Get_Instance_Metrics(self->instance, &self->metrics))
    onError("cannot get instance metrics")
  Py_INCREF(Py_None);
  return Py_None;
}

static PyObject *
Instance_new_glyph(self, args)
     InstanceObject *self;
     PyObject *args;
{
  int index, scaling = 1;  /* scaling is enabled by default */

  if (!PyArg_ParseTuple(args, "i|i:new_glyph", &index, &scaling))
    return NULL;
  return (PyObject *)newGlyphObject(self, index, scaling);
}

static PyObject *
Instance_close(self, args)
     InstanceObject *self;
     PyObject *args;
{
  if (!PyArg_ParseTuple(args, ":close"))
    return NULL;
  if (!self->closed &&
      !self->parent->closed &&
      !self->parent->parent->closed) {
    if (TT_Done_Instance(self->instance))
      onError("cannot close instance")
    self->closed = 1;
  }
  Py_INCREF(Py_None);
  return Py_None;
}

static PyMethodDef Instance_methods[] = {
  {"set_resolutions",	(PyCFunction)Instance_set_resolutions,	1},
  {"set_charsize",	(PyCFunction)Instance_set_charsize,	1},
  {"set_charsizes",	(PyCFunction)Instance_set_charsizes,	1},
  {"new_glyph",		(PyCFunction)Instance_new_glyph,	1},
  {"close",		(PyCFunction)Instance_close,		1},
  {NULL, NULL} /* sentinel */
};

static void
Instance_dealloc(self)
     InstanceObject *self;
{
  Py_DECREF(self->parent);
  PyMem_DEL(self);
}

static PyObject *
Instance_getattr(self, name)
     InstanceObject *self;
     char *name;
{
  if (strcmp(name, "pointsize") == 0)
    return PyFloat_FromDouble((double)self->metrics.pointSize / 64);
  else if (strcmp(name, "x_ppem") == 0)
    return PyInt_FromLong(self->metrics.x_ppem);
  else if (strcmp(name, "y_ppem") == 0)
    return PyInt_FromLong(self->metrics.y_ppem);
  else if (strcmp(name, "x_scale") == 0)
    return PyFloat_FromDouble((double)self->metrics.x_scale / 0x10000);
  else if (strcmp(name, "y_scale") == 0)
    return PyFloat_FromDouble((double)self->metrics.y_scale / 0x10000);
  else if (strcmp(name, "x_resolution") == 0)
    return PyInt_FromLong(self->metrics.x_resolution);
  else if (strcmp(name, "y_resolution") == 0)
    return PyInt_FromLong(self->metrics.y_resolution);
  else
    return Py_FindMethod(Instance_methods, (PyObject *)self, name);
}

statichere PyTypeObject Instance_Type = {
  PyObject_HEAD_INIT(NULL)
  0,					/*ob_size*/
  "Instance",				/*tp_name*/
  sizeof(InstanceObject),		/*tp_basicsize*/
  0,					/*tp_itemsize*/
  /* methods */
  (destructor)Instance_dealloc,		/*tp_dealloc*/
  0,					/*tp_print*/
  (getattrfunc)Instance_getattr,	/*tp_getattr*/
  0,					/*tp_setattr*/
  0,					/*tp_compare*/
  0,					/*tp_repr*/
  0,					/*tp_as_number*/
  0,					/*tp_as_sequence*/
  0,					/*tp_as_mapping*/
  0,					/*tp_hash*/
};

/*----------------------------------------------------------------------
 *  CharMap object
 */

staticforward PyTypeObject CharMap_Type;

#define CharMapObject_Check(v) ((v)->ob_type == &CharMap_Type)

static CharMapObject *
newCharMapObject(face, index)
     FaceObject *face;
     int index;
{
  CharMapObject *self;

  self = PyObject_NEW(CharMapObject, &CharMap_Type);
  if (self == NULL)
    return NULL;
  if (TT_Get_CharMap(face->face, index, &self->charmap))
    onError("cannot get charmap")
  return self;
}

static PyObject *
CharMap_index(self, args)
     CharMapObject *self;
     PyObject *args;
{
  int charcode;

  if (!PyArg_ParseTuple(args, "i:index", &charcode))
    return NULL;
  return PyInt_FromLong(TT_Char_Index(self->charmap, charcode));
}

static PyMethodDef CharMap_methods[] = {
  {"index",	(PyCFunction)CharMap_index,	1},
  {NULL, NULL} /* sentinel */
};

static void
CharMap_dealloc(self)
     CharMapObject *self;
{
  PyMem_DEL(self);
}

static PyObject *
CharMap_getattr(self, name)
     CharMapObject *self;
     char *name;
{
  return Py_FindMethod(CharMap_methods, (PyObject *)self, name);
}

statichere PyTypeObject CharMap_Type = {
  PyObject_HEAD_INIT(NULL)
  0,				/*ob_size*/
  "CharMap",			/*tp_name*/
  sizeof(CharMapObject),	/*tp_basicsize*/
  0,				/*tp_itemsize*/
  /* methods */
  (destructor)CharMap_dealloc,	/*tp_dealloc*/
  0,				/*tp_print*/
  (getattrfunc)CharMap_getattr,	/*tp_getattr*/
  0,				/*tp_setattr*/
  0,				/*tp_compare*/
  0,				/*tp_repr*/
  0,				/*tp_as_number*/
  0,				/*tp_as_sequence*/
  0,				/*tp_as_mapping*/
  0,				/*tp_hash*/
};

/*----------------------------------------------------------------------
 *  Glyph object
 */

staticforward PyTypeObject Glyph_Type;

#define GlyphObject_Check(v) ((v)->ob_type == &Glyph_Type)

static GlyphObject *
newGlyphObject(instance, index, scaling)
     InstanceObject *instance;
     int index, scaling;
{
  GlyphObject *self;
  TT_Outline outline;
  PyObject *contour, *p;
  int load_flags = 0;
  int i, j, k;

  self = PyObject_NEW(GlyphObject, &Glyph_Type);
  if (self == NULL)
    return NULL;
  if (TT_New_Glyph(instance->parent->face, &self->glyph))
    onError("cannot create new glyph")
  if (scaling)
    load_flags = TTLOAD_DEFAULT;
  if (TT_Load_Glyph(instance->instance, self->glyph, index, load_flags))
    onError("cannot load glyph")
  if (TT_Get_Glyph_Metrics(self->glyph, &self->metrics))
    onError("cannot get glyph metrics")
  if (TT_Get_Glyph_Outline(self->glyph, &outline))
    onError("cannot get glyph outline")
  /* convert outline contours into tuples */
  self->outline = PyTuple_New(outline.n_contours);
  for (i = j = 0; i < outline.n_contours; i++) {
    contour = PyTuple_New(outline.contours[i] - j + 1);
    for (k = 0; j <= outline.contours[i]; k++, j++) {
      if (scaling)
        p = Py_BuildValue("ffi",
                          (double)outline.points[j].x / 64,
                          (double)outline.points[j].y / 64,
                          outline.flags[j] & 1);
      else
        p = Py_BuildValue("iii",
                          outline.points[j].x,
                          outline.points[j].y,
                          outline.flags[j] & 1);
      PyTuple_SetItem(contour, k, p);
    }
    PyTuple_SetItem(self->outline, i, contour);
  }
  Py_INCREF(instance);
  self->parent = instance;
  self->closed = 0;
  return self;
}

static PyObject *
Glyph_close(self, args)
     GlyphObject *self;
     PyObject *args;
{
  if (!PyArg_ParseTuple(args, ":close"))
    return NULL;
  if (!self->closed &&
      !self->parent->closed &&
      !self->parent->parent->closed &&
      !self->parent->parent->parent->closed) {
    if (TT_Done_Glyph(self->glyph))
      onError("cannot close glyph")
    self->closed = 1;
  }
  Py_INCREF(Py_None);
  return Py_None;
}

static PyMethodDef Glyph_methods[] = {
  {"close",	(PyCFunction)Glyph_close,	1},
  {NULL, NULL} /* sentinel */
};

static void
Glyph_dealloc(self)
     GlyphObject *self;
{
  Py_DECREF(self->outline);
  Py_DECREF(self->parent);
  PyMem_DEL(self);
}

static PyObject *
Glyph_getattr(self, name)
     GlyphObject *self;
     char *name;
{
  if (strcmp(name, "outline") == 0) {
    Py_INCREF(self->outline);
    return self->outline;
  }
  else if (strcmp(name, "bbox") == 0)
    return Py_BuildValue("ffff",
                         (double)self->metrics.bbox.xMin / 64,
                         (double)self->metrics.bbox.yMin / 64,
                         (double)self->metrics.bbox.xMax / 64,
                         (double)self->metrics.bbox.yMax / 64);
  else if (strcmp(name, "bearingx") == 0)
    return PyFloat_FromDouble((double)self->metrics.bearingX / 64);
  else if (strcmp(name, "bearingy") == 0)
    return PyFloat_FromDouble((double)self->metrics.bearingY / 64);
  else if (strcmp(name, "advance") == 0)
    return PyFloat_FromDouble((double)self->metrics.advance / 64);
  else
    return Py_FindMethod(Glyph_methods, (PyObject *)self, name);
}

statichere PyTypeObject Glyph_Type = {
  PyObject_HEAD_INIT(NULL)
  0,				/*ob_size*/
  "Glyph",			/*tp_name*/
  sizeof(GlyphObject),		/*tp_basicsize*/
  0,				/*tp_itemsize*/
  /* methods */
  (destructor)Glyph_dealloc,	/*tp_dealloc*/
  0,				/*tp_print*/
  (getattrfunc)Glyph_getattr,	/*tp_getattr*/
  0,				/*tp_setattr*/
  0,				/*tp_compare*/
  0,				/*tp_repr*/
  0,				/*tp_as_number*/
  0,				/*tp_as_sequence*/
  0,				/*tp_as_mapping*/
  0,				/*tp_hash*/
};

/*----------------------------------------------------------------------
 *  Module functions
 */

static PyObject *
freetype_FreeType(self, args)
     PyObject *self; /* not used */
     PyObject *args;
{
  if (!PyArg_ParseTuple(args, ":FreeType"))
    return NULL;
  return (PyObject *)newFreeTypeObject();
}

/* List of module functions */
static PyMethodDef freetype_methods[] = {
  {"FreeType",	freetype_FreeType,	1},
  {NULL, NULL} /* sentinel */
};

/* Initialization function for the freetype module  */
DL_EXPORT(void)
initfreetype()
{
  PyObject *m, *d, *v;

  /* Initialize the type of the new type objects */
  FreeType_Type.ob_type = Face_Type.ob_type = \
    Instance_Type.ob_type = CharMap_Type.ob_type = \
    Glyph_Type.ob_type = &PyType_Type;

  /* Create a module and add functions */
  m = Py_InitModule("freetype", freetype_methods);

  /* Add constants to the module */
  d = PyModule_GetDict(m);
  ErrorObject = PyErr_NewException("freetype.error", NULL, NULL);
  PyDict_SetItemString(d, "error", ErrorObject);
  v = Py_BuildValue("ii", TT_FREETYPE_MAJOR, TT_FREETYPE_MINOR);
  PyDict_SetItemString(d, "version", v);
}
