#
#  JapaneseText.py - a Sketch script for drawing Japanese characters
#                    Version 1.0 <20 April 2000>
#
#  Copyright (c) Tamito KAJIYAMA <kajiyama@grad.sccs.chukyo-u.ac.jp>
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License as
#  published by the Free Software Foundation; either version 2 of the
#  License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful, but
#  WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#  General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#

__version__ = "1.0.2"

import freetype
import euc2unicode
import os

from Tkinter import *

from Sketch import Scale, TrafoPlugin, PolyBezier, \
     CreatePath, Point, ContAngle, ContSmooth, \
     EmptyLineStyle, SolidPattern, StandardColors
from Sketch.UI.sketchdlg import SKModal
from Sketch.UI.miniscroll import MiniScroller

# User configurations (to be customized in ~/.sketch/userhooks.py)
kanjifonts = []
asciifonts = []
preview_size = 60.0

def create_listbox_with_label(master, label):
    frame = Frame(master)
    frame.label = Label(frame, text=label, anchor=W)
    frame.label.pack(fill=X)
    frame.scrollbar = Scrollbar(frame)
    frame.scrollbar.pack(side=RIGHT, fill=Y)
    frame.listbox = Listbox(frame, yscrollcommand=frame.scrollbar.set)
    frame.listbox.pack(side=LEFT, fill=BOTH, expand=1)
    frame.scrollbar.config(command=frame.listbox.yview)
    return frame

parameters = None

class ParameterDialog(SKModal):
    title = "JapaneseText"
    def build_dlg(self):
        # text entry
        frame = Frame(self.top)
        frame.pack(fill=X)
        label = Label(frame, text="Text", width=4)
        label.pack(side=LEFT)
        self.text = Entry(frame)
        self.text.pack(side=RIGHT, fill=X, expand=1)
        # size entry
        self.size = DoubleVar(self.top)
        self.size.set(12.0)
        frame = Frame(self.top)
        frame.pack(fill=X)
        label = Label(frame, text="Size", width=4)
        label.pack(side=LEFT)
        entry = Entry(frame, textvariable=self.size, width=8, justify=RIGHT)
        entry.pack(side=LEFT)
        scroll = MiniScroller(frame, variable=self.size, min=0)
        scroll.pack(side=LEFT, fill=Y)
        self.fill = BooleanVar(self.top)
        self.fill.set(0)
        checkbutton = Checkbutton(frame, text="Fill", variable=self.fill)
        checkbutton.pack(side=LEFT)
        # buttons
        frame = Frame(self.top)
        frame.pack(side=BOTTOM, fill=X)
	button = Button(frame, text=" OK ", command=self.ok)
	button.pack(side=LEFT)
	button = Button(frame, text="Cancel", command=self.cancel)
	button.pack(side=RIGHT)
        # preview
        scrollbar = Scrollbar(self.top, orient=HORIZONTAL)
        scrollbar.pack(side=BOTTOM, fill=X)
        self.canvas = Canvas(self.top, height=int(preview_size * 2),
                             bg="white", xscrollcommand=scrollbar.set)
        self.canvas.pack(side=BOTTOM, fill=X)
        scrollbar.config(command=self.canvas.xview)
        button = Button(self.top, text="Preview", command=self.redraw)
        button.pack(side=BOTTOM, anchor=W)
        # font lists
        frame = create_listbox_with_label(self.top, "Kanji Fonts")
        frame.pack(side=LEFT, fill=BOTH, expand=1)
        self.kanjifonts = frame.listbox
        self.kanjifonts.config(height=6)
        for info in kanjifonts:
            self.kanjifonts.insert(END, info[0])
        frame = create_listbox_with_label(self.top, "ASCII Fonts")
        frame.pack(side=RIGHT, fill=BOTH, expand=1)
        self.asciifonts = frame.listbox
        self.asciifonts.config(height=6)
        for info in asciifonts:
            self.asciifonts.insert(END, info[0])
        #  restore previous parameters
        if parameters is not None:
            self.text.insert(END, parameters[0])
            self.size.set(parameters[1])
            self.kanjifonts.activate(parameters[2])
            self.asciifonts.activate(parameters[3])
            self.fill.set(parameters[4])
        self.kanjifonts.selection_set(ACTIVE)
        self.asciifonts.selection_set(ACTIVE)
    def get_parameters(self, save=0):
        text = self.text.get()
        size = self.size.get()
        kindex = int(self.kanjifonts.index(ACTIVE))
        aindex = int(self.asciifonts.index(ACTIVE))
        fill = self.fill.get()
        if save:
            global parameters
            parameters = (text, size, kindex, aindex, fill)
        kface = self.get_face(kanjifonts, kindex)
        aface = self.get_face(asciifonts, aindex)
        return (text, size, kface, aface, fill)
    def redraw(self, event=None):
        # get parameters
        text, size, kface, aface, fill = self.get_parameters()
        # prepare fonts
        kfont = kface.new_instance()
        kfont.set_charsize(preview_size)
        kfont_charmap = get_charmap(kface)
        afont = aface.new_instance()
        afont.set_charsize(preview_size)
        afont_charmap = get_charmap(aface)
        # clear canvas
        self.canvas.delete("lines")
        # draw glyphs
        i = 0
        margin = preview_size / 2
        x_offset = margin
        y_offset = preview_size + margin
        while i < len(text):
            if text[i] < '\200':
                code = ord(text[i])
                glyph = afont.new_glyph(afont_charmap.index(code))
                i = i + 1
            else:
                code = euc2unicode.mapping[text[i:i+2]]
                glyph = kfont.new_glyph(kfont_charmap.index(code))
                i = i + 2
            for contour in glyph.outline:
                x, y, onpoint = contour[0]
                last_point = Point(x, y)
                j = 1
                npoints = len(contour)
                while j <= npoints:
                    x, y, onpoint = contour[j % npoints]
                    point = Point(x, y)
                    j = j + 1
                    if onpoint:
                        self.canvas.create_line(x_offset + last_point.x,
                                                y_offset - last_point.y,
                                                x_offset + point.x,
                                                y_offset - point.y,
                                                tags="lines")
                        last_point = point
                    else:
                        x, y, onpoint = contour[j % npoints]
                        if onpoint:
                            j = j + 1
                        else:
                            x = point.x + (x - point.x) * 0.5
                            y = point.y + (y - point.y) * 0.5
                        self.canvas.create_line(x_offset + last_point.x,
                                                y_offset - last_point.y,
                                                x_offset + point.x,
                                                y_offset - point.y,
                                                x_offset + x,
                                                y_offset - y,
                                                tags="lines", smooth=1)
                        last_point = Point(x, y)
            x_offset = x_offset + glyph.advance
        w = x_offset + margin
        h = preview_size + margin * 2
        self.canvas.config(scrollregion=(1, 1, w, h))
    def get_face(self, fontlist, index):
        engine = freetype.FreeType()
        fontfile = os.path.expanduser(fontlist[index][1])
        if len(fontlist[index]) == 3:
            return engine.open_collection(fontfile, fontlist[index][2])
        else:
            return engine.open_face(fontfile)
    def ok(self, *args):
	self.close_dlg(self.get_parameters(save=1))

def main(context):
    # check font configurations
    if len(kanjifonts) == 0 or len(asciifonts) == 0:
        raise RuntimeError
    # inquiry parameteres
    args = ParameterDialog(context.application.root).RunDialog()
    if args is None:
        return
    # create paths
    text, size, kface, aface, fill = args
    paths = create_path(text, size, kface, aface)
    polygon = PolyBezier(paths)
    if fill:
        polygon.AddStyle(EmptyLineStyle)
        polygon.SetProperties(fill_pattern=SolidPattern(StandardColors.black))
    context.main_window.PlaceObject(polygon)

def create_path(text, size, kface, aface):
    # prepare fonts
    kfont = kface.new_instance()
    kfont_charmap = get_charmap(kface)
    kfont.set_resolutions(72, 72)
    kfont.set_charsize(1024.0)
    afont = aface.new_instance()
    afont.set_resolutions(72, 72)
    afont.set_charsize(1024.0)
    afont_charmap = get_charmap(aface)
    # convert glyph data into bezier polygons
    paths = []
    offset = i = 0
    while i < len(text):
        if text[i] < '\200':
            code = ord(text[i])
            glyph = afont.new_glyph(afont_charmap.index(code))
            i = i + 1
        else:
            code = euc2unicode.mapping[text[i:i+2]]
            glyph = kfont.new_glyph(kfont_charmap.index(code))
            i = i + 2
        for contour in glyph.outline:
            # rotate contour so that it begins with an onpoint
            x, y, onpoint = contour[0]
            if not onpoint:
                for j in range(1, len(contour)):
                    x, y, onpoint = contour[j]
                    if onpoint:
                        contour = contour[j:] + contour[:j]
                        break
                else:
                    print "unsupported type of contour (no onpoint)"
            # create a Sketch path object
            path = CreatePath()
            j = 0
            npoints = len(contour)
            x, y, onpoint = contour[0]
            last_point = Point(x, y)
            while j <= npoints:
                if j == npoints:
                    x, y, onpoint = contour[0]
                else:
                    x, y, onpoint = contour[j]
                point = Point(x, y)
                j = j + 1
                if onpoint:
                    path.AppendLine(point)
                    last_point = point
                else:
                    c1 = last_point + (point - last_point) * 2.0 / 3.0
                    x, y, onpoint = contour[j % npoints]
                    if onpoint:
                        j = j + 1
                        cont = ContAngle
                    else:
                        x = point.x + (x - point.x) * 0.5
                        y = point.y + (y - point.y) * 0.5
                        cont = ContSmooth
                    last_point = Point(x, y)
                    c2 = last_point + (point - last_point) * 2.0 / 3.0
                    path.AppendBezier(c1, c2, last_point, cont)
            path.ClosePath()
            path.Translate(offset, 0)
            path.Transform(Scale(size/1024.0))
            paths.append(path)
        offset = offset + glyph.advance
    return tuple(paths)

def get_charmap(face):
    for i in range(face.num_charmaps):
        pid, eid = face.get_charmap_id(i)
        if pid == 3 and eid == 1:  # Windows Unicode
            return face.get_charmap(i)
    else:
        raise RuntimeError, "no Windows Unicode charmap"

import Sketch.Scripting
Sketch.Scripting.AddFunction('JapaneseText', 'JapaneseText', main,
                             script_type = Sketch.Scripting.AdvancedScript)
