/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%               CCCC   OOO   M   M  PPPP    AAA   RRRR    EEEEE               %
%              C      O   O  MM MM  P   P  A   A  R   R   E                   %
%              C      O   O  M M M  PPPP   AAAAA  RRRR    EEE                 %
%              C      O   O  M   M  P      A   A  R R     E                   %
%               CCCC   OOO   M   M  P      A   A  R  R    EEEEE               %
%                                                                             %
%                                                                             %
%                         Image Comparison Methods                            %
%                                                                             %
%                              Software Design                                %
%                                John Cristy                                  %
%                               December 2003                                 %
%                                                                             %
%                                                                             %
%  Copyright 1999-2004 ImageMagick Studio LLC, a non-profit organization      %
%  dedicated to making software imaging solutions freely available.           %
%                                                                             %
%  You may not use this file except in compliance with the License.  You may  %
%  obtain a copy of the License at                                            %
%                                                                             %
%    http://www.imagemagick.org/www/Copyright.html                            %
%                                                                             %
%  Unless required by applicable law or agreed to in writing, software        %
%  distributed under the License is distributed on an "AS IS" BASIS,          %
%  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.   %
%  See the License for the specific language governing permissions and        %
%  limitations under the License.                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%
%
*/

/*
  Include declarations.
*/
#include "magick/studio.h"
#include "magick/client.h"
#include "magick/color.h"
#include "magick/compare.h"
#include "magick/composite.h"
#include "magick/composite_private.h"
#include "magick/constitute.h"
#include "magick/geometry.h"
#include "magick/list.h"
#include "magick/log.h"
#include "magick/memory_.h"
#include "magick/mogrify.h"
#include "magick/option.h"
#include "magick/resource_.h"
#include "magick/string_.h"
#include "magick/utility.h"
#include "magick/version.h"

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   C o m p a r e I m a g e C h a n n e l s                                   %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  CompareImageChannels() compares one or more image channels and returns
%  the specified distortion metric.
%
%  The format of the CompareImageChannels method is:
%
%      unsigned int CompareImageChannels(const Image *image,
%        const Image *reconstruct_image,const ChannelType channel,
%        const MetricType metric,double *distortion,ExceptionInfo *exception)
%
%  A description of each parameter follows:
%
%    o image: The image.
%
%    o reconstruct_image: The reconstruct image.
%
%    o channel: The channel.
%
%    o metric: The metric.
%
%    o distortion: The computed distortion between the images.
%
%    o exception: Return any errors or warnings in this structure.
%
%
*/

MagickExport Image *CompareImages(Image *image,const Image *reconstruct_image,
  const MetricType metric,double *distortion,ExceptionInfo *exception)
{
  Image
    *difference_image;

  difference_image=CompareImageChannels(image,reconstruct_image,AllChannels,
    metric,distortion,exception);
  return(difference_image);
}

static MagickRealType GetMeanAbsoluteError(const Image *image,
  const Image *reconstruct_image,const ChannelType channel,
  ExceptionInfo *exception)
{
  IndexPacket
    *indexes,
    *reconstruct_indexes;

  long
    y;

  MagickRealType
    area,
    distortion;

  register const PixelPacket
    *p,
    *q;

  register long
    x;

  area=0.0;
  distortion=0.0;
  for (y=0; y < (long) image->rows; y++)
  {
    p=AcquireImagePixels(image,0,y,image->columns,1,exception);
    q=AcquireImagePixels(reconstruct_image,0,y,reconstruct_image->columns,1,
      exception);
    if ((p == (const PixelPacket *) NULL) || (q == (const PixelPacket *) NULL))
      break;
    indexes=GetIndexes(image);
    reconstruct_indexes=GetIndexes(reconstruct_image);
    for (x=0; x < (long) image->columns; x++)
    {
      if ((channel & RedChannel) != 0)
        {
          distortion+=fabs(p->red-(MagickRealType) q->red);
          area++;
        }
      if ((channel & GreenChannel) != 0)
        {
          distortion+=fabs(p->green-(MagickRealType) q->green);
          area++;
        }
      if ((channel & BlueChannel) != 0)
        {
          distortion+=fabs(p->blue-(MagickRealType) q->blue);
          area++;
        }
      if (((channel & OpacityChannel) != 0) && (image->matte != False))
        {
          distortion+=fabs(p->opacity-(MagickRealType) q->opacity);
          area++;
        }
      if (((channel & IndexChannel) != 0) &&
          (image->colorspace == CMYKColorspace))
        {
          distortion+=fabs(indexes[x]-(MagickRealType) reconstruct_indexes[x]);
          area++;
        }
      p++;
      q++;
    }
  }
  return(distortion/area);
}

static MagickRealType GetMeanSquaredError(const Image *image,
  const Image *reconstruct_image,const ChannelType channel,
  ExceptionInfo *exception)
{
  IndexPacket
    *indexes,
    *reconstruct_indexes;

  long
    y;

  MagickRealType
    area,
    distance,
    distortion;

  register const PixelPacket
    *p,
    *q;

  register long
    x;

  area=0.0;
  distortion=0.0;
  for (y=0; y < (long) image->rows; y++)
  {
    p=AcquireImagePixels(image,0,y,image->columns,1,exception);
    q=AcquireImagePixels(reconstruct_image,0,y,reconstruct_image->columns,1,
      exception);
    if ((p == (const PixelPacket *) NULL) || (q == (const PixelPacket *) NULL))
      break;
    indexes=GetIndexes(image);
    reconstruct_indexes=GetIndexes(reconstruct_image);
    for (x=0; x < (long) image->columns; x++)
    {
      if ((channel & RedChannel) != 0)
        {
          distance=p->red-(MagickRealType) q->red;
          distortion+=distance*distance;
          area++;
        }
      if ((channel & GreenChannel) != 0)
        {
          distance=p->green-(MagickRealType) q->green;
          distortion+=distance*distance;
          area++;
        }
      if ((channel & BlueChannel) != 0)
        {
          distance=p->blue-(MagickRealType) q->blue;
          distortion+=distance*distance;
          area++;
        }
      if (((channel & OpacityChannel) != 0) && (image->matte != False))
        {
          distance=p->opacity-(MagickRealType) q->opacity;
          distortion+=distance*distance;
          area++;
        }
      if (((channel & IndexChannel) != 0) &&
          (image->colorspace == CMYKColorspace))
        {
          distance=indexes[x]-(MagickRealType) reconstruct_indexes[x];
          distortion+=distance*distance;
          area++;
        }
      p++;
      q++;
    }
  }
  return(distortion/area);
}

static MagickRealType GetPeakAbsoluteError(const Image *image,
  const Image *reconstruct_image,const ChannelType channel,
  ExceptionInfo *exception)
{
  IndexPacket
    *indexes,
    *reconstruct_indexes;

  long
    y;

  MagickRealType
    distance,
    distortion;

  register const PixelPacket
    *p,
    *q;

  register long
    x;

  distortion=0.0;
  for (y=0; y < (long) image->rows; y++)
  {
    p=AcquireImagePixels(image,0,y,image->columns,1,exception);
    q=AcquireImagePixels(reconstruct_image,0,y,reconstruct_image->columns,1,
      exception);
    if ((p == (const PixelPacket *) NULL) || (q == (const PixelPacket *) NULL))
      break;
    indexes=GetIndexes(image);
    reconstruct_indexes=GetIndexes(reconstruct_image);
    for (x=0; x < (long) image->columns; x++)
    {
      if ((channel & RedChannel) != 0)
        {
          distance=fabs(p->red-(MagickRealType) q->red);
          if (distance > distortion)
            distortion=distance;
        }
      if ((channel & GreenChannel) != 0)
        {
          distance=fabs(p->green-(MagickRealType) q->green);
          if (distance > distortion)
            distortion=distance;
        }
      if ((channel & BlueChannel) != 0)
        {
          distance=fabs(p->blue-(MagickRealType) q->blue);
          if (distance > distortion)
            distortion=distance;
        }
      if (((channel & OpacityChannel) != 0) && (image->matte != False))
        {
          distance=fabs(p->opacity-(MagickRealType) q->opacity);
          if (distance > distortion)
            distortion=distance;
        }
      if (((channel & IndexChannel) != 0) &&
          (image->colorspace == CMYKColorspace))
        {
          distance=
            fabs(indexes[x]-(MagickRealType) reconstruct_indexes[x]);
          if (distance > distortion)
            distortion=distance;
        }
      p++;
      q++;
    }
  }
  return(distortion);
}

static MagickRealType GetPeakSignalToNoiseRatio(const Image *image,
  const Image *reconstruct_image,const ChannelType channel,
  ExceptionInfo *exception)
{
  MagickRealType
    distortion;

  distortion=GetMeanSquaredError(image,reconstruct_image,channel,exception);
  return(20.0*log10(MaxRGB/sqrt(distortion)));
}

static MagickRealType GetRootMeanSquaredError(const Image *image,
  const Image *reconstruct_image,const ChannelType channel,
  ExceptionInfo *exception)
{
  MagickRealType
    distortion;

  distortion=GetMeanSquaredError(image,reconstruct_image,channel,exception);
  return(sqrt(distortion));
}

MagickExport Image *CompareImageChannels(Image *image,
  const Image *reconstruct_image,const ChannelType channel,
  const MetricType metric,double *distortion,ExceptionInfo *exception)
{
  Image
    *difference_image;

  IndexPacket
    *indexes,
    *reconstruct_indexes;

  long
    y;

  MagickRealType
    alpha,
    beta;

  PixelPacket
    red,
    white;

  register const PixelPacket
    *p,
    *q;

  register long
    x;

  register PixelPacket
    *r;

  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  if (image->debug != False)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),image->filename);
  assert(reconstruct_image != (const Image *) NULL);
  assert(reconstruct_image->signature == MagickSignature);
  assert(distortion != (double *) NULL);
  if (image->debug != False)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),image->filename);
  if ((reconstruct_image->columns != image->columns) ||
      (reconstruct_image->rows != image->rows))
    ThrowImageException(ImageError,"ImageSizeDiffers");
  if (image->colorspace != reconstruct_image->colorspace)
    ThrowImageException(ImageError,"ImageColorspaceDiffers");
  if (image->matte != reconstruct_image->matte)
    ThrowImageException(ImageError,"ImageOpacityDiffers");
  difference_image=CloneImage(image,0,0,True,exception);
  if (difference_image == (Image *) NULL)
    return((Image *) NULL);
  difference_image->storage_class=DirectClass;
  (void) QueryColorDatabase("#f1001e",&red,exception);
  (void) QueryColorDatabase("#ffffff",&white,exception);
  /*
    Get image distortion.
  */
  *distortion=0.0;
  switch (metric)
  {
    case MeanAbsoluteErrorMetric:
    {
      *distortion=GetMeanAbsoluteError(image,reconstruct_image,channel,
        exception);
      break;
    }
    case MeanSquaredErrorMetric:
    {
      *distortion=GetMeanSquaredError(image,reconstruct_image,channel,
        exception);
      break;
    }
    case PeakAbsoluteErrorMetric:
    default:
    {
      *distortion=GetPeakAbsoluteError(image,reconstruct_image,channel,
        exception);
      break;
    }
    case PeakSignalToNoiseRatioMetric:
    {
      *distortion=GetPeakSignalToNoiseRatio(image,reconstruct_image,channel,
        exception);
      break;
    }
    case RootMeanSquaredErrorMetric:
    {
      *distortion=GetRootMeanSquaredError(image,reconstruct_image,channel,
        exception);
      break;
    }
  }
  /*
    Generate difference image.
  */
  for (y=0; y < (long) image->rows; y++)
  {
    p=AcquireImagePixels(image,0,y,image->columns,1,exception);
    q=AcquireImagePixels(reconstruct_image,0,y,reconstruct_image->columns,1,
      exception);
    if ((p == (const PixelPacket *) NULL) || (q == (const PixelPacket *) NULL))
      break;
    r=SetImagePixels(difference_image,0,y,difference_image->columns,1);
    if ((r == (const PixelPacket *) NULL) || (r == (const PixelPacket *) NULL))
      break;
    indexes=GetIndexes(image);
    reconstruct_indexes=GetIndexes(reconstruct_image);
    for (x=0; x < (long) image->columns; x++)
    {
      alpha=0.0;
      beta=0.0;
      if ((channel & RedChannel) != 0)
        {
          alpha+=p->red;
          beta+=q->red;
        }
      if ((channel & GreenChannel) != 0)
        {
          alpha+=p->green;
          beta+=q->green;
        }
      if ((channel & BlueChannel) != 0)
        {
          alpha+=p->blue;
          beta+=q->blue;
        }
      if (((channel & OpacityChannel) != 0) && (image->matte != False))
        {
          alpha+=p->opacity;
          beta+=q->opacity;
        }
      if (((channel & IndexChannel) != 0) &&
          (image->colorspace == CMYKColorspace))
        {
          alpha+=indexes[x];
          beta+=reconstruct_indexes[x];
        }
      if (alpha != beta)
        *r=MagickCompositeOver(q,9.0*MaxRGB/10.0,&red,(double) red.opacity);
      else
        *r=MagickCompositeOver(q,9.0*MaxRGB/10.0,&white,(double) white.opacity);
      p++;
      q++;
      r++;
    }
  }
  return(difference_image);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   C o m p a r e I m a g e C o m m a n d                                     %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  CompareImageCommand() compares two images and returns the difference between
%  them as a distortion metric and as a new image visually annotating their
%  differences.
%
%  The format of the CompareImageCommand method is:
%
%      unsigned int CompareImageCommand(ImageInfo *image_info,int argc,
%        char **argv,char **metadata,ExceptionInfo *exception)
%
%  A description of each parameter follows:
%
%    o image_info: The image info.
%
%    o argc: The number of elements in the argument vector.
%
%    o argv: A text array containing the command line arguments.
%
%    o metadata: any metadata is returned here.
%
%    o exception: Return any errors or warnings in this structure.
%
%
*/

static void CompareUsage(void)
{
  const char
    **p;

  static const char
    *options[]=
    {
      "-authenticate value  decrypt image with this password",
      "-channel type        Red, Green, Blue, Opacity, Index, Cyan, Yellow, ",
      "                     Magenta, Black, or All",
      "-colorspace type     alternate image colorspace",
      "-compress type       image compression type",
      "-debug events        display copious debugging information",
      "-define format:option",
      "                     define one or more image format options",
      "-density geometry    horizontal and vertical density of the image",
      "-depth value         image depth",
      "-extract geometry    extract area from image",
      "-help                print program options",
      "-interlace type      None, Line, Plane, or Partition",
      "-limit type value    Area, Disk, Map, or Memory resource limit",
      "-log format          format of debugging information",
      "-metric type         MAE, MSE, PSE, PSNR, or RMSE",
      "-quality value       JPEG/MIFF/PNG compression level",
      "-profile filename    add, delete, or apply an image profile",
      "-sampling-factor geometry",
      "                     horizontal and vertical sampling factor",
      "-size geometry       width and height of image",
      "-verbose             print detailed information about the image",
      "-version             print version information",
      "-virtual-pixel method",
      "                     Constant, Edge, Mirror, or Tile",
      (char *) NULL
    };

  (void) printf("Version: %s\n",GetMagickVersion((unsigned long *) NULL));
  (void) printf("Copyright: %s\n\n",GetMagickCopyright());
  (void) printf("Usage: %s [options ...] image reconstruct difference\n",
    GetClientName());
  (void) printf("\nWhere options include:\n");
  for (p=options; *p != (char *) NULL; p++)
    (void) printf("  %s\n",*p);
  Exit(0);
}

MagickExport unsigned int CompareImageCommand(ImageInfo *image_info,int argc,
  char **argv,char **metadata,ExceptionInfo *exception)
{
#define DestroyCompare() \
{ \
  DestroyImageList(image); \
  for (i=0; i < argc; i++) \
    argv[i]=(char *) RelinquishMagickMemory(argv[i]); \
  argv=(char **) RelinquishMagickMemory(argv); \
}
#define ThrowCompareException(severity,tag,option) \
{ \
  (void) ThrowMagickException(exception,GetMagickModule(),severity,tag,option);\
  DestroyCompare(); \
  return(False); \
}
#define ThrowCompareInvalidArgumentException(option,argument) \
{ \
  (void) ThrowMagickException(exception,GetMagickModule(),OptionError, \
    "InvalidArgument",argument,option); \
  DestroyCompare(); \
  return(False); \
}

  char
    *filename,
    *option;

  ChannelType
    channel;

  double
    distortion;

  Image
    *difference_image,
    *image,
    *reconstruct_image;

  long
    j;

  MetricType
    metric;

  register long
    i;

  unsigned int
    status;

  /*
    Set defaults.
  */
  assert(image_info != (ImageInfo *) NULL);
  assert(image_info->signature == MagickSignature);
  if (image_info->debug != False)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),"");
  assert(exception != (ExceptionInfo *) NULL);
  if (argc < 2)
    CompareUsage();
  channel=AllChannels;
  difference_image=(Image *) NULL;
  distortion=0.0;
  image=(Image *) NULL;
  metric=UndefinedMetric;
  option=(char *) NULL;
  reconstruct_image=(Image *) NULL;
  /*
    Compare an image.
  */
  ReadCommandlLine(argc,&argv);
  status=ExpandFilenames(&argc,&argv);
  if (status == False)
    ThrowCompareException(ResourceLimitError,"MemoryAllocationFailed",
      strerror(errno));
  j=1;
  for (i=1; i < (argc-1); i++)
  {
    option=argv[i];
    if ((strlen(option) == 1) || ((*option != '-') && (*option != '+')))
      {
        /*
          Read input images.
        */
        filename=argv[i];
        (void) CopyMagickString(image_info->filename,filename,MaxTextExtent);
        if (image == (Image *) NULL)
          {
            image=ReadImage(image_info,exception);
            if (exception->severity != UndefinedException)
              CatchException(exception);
            if (image != (Image *) NULL)
              {
                status&=MogrifyImages(image_info,(int) (i-j),argv+j,&image);
                GetImageException(image,exception);
              }
            j=i+1;
            continue;
          }
        if (reconstruct_image != (Image *) NULL)
          ThrowCompareException(OptionError,"InputImagesAlreadySpecified",
            filename);
        reconstruct_image=ReadImage(image_info,exception);
        if (exception->severity != UndefinedException)
          CatchException(exception);
        status&=reconstruct_image != (Image *) NULL;
        continue;
      }
    switch (*(option+1))
    {
      case 'c':
      {
        if (LocaleCompare("cache",option+1) == 0)
          {
            if (*option == '-')
              {
                unsigned long
                  limit;

                i++;
                if (i == argc)
                  ThrowCompareException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == False)
                  ThrowCompareInvalidArgumentException(option,argv[i]);
                limit=(~0UL);
                if (LocaleCompare("unlimited",argv[i]) != 0)
                  limit=(unsigned long) atol(argv[i]);
                (void) SetMagickResourceLimit(MemoryResource,limit);
                (void) SetMagickResourceLimit(MapResource,2*limit);
              }
            break;
          }
        if (LocaleCompare("channel",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == argc)
                  ThrowCompareException(OptionError,"MissingArgument",option);
                channel=(ChannelType) ParseMagickOption(MagickChannelOptions,
                  True,argv[i]);
                if (channel <= UndefinedChannel)
                  ThrowCompareException(OptionError,"UnrecognizedChannelType",
                    argv[i]);
              }
            break;
          }
        if (LocaleCompare("colorspace",option+1) == 0)
          {
            image_info->colorspace=UndefinedColorspace;
            if (*option == '-')
              {
                long
                  colorspace;

                i++;
                if (i == (argc-1))
                  ThrowCompareException(OptionError,"MissingArgument",option);
                colorspace=ParseMagickOption(MagickColorspaceOptions,False,
                  argv[i]);
                if (colorspace < 0)
                  ThrowCompareException(OptionError,"UnrecognizedColorspace",
                    argv[i]);
                image_info->colorspace=(ColorspaceType) colorspace;
              }
            break;
          }
        if (LocaleCompare("compress",option+1) == 0)
          {
            image_info->compression=UndefinedCompression;
            if (*option == '-')
              {
                long
                  compression;

                i++;
                if (i == (argc-1))
                  ThrowCompareException(OptionError,"MissingArgument",option);
                compression=ParseMagickOption(MagickCompressionOptions,False,
                  argv[i]);
                if (compression < 0)
                  ThrowCompareException(OptionError,
                    "UnrecognizedImageCompression",argv[i]);
                image_info->compression=(CompressionType) compression;
              }
            break;
          }
        ThrowCompareException(OptionError,"UnrecognizedOption",option)
      }
      case 'd':
      {
        if (LocaleCompare("debug",option+1) == 0)
          {
            (void) SetLogEventMask("None");
            if (*option == '-')
              {
                LogEventType
                  event_mask;

                i++;
                if (i == argc)
                  ThrowCompareException(OptionError,"MissingArgument",option);
                event_mask=SetLogEventMask(argv[i]);
                if (event_mask == UndefinedEvents)
                  ThrowCompareException(OptionError,"UnrecognizedEventType",
                    option);
              }
            image_info->debug=IsEventLogging();
            break;
          }
        if (LocaleCompare("define",option+1) == 0)
          {
            i++;
            if (i == argc)
              ThrowCompareException(OptionError,"MissingArgument",option);
            if (*option == '+')
              {
                char
                  *define;

                define=RemoveImageOption(image_info,argv[i]);
                if (define == (char *) NULL)
                  ThrowCompareException(OptionError,"NoSuchOption",argv[i]);
                define=(char *) RelinquishMagickMemory(define);
                break;
              }
            status=DefineImageOption(image_info,argv[i]);
            if (status == False)
              ThrowCompareException(OptionError,"UnrecognizedOption",argv[i]);
            break;
          }
        if (LocaleCompare("density",option+1) == 0)
          {
            (void) CloneString(&image_info->density,(char *) NULL);
            if (*option == '-')
              {
                i++;
                if (i == argc)
                  ThrowCompareException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == False)
                  ThrowCompareInvalidArgumentException(option,argv[i]);
                (void) CloneString(&image_info->density,argv[i]);
              }
            break;
          }
        if (LocaleCompare("depth",option+1) == 0)
          {
            image_info->depth=QuantumDepth;
            if (*option == '-')
              {
                i++;
                if (i == argc)
                  ThrowCompareException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == False)
                  ThrowCompareInvalidArgumentException(option,argv[i]);
                image_info->depth=(unsigned long) atol(argv[i]);
              }
            break;
          }
        ThrowCompareException(OptionError,"UnrecognizedOption",option)
      }
      case 'f':
      {
        if (LocaleCompare("format",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == argc)
                  ThrowCompareException(OptionError,"MissingArgument",option);
              }
            break;
          }
        ThrowCompareException(OptionError,"UnrecognizedOption",option)
      }
      case 'h':
      {
        if (LocaleCompare("help",option+1) == 0)
          CompareUsage();
        ThrowCompareException(OptionError,"UnrecognizedOption",option)
      }
      case 'i':
      {
        if (LocaleCompare("interlace",option+1) == 0)
          {
            image_info->interlace=UndefinedInterlace;
            if (*option == '-')
              {
                long
                  interlace;

                i++;
                if (i == argc)
                  ThrowCompareException(OptionError,"MissingArgument",option);
                interlace=ParseMagickOption(MagickInterlaceOptions,False,
                  argv[i]);
                if (interlace < 0)
                  ThrowCompareException(OptionError,"UnrecognizedInterlaceType",
                    argv[i]);
                image->interlace=(InterlaceType) interlace;
              }
            break;
          }
        ThrowCompareException(OptionError,"UnrecognizedOption",option)
      }
      case 'l':
      {
        if (LocaleCompare("limit",option+1) == 0)
          {
            if (*option == '-')
              {
                char
                  *type;

                long
                  resource;

                unsigned long
                  limit;

                i++;
                if (i == argc)
                  ThrowCompareException(OptionError,"MissingArgument",option);
                type=argv[i];
                i++;
                if (i == argc)
                  ThrowCompareException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == False)
                  ThrowCompareInvalidArgumentException(option,argv[i]);
                limit=(~0UL);
                if (LocaleCompare("unlimited",argv[i]) != 0)
                  limit=(unsigned long) atol(argv[i]);
                resource=ParseMagickOption(MagickResourceOptions,False,type);
                if (resource < 0)
                  ThrowCompareException(OptionError,
                    "UnrecognizedResourceType",type);
                (void) SetMagickResourceLimit((ResourceType) resource,limit);
                break;
              }
            break;
          }
        if (LocaleCompare("log",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if ((i == argc) || (strchr(argv[i],'%') == (char *) NULL))
                  ThrowCompareException(OptionError,"MissingArgument",option);
                (void) SetLogFormat(argv[i]);
              }
            break;
          }
        ThrowCompareException(OptionError,"UnrecognizedOption",option)
      }
      case 'm':
      {
        if (LocaleCompare("metric",option+1) == 0)
          {
            if (*option == '-')
              {
                i++;
                if (i == argc)
                  ThrowCompareException(OptionError,"MissingArgument",option);
                metric=(MetricType) ParseMagickOption(MagickMetricOptions,True,
                  argv[i]);
                if (metric <= UndefinedMetric)
                  ThrowCompareException(OptionError,"UnrecognizedMetricType",
                    argv[i]);
              }
            break;
          }
        ThrowCompareException(OptionError,"UnrecognizedOption",option)
      }
      case 'p':
      {
        if (LocaleCompare("profile",option+1) == 0)
          {
            i++;
            if (i == (argc-1))
              ThrowCompareException(OptionError,"MissingArgument",option);
            break;
          }
        ThrowCompareException(OptionError,"UnrecognizedOption",option)
      }
      case 'q':
      {
        if (LocaleCompare("quality",option+1) == 0)
          {
            image_info->quality=UndefinedCompressionQuality;
            if (*option == '-')
              {
                i++;
                if (i == (argc-1))
                  ThrowCompareException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == False)
                  ThrowCompareInvalidArgumentException(option,argv[i]);
                image_info->quality=(unsigned long) atol(argv[i]);
              }
            break;
          }
        ThrowCompareException(OptionError,"UnrecognizedOption",option)
      }
      case 's':
      {
        if (LocaleCompare("sampling-factor",option+1) == 0)
          {
            (void) CloneString(&image_info->sampling_factor,(char *) NULL);
            if (*option == '-')
              {
                i++;
                if (i == argc)
                  ThrowCompareException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == False)
                  ThrowCompareInvalidArgumentException(option,argv[i]);
                (void) CloneString(&image_info->sampling_factor,argv[i]);
              }
            break;
          }
        if (LocaleCompare("size",option+1) == 0)
          {
            (void) CloneString(&image_info->size,(char *) NULL);
            if (*option == '-')
              {
                i++;
                if (i == argc)
                  ThrowCompareException(OptionError,"MissingArgument",option);
                if (IsGeometry(argv[i]) == False)
                  ThrowCompareInvalidArgumentException(option,argv[i]);
                (void) CloneString(&image_info->size,argv[i]);
              }
            break;
          }
        ThrowCompareException(OptionError,"UnrecognizedOption",option)
      }
      case 'v':
      {
        if (LocaleCompare("verbose",option+1) == 0)
          {
            image_info->verbose=(unsigned int) (*option == '-');
            break;
          }
        if (LocaleCompare("version",option+1) == 0)
          break;
        if (LocaleCompare("virtual-pixel",option+1) == 0)
          {
            if (*option == '-')
              {
                long
                  method;

                i++;
                if (i == (argc-1))
                  ThrowCompareException(OptionError,"MissingArgument",option);
                method=ParseMagickOption(MagickVirtualPixelOptions,False,
                  argv[i]);
                if (method < 0)
                  ThrowCompareException(OptionError,
                    "UnrecognizedVirtualPixelMethod",argv[i]);
              }
            break;
          }
        ThrowCompareException(OptionError,"UnrecognizedOption",option)
      }
      case '?':
        break;
      default:
        ThrowCompareException(OptionError,"UnrecognizedOption",option)
    }
  }
  if (i != (argc-1))
    ThrowCompareException(OptionError,"MissingAnImageFilename",argv[i]);
  if ((image == (Image *) NULL) || (reconstruct_image == (Image *) NULL))
    ThrowCompareException(OptionError,"MissingAnImageFilename",argv[i]);
  status&=MogrifyImages(image_info,(int) (i-j),argv+j,&reconstruct_image);
  GetImageException(image,exception);
  difference_image=CompareImageChannels(image,reconstruct_image,channel,
    metric,&distortion,exception);
  if (difference_image != (Image *) NULL)
    {
      if (image_info->verbose != False)
        (void) IsImagesEqual(image,reconstruct_image);
      difference_image->error=image->error;
      status&=WriteImages(image_info,difference_image,argv[argc-1],exception);
      DestroyImageList(difference_image);
    }
  if (metric != UndefinedMetric)
    (void) fprintf(stdout,"%g dB\n",distortion);
  DestroyCompare();
  return(status);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%  I s I m a g e s E q u a l                                                  %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  IsImagesEqual() measures the difference between colors at each pixel
%  location of two images.  A value other than 0 means the colors match
%  exactly.  Otherwise an error measure is computed by summing over all
%  pixels in an image the distance squared in RGB space between each image
%  pixel and its corresponding pixel in the reconstruct image.  The error
%  measure is assigned to these image members:
%
%    o mean_error_per_pixel:  The mean error for any single pixel in
%      the image.
%
%    o normalized_mean_error:  The normalized mean quantization error for
%      any single pixel in the image.  This distance measure is normalized to
%      a range between 0 and 1.  It is independent of the range of red, green,
%      and blue values in the image.
%
%    o normalized_maximum_error:  The normalized maximum quantization
%      error for any single pixel in the image.  This distance measure is
%      normalized to a range between 0 and 1.  It is independent of the range
%      of red, green, and blue values in your image.
%
%  A small normalized mean square error, accessed as
%  image->normalized_mean_error, suggests the images are very similiar in
%  spatial layout and color.
%
%  The format of the IsImagesEqual method is:
%
%      unsigned int IsImagesEqual(Image *image,const Image *reconstruct_image)
%
%  A description of each parameter follows.
%
%    o image: The image.
%
%    o reconstruct_image: The reconstruct image.
%
*/
MagickExport unsigned int IsImagesEqual(Image *image,
  const Image *reconstruct_image)
{
  IndexPacket
    *indexes,
    *reconstruct_indexes;

  long
    y;

  MagickRealType
    area,
    distance,
    maximum_error,
    mean_error,
    mean_error_per_pixel;

  register const PixelPacket
    *p,
    *q;

  register long
    x;

  unsigned int
    status;

  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  assert(reconstruct_image != (const Image *) NULL);
  assert(reconstruct_image->signature == MagickSignature);
  if ((reconstruct_image->columns != image->columns) ||
      (reconstruct_image->rows != image->rows))
    ThrowBinaryException(ImageError,"ImageSizeDiffers",image->filename);
  if (image->colorspace != reconstruct_image->colorspace)
    ThrowBinaryException(ImageError,"ImageColorspaceDiffers",image->filename);
  if (image->matte != reconstruct_image->matte)
    ThrowBinaryException(ImageError,"ImageOpacityDiffers",image->filename);
  area=0.0;
  maximum_error=0.0;
  mean_error_per_pixel=0.0;
  mean_error=0.0;
  for (y=0; y < (long) image->rows; y++)
  {
    p=AcquireImagePixels(image,0,y,image->columns,1,&image->exception);
    q=AcquireImagePixels(reconstruct_image,0,y,reconstruct_image->columns,1,
      &image->exception);
    if ((p == (const PixelPacket *) NULL) || (q == (const PixelPacket *) NULL))
      break;
    indexes=GetIndexes(image);
    reconstruct_indexes=GetIndexes(reconstruct_image);
    for (x=0; x < (long) image->columns; x++)
    {
      distance=fabs(p->red-(MagickRealType) q->red);
      mean_error_per_pixel+=distance;
      mean_error+=distance*distance;
      if (distance > maximum_error)
        maximum_error=distance;
      area++;
      distance=fabs(p->green-(MagickRealType) q->green);
      mean_error_per_pixel+=distance;
      mean_error+=distance*distance;
      if (distance > maximum_error)
        maximum_error=distance;
      area++;
      distance=fabs(p->blue-(MagickRealType) q->blue);
      mean_error_per_pixel+=distance;
      mean_error+=distance*distance;
      if (distance > maximum_error)
        maximum_error=distance;
      area++;
      if (image->matte != False)
        {
          distance=fabs(p->opacity-(MagickRealType) q->opacity);
          mean_error_per_pixel+=distance;
          mean_error+=distance*distance;
          if (distance > maximum_error)
            maximum_error=distance;
          area++;
        }
      if (image->colorspace == CMYKColorspace)
        {
          distance=fabs(indexes[x]-(MagickRealType) reconstruct_indexes[x]);
          mean_error_per_pixel+=distance;
          mean_error+=distance*distance;
          if (distance > maximum_error)
            maximum_error=distance;
          area++;
        }
      p++;
      q++;
    }
  }
  image->error.mean_error_per_pixel=mean_error_per_pixel/area;
  image->error.normalized_mean_error=mean_error/area/MaxRGB/MaxRGB;
  image->error.normalized_maximum_error=maximum_error/MaxRGB;
  status=(unsigned int)
    (image->error.mean_error_per_pixel == 0.0 ? True : False);
  return(status);
}
