/*
 *  filter_logo.c
 *
 *  Copyright (C) Tilmann Bitterberg - April 2002
 *
 *  This file is part of transcode, a linux video stream processing tool
 *      
 *  transcode is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  transcode is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *
 */

    /* TODO: 
       - animated gif/png support / sequences of jpgs maybe
	  would be nice.
     */

#define MOD_NAME    "filter_logo.so"
#define MOD_VERSION "v0.4 (07/04/2002)"
#define MOD_CAP     "render image in videostream"

#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <unistd.h>
#include <magick/api.h>

/* -------------------------------------------------
 *
 * mandatory include files
 *
 *-------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <unistd.h>
#include <inttypes.h>

#include "transcode.h"
#include "framebuffer.h"
#include "../export/vid_aux.h"


// basic parameter

enum POS { NONE, TOP_LEFT, TOP_RIGHT, BOT_LEFT, BOT_RIGHT, CENTER };

typedef struct MyFilterData {
	char file[PATH_MAX];
	int posx;
	int posy;
	enum POS pos;
	int flip;
	unsigned int start, end;
	char *yuv;
} MyFilterData;
	
static MyFilterData *mfd = NULL;

/* from /src/transcode.c */
extern int rgbswap;
extern int flip;
/* should probably honor the other flags too */ 

/*-------------------------------------------------
 *
 * single function interface
 *
 *-------------------------------------------------*/

static void help_optstr(void) 
{
   printf ("[%s] (%s) help\n", MOD_NAME, MOD_CAP);
   printf ("* Overview\n");
   printf("    This filter renders an user specified image into the video.\n");
   printf("    Any image format ImageMagick can read is accepted.\n");
   printf("    Transparent images are also supported.\n");
   printf("    Image origin is at the very top left.\n");

   printf ("* Options\n");
   printf ("   'file' Image filename (required) [logo.png]\n");
   printf ("    'pos' Position (0-width x 0-height) [0x0]\n");
   printf (" 'posdef' Position (0=None, 1=TopL, 2=TopR, 3=BotL, 4=BotR, 5=Center) [0]\n");
   printf ("  'range' Restrict rendering to framerange (0-oo) [0-end]\n");
   printf ("   'flip' Mirror image (0=off, 1=on) [0]\n");
}

int tc_filter(vframe_list_t *ptr, char *options)
{

  static vob_t *vob=NULL;

  static ExceptionInfo exception_info;
  static ImageInfo *image_info;
  static Image *image;
  static PixelPacket *pixel_packet;

  int column, row;
  int rgb_off = 0;
  
  //----------------------------------
  //
  // filter init
  //
  //----------------------------------


  if(ptr->tag & TC_FILTER_INIT) {

    if((vob = tc_get_vob())==NULL) return(-1);

    if((mfd = (MyFilterData *)malloc (sizeof(MyFilterData))) == NULL) return (-1);

    //mfd->file = filename;
    strcpy (mfd->file, "logo.png");
    mfd->flip  = 0;
    mfd->pos   = 0;
    mfd->posx  = 0;
    mfd->posy  = 0;
    mfd->start = 0;
    mfd->end   = (unsigned int)-1;

    if (options != NULL) {
    
	if(verbose) printf("[%s] options=%s\n", MOD_NAME, options);

	optstr_get (options, "file",     "%[^:]s",    &mfd->file);
	optstr_get (options, "flip",     "%d",        &mfd->flip);
	optstr_get (options, "posdef",   "%d",        &mfd->pos);
	optstr_get (options, "pos",      "%dx%d",     &mfd->posx,  &mfd->posy);
	optstr_get (options, "range",    "%u-%u",     &mfd->start, &mfd->end);

	if (optstr_get (options, "help", "") >= 0) {
	    help_optstr();
	}
    }

    if (verbose > 1) {
	printf (" Logo renderer Settings:\n");
	printf ("              file = %s\n", mfd->file);
	printf ("            posdef = %d\n", mfd->pos);
	printf ("               pos = %dx%d\n", mfd->posx, mfd->posy);
	printf ("             range = %u-%u\n", mfd->start, mfd->end);
	printf ("              flip = %d\n", mfd->flip);
    }

    InitializeMagick("");

    GetExceptionInfo (&exception_info);
    image_info = CloneImageInfo ((ImageInfo *) NULL);
    strcpy(image_info->filename, mfd->file);

    image = ReadImage (image_info, &exception_info);
    if (image == (Image *) NULL)
        MagickError (exception_info.severity, exception_info.reason, exception_info.description);


    if (image->columns > vob->ex_v_width || image->rows > vob->ex_v_height) {
	fprintf(stderr, "[%s] \"%s\" is too large", MOD_NAME, mfd->file);
	return(-1);
    }

    if (vob->im_v_codec == CODEC_YUV) {
	if ( (image->columns&1) || (image->rows&1)) {
	    fprintf(stderr, "[%s] \"%s\" has odd sizes", MOD_NAME, mfd->file);
	    return(-1);
	}
    }

    if (mfd->flip || flip) {
	image = FlipImage (image, &exception_info);
	if (image == (Image *) NULL)
	    MagickError (exception_info.severity, exception_info.reason, exception_info.description);
    }

    pixel_packet = GetImagePixels(image, 0, 0, image->columns, image->rows);

    if (vob->im_v_codec == CODEC_YUV) {

	if (!mfd->yuv)
	    mfd->yuv=malloc(sizeof(char)*image->columns*image->rows*3);

	if (tc_rgb2yuv_init(image->columns, image->rows)<0) {
	    fprintf(stderr, "[%s] rgb2yuv init failed\n", MOD_NAME);
	    return(-1); 
	}

	/* convert Magick RGB format to 24bit RGB */
	if (rgbswap) {
	  for (row = 0; row < image->rows; row++) {
	    for (column = 0; column < image->columns; column++) {

		mfd->yuv[(row * image->columns + column) * 3 + 0] =
		    pixel_packet[image->columns*row + column].blue;

		mfd->yuv[(row * image->columns + column) * 3 + 1] =
		    pixel_packet[image->columns*row + column].green;

		mfd->yuv[(row * image->columns + column) * 3 + 2] =
		    pixel_packet[image->columns*row + column].red;
	    }
	  }

	} else {

	  for (row = 0; row < image->rows; row++) {
	    for (column = 0; column < image->columns; column++) {

		mfd->yuv[(row * image->columns + column) * 3 + 0] =
		    pixel_packet[image->columns*row + column].red;

		mfd->yuv[(row * image->columns + column) * 3 + 1] =
		    pixel_packet[image->columns*row + column].green;

		mfd->yuv[(row * image->columns + column) * 3 + 2] =
		    pixel_packet[image->columns*row + column].blue;
	    }
	  }
	}

	if(tc_rgb2yuv_core(mfd->yuv)<0) {
	    fprintf(stderr, "[%s] rgb2yuv conversion failed\n", MOD_NAME);
	    return(-1);
	}

    }  else {
	/* for RGB format is origin bottom left */
	rgb_off = vob->ex_v_height - image->rows;
	mfd->posy = rgb_off - mfd->posy;
    }

    switch (mfd->pos) {
	case NONE: /* 0 */
	    break;
	case TOP_LEFT:
	    mfd->posx = 0;
	    mfd->posy = rgb_off;
	    break;
	case TOP_RIGHT:
	    mfd->posx = vob->ex_v_width  - image->columns;
	    break;
	case BOT_LEFT:
	    mfd->posy = vob->ex_v_height - image->rows - rgb_off;
	    break;
	case BOT_RIGHT:
	    mfd->posx = vob->ex_v_width  - image->columns;
	    mfd->posy = vob->ex_v_height - image->rows - rgb_off;
	    break;
	case CENTER:
	    mfd->posx = (vob->ex_v_width - image->columns)/2;
	    mfd->posy = (vob->ex_v_height- image->rows)/2;
	    break;
    }

 
    
    if ( mfd->posy < 0 || mfd->posx < 0 || 
	    mfd->posx+image->columns > vob->ex_v_width ||
	    mfd->posy+image->rows > vob->ex_v_height) {
	fprintf(stderr, "[%s] invalid position\n", MOD_NAME);
	return (-1);
    }

    // filter init ok.
    if (verbose) printf("[%s] %s %s\n", MOD_NAME, MOD_VERSION, MOD_CAP);

    
    return(0);
  }

  //----------------------------------
  //
  // filter close
  //
  //----------------------------------

  
  if(ptr->tag & TC_FILTER_CLOSE) {

    if (mfd) { 
	if (mfd->yuv)
	    free(mfd->yuv);
	free(mfd);
    }

    if (vob->ex_v_codec == CODEC_YUV)
	tc_rgb2yuv_close();

    DestroyImage(image);
    DestroyImageInfo(image_info);
    DestroyMagick();

    return(0);

  } /* filter close */
  
  //----------------------------------
  //
  // filter frame routine
  //
  //----------------------------------

    
  // tag variable indicates, if we are called before
  // transcodes internal video/audo frame processing routines
  // or after and determines video/audio context
  
  if((ptr->tag & TC_POST_PROCESS) && (ptr->tag & TC_VIDEO))  {
    
    if (ptr->id < mfd->start || ptr->id > mfd->end)
	return (0);
    
    if(vob->im_v_codec==CODEC_RGB) {

	if (rgbswap) {
	  for (row = 0; row < image->rows; row++) {
	    for (column = 0; column < image->columns; column++) {
		if (pixel_packet[(image->rows - row - 1) * image->columns + column].opacity == 0) {
		    
		    int packet_off = (image->rows - row - 1) * image->columns + column;
		    int ptr_off    = ((row+mfd->posy)* vob->ex_v_width + column+mfd->posx) * 3;

		    ptr->video_buf[ptr_off + 0] = pixel_packet[packet_off].red;
		    ptr->video_buf[ptr_off + 1] = pixel_packet[packet_off].green;
		    ptr->video_buf[ptr_off + 2] = pixel_packet[packet_off].blue;
		} /* !opaque */
	    }
	  }
	} else {
	  for (row = 0; row < image->rows; row++) {
	    for (column = 0; column < image->columns; column++) {
		if (pixel_packet[(image->rows - row - 1) * image->columns + column].opacity == 0) {
		    
		    int packet_off = (image->rows - row - 1) * image->columns + column;
		    int ptr_off    = ((row+mfd->posy)* vob->ex_v_width + column+mfd->posx) * 3;

		    ptr->video_buf[ptr_off + 0] = pixel_packet[packet_off].blue;
		    ptr->video_buf[ptr_off + 1] = pixel_packet[packet_off].green;
		    ptr->video_buf[ptr_off + 2] = pixel_packet[packet_off].red;
		} /* !opaque */
	    }
	  }
	}

    } else { /* !RGB */

	int size  = vob->ex_v_width*vob->ex_v_height;
	int block = image->columns*image->rows;
	char *p1, *p2;
	char *y1, *y2;

	/* Y' */
	for (row = 0; row < image->rows; row++) {
	    for (column = 0; column < image->columns; column++) {

		if (pixel_packet[image->columns*row + column].opacity == 0) {

		    *(ptr->video_buf + (row+mfd->posy)*vob->ex_v_width + column + mfd->posx) = 
			mfd->yuv[image->columns*row + column];

		}
	    }
	}

	/* Cb, Cr */
	p1 = ptr->video_buf + size + mfd->posy*vob->ex_v_width/4 + mfd->posx/2;
	y1 = mfd->yuv       + block;
	p2 = ptr->video_buf + 5*size/4 + mfd->posy*vob->ex_v_width/4 + mfd->posx/2;
	y2 = mfd->yuv       + 5*block/4;
	
	for (row = 0; row < image->rows/2; row++) {
	    for (column = 0; column < image->columns/2; column++) {
		if (pixel_packet[image->columns*row*2 + column*2].opacity == 0) {
		    p1[column] = y1[column];
		    p2[column] = y2[column];
		}
	    }

	    p1 += vob->ex_v_width/2;
	    y1 += image->columns/2;

	    p2 += vob->ex_v_width/2;
	    y2 += image->columns/2;
	}

    }
  }
  
  return(0);
}

