/*
 * lib/dyn_pageflags.c
 *
 * Copyright (C) 2004-2006 Nigel Cunningham <nigel@suspend2.net>
 * 
 * This file is released under the GPLv2.
 *
 * Routines for dynamically allocating and releasing bitmaps
 * used as pseudo-pageflags.
 */

#include <linux/module.h>
#include <linux/dyn_pageflags.h>
#include <linux/bootmem.h>
#include <linux/mm.h>

#define page_to_zone_offset(pg) (page_to_pfn(pg) - page_zone(pg)->zone_start_pfn)

/* 
 * pages_for_zone(struct zone *zone)
 * 
 * How many pages do we need for a bitmap for this zone?
 *
 */

static int pages_for_zone(struct zone *zone)
{
	return (zone->spanned_pages + (PAGE_SIZE << 3) - 1) >>
			(PAGE_SHIFT + 3);
}

/* 
 * clear_dyn_pageflags(dyn_pageflags_t pagemap)
 *
 * Clear an array used to store local page flags.
 *
 */

void clear_dyn_pageflags(dyn_pageflags_t pagemap)
{
	int i = 0, zone_num = 0;
	struct zone *zone;
	
	BUG_ON(!pagemap);

	for_each_zone(zone) {
		if (!populated_zone(zone))
			continue;

		zone_num = page_zone_id(pfn_to_page(zone->zone_start_pfn));

		for (i = 0; i < pages_for_zone(zone); i++)
			memset((pagemap[zone_num][i]), 0, PAGE_SIZE);
	}
}

/* 
 * allocate_dyn_pageflags(dyn_pageflags_t *pagemap)
 *
 * Allocate a bitmap for dynamic page flags.
 *
 */
int allocate_dyn_pageflags(dyn_pageflags_t *pagemap)
{
	int i, zone_num, zone_pages;
	struct zone *zone;

	BUG_ON(*pagemap);

	*pagemap = kmalloc(sizeof(void *) * (1 << ZONETABLE_SHIFT), GFP_ATOMIC);

	if (!*pagemap)
		return -ENOMEM;

	for_each_zone(zone) {
		if (!populated_zone(zone))
			continue;

		zone_pages = pages_for_zone(zone);
		zone_num = page_zone_id(pfn_to_page(zone->zone_start_pfn));

		(*pagemap)[zone_num] = kmalloc(sizeof(void *) * zone_pages,
					       GFP_ATOMIC);

		if (!(*pagemap)[zone_num]) {
			free_dyn_pageflags(pagemap);
			return -ENOMEM;
		}

		for (i = 0; i < zone_pages; i++) {
			unsigned long address = get_zeroed_page(GFP_ATOMIC);
			(*pagemap)[zone_num][i] = (unsigned long *) address;
			if (!(*pagemap)[zone_num][i]) {
				printk("Error. Unable to allocate memory for "
					"dynamic pageflags.");
				free_dyn_pageflags(pagemap);
				return -ENOMEM;
			}
		}
	}

	return 0;
}

/* 
 * free_dyn_pageflags(dyn_pageflags_t *pagemap)
 *
 * Free a dynamically allocated pageflags bitmap. For Suspend2 usage, we
 * support data being relocated from slab to pages that don't conflict
 * with the image that will be copied back. This is the reason for the
 * PageSlab tests below.
 *
 */
void free_dyn_pageflags(dyn_pageflags_t *pagemap)
{
	int i = 0, zone_num, zone_pages;
	struct zone *zone;

	if (!*pagemap)
		return;
	
	for_each_zone(zone) {
		if (!populated_zone(zone))
			continue;

		zone_pages = pages_for_zone(zone);
		zone_num = page_zone_id(pfn_to_page(zone->zone_start_pfn));

		/* 
		 * May be called on an error path in allocating, so this
		 * isn't redundant.
		 */
		if (!((*pagemap)[zone_num]))
			continue;

		for (i = 0; i < zone_pages; i++)
			if ((*pagemap)[zone_num][i])
				free_page((unsigned long) (*pagemap)[zone_num][i]);
	
		if (PageSlab(virt_to_page((*pagemap)[zone_num])))
			kfree((*pagemap)[zone_num]);
		else
			free_page((unsigned long) (*pagemap)[zone_num]);
	}

	if (PageSlab(virt_to_page((*pagemap))))
		kfree(*pagemap);
	else
		free_page((unsigned long) (*pagemap));

	*pagemap = NULL;
	return;
}

#define GET_BIT_AND_UL(bitmap, page) \
	unsigned long zone_pfn = page_to_zone_offset(page); \
	int zone_num = page_zone_id(page); \
	int pagenum = PAGENUMBER(zone_pfn); \
	int page_offset = PAGEINDEX(zone_pfn); \
	unsigned long *ul = ((*bitmap)[zone_num][pagenum]) + page_offset; \
	int bit = PAGEBIT(zone_pfn);

/*
 * test_dynpageflag(dyn_pageflags_t *bitmap, struct page *page)
 *
 * Is the page flagged in the given bitmap?
 *
 */

int test_dynpageflag(dyn_pageflags_t *bitmap, struct page *page)
{
	GET_BIT_AND_UL(bitmap, page);
	return test_bit(bit, ul);
}

/*
 * set_dynpageflag(dyn_pageflags_t *bitmap, struct page *page)
 *
 * Set the flag for the page in the given bitmap.
 *
 */

void set_dynpageflag(dyn_pageflags_t *bitmap, struct page *page)
{
	GET_BIT_AND_UL(bitmap, page);
	set_bit(bit, ul);
}

/*
 * clear_dynpageflags(dyn_pageflags_t *bitmap, struct page *page)
 *
 * Clear the flag for the page in the given bitmap.
 *
 */

void clear_dynpageflag(dyn_pageflags_t *bitmap, struct page *page)
{
	GET_BIT_AND_UL(bitmap, page);
	clear_bit(bit, ul);
}

/*
 * get_next_bit_on(dyn_pageflags_t bitmap, int counter)
 *
 * Given a pfn (possibly -1), find the next pfn in the bitmap that
 * is set. If there are no more flags set, return -1.
 *
 */

unsigned long get_next_bit_on(dyn_pageflags_t bitmap, unsigned long counter)
{
	struct page *page;
	struct zone *zone;
	unsigned long *ul = NULL;
	unsigned long zone_offset;
	int pagebit, zone_num, first = (counter == max_pfn);

	if (first)
		counter = first_online_pgdat()->node_zones->zone_start_pfn;

	page = pfn_to_page(counter);
	zone = page_zone(page);
	zone_num = page_zone_id(page);
	zone_offset = counter - zone->zone_start_pfn;

	if (first)
		goto test;

	do {
		zone_offset++;
	
		if (zone_offset >= zone->spanned_pages) {
			do {
				zone = next_zone(zone);
				if (!zone)
					return max_pfn;
			} while(!zone->spanned_pages);
			
			zone_num = page_zone_id(pfn_to_page(zone->zone_start_pfn));
			zone_offset = 0;
		}
test:
		pagebit = PAGEBIT(zone_offset);

		if (!pagebit || !ul)
			ul = (bitmap[zone_num][PAGENUMBER(zone_offset)]) + PAGEINDEX(zone_offset);

		if (!(*ul & ~((1 << pagebit) - 1))) {
			zone_offset += BITS_PER_LONG - pagebit - 1;
			continue;
		}

	} while(!test_bit(pagebit, ul));

	return zone->zone_start_pfn + zone_offset;
}

