/*
 * kernel/power/sysfs.h
 *
 * Copyright (C) 2004-2006 Nigel Cunningham <nigel@suspend2.net>
 *
 * This file is released under the GPLv2.
 *
 * It provides declarations for suspend to use in managing
 * /sysfs/suspend2. When we switch to kobjects,
 * this will become redundant.
 *
 */

#include <linux/sysfs.h>
#include "power.h"

struct suspend_sysfs_data {
	struct attribute attr;
	int type;
	int needs_storage_manager;
	union {
		struct {
			unsigned long *bit_vector;
			int bit;
		} bit;
		struct {
			int *variable;
			int minimum;
			int maximum;
		} integer;
		struct {
			long *variable;
			long minimum;
			long maximum;
		} a_long;
		struct {
			unsigned long *variable;
			unsigned long minimum;
			unsigned long maximum;
		} ul;
		struct {
			char *variable;
			int max_length;
		} string;
		struct {
			int (*read_sysfs) (const char *buffer, int count);
			int (*write_sysfs) (const char *buffer, int count);
			void *data;
		} special;
	} data;
	
	/* Side effects routines. Used, eg, for reparsing the
	 * resume2 entry when it changes */
	void (*read_side_effect) (void);
	void (*write_side_effect) (void); 
	struct list_head sysfs_data_list;
};

enum {
	SUSPEND_SYSFS_DATA_NONE,
	SUSPEND_SYSFS_DATA_CUSTOM,
	SUSPEND_SYSFS_DATA_BIT,
	SUSPEND_SYSFS_DATA_INTEGER,
	SUSPEND_SYSFS_DATA_UL,
	SUSPEND_SYSFS_DATA_LONG,
	SUSPEND_SYSFS_DATA_STRING
};

#define SUSPEND2_ATTR(_name, _mode)      \
        .attr = {.name  = _name , .mode   = _mode }

#define SYSFS_BIT(_ul, _bit) \
	.type = SUSPEND_SYSFS_DATA_BIT, \
	.data = { .bit = { .bit_vector = _ul, .bit = _bit } }

#define SYSFS_INT(_int, _min, _max) \
	.type = SUSPEND_SYSFS_DATA_INTEGER, \
	.data = { .integer = { .variable = _int, .minimum = _min, \
			.maximum = _max } }

#define SYSFS_UL(_ul, _min, _max) \
	.type = SUSPEND_SYSFS_DATA_UL, \
	.data = { .ul = { .variable = _ul, .minimum = _min, \
			.maximum = _max } }

#define SYSFS_LONG(_long, _min, _max) \
	.type = SUSPEND_SYSFS_DATA_LONG, \
	.data = { .a_long = { .variable = _long, .minimum = _min, \
			.maximum = _max } }

#define SYSFS_STRING(_string, _max_len, _needs_usm) \
	.type = SUSPEND_SYSFS_DATA_STRING, \
	.needs_storage_manager = _needs_usm, \
	.data = { .string = { .variable = _string, .max_length = _max_len } }

#define SYSFS_CUSTOM(_read, _write, _needs_storage_manager) \
	.type = SUSPEND_SYSFS_DATA_CUSTOM, \
	.needs_storage_manager = _needs_storage_manager, \
	.data = { .special = { .read_sysfs = _read, .write_sysfs = _write } }

#define SYSFS_WRITEONLY 0200
#define SYSFS_READONLY 0444
#define SYSFS_RW 0644

/* Storage manager needed? */
#define SYSFS_SM_NOT_NEEDED 0
#define SYSFS_NEEDS_FOR_READ 1
#define SYSFS_NEEDS_FOR_WRITE 2
#define SYSFS_NEEDS_FOR_BOTH 3

void suspend_register_sysfs_file(struct kobject *kobj,
		struct suspend_sysfs_data *suspend_sysfs_data);
void suspend_unregister_sysfs_file(struct suspend_sysfs_data *suspend_sysfs_data);

extern struct subsystem suspend2_subsys;

struct kobject *make_suspend2_sysdir(char *name);
