/*
 * kernel/power/suspend_file.c
 *
 * Copyright 2005-2006 Nigel Cunningham <nigel@suspend2.net>
 *
 * Distributed under GPLv2.
 * 
 * This file encapsulates functions for usage of a simple file as a
 * backing store. It is based upon the swapwriter, and shares the
 * same basic working. Here, though, we have nothing to do with
 * swapspace, and only one device to worry about.
 *
 * The user can just
 *
 * echo Suspend2 > /path/to/my_file
 *
 * and
 *
 * echo /path/to/my_file > /sys/power/suspend2/filewriter/target
 *
 * then put what they find in /sys/power/suspend2/resume2
 * as their resume2= parameter in lilo.conf (and rerun lilo if using it).
 *
 * Having done this, they're ready to suspend and resume.
 *
 * TODO:
 * - File resizing.
 */

#include <linux/suspend.h>
#include <linux/module.h>
#include <linux/blkdev.h>
#include <linux/file.h>
#include <linux/stat.h>
#include <linux/mount.h>
#include <linux/statfs.h>
#include <linux/syscalls.h>
#include <linux/namei.h>
#include <linux/fs.h>

#include "suspend.h"
#include "sysfs.h"
#include "modules.h"
#include "ui.h"
#include "extent.h"
#include "io.h"
#include "storage.h"
#include "block_io.h"

static struct suspend_module_ops filewriterops;

/* Details of our target.  */

char filewriter_target[256];
static struct inode *target_inode;
static struct file *target_file;
static struct block_device *filewriter_target_bdev;
static dev_t resume_dev_t;
static int used_devt = 0;
static int setting_filewriter_target = 0;
static sector_t target_firstblock = 0, target_header_start = 0;
static int target_storage_available = 0;
static int target_claim = 0;

static char HaveImage[] = "HaveImage\n";
static char NoImage[] =   "Suspend2\n";
#define sig_size (sizeof(HaveImage) + 1)

struct filewriter_header {
	char sig[sig_size];
	int resumed_before;
	unsigned long first_header_block;
};

extern dev_t ROOT_DEV;
extern char *__initdata root_device_name;

/* Header_pages must be big enough for signature */
static int header_pages, main_pages;

#define target_is_normal_file() (S_ISREG(target_inode->i_mode))

static struct suspend_bdev_info devinfo;

/* Extent chain for blocks */
static struct extent_chain block_chain;

/* Signature operations */
enum {
	GET_IMAGE_EXISTS,
	INVALIDATE,
	MARK_RESUME_ATTEMPTED,
};

static void set_devinfo(struct block_device *bdev, int target_blkbits)
{
	devinfo.bdev = bdev;
	if (!target_blkbits) {
		devinfo.bmap_shift = devinfo.blocks_per_page = 0;
	} else {
		devinfo.bmap_shift = target_blkbits - 9;
		devinfo.blocks_per_page = (1 << (PAGE_SHIFT - target_blkbits));
	}
}

static int filewriter_storage_available(void)
{
	int result = 0;
	struct block_device *bdev=filewriter_target_bdev;

	if (!target_inode)
		return 0;

	switch (target_inode->i_mode & S_IFMT) {
		case S_IFSOCK:
		case S_IFCHR:
		case S_IFIFO: /* Socket, Char, Fifo */
			return -1;
		case S_IFREG: /* Regular file: current size - holes + free
				 space on part */
			result = target_storage_available;
			break;
		case S_IFBLK: /* Block device */
			if (!bdev->bd_disk) {
				printk("bdev->bd_disk null.\n");
				return 0;
			}

			result = (bdev->bd_part ?
				bdev->bd_part->nr_sects :
				bdev->bd_disk->capacity) >> (PAGE_SHIFT - 9);
	}

	return result;
}

static int has_contiguous_blocks(int page_num)
{
	int j;
	sector_t last = 0;

	for (j = 0; j < devinfo.blocks_per_page; j++) {
		sector_t this = bmap(target_inode,
				page_num * devinfo.blocks_per_page + j);

		if (!this || (last && (last + 1) != this))
			break;

		last = this;
	}
			
	return (j == devinfo.blocks_per_page);
}

static int size_ignoring_ignored_pages(void)
{
	int mappable = 0, i;
	
	if (target_is_normal_file()) {
		for (i = 0; i < (target_inode->i_size >> PAGE_SHIFT) ; i++)
			if (has_contiguous_blocks(i))
				mappable++;
	
		return mappable;
	} else
		return filewriter_storage_available();
}

static void __populate_block_list(int min, int max)
{
	if (test_action_state(SUSPEND_TEST_BIO))
		printk("Adding extent %d-%d.\n", min << devinfo.bmap_shift,
		        ((max + 1) << devinfo.bmap_shift) - 1);

	suspend_add_to_extent_chain(&block_chain, min, max);
}

static void populate_block_list(void)
{
	int i;
	
	if (block_chain.first)
		suspend_put_extent_chain(&block_chain);

	if (target_is_normal_file()) {
		int extent_min = -1, extent_max = -1, got_header = 0;

		for (i = 0;
		     i < (target_inode->i_size >> PAGE_SHIFT);
		     i++) {
			sector_t new_sector;

			if (!has_contiguous_blocks(i))
				continue;

			new_sector = bmap(target_inode,
				(i * devinfo.blocks_per_page));

			/* 
			 * Ignore the first block in the file.
			 * It gets the header.
			 */
			if (new_sector == target_firstblock >> devinfo.bmap_shift) {
				got_header = 1;
				continue;
			}

			/* 
			 * I'd love to be able to fill in holes and resize 
			 * files, but not yet...
			 */

			if (new_sector == extent_max + 1)
				extent_max+= devinfo.blocks_per_page;
			else {
				if (extent_min > -1)
					__populate_block_list(extent_min,
							extent_max);

				extent_min = new_sector;
				extent_max = extent_min +
					devinfo.blocks_per_page - 1;
			}
		}
		if (extent_min > -1)
			__populate_block_list(extent_min, extent_max);

		BUG_ON(!got_header);
	} else
		if (target_storage_available > 0)
			__populate_block_list(devinfo.blocks_per_page, 
				(min(main_pages, target_storage_available) + 1) *
			 	devinfo.blocks_per_page - 1);
}

static void filewriter_cleanup(int finishing_cycle)
{
	if (filewriter_target_bdev) {
		if (target_claim) {
			bd_release(filewriter_target_bdev);
			target_claim = 0;
		}

		if (used_devt) {
			blkdev_put(filewriter_target_bdev);
			used_devt = 0;
		}
		filewriter_target_bdev = NULL;
		target_inode = NULL;
		set_devinfo(NULL, 0);
		target_storage_available = 0;
	}

	if (target_file > 0) {
		filp_close(target_file, NULL);
		target_file = NULL;
	}
}

/* 
 * reopen_resume_devt
 *
 * Having opened resume2= once, we remember the major and
 * minor nodes and use them to reopen the bdev for checking
 * whether an image exists (possibly when starting a resume).
 */
static void reopen_resume_devt(void)
{
	filewriter_target_bdev = open_by_devnum(resume_dev_t, FMODE_READ);
	if (IS_ERR(filewriter_target_bdev)) {
		printk("Got a dev_num (%lx) but failed to open it.\n",
				(unsigned long) resume_dev_t);
		return;
	}
	target_inode = filewriter_target_bdev->bd_inode;
	set_devinfo(filewriter_target_bdev, target_inode->i_blkbits);
}

static void filewriter_get_target_info(char *target, int get_size,
		int resume2)
{
	if (target_file)
		filewriter_cleanup(0);

	if (!target || !strlen(target))
		return;

	target_file = filp_open(target, O_RDWR, 0);

	if (IS_ERR(target_file) || !target_file) {

		if (!resume2) {
			printk("Open file %s returned %p.\n",
					target, target_file);
			target_file = NULL;
			return;
		}

		target_file = NULL;
		resume_dev_t = name_to_dev_t(target);
		if (!resume_dev_t) {
			printk("Open file %s returned %p and name_to_devt "
					"failed.\n",
					target, target_file);
			if (!resume_dev_t) {
				struct kstat stat;
				int error = vfs_stat(target, &stat);
				if (error) {
					printk("Stating the file also failed."
						" Nothing more we can do.\n");
					return;
				}
				resume_dev_t = stat.rdev;
			}
			return;
		}
	     	filewriter_target_bdev = open_by_devnum(resume_dev_t, FMODE_READ);
		if (IS_ERR(filewriter_target_bdev)) {
			printk("Got a dev_num (%lx) but failed to open it.\n",
					(unsigned long) resume_dev_t);
			return;
		}
		used_devt = 1;
		target_inode = filewriter_target_bdev->bd_inode;
	} else
		target_inode = target_file->f_mapping->host;

	if (S_ISLNK(target_inode->i_mode) ||
	    S_ISDIR(target_inode->i_mode) ||
	    S_ISSOCK(target_inode->i_mode) ||
	    S_ISFIFO(target_inode->i_mode)) {
		printk("The filewriter works with regular files, character "
				"files and block devices.\n");
		goto cleanup;
	}

	if (!used_devt) {
		if (S_ISBLK(target_inode->i_mode)) {
			filewriter_target_bdev = I_BDEV(target_inode);
			if (!bd_claim(filewriter_target_bdev, &filewriterops))
				target_claim = 1;
		} else
			filewriter_target_bdev = target_inode->i_sb->s_bdev;
		resume_dev_t = filewriter_target_bdev->bd_dev;
	}

	set_devinfo(filewriter_target_bdev, target_inode->i_blkbits);

	if (get_size)
		target_storage_available = size_ignoring_ignored_pages();

	if (!resume2)
		target_firstblock = bmap(target_inode, 0) << devinfo.bmap_shift;
	
	return;
cleanup:
	target_inode = NULL;
	if (target_file) {
		filp_close(target_file, NULL);
		target_file = NULL;
	}
	set_devinfo(NULL, 0);
	target_storage_available = 0;
}

static int parse_signature(struct filewriter_header *header)
{
	int have_image = !memcmp(HaveImage, header->sig, sizeof(HaveImage) - 1);
	int no_image_header = !memcmp(NoImage, header->sig, sizeof(NoImage) - 1);

	if (no_image_header)
		return 0;

	if (!have_image)
		return -1;

	if (header->resumed_before)
		set_suspend_state(SUSPEND_RESUMED_BEFORE);
	else
		clear_suspend_state(SUSPEND_RESUMED_BEFORE);

	target_header_start = header->first_header_block;
	return 1;
}

/* prepare_signature */

static int prepare_signature(struct filewriter_header *current_header,
		unsigned long first_header_block)
{
	strncpy(current_header->sig, HaveImage, sizeof(HaveImage));
	current_header->resumed_before = 0;
	current_header->first_header_block = first_header_block;
	return 0;
}

static int filewriter_storage_allocated(void)
{
	int result;

	if (!target_inode)
		return 0;

	if (target_is_normal_file()) {
		result = (int) target_storage_available;
	} else
		result = header_pages + main_pages;

	return result;
}

static int filewriter_release_storage(void)
{
	if ((test_action_state(SUSPEND_KEEP_IMAGE)) &&
	     test_suspend_state(SUSPEND_NOW_RESUMING))
		return 0;

	suspend_put_extent_chain(&block_chain);

	header_pages = main_pages = 0;
	return 0;
}

static int filewriter_allocate_header_space(int space_requested)
{
	int i;

	if (!block_chain.first)
		return 0;

	suspend_extent_state_goto_start(&suspend_writer_posn);
	suspend_bio_ops.forward_one_page(); /* To first page */
	
	for (i = 0; i < space_requested; i++)
		if (suspend_bio_ops.forward_one_page())
			return -ENOSPC;

	/* The end of header pages will be the start of pageset 2 */
	suspend_extent_state_save(&suspend_writer_posn,
			&suspend_writer_posn_save[2]);
	header_pages = space_requested;
	return 0;
}

static int filewriter_allocate_storage(int space_requested)
{
	int result = 0, prev_header_pages;
	int blocks_to_get = (space_requested << devinfo.bmap_shift) -
		block_chain.size;
	
	/* Only release_storage reduces the size */
	if (blocks_to_get < 1)
		return 0;

	main_pages = space_requested;

	populate_block_list();

	suspend_message(SUSPEND_WRITER, SUSPEND_MEDIUM, 0,
		"Finished with block_chain.size == %d.\n",
		block_chain.size);

	if (block_chain.size < (header_pages + main_pages)) {
		printk("Block chain size (%d) < header pages (%d) + main pages (%d) (=%d).\n",
				block_chain.size,
				header_pages, main_pages,
				header_pages + main_pages);
		result = -ENOSPC;
	}

	prev_header_pages = header_pages;
	header_pages = 0;
	filewriter_allocate_header_space(prev_header_pages);
	return result;
}

static int filewriter_write_header_init(void)
{
	suspend_extent_state_goto_start(&suspend_writer_posn);

	suspend_writer_buffer = (char *) get_zeroed_page(GFP_ATOMIC);
	suspend_writer_buffer_posn = suspend_header_bytes_used = 0;

	/* Info needed to bootstrap goes at the start of the header.
	 * First we save the basic info needed for reading, including the number
	 * of header pages. Then we save the structs containing data needed
	 * for reading the header pages back.
	 * Note that even if header pages take more than one page, when we
	 * read back the info, we will have restored the location of the
	 * next header page by the time we go to use it.
	 */

	suspend_bio_ops.rw_header_chunk(WRITE, &filewriterops,
			(char *) &suspend_writer_posn_save, 
			sizeof(suspend_writer_posn_save));

	suspend_bio_ops.rw_header_chunk(WRITE, &filewriterops,
			(char *) &devinfo, sizeof(devinfo));

	suspend_serialise_extent_chain(&filewriterops, &block_chain);
	
	return 0;
}

static int filewriter_write_header_cleanup(void)
{
	struct filewriter_header *header;

	/* Write any unsaved data */
	if (suspend_writer_buffer_posn)
		suspend_bio_ops.write_header_chunk_finish();

	suspend_bio_ops.finish_all_io();

	suspend_extent_state_goto_start(&suspend_writer_posn);
	suspend_bio_ops.forward_one_page();

	/* Adjust image header */
	suspend_bio_ops.bdev_page_io(READ, filewriter_target_bdev,
			target_firstblock,
			virt_to_page(suspend_writer_buffer));

	header = (struct filewriter_header *) suspend_writer_buffer;

	prepare_signature(header,
			suspend_writer_posn.current_offset <<
			devinfo.bmap_shift);
		
	suspend_bio_ops.bdev_page_io(WRITE, filewriter_target_bdev,
			target_firstblock,
			virt_to_page(suspend_writer_buffer));

	free_page((unsigned long) suspend_writer_buffer);
	suspend_writer_buffer = NULL;
	
	suspend_bio_ops.finish_all_io();

	return 0;
}

/* HEADER READING */

#ifdef CONFIG_DEVFS_FS
int  create_dev(char *name, dev_t dev, char *devfs_name);
#else
static int create_dev(char *name, dev_t dev, char *devfs_name)
{
	sys_unlink(name);
	return sys_mknod(name, S_IFBLK|0600, new_encode_dev(dev));
}
#endif

static int rd_init(void)
{
	suspend_writer_buffer_posn = 0;

	create_dev("/dev/root", ROOT_DEV, root_device_name);
	create_dev("/dev/ram", MKDEV(RAMDISK_MAJOR, 0), NULL);

	suspend_read_fd = sys_open("/dev/root", O_RDONLY, 0);
	if (suspend_read_fd < 0)
		goto out;
	
	sys_read(suspend_read_fd, suspend_writer_buffer, BLOCK_SIZE);

	memcpy(&suspend_writer_posn_save,
		suspend_writer_buffer + suspend_writer_buffer_posn,
		sizeof(suspend_writer_posn_save));
	
	suspend_writer_buffer_posn += sizeof(suspend_writer_posn_save);

	return 0;
out:
	sys_unlink("/dev/ram");
	sys_unlink("/dev/root");
	return -EIO;
}

static int file_init(void)
{
	suspend_writer_buffer_posn = 0;

	/* Read filewriter configuration */
	suspend_bio_ops.bdev_page_io(READ, filewriter_target_bdev,
			target_header_start,
			virt_to_page((unsigned long) suspend_writer_buffer));
	
	return 0;
}

/*
 * read_header_init()
 * 
 * Ramdisk support based heavily on init/do_mounts_rd.c
 *
 * Description:
 * 1. Attempt to read the device specified with resume2=.
 * 2. Check the contents of the header for our signature.
 * 3. Warn, ignore, reset and/or continue as appropriate.
 * 4. If continuing, read the filewriter configuration section
 *    of the header and set up block device info so we can read
 *    the rest of the header & image.
 *
 * Returns:
 * May not return if user choose to reboot at a warning.
 * -EINVAL if cannot resume at this time. Booting should continue
 * normally.
 */

static int filewriter_read_header_init(void)
{
	int result;
	struct block_device *tmp;

	suspend_writer_buffer = (char *) get_zeroed_page(GFP_ATOMIC);
	
	if (test_suspend_state(SUSPEND_TRY_RESUME_RD))
		result = rd_init();
	else
		result = file_init();
	
	if (result) {
		printk("Filewriter read header init: Failed to initialise "
				"reading the first page of data.\n");
		return result;
	}

	memcpy(&suspend_writer_posn_save,
	       suspend_writer_buffer + suspend_writer_buffer_posn,
	       sizeof(suspend_writer_posn_save));
	
	suspend_writer_buffer_posn += sizeof(suspend_writer_posn_save);

	tmp = devinfo.bdev;

	memcpy(&devinfo,
	       suspend_writer_buffer + suspend_writer_buffer_posn,
	       sizeof(devinfo));

	devinfo.bdev = tmp;
	suspend_writer_buffer_posn += sizeof(devinfo);

	suspend_extent_state_goto_start(&suspend_writer_posn);
	suspend_bio_ops.set_extra_page_forward();

	suspend_header_bytes_used = suspend_writer_buffer_posn;

	suspend_load_extent_chain(&block_chain);
	
	return 0;
}

static int filewriter_read_header_cleanup(void)
{
	free_page((unsigned long) suspend_writer_buffer);
	suspend_writer_buffer = NULL;
	return 0;
}

static int filewriter_signature_op(int op)
{
	char *cur;
	int result = 0, changed = 0;
	struct filewriter_header *header;
	
	if(filewriter_target_bdev <= 0)
		return -1;

	cur = (char *) get_zeroed_page(GFP_ATOMIC);
	if (!cur) {
		printk("Unable to allocate a page for reading the image "
				"signature.\n");
		return -ENOMEM;
	}

	suspend_bio_ops.bdev_page_io(READ, filewriter_target_bdev,
			target_firstblock,
			virt_to_page(cur));

	header = (struct filewriter_header *) cur;
	result = parse_signature(header);
		
	switch (op) {
		case INVALIDATE:
			if (result == -1)
				goto out;

			strcpy(header->sig, NoImage);
			header->resumed_before = 0;
			result = changed = 1;
			break;
		case MARK_RESUME_ATTEMPTED:
			if (result == 1) {
				header->resumed_before = 1;
				changed = 1;
			}
			break;
	}

	if (changed)
		suspend_bio_ops.bdev_page_io(WRITE, filewriter_target_bdev,
				target_firstblock,
				virt_to_page(cur));

out:
	suspend_bio_ops.finish_all_io();
	free_page((unsigned long) cur);
	return result;
}

/* Print debug info
 *
 * Description:
 */

static int filewriter_print_debug_stats(char *buffer, int size)
{
	int len = 0;
	
	if (suspend_active_writer != &filewriterops) {
		len = snprintf_used(buffer, size, "- Filewriter inactive.\n");
		return len;
	}

	len = snprintf_used(buffer, size, "- Filewriter active.\n");

	len+= snprintf_used(buffer+len, size-len, "  Storage available for image: "
			"%ld pages.\n",
			filewriter_storage_allocated());

	return len;
}

/*
 * Storage needed
 *
 * Returns amount of space in the image header required
 * for the filewriter's data.
 *
 * We ensure the space is allocated, but actually save the
 * data from write_header_init and therefore don't also define a
 * save_config_info routine.
 */
static unsigned long filewriter_storage_needed(void)
{
	return sig_size + strlen(filewriter_target) + 1 +
		3 * sizeof(struct extent_iterate_saved_state) +
		sizeof(devinfo) +
		sizeof(struct extent_chain) - 2 * sizeof(void *) +
		(2 * sizeof(unsigned long) *
		 (block_chain.allocs - block_chain.frees));
}

/* 
 * filewriter_invalidate_image
 * 
 */
static int filewriter_invalidate_image(void)
{
	int result;

	if (nr_suspends > 0)
		filewriter_release_storage();

	result = filewriter_signature_op(INVALIDATE);
	if (result == 1 && !nr_suspends)
		printk(KERN_WARNING name_suspend "Image invalidated.\n");

	return result;
}

/*
 * Image_exists
 *
 */

static int filewriter_image_exists(void)
{
	if (!filewriter_target_bdev)
		reopen_resume_devt();

	return filewriter_signature_op(GET_IMAGE_EXISTS);
}

/*
 * Mark resume attempted.
 *
 * Record that we tried to resume from this image.
 */

static void filewriter_mark_resume_attempted(void)
{
	filewriter_signature_op(MARK_RESUME_ATTEMPTED);
}

static void filewriter_set_resume2(void)
{
	char *buffer = (char *) get_zeroed_page(GFP_ATOMIC);
	char *buffer2 = (char *) get_zeroed_page(GFP_ATOMIC);
	unsigned long sector = bmap(target_inode, 0);
	int offset = 0;

	if (filewriter_target_bdev) {
		set_devinfo(filewriter_target_bdev, target_inode->i_blkbits);

		bdevname(filewriter_target_bdev, buffer2);
		offset += snprintf(buffer + offset, PAGE_SIZE - offset, 
				"/dev/%s", buffer2);
		
		if (sector)
			offset += snprintf(buffer + offset, PAGE_SIZE - offset,
				":0x%lx", sector << devinfo.bmap_shift);
	} else
		offset += snprintf(buffer + offset, PAGE_SIZE - offset,
				"%s is not a valid target.", filewriter_target);
			
	sprintf(resume2_file, "file:%s", buffer);

	free_page((unsigned long) buffer);
	free_page((unsigned long) buffer2);

	suspend_attempt_to_parse_resume_device();
}

static int __test_filewriter_target(char *target, int resume_time)
{
	filewriter_get_target_info(target, 0, resume_time);
	if (filewriter_signature_op(GET_IMAGE_EXISTS) > -1) {
		printk(name_suspend "Filewriter: File signature found.\n");
		if (!resume_time)
			filewriter_set_resume2();
		
		suspend_bio_ops.set_devinfo(&devinfo);
		suspend_writer_posn.chains = &block_chain;
		suspend_writer_posn.num_chains = 1;

		set_suspend_state(SUSPEND_CAN_SUSPEND);
		return 0;
	}

	clear_suspend_state(SUSPEND_CAN_SUSPEND);

	if (*target)
		printk(name_suspend
			"Filewriter: Sorry. No signature found at %s.\n",
			target);
	else
		if (!resume_time)
			printk(name_suspend
				"Filewriter: Sorry. Target is not set for suspending.\n");

	return 1;
}

static void test_filewriter_target(void)
{
	int cant_suspend;

	setting_filewriter_target = 1;
       	
	printk(name_suspend "Filewriter: Testing whether you can suspend:\n");
	cant_suspend =__test_filewriter_target(filewriter_target, 0);

	printk(name_suspend "Suspending %sabled.\n",  cant_suspend ? "dis" : "en");
	
	setting_filewriter_target = 0;
}

/*
 * Parse Image Location
 *
 * Attempt to parse a resume2= parameter.
 * Swap Writer accepts:
 * resume2=file:DEVNAME[:FIRSTBLOCK]
 *
 * Where:
 * DEVNAME is convertable to a dev_t by name_to_dev_t
 * FIRSTBLOCK is the location of the first block in the file.
 * BLOCKSIZE is the logical blocksize >= SECTOR_SIZE & <= PAGE_SIZE, 
 * mod SECTOR_SIZE == 0 of the device.
 * Data is validated by attempting to read a header from the
 * location given. Failure will result in filewriter refusing to
 * save an image, and a reboot with correct parameters will be
 * necessary.
 */

static int filewriter_parse_sig_location(char *commandline, int only_writer)
{
	char *thischar, *devstart = NULL, *colon = NULL, *at_symbol = NULL;
	int result = -EINVAL, target_blocksize = 0;

	if (strncmp(commandline, "file:", 5)) {
		if (!only_writer)
			return 1;
	} else
		commandline += 5;

	/* 
	 * Don't check signature again if we're beginning a cycle. If we already
	 * did the initialisation successfully, assume we'll be okay when it comes
	 * to resuming.
	 */
	if (filewriter_target_bdev)
		return 0;
	
	devstart = thischar = commandline;
	while ((*thischar != ':') && (*thischar != '@') &&
		((thischar - commandline) < 250) && (*thischar))
		thischar++;

	if (*thischar == ':') {
		colon = thischar;
		*colon = 0;
		thischar++;
	}

	while ((*thischar != '@') && ((thischar - commandline) < 250) && (*thischar))
		thischar++;

	if (*thischar == '@') {
		at_symbol = thischar;
		*at_symbol = 0;
	}
	
	/* 
	 * For the filewriter, you can be able to resume, but not suspend,
	 * because the resume2= is set correctly, but the filewriter_target
	 * isn't. 
	 *
	 * We may have come here as a result of setting resume2 or
	 * filewriter_target. We only test the filewriter target in the
	 * former case (it's already done in the later), and we do it before
	 * setting the block number ourselves. It will overwrite the values
	 * given on the command line if we don't.
	 */

	if (!setting_filewriter_target)
		__test_filewriter_target(filewriter_target, 1);

	if (colon)
		target_firstblock = (int) simple_strtoul(colon + 1, NULL, 0);
	else
		target_firstblock = 0;

	if (at_symbol) {
		target_blocksize = (int) simple_strtoul(at_symbol + 1, NULL, 0);
		if (target_blocksize & (SECTOR_SIZE - 1)) {
			printk("Filewriter: Blocksizes are multiples of %d.\n", SECTOR_SIZE);
			result = -EINVAL;
			goto out;
		}
	}
	
	printk("Suspend2 Filewriter: Testing whether you can resume:\n");

	filewriter_get_target_info(commandline, 0, 1);

	if (!filewriter_target_bdev || IS_ERR(filewriter_target_bdev)) {
		filewriter_target_bdev = NULL;
		result = -1;
		goto out;
	}

	if (target_blocksize)
		set_devinfo(filewriter_target_bdev, ffs(target_blocksize));

	result = __test_filewriter_target(commandline, 1);

out:
	if (result)
		clear_suspend_state(SUSPEND_CAN_SUSPEND);
	else
		set_suspend_state(SUSPEND_CAN_SUSPEND);

	printk("Resuming %sabled.\n",  result ? "dis" : "en");

	if (colon)
		*colon = ':';
	if (at_symbol)
		*at_symbol = '@';

	return result;
}

/* filewriter_save_config_info
 *
 * Description:	Save the target's name, not for resume time, but for all_settings.
 * Arguments:	Buffer:		Pointer to a buffer of size PAGE_SIZE.
 * Returns:	Number of bytes used for saving our data.
 */

static int filewriter_save_config_info(char *buffer)
{
	strcpy(buffer, filewriter_target);
	return strlen(filewriter_target) + 1;
}

/* filewriter_load_config_info
 *
 * Description:	Reload target's name.
 * Arguments:	Buffer:		Pointer to the start of the data.
 *		Size:		Number of bytes that were saved.
 */

static void filewriter_load_config_info(char *buffer, int size)
{
	strcpy(filewriter_target, buffer);
}

static int filewriter_initialise(int starting_cycle)
{
	int result = 0;

	if (starting_cycle) {
	       if (suspend_active_writer != &filewriterops)
			return 0;

		if (!*filewriter_target) {
			printk("Filewriter is the active writer,  but no filename has been set.\n");
			return 1;
		}
	}

	if (filewriter_target)
		filewriter_get_target_info(filewriter_target, starting_cycle, 0);

	if (starting_cycle && (filewriter_image_exists() == -1)) {
		printk("%s is does not have a valid signature for suspending.\n",
				filewriter_target);
		result = 1;
	}

	return result;
}

static struct suspend_sysfs_data filewriter_sysfs_data[] = {

	{
	 SUSPEND2_ATTR("target", SYSFS_RW),
	 SYSFS_STRING(filewriter_target, 256, SYSFS_NEEDS_FOR_WRITE),
	 .write_side_effect		= test_filewriter_target,
	},

	{
	  SUSPEND2_ATTR("enabled", SYSFS_RW),
	  SYSFS_INT(&filewriterops.enabled, 0, 1),
	  .write_side_effect		= attempt_to_parse_resume_device2,
	}
};

static struct suspend_module_ops filewriterops = {
	.type					= WRITER_MODULE,
	.name					= "File Writer",
	.module					= THIS_MODULE,
	.print_debug_info			= filewriter_print_debug_stats,
	.save_config_info			= filewriter_save_config_info,
	.load_config_info			= filewriter_load_config_info,
	.storage_needed				= filewriter_storage_needed,
	.initialise				= filewriter_initialise,
	.cleanup				= filewriter_cleanup,

	.storage_available 	= filewriter_storage_available,
	.storage_allocated	= filewriter_storage_allocated,
	.release_storage	= filewriter_release_storage,
	.allocate_header_space	= filewriter_allocate_header_space,
	.allocate_storage	= filewriter_allocate_storage,
	.image_exists		= filewriter_image_exists,
	.mark_resume_attempted	= filewriter_mark_resume_attempted,
	.write_header_init	= filewriter_write_header_init,
	.write_header_cleanup	= filewriter_write_header_cleanup,
	.read_header_init	= filewriter_read_header_init,
	.read_header_cleanup	= filewriter_read_header_cleanup,
	.invalidate_image	= filewriter_invalidate_image,
	.parse_sig_location	= filewriter_parse_sig_location,
};

/* ---- Registration ---- */
static __init int filewriter_load(void)
{
	int result;
	int i,
	    numfiles = sizeof(filewriter_sysfs_data) / 
		    sizeof(struct suspend_sysfs_data);
	
	printk("Suspend2 FileWriter loading.\n");

	filewriterops.read_chunk = suspend_bio_ops.read_chunk;
	filewriterops.write_chunk = suspend_bio_ops.write_chunk;
	filewriterops.rw_init = suspend_bio_ops.rw_init;
	filewriterops.rw_cleanup = suspend_bio_ops.rw_cleanup;
	filewriterops.rw_header_chunk =
		suspend_bio_ops.rw_header_chunk;

	if (!(result = suspend_register_module(&filewriterops))) {
		struct kobject *kobj = make_suspend2_sysdir("filewriter");
		for (i=0; i< numfiles; i++)
			suspend_register_sysfs_file(kobj,
					&filewriter_sysfs_data[i]);
	} else
		printk("Suspend2 FileWriter unable to register!\n");

	return result;
}

#ifdef MODULE
static __exit void filewriter_unload(void)
{
	int i, numfiles = sizeof(filewriter_sysfs_data) /
		sizeof(struct suspend_sysfs_data);

	printk("Suspend2 FileWriter unloading.\n");

	for (i=0; i< numfiles; i++)
		suspend_unregister_sysfs_file(&filewriter_sysfs_data[i]);
	suspend_unregister_module(&filewriterops);
}

module_init(filewriter_load);
module_exit(filewriter_unload);
MODULE_LICENSE("GPL");
MODULE_AUTHOR("Nigel Cunningham");
MODULE_DESCRIPTION("Suspend2 filewriter");
#else
late_initcall(filewriter_load);
#endif
