/*
 * kernel/power/suspend.c
 */
/** \mainpage Suspend2.
 *
 * Suspend2 provides support for saving and restoring an image of
 * system memory to an arbitrary storage device, either on the local computer,
 * or across some network. The support is entirely OS based, so Suspend2 
 * works without requiring BIOS, APM or ACPI support. The vast majority of the
 * code is also architecture independant, so it should be very easy to port
 * the code to new architectures. Suspend includes support for SMP, 4G HighMem
 * and preemption. Initramfses and initrds are also supported.
 *
 * Suspend2 uses a modular design, in which the method of storing the image is
 * completely abstracted from the core code, as are transformations on the data
 * such as compression and/or encryption (multiple 'modules' can be used to
 * provide arbitrary combinations of functionality). The user interface is also
 * modular, so that arbitrarily simple or complex interfaces can be used to
 * provide anything from debugging information through to eye candy.
 * 
 * \section Copyright
 *
 * Suspend2 is released under the GPLv2.
 *
 * Copyright (C) 1998-2001 Gabor Kuti <seasons@fornax.hu><BR>
 * Copyright (C) 1998,2001,2002 Pavel Machek <pavel@suse.cz><BR>
 * Copyright (C) 2002-2003 Florent Chabaud <fchabaud@free.fr><BR>
 * Copyright (C) 2002-2006 Nigel Cunningham <nigel@suspend2.net><BR>
 *
 * \section Credits
 * 
 * Nigel would like to thank the following people for their work:
 * 
 * Pavel Machek <pavel@ucw.cz><BR>
 * Modifications, defectiveness pointing, being with Gabor at the very beginning,
 * suspend to swap space, stop all tasks. Port to 2.4.18-ac and 2.5.17.
 *
 * Steve Doddi <dirk@loth.demon.co.uk><BR> 
 * Support the possibility of hardware state restoring.
 *
 * Raph <grey.havens@earthling.net><BR>
 * Support for preserving states of network devices and virtual console
 * (including X and svgatextmode)
 *
 * Kurt Garloff <garloff@suse.de><BR>
 * Straightened the critical function in order to prevent compilers from
 * playing tricks with local variables.
 *
 * Andreas Mohr <a.mohr@mailto.de>
 *
 * Alex Badea <vampire@go.ro><BR>
 * Fixed runaway init
 *
 * Jeff Snyder <je4d@pobox.com><BR>
 * ACPI patch
 *
 * Nathan Friess <natmanz@shaw.ca><BR>
 * Some patches.
 *
 * Michael Frank <mhf@linuxmail.org><BR>
 * Extensive testing and help with improving stability. Nigel was constantly
 * amazed by the quality and quantity of Michael's help.
 *
 * Bernard Blackham <bernard@blackham.com.au><BR>
 * Web page & Wiki administration, some coding. Another person without whom
 * Suspend would not be where it is.
 *
 * ..and of course the myriads of Suspend2 users who have helped diagnose
 * and fix bugs, made suggestions on how to improve the code, proofread
 * documentation, and donated time and money.
 *
 * Thanks also to corporate sponsors:
 *
 * <B>Cyclades.com.</B> Nigel's employers from Dec 2004, who allow him to work on
 * Suspend and PM related issues on company time.
 * 
 * <B>LinuxFund.org.</B> Sponsored Nigel's work on Suspend for four months Oct 2003
 * to Jan 2004.
 *
 * <B>LAC Linux.</B> Donated P4 hardware that enabled development and ongoing
 * maintenance of SMP and Highmem support.
 *
 * <B>OSDL.</B> Provided access to various hardware configurations, make occasional
 * small donations to the project.
 */

#define SUSPEND_MAIN_C

#include <linux/suspend.h>
#include <linux/module.h>
#include <linux/console.h>
#include <linux/version.h>
#include <linux/reboot.h>
#include <linux/mm.h>
#include <linux/highmem.h>
#include <linux/freezer.h>
#include <linux/utsrelease.h>
#include <linux/swap.h> /* For LRU unlinking prototypes */
#include <asm/uaccess.h>
#include <asm/setup.h>

#include "suspend.h"
#include "modules.h"
#include "sysfs.h"
#include "pageflags.h"
#include "prepare_image.h"
#include "io.h"
#include "ui.h"
#include "extent.h"
#include "power_off.h"
#include "storage.h"

#ifdef  CONFIG_X86
#include <asm/i387.h> /* for kernel_fpu_end */
#endif

int suspend2_running;
 
/* Variables to be preserved over suspend */
long pageset1_sizelow = 0, pageset2_sizelow = 0, image_size_limit = 0;
unsigned long suspend_orig_mem_free = 0;

static char suspend_core_version[] = SUSPEND_CORE_VERSION;

extern __nosavedata char suspend_resume_commandline[COMMAND_LINE_SIZE];

#ifdef CONFIG_SUSPEND2_REPLACE_SWSUSP
unsigned long suspend_action = 1 << SUSPEND_REPLACE_SWSUSP;
#else
unsigned long suspend_action = 0;
#endif
unsigned long suspend_result = 0;
unsigned long suspend_debug_state = 0;

int suspend2_in_suspend __nosavedata;
extern void copyback_post(void);
extern int suspend2_suspend(void);
extern int extra_pd1_pages_used;

static int orig_system_state;
extern int driver_model_beeping;

/* 
 * ---  Variables -----
 * 
 * The following are used by the arch specific low level routines 
 * and only needed if suspend2 is compiled in. Other variables,
 * used by the freezer even if suspend2 is not compiled in, are
 * found in process.c
 */

/*! How long I/O took. */
int suspend_io_time[2][2];

/* Compression ratio */
__nosavedata unsigned long bytes_in = 0, bytes_out = 0;

/*! Pageset metadata. */
struct pagedir pagedir1 = { 0, 0}, pagedir2 = { 0, 0}; 

/* Suspend2 variables used by built-in routines. */

/*! The number of suspends we have started (some may have been cancelled) */
unsigned int nr_suspends = 0;

/* 
 * For resume2= kernel option. It's pointless to compile
 * suspend2 without any writers, but compilation shouldn't
 * fail if you do.
 */

unsigned long suspend_state = ((1 << SUSPEND_BOOT_TIME) |
		(1 << SUSPEND_RESUME_NOT_DONE) | (1 << SUSPEND_IGNORE_LOGLEVEL));

mm_segment_t	oldfs;

char resume2_file[256] = CONFIG_SUSPEND2_DEFAULT_RESUME2;
#ifdef CONFIG_SOFTWARE_SUSPEND
extern char resume_file[256];
#endif

static atomic_t actions_running;

extern int block_dump;

int block_dump_save;

/*
 * Basic clean-up routine.
 */
void suspend_finish_anything(int finishing_cycle)
{
	if (atomic_dec_and_test(&actions_running)) {
		suspend_cleanup_modules(finishing_cycle);
		suspend_put_modules();
		clear_suspend_state(SUSPEND_RUNNING);
	}

	set_fs(oldfs);

	if (finishing_cycle)
		block_dump = block_dump_save;
}

/*
 * Basic set-up routine.
 */
int suspend_start_anything(int starting_cycle)
{
	oldfs = get_fs();

	if (atomic_add_return(1, &actions_running) == 1) {
       		set_fs(KERNEL_DS);

		set_suspend_state(SUSPEND_RUNNING);

		if (suspend_get_modules()) {
			printk(name_suspend "Get modules failed!\n");
			clear_suspend_state(SUSPEND_RUNNING);
			set_fs(oldfs);
			return -EBUSY;
		}

		if (suspend_initialise_modules(starting_cycle)) {
			printk(name_suspend "Initialise modules failed!\n");
			suspend_finish_anything(starting_cycle);
			return -EBUSY;
		}

		if (starting_cycle) {
			block_dump_save = block_dump;
			block_dump = 0;
		}
	}

	return 0;
}

/*
 * save_image
 * Functionality    : High level routine which performs the steps necessary
 *                    to prepare and save the image after preparatory steps
 *                    have been taken.
 * Key Assumptions  : Processes frozen, sufficient memory available, drivers
 *                    suspended.
 * Called from      : suspend_suspend_2
 */
static void save_image(void)
{
	int temp_result;

	suspend_message(SUSPEND_ANY_SECTION, SUSPEND_LOW, 1,
		" - Final values: %d and %d.\n",
		pagedir1.pageset_size, 
		pagedir2.pageset_size);

	suspend_cond_pause(1, "About to write pagedir2.");

	temp_result = write_pageset(&pagedir2, 2);
	
	if (temp_result == -1 || test_result_state(SUSPEND_ABORTED))
		goto backout;

	suspend_cond_pause(1, "About to copy pageset 1.");

	if (test_result_state(SUSPEND_ABORTED))
		goto backout;

	suspend_deactivate_storage(1);

	suspend_prepare_status(DONT_CLEAR_BAR, "Doing atomic copy.");
	
	suspend2_running = 1; /* For the swsusp code we use :< */

	suspend2_in_suspend = 1;
	
	if (device_suspend(PMSG_FREEZE)) {
		set_result_state(SUSPEND_DEVICE_REFUSED);
		set_result_state(SUSPEND_ABORTED);
		goto backout;
	}
	
	if (suspend2_suspend()) {
		device_resume();
		return;
	}

	suspend2_running = 0;

	device_resume();
	
	/* Resume time? */
	if (!suspend2_in_suspend) {
		copyback_post();
		return;
	}

	/* Nope. Suspending. So, see if we can save the image... */
	if (!save_image_part1()) {
		suspend_power_down();

		if (read_pageset2(1))
			panic("Attempt to reload pagedir 2 failed. Try rebooting.");

		if (!test_result_state(SUSPEND_ABORT_REQUESTED) &&
		    !test_action_state(SUSPEND_TEST_FILTER_SPEED) &&
		    !test_action_state(SUSPEND_TEST_BIO) &&
		    suspend_powerdown_method != PM_SUSPEND_MEM)
			printk(KERN_EMERG name_suspend
				"Suspend failed, trying to recover...\n");
		barrier();
		mb();
	}

	return;
	
backout:
	return;
}

/*
 * Save the second part of the image.
 */
int save_image_part1(void)
{
	int temp_result;

	if (suspend_activate_storage(1))
		panic("Failed to reactivate our storage.");
	
	suspend_update_status(pagedir2.pageset_size,
			pagedir1.pageset_size + pagedir2.pageset_size,
			NULL);
	
	if (test_result_state(SUSPEND_ABORTED))
		goto abort_reloading_pagedir_two;

	suspend_cond_pause(1, "About to write pageset1.");

	/*
	 * End of critical section.
	 */
	
	suspend_message(SUSPEND_ANY_SECTION, SUSPEND_LOW, 1,
			"-- Writing pageset1\n");

	temp_result = write_pageset(&pagedir1, 1);

	/* We didn't overwrite any memory, so no reread needs to be done. */
	if (test_action_state(SUSPEND_TEST_FILTER_SPEED))
		return -1;

	if (temp_result == -1 || test_result_state(SUSPEND_ABORTED))
		goto abort_reloading_pagedir_two;

	suspend_cond_pause(1, "About to write header.");

	if (test_result_state(SUSPEND_ABORTED))
		goto abort_reloading_pagedir_two;

	temp_result = write_image_header();

	if (test_action_state(SUSPEND_TEST_BIO))
		return -1;

	if (temp_result || (test_result_state(SUSPEND_ABORTED)))
		goto abort_reloading_pagedir_two;

	suspend_cond_pause(1, "About to power down or reboot.");

	return 0;

abort_reloading_pagedir_two:
	temp_result = read_pageset2(1);

	/* If that failed, we're sunk. Panic! */
	if (temp_result)
		panic("Attempt to reload pagedir 2 while aborting "
				"a suspend failed.");

	return -1;		

}

static int io_MB_per_second(int read_write)
{
	if (!suspend_io_time[read_write][1])
		return 0;

	return MB((unsigned long) suspend_io_time[read_write][0]) * HZ /
		suspend_io_time[read_write][1];
}

/* get_debug_info
 * Functionality:	Store debug info in a buffer.
 * Called from:		suspend2_try_suspend.
 */
#define SNPRINTF(a...) 	len += snprintf_used(((char *)buffer) + len, \
		count - len - 1, ## a)
static int get_suspend_debug_info(const char *buffer, int count)
{
	int len = 0;

	SNPRINTF("Suspend2 debugging info:\n");
	SNPRINTF("- SUSPEND core   : %s\n", SUSPEND_CORE_VERSION);
	SNPRINTF("- Kernel Version : %s\n", UTS_RELEASE);
	SNPRINTF("- Compiler vers. : %d.%d\n", __GNUC__, __GNUC_MINOR__);
	SNPRINTF("- Attempt number : %d\n", nr_suspends);
	SNPRINTF("- Parameters     : %ld %ld %ld %d %d %ld\n",
			suspend_result,
			suspend_action,
			suspend_debug_state,
			suspend_default_console_level,
			image_size_limit,
			suspend_powerdown_method);
	SNPRINTF("- Overall expected compression percentage: %d.\n",
			100 - suspend_expected_compression_ratio());
	len+= suspend_print_module_debug_info(((char *) buffer) + len, 
			PAGE_SIZE - len - 1);
	if (suspend_io_time[0][1]) {
		if ((io_MB_per_second(0) < 5) || (io_MB_per_second(1) < 5)) {
			SNPRINTF("- I/O speed: Write %d KB/s",
			  (KB((unsigned long) suspend_io_time[0][0]) * HZ /
			  suspend_io_time[0][1]));
			if (suspend_io_time[1][1])
				SNPRINTF(", Read %d KB/s",
				  (KB((unsigned long) suspend_io_time[1][0]) * HZ /
				  suspend_io_time[1][1]));
		} else {
			SNPRINTF("- I/O speed: Write %d MB/s",
			 (MB((unsigned long) suspend_io_time[0][0]) * HZ /
			  suspend_io_time[0][1]));
			if (suspend_io_time[1][1])
				SNPRINTF(", Read %d MB/s",
				 (MB((unsigned long) suspend_io_time[1][0]) * HZ /
				  suspend_io_time[1][1]));
		}
		SNPRINTF(".\n");
	}
	else
		SNPRINTF("- No I/O speed stats available.\n");
	SNPRINTF("- Extra pages    : %d used/%d.\n",
			extra_pd1_pages_used, extra_pd1_pages_allowance);

	return len;
}

static int allocate_bitmaps(void)
{
	if (allocate_dyn_pageflags(&in_use_map) ||
	    allocate_dyn_pageflags(&pageset1_map) ||
	    allocate_dyn_pageflags(&pageset1_copy_map) ||
	    allocate_dyn_pageflags(&pageset2_map) ||
	    allocate_dyn_pageflags(&pageset2_rw_map))
		return 1;

	return 0;
}

static void free_metadata(void)
{
	free_dyn_pageflags(&pageset1_map);
	free_dyn_pageflags(&pageset1_copy_map);
	free_dyn_pageflags(&pageset2_map);
	free_dyn_pageflags(&pageset2_rw_map);
	free_dyn_pageflags(&in_use_map);
}

static int check_still_keeping_image(void)
{
	if (test_action_state(SUSPEND_KEEP_IMAGE)) {
		printk("Image already stored: powering down immediately.");
		suspend_power_down();
		return 1;	/* Just in case we're using S3 */
	}

	printk("Invalidating previous image.\n");
	suspend_active_writer->invalidate_image();

	return 0;
}

static int suspend_init(void)
{
	suspend_result = 0;

	printk(name_suspend "Initiating a software suspend cycle.\n");

	nr_suspends++;
	clear_suspend_state(SUSPEND_NOW_RESUMING);
	
	orig_system_state = system_state;
	
	suspend_io_time[0][0] = suspend_io_time[0][1] = 
		suspend_io_time[1][0] =
		suspend_io_time[1][1] = 0;

	free_metadata();	/* We might have kept it */

	if (!test_suspend_state(SUSPEND_CAN_SUSPEND))
		return 0;
	
	if (allocate_bitmaps())
		return 0;
	
	suspend_prepare_console();
	disable_nonboot_cpus();

	return 1;
}

void suspend_cleanup(int had_pmsem)
{
	int i = 0;
	char *buffer = (char *) get_zeroed_page(GFP_ATOMIC);

	if (buffer)
		i = get_suspend_debug_info(buffer, PAGE_SIZE);

	suspend_free_extra_pagedir_memory();
	
	pagedir1.pageset_size = pagedir2.pageset_size = 0;

	system_state = orig_system_state;

	thaw_processes(FREEZER_KERNEL_THREADS);

#ifdef CONFIG_SUSPEND2_KEEP_IMAGE
	if (test_action_state(SUSPEND_KEEP_IMAGE) &&
	    !test_result_state(SUSPEND_ABORTED)) {
		suspend_message(SUSPEND_ANY_SECTION, SUSPEND_LOW, 1,
			name_suspend "Not invalidating the image due "
			"to Keep Image being enabled.\n");
		set_result_state(SUSPEND_KEPT_IMAGE);
	} else
#endif
		if (suspend_active_writer)
			suspend_active_writer->invalidate_image();

	free_metadata();

	if (buffer && i) {
		/* Printk can only handle 1023 bytes, including
		 * its level mangling. */
		for (i = 0; i < 3; i++)
			printk("%s", buffer + (1023 * i));
		free_page((unsigned long) buffer);
		buffer = NULL;
	}

	thaw_processes(FREEZER_ALL_THREADS);
	enable_nonboot_cpus();
	suspend_cleanup_console();
	suspend2_running = 0;
	if (!had_pmsem)
		up(&pm_sem);
}

static int can_suspend(int had_pmsem)
{
	if (!had_pmsem && down_trylock(&pm_sem)) {
		set_result_state(SUSPEND_ABORTED);
		set_result_state(SUSPEND_PM_SEM);
		return 0;
	}

	if (!test_suspend_state(SUSPEND_CAN_SUSPEND))
		suspend_attempt_to_parse_resume_device();

	if (!test_suspend_state(SUSPEND_CAN_SUSPEND)) {
		printk(name_suspend "Software suspend is disabled.\n"
			"This may be because you haven't put something along "
			"the lines of\n\nresume2=swap:/dev/hda1\n\n"
			"in lilo.conf or equivalent. (Where /dev/hda1 is your "
			"swap partition).\n");
		set_result_state(SUSPEND_ABORTED);
		if (!had_pmsem)
			up(&pm_sem);
		return 0;
	}
	
	return 1;
}

/*
 * suspend_main
 * Functionality   : First level of code for software suspend invocations.
 *                   Stores and restores load averages (to avoid a spike),
 *                   allocates bitmaps, freezes processes and eats memory
 *                   as required before suspending drivers and invoking
 *                   the 'low level' code to save the state to disk.
 *                   By the time we return from do_suspend2_suspend, we
 *                   have either failed to save the image or successfully
 *                   suspended and reloaded the image. The difference can
 *                   be discerned by checking SUSPEND_ABORTED.
 * Called From     : 
 */
void suspend_main(int had_pmsem)
{
	if (suspend_activate_storage(0))
		return;

	if (!can_suspend(had_pmsem))
		goto cleanup_deactivate_storage;

	/*
	 * If kept image and still keeping image and suspending to RAM, we will 
	 * return 1 after suspending and resuming (provided the power doesn't
	 * run out.
	 */
	if (test_result_state(SUSPEND_KEPT_IMAGE) && check_still_keeping_image()) 
		goto cleanup;


	if (suspend_init() && !suspend_prepare_image() && !test_result_state(SUSPEND_ABORTED) &&
		!test_action_state(SUSPEND_FREEZER_TEST)) {
		suspend_prepare_status(DONT_CLEAR_BAR, "Starting to save the image..");
		unlink_lru_lists();
		save_image();
		relink_lru_lists();
	}
	
cleanup:
	suspend_cleanup(had_pmsem);
cleanup_deactivate_storage:
	suspend_deactivate_storage(0);
}

/* image_exists_read
 * 
 * Return 0 or 1, depending on whether an image is found.
 * Incoming buffer is PAGE_SIZE and result is guaranteed
 * to be far less than that, so we don't worry about
 * overflow.
 */
static int image_exists_read(const char *page, int count)
{
	int len = 0;
	char *result;
	
	if (suspend_activate_storage(0))
		return count;

	if (!test_suspend_state(SUSPEND_RESUME_DEVICE_OK))
		suspend_attempt_to_parse_resume_device();

	if (!suspend_active_writer) {
		len = sprintf((char *) page, "-1\n");
	} else {
		result = get_have_image_data();
		if (result) {
			len = sprintf((char *) page, "%s",  result);
			free_page((unsigned long) result);
		}
	}

	suspend_deactivate_storage(0);

	return len;
}

/* image_exists_write
 * 
 * Invalidate an image if one exists.
 */
static int image_exists_write(const char *buffer, int count)
{
	if (suspend_activate_storage(0))
		return count;

	if (suspend_active_writer && suspend_active_writer->image_exists())
		suspend_active_writer->invalidate_image();

	suspend_deactivate_storage(0);

	return count;
}

/*
 * Core sysfs entries that aren't built in.
 *
 * This array contains entries that are automatically registered at
 * boot. Modules and the console code register their own entries separately.
 */
static struct suspend_sysfs_data sysfs_params[] = {
	{ SUSPEND2_ATTR("driver_model_beeping", SYSFS_RW),
	  SYSFS_INT(&driver_model_beeping, 0, 1)
	},

	{ SUSPEND2_ATTR("debug_info", SYSFS_READONLY),
	  SYSFS_CUSTOM(get_suspend_debug_info, NULL, SYSFS_SM_NOT_NEEDED)
	},
	
	{ SUSPEND2_ATTR("extra_pages_allowance", SYSFS_RW),
	  SYSFS_UL(&extra_pd1_pages_allowance, 0, INT_MAX)
	},
	
	{ SUSPEND2_ATTR("ignore_rootfs", SYSFS_RW),
	  SYSFS_BIT(&suspend_action, SUSPEND_IGNORE_ROOTFS)
	},
	
	{ SUSPEND2_ATTR("image_exists", SYSFS_RW),
	  SYSFS_CUSTOM(image_exists_read, image_exists_write,
			  SYSFS_NEEDS_FOR_BOTH)
	},

	{ SUSPEND2_ATTR("image_size_limit", SYSFS_RW),
	  SYSFS_LONG(&image_size_limit, -2, INT_MAX)
	},

	{ SUSPEND2_ATTR("last_result", SYSFS_READONLY),
	  SYSFS_UL(&suspend_result, 0, 0)
	},
	
	{ SUSPEND2_ATTR("reboot", SYSFS_RW),
	  SYSFS_BIT(&suspend_action, SUSPEND_REBOOT)
	},

#ifdef CONFIG_SOFTWARE_SUSPEND
	{ SUSPEND2_ATTR("replace_swsusp", SYSFS_RW),
	  SYSFS_BIT(&suspend_action, SUSPEND_REPLACE_SWSUSP)
	},
#endif

	{ SUSPEND2_ATTR("resume2", SYSFS_RW),
	  SYSFS_STRING(resume2_file, 255, SYSFS_NEEDS_FOR_WRITE),
	  .write_side_effect = attempt_to_parse_resume_device2,
	},

	{ SUSPEND2_ATTR("resume_commandline", SYSFS_RW),
	  SYSFS_STRING(suspend_resume_commandline, COMMAND_LINE_SIZE,
			  SYSFS_SM_NOT_NEEDED)
	},

	{ SUSPEND2_ATTR("version", SYSFS_READONLY),
	  SYSFS_STRING(suspend_core_version, 0, SYSFS_SM_NOT_NEEDED)
	},

#ifdef CONFIG_PM_DEBUG
	{ SUSPEND2_ATTR("freezer_test", SYSFS_RW),
	  SYSFS_BIT(&suspend_action, SUSPEND_FREEZER_TEST)
	},

	{ SUSPEND2_ATTR("test_bio", SYSFS_RW),
	  SYSFS_BIT(&suspend_action, SUSPEND_TEST_BIO)
	},

	{ SUSPEND2_ATTR("test_filter_speed", SYSFS_RW),
	  SYSFS_BIT(&suspend_action, SUSPEND_TEST_FILTER_SPEED)
	},

	{ SUSPEND2_ATTR("slow", SYSFS_RW),
	  SYSFS_BIT(&suspend_action, SUSPEND_SLOW)
	},

	{ SUSPEND2_ATTR("no_pageset2", SYSFS_RW),
	  SYSFS_BIT(&suspend_action, SUSPEND_NO_PAGESET2)
	},
#endif
	  
#if defined(CONFIG_ACPI)
	{ SUSPEND2_ATTR("powerdown_method", SYSFS_RW),
	  SYSFS_UL(&suspend_powerdown_method, 0, 5)
	},
#endif

#ifdef CONFIG_SUSPEND2_KEEP_IMAGE
	{ SUSPEND2_ATTR("keep_image", SYSFS_RW),
	  SYSFS_BIT(&suspend_action, SUSPEND_KEEP_IMAGE)
	},
#endif
};
 
static __init int core_load(void)
{
	int i,
	    numfiles = sizeof(sysfs_params) / sizeof(struct suspend_sysfs_data);

	printk("Suspend2 Core.\n");

	suspend_initialise_module_lists();

	for (i=0; i< numfiles; i++)
		suspend_register_sysfs_file(&suspend2_subsys.kset.kobj,
				&sysfs_params[i]);

#ifdef CONFIG_SOFTWARE_SUSPEND
	/* Overriding resume2= with resume=? */
	if (test_action_state(SUSPEND_REPLACE_SWSUSP) && resume_file[0])
		strncpy(resume2_file, resume_file, 256);
#endif

	return 0;
}

/*
 * Called from init kernel_thread.
 * We check if we have an image and if so we try to resume.
 * We also start ksuspendd if configuration looks right.
 */
int suspend_resume(void)
{
	int read_image_result = 0;

	if (sizeof(swp_entry_t) != sizeof(long)) {
		printk(KERN_WARNING name_suspend
			"The size of swp_entry_t != size of long. "
			"Please report this!\n");
		return 1;
	}
	
	if (!resume2_file[0])
		printk(KERN_WARNING name_suspend
			"You need to use a resume2= command line parameter to "
			"tell Suspend2 where to look for an image.\n");

	suspend_activate_storage(0);

	if (!(test_suspend_state(SUSPEND_RESUME_DEVICE_OK)) &&
		!suspend_attempt_to_parse_resume_device()) {
		/* 
		 * Without a usable storage device we can do nothing - 
		 * even if noresume is given
		 */

		if (!suspend_num_writers)
			printk(KERN_ALERT name_suspend
				"No writers have been registered.\n");
		else
			printk(KERN_ALERT name_suspend
				"Missing or invalid storage location "
				"(resume2= parameter). Please correct and "
				"rerun lilo (or equivalent) before "
				"suspending.\n");
		suspend_deactivate_storage(0);
		return 1;
	}

	suspend_orig_mem_free = real_nr_free_pages();

	read_image_result = read_pageset1(); /* non fatal error ignored */

	if (test_suspend_state(SUSPEND_NORESUME_SPECIFIED))
		printk(KERN_WARNING name_suspend "Resuming disabled as requested.\n");

	suspend_deactivate_storage(0);
	
	if (read_image_result) {
		printk(KERN_WARNING name_suspend "Read image returned %d.\n", read_image_result);
		return 1;
	}

	suspend2_running = 1;

	suspend_atomic_restore();

	BUG();

	return 0;
}

/* -- Functions for kickstarting a suspend or resume --- */

/*
 * Check if we have an image and if so try to resume.
 */
void __suspend_try_resume(void)
{
	set_suspend_state(SUSPEND_TRYING_TO_RESUME);
	clear_suspend_state(SUSPEND_RESUME_NOT_DONE);

	suspend_resume();

	clear_suspend_state(SUSPEND_IGNORE_LOGLEVEL);
	clear_suspend_state(SUSPEND_TRYING_TO_RESUME);
}

/* Wrapper for when called from init/do_mounts.c */
void suspend2_try_resume(void)
{
	if (suspend_start_anything(0))
		return;

	down(&pm_sem);
	__suspend_try_resume();

	/* 
	 * For initramfs, we have to clear the boot time
	 * flag after trying to resume
	 */
	clear_suspend_state(SUSPEND_BOOT_TIME);

	up(&pm_sem);

	suspend_finish_anything(0);
}

/*
 * suspend2_try_suspend
 * Functionality   : Wrapper around suspend_main.
 * Called From     : drivers/acpi/sleep/main.c
 *                   kernel/reboot.c
 */
void suspend2_try_suspend(int have_pmsem)
{
	if (suspend_start_anything(0))
		return;

	suspend_main(0);

	suspend_finish_anything(0);
}

/* --  Commandline Parameter Handling ---
 *
 * Resume setup: obtain the storage device.
 */
static int __init resume2_setup(char *str)
{
	if (!*str)
		return 0;
	
	strncpy(resume2_file, str, 255);
	return 0;
}

/*
 * Allow the user to set the debug parameter from lilo, prior to resuming.
 */
/*
 * Allow the user to specify that we should ignore any image found and
 * invalidate the image if necesssary. This is equivalent to running
 * the task queue and a sync and then turning off the power. The same
 * precautions should be taken: fsck if you're not journalled.
 */
static int __init noresume2_setup(char *str)
{
	set_suspend_state(SUSPEND_NORESUME_SPECIFIED);
	return 0;
}

static int __init suspend_retry_resume_setup(char *str)
{
	set_suspend_state(SUSPEND_RETRY_RESUME);
	return 0;
}

__setup("noresume2", noresume2_setup);
__setup("resume2=", resume2_setup);
__setup("suspend_retry_resume", suspend_retry_resume_setup);

late_initcall(core_load);
EXPORT_SYMBOL(suspend_state);
