/*
 * kernel/power/storage.c
 *
 * Copyright (C) 2005-2006 Nigel Cunningham <nigel@suspend2.net>
 *
 * This file is released under the GPLv2.
 *
 * Routines for talking to a userspace program that manages storage.
 *
 * The kernel side:
 * - starts the userspace program;
 * - sends messages telling it when to open and close the connection;
 * - tells it when to quit;
 *
 * The user space side:
 * - passes messages regarding status;
 *
 */

#include <linux/suspend.h>
#include <linux/freezer.h>
 
#include "sysfs.h"
#include "modules.h"
#include "netlink.h"
#include "storage.h"
#include "ui.h"

static struct user_helper_data usm_helper_data;
static struct suspend_module_ops usm_ops;
static int message_received = 0;
static int activations = 0;
static int usm_prepare_count = 0;
static int storage_manager_last_action = 0;
static int storage_manager_action = 0;
       
static int usm_user_rcv_msg(struct sk_buff *skb, struct nlmsghdr *nlh)
{
	int type;
	int *data;

	type = nlh->nlmsg_type;

	/* A control message: ignore them */
	if (type < NETLINK_MSG_BASE)
		return 0;

	/* Unknown message: reply with EINVAL */
	if (type >= USM_MSG_MAX)
		return -EINVAL;

	/* All operations require privileges, even GET */
	if (security_netlink_recv(skb, CAP_NET_ADMIN))
		return -EPERM;

	/* Only allow one task to receive NOFREEZE privileges */
	if (type == NETLINK_MSG_NOFREEZE_ME && usm_helper_data.pid != -1)
		return -EBUSY;

	data = (int*)NLMSG_DATA(nlh);

	switch (type) {
		case USM_MSG_SUCCESS:
		case USM_MSG_FAILED:
			message_received = type;
			complete(&usm_helper_data.wait_for_process);
			break;
		default:
			printk("Storage manager doesn't recognise message %d.\n", type);
	}

	return 1;
}

int suspend_activate_storage(int force)
{
	int tries = 1;

	if (usm_helper_data.pid == -1 || !usm_ops.enabled)
		return 0;

	message_received = 0;
	activations++;

	if (activations > 1 && !force)
		return 0;

	while ((!message_received || message_received == USM_MSG_FAILED) && tries < 2) {
		suspend_prepare_status(DONT_CLEAR_BAR, "Activate storage attempt %d.\n", tries);

		init_completion(&usm_helper_data.wait_for_process);

		suspend_send_netlink_message(&usm_helper_data,
			USM_MSG_CONNECT,
			NULL, 0);

		/* Wait 2 seconds for the userspace process to make contact */
		wait_for_completion_timeout(&usm_helper_data.wait_for_process, 2*HZ);

		tries++;
	}

	return 0;
}

int suspend_deactivate_storage(int force)
{
	if (usm_helper_data.pid == -1 || !usm_ops.enabled)
		return 0;
	
	message_received = 0;
	activations--;

	if (activations && !force)
		return 0;

	init_completion(&usm_helper_data.wait_for_process);

	suspend_send_netlink_message(&usm_helper_data,
			USM_MSG_DISCONNECT,
			NULL, 0);

	wait_for_completion_timeout(&usm_helper_data.wait_for_process, 2*HZ);

	if (!message_received || message_received == USM_MSG_FAILED) {
		printk("Returning failure disconnecting storage.\n");
		return 1;
	}

	return 0;
}

#ifdef CONFIG_PM_DEBUG
static void storage_manager_simulate(void)
{
	printk("--- Storage manager simulate ---\n");
	suspend_prepare_usm();
	schedule();
	printk("--- Activate storage 1 ---\n");
	suspend_activate_storage(1);
	schedule();
	printk("--- Deactivate storage 1 ---\n");
	suspend_deactivate_storage(1);
	schedule();
	printk("--- Cleanup usm ---\n");
	suspend_cleanup_usm();
	schedule();
	printk("--- Storage manager simulate ends ---\n");
}
#endif

static unsigned long usm_storage_needed(void)
{
	return strlen(usm_helper_data.program);
}

static int usm_save_config_info(char *buf)
{
	int len = strlen(usm_helper_data.program);
	memcpy(buf, usm_helper_data.program, len);
	return len;
}

static void usm_load_config_info(char *buf, int size)
{
	/* Don't load the saved path if one has already been set */
	if (usm_helper_data.program[0])
		return;

	memcpy(usm_helper_data.program, buf, size);
}

static unsigned long usm_memory_needed(void)
{
	/* ball park figure of 32 pages */
	return (32 * PAGE_SIZE);
}

/* suspend_prepare_usm
 */
int suspend_prepare_usm(void)
{
	usm_prepare_count++;

	if (usm_prepare_count > 1 || !usm_ops.enabled)
		return 0;
	
	usm_helper_data.pid = -1;

	if (!*usm_helper_data.program)
		return 0;

	suspend_netlink_setup(&usm_helper_data);

	if (usm_helper_data.pid == -1)
		printk("Suspend2 Storage Manager wanted, but couldn't start it.\n");

	suspend_activate_storage(0);

	return (usm_helper_data.pid != -1);
}

void suspend_cleanup_usm(void)
{
	usm_prepare_count--;

	if (usm_helper_data.pid > -1 && !usm_prepare_count) {
		struct task_struct *t;

		suspend_deactivate_storage(0);

		suspend_send_netlink_message(&usm_helper_data,
				NETLINK_MSG_CLEANUP, NULL, 0);

		read_lock(&tasklist_lock);
		if ((t = find_task_by_pid(usm_helper_data.pid)))
			t->flags &= ~PF_NOFREEZE;
		read_unlock(&tasklist_lock);

		suspend_netlink_close(&usm_helper_data);

		usm_helper_data.pid = -1;
	}
}

static void storage_manager_activate(void)
{
	if (storage_manager_action == storage_manager_last_action)
		return;

	if (storage_manager_action)
		suspend_prepare_usm();
	else
		suspend_cleanup_usm();

	storage_manager_last_action = storage_manager_action;
}

/*
 * User interface specific /sys/power/suspend2 entries.
 */

static struct suspend_sysfs_data sysfs_params[] = {
	{ SUSPEND2_ATTR("enabled", SYSFS_RW),
	  SYSFS_INT(&usm_ops.enabled, 0, 1)
	},

	{ SUSPEND2_ATTR("program", SYSFS_RW),
	  SYSFS_STRING(usm_helper_data.program, 254, SYSFS_SM_NOT_NEEDED)
	},

	{ SUSPEND2_ATTR("activate_storage", SYSFS_RW),
	  SYSFS_INT(&storage_manager_action, 0, 1),
	  .write_side_effect		= storage_manager_activate,
	},

#ifdef CONFIG_PM_DEBUG
	{ SUSPEND2_ATTR("simulate_atomic_copy", SYSFS_RW),
	  .type				= SUSPEND_SYSFS_DATA_NONE,
	  .write_side_effect		= storage_manager_simulate,
	}
#endif
};

static struct suspend_module_ops usm_ops = {
	.type				= MISC_MODULE,
	.name				= "Userspace Storage Manager",
	.module				= THIS_MODULE,
	.storage_needed			= usm_storage_needed,
	.save_config_info		= usm_save_config_info,
	.load_config_info		= usm_load_config_info,
	.memory_needed			= usm_memory_needed,
};
       
/* suspend_usm_sysfs_init
 * Description: Boot time initialisation for user interface.
 */
static __init int suspend_usm_sysfs_init(void)
{
	int result, i,
	    numfiles = sizeof(sysfs_params) / sizeof(struct suspend_sysfs_data);

	if (!(result = suspend_register_module(&usm_ops))) {
		struct kobject *kobj = make_suspend2_sysdir("storage_manager");
		for (i=0; i< numfiles; i++)
			suspend_register_sysfs_file(kobj, &sysfs_params[i]);
	}

	usm_helper_data.nl = NULL;
	usm_helper_data.program[0] = '\0';
	usm_helper_data.pid = -1;
	usm_helper_data.skb_size = 0;
	usm_helper_data.pool_limit = 6;
	usm_helper_data.netlink_id = NETLINK_SUSPEND2_USM;
	usm_helper_data.name = "userspace storage manager";
	usm_helper_data.rcv_msg = usm_user_rcv_msg;
	usm_helper_data.interface_version = 1;
	usm_helper_data.must_init = 0;
	init_completion(&usm_helper_data.wait_for_process);

	return result;
}

late_initcall(suspend_usm_sysfs_init);
