/*
 * kernel/power/modules.c
 *
 * Copyright (C) 2004-2006 Nigel Cunningham <nigel@suspend2.net>
 *
 */

#include <linux/suspend.h>
#include <linux/module.h>
#include "suspend.h"
#include "modules.h"
#include "ui.h"

struct list_head suspend_filters, suspend_writers, suspend_modules;
struct suspend_module_ops *suspend_active_writer;
static int suspend_num_filters;
int suspend_num_writers, suspend_num_modules;
       
/*
 * suspend_header_storage_for_modules
 *
 * Returns the amount of space needed to store configuration
 * data needed by the modules prior to copying back the original
 * kernel. We can exclude data for pageset2 because it will be
 * available anyway once the kernel is copied back.
 */
unsigned long suspend_header_storage_for_modules(void)
{
	struct suspend_module_ops *this_module;
	unsigned long bytes = 0;
	
	list_for_each_entry(this_module, &suspend_modules, module_list) {
		if (!this_module->enabled ||
		    (this_module->type == WRITER_MODULE &&
		     suspend_active_writer != this_module))
			continue;
		if (this_module->storage_needed) {
			int this = this_module->storage_needed() +
				sizeof(struct suspend_module_header) +
				sizeof(int);
			this_module->header_requested = this;
			bytes += this;
		}
	}

	/* One more for the empty terminator */
	return bytes + sizeof(struct suspend_module_header);
}

/*
 * suspend_memory_for_modules
 *
 * Returns the amount of memory requested by modules for
 * doing their work during the cycle.
 */

unsigned long suspend_memory_for_modules(void)
{
	unsigned long bytes = 0;
	struct suspend_module_ops *this_module;

	list_for_each_entry(this_module, &suspend_modules, module_list) {
		if (!this_module->enabled)
			continue;
		if (this_module->memory_needed)
			bytes += this_module->memory_needed();
	}

	return ((bytes + PAGE_SIZE - 1) >> PAGE_SHIFT);
}

/* suspend_find_module_given_name
 * Functionality :	Return a module (if found), given a pointer
 * 			to its name
 */

struct suspend_module_ops *suspend_find_module_given_name(char *name)
{
	struct suspend_module_ops *this_module, *found_module = NULL;
	
	list_for_each_entry(this_module, &suspend_modules, module_list) {
		if (!strcmp(name, this_module->name)) {
			found_module = this_module;
			break;
		}			
	}

	return found_module;
}

/*
 * suspend_print_module_debug_info
 * Functionality   : Get debugging info from modules into a buffer.
 */
int suspend_print_module_debug_info(char *buffer, int buffer_size)
{
	struct suspend_module_ops *this_module;
	int len = 0;

	list_for_each_entry(this_module, &suspend_modules, module_list) {
		if (!this_module->enabled)
			continue;
		if (this_module->print_debug_info) {
			int result;
			result = this_module->print_debug_info(buffer + len, 
					buffer_size - len);
			len += result;
		}
	}

	/* Ensure null terminated */
	buffer[buffer_size] = 0;

	return len;
}

/*
 * suspend_register_module
 *
 * Register a module.
 */
int suspend_register_module(struct suspend_module_ops *module)
{
	module->enabled = 1;
	
	if (suspend_find_module_given_name(module->name)) {
		printk(name_suspend "Trying to load module %s,"
				" which is already registered.\n",
				module->name);
		return -EBUSY;
	}

	switch (module->type) {
		case FILTER_MODULE:
			list_add_tail(&module->type_list,
					&suspend_filters);
			suspend_num_filters++;
			break;

		case WRITER_MODULE:
			list_add_tail(&module->type_list,
					&suspend_writers);
			suspend_num_writers++;
			break;

		case MISC_MODULE:
			break;

		default:
			printk("Hmmm. Module '%s' has an invalid type."
				" It has been ignored.\n", module->name);
			return -EINVAL;
	}
	list_add_tail(&module->module_list, &suspend_modules);
	suspend_num_modules++;

	return 0;	
}

/*
 * suspend_unregister_module
 *
 * Remove a module.
 */
void suspend_unregister_module(struct suspend_module_ops *module)
{
	switch (module->type) {
		case FILTER_MODULE:
			list_del(&module->type_list);
			suspend_num_filters--;
			break;

		case WRITER_MODULE:
			list_del(&module->type_list);
			suspend_num_writers--;
			if (suspend_active_writer == module) {
				suspend_active_writer = NULL;
				clear_suspend_state(SUSPEND_CAN_RESUME);
				clear_suspend_state(SUSPEND_CAN_SUSPEND);
			}
			break;
		
		case MISC_MODULE:
			break;

		default:
			printk("Hmmm. Module '%s' has an invalid type."
				" It has been ignored.\n", module->name);
			return;
	}
	list_del(&module->module_list);
	suspend_num_modules--;
}

/*
 * suspend_move_module_tail
 *
 * Rearrange modules when reloading the config.
 */
void suspend_move_module_tail(struct suspend_module_ops *module)
{
	switch (module->type) {
		case FILTER_MODULE:
			if (suspend_num_filters > 1)
				list_move_tail(&module->type_list,
						&suspend_filters);
			break;

		case WRITER_MODULE:
			if (suspend_num_writers > 1)
				list_move_tail(&module->type_list,
						&suspend_writers);
			break;
		
		case MISC_MODULE:
			break;
		default:
			printk("Hmmm. Module '%s' has an invalid type."
				" It has been ignored.\n", module->name);
			return;
	}
	if ((suspend_num_filters + suspend_num_writers) > 1)
		list_move_tail(&module->module_list, &suspend_modules);
}

/*
 * suspend_initialise_modules
 *
 * Get ready to do some work!
 */
int suspend_initialise_modules(int starting_cycle)
{
	struct suspend_module_ops *this_module;
	int result;
	
	list_for_each_entry(this_module, &suspend_modules, module_list) {
		this_module->header_requested = 0;
		this_module->header_used = 0;
		if (!this_module->enabled)
			continue;
		if (this_module->initialise) {
			suspend_message(SUSPEND_MEMORY, SUSPEND_MEDIUM, 1,
				"Initialising module %s.\n",
				this_module->name);
			if ((result = this_module->initialise(starting_cycle))) {
				printk("%s didn't initialise okay.\n",
						this_module->name);
				return result;
			}
		}
	}

	return 0;
}

/* 
 * suspend_cleanup_modules
 *
 * Tell modules the work is done.
 */
void suspend_cleanup_modules(int finishing_cycle)
{
	struct suspend_module_ops *this_module;
	
	list_for_each_entry(this_module, &suspend_modules, module_list) {
		if (!this_module->enabled)
			continue;
		if (this_module->cleanup) {
			suspend_message(SUSPEND_MEMORY, SUSPEND_MEDIUM, 1,
				"Cleaning up module %s.\n",
				this_module->name);
			this_module->cleanup(finishing_cycle);
		}
	}
}

/*
 * suspend_get_next_filter
 *
 * Get the next filter in the pipeline.
 */
struct suspend_module_ops *suspend_get_next_filter(struct suspend_module_ops *filter_sought)
{
	struct suspend_module_ops *last_filter = NULL, *this_filter = NULL;

	list_for_each_entry(this_filter, &suspend_filters, type_list) {
		if (!this_filter->enabled)
			continue;
		if ((last_filter == filter_sought) || (!filter_sought))
			return this_filter;
		last_filter = this_filter;
	}

	return suspend_active_writer;
}

/* suspend_get_modules
 * 
 * Take a reference to modules so they can't go away under us.
 */

int suspend_get_modules(void)
{
	struct suspend_module_ops *this_module;
	
	list_for_each_entry(this_module, &suspend_modules, module_list) {
		if (!try_module_get(this_module->module)) {
			/* Failed! Reverse gets and return error */
			struct suspend_module_ops *this_module2;
			list_for_each_entry(this_module2, &suspend_modules, module_list) {
				if (this_module == this_module2)
					return -EINVAL;
				module_put(this_module2->module);
			}
		}
	}

	return 0;
}

/* suspend_put_modules
 *
 * Release our references to modules we used.
 */

void suspend_put_modules(void)
{
	struct suspend_module_ops *this_module;
	
	list_for_each_entry(this_module, &suspend_modules, module_list)
		module_put(this_module->module);
}

