/*
 *	MP3 huffman table selecting and bit counting
 *
 *	Copyright (c) 1999 Takehiro TOMINAGA
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "util.h"
#include "l3side.h"
#include "huffman.h"
#include "quantize.h"


extern int *scalefac_band_long; 
extern int *scalefac_band_short;



struct
{
    unsigned region0_count;
    unsigned region1_count;
} subdv_table[ 23 ] =
{
{0, 0}, /* 0 bands */
{0, 0}, /* 1 bands */
{0, 0}, /* 2 bands */
{0, 0}, /* 3 bands */
{0, 0}, /* 4 bands */
{0, 1}, /* 5 bands */
{1, 1}, /* 6 bands */
{1, 1}, /* 7 bands */
{1, 2}, /* 8 bands */
{2, 2}, /* 9 bands */
{2, 3}, /* 10 bands */
{2, 3}, /* 11 bands */
{3, 4}, /* 12 bands */
{3, 4}, /* 13 bands */
{3, 4}, /* 14 bands */
{4, 5}, /* 15 bands */
{4, 5}, /* 16 bands */
{4, 6}, /* 17 bands */
{5, 6}, /* 18 bands */
{5, 6}, /* 19 bands */
{5, 7}, /* 20 bands */
{6, 7}, /* 21 bands */
{6, 7}, /* 22 bands */
};


/*************************************************************************/
/*	      ix_max							 */
/*************************************************************************/

static int ix_max_takehiro(int *ix, int *end) 
{
    int max = 0;

    while (ix < end) {
	int x =	 *ix++;
	if (max < x) 
	    max = x;
    }
    return max;
}


/*************************************************************************/
/*	      count_bit							 */
/*************************************************************************/

/*
 Function: Count the number of bits necessary to code the subregion. 
*/

static int count_bit_ESC(int *ix, int *end, unsigned int table) 
{
    /* ESC-table is used */
    unsigned char *hlen = ht[table].hlen;
    int linbits = ht[table].linbits;
    int	sum = 0;

    while (ix < end) {
	int x = *ix++;
	int y = *ix++;

	if (x != 0) {
	    sum++;
	    if (x > 14) {
		x = 15;
		sum += linbits;
	    }
	    x *= 16;
	}

	if (y != 0) {
	    sum++;
	    if (y > 14) {
		y = 15;
		sum += linbits;
	    }
	    x += y;
	}

	sum += hlen[x];
    }

    return sum;
}



static int count_bit_noESC(int *ix, int *end, unsigned int table) 
{
    /* No ESC-words */
    int	sum = 0;
    unsigned char *hlen = ht[table].hlen;

    while (ix < end) {
	int x = *ix++;
	int y = *ix++;
	if (x != 0) {
	    sum++;
	    x *= 16;
	}

	if (y != 0) {
	    sum++;
	    x += y;
	}

	sum += hlen[x];
    }

    return sum;
}



static int count_bit_short_ESC(int *ix, int *end, unsigned int table)
{
    /* ESC-table is used */
    unsigned char *hlen = ht[table].hlen;
    unsigned int linbits = ht[table].linbits;
    int	sum = 0;

    while (ix < end) {
	int i;
	for (i = 0; i < 3; i++) {
	    int y = *(ix + 3);
	    int x = *ix++;

	    if (x != 0) {
		sum++;
		if (x > 14) {
		    x = 15;
		    sum += linbits;
		}
		x *= 16;
	    }

	    if (y != 0) {
		sum++;
		if (y > 14) {
		    y = 15;
		    sum += linbits;
		}
		x += y;
	    }

	    sum += hlen[x];
	}
	ix += 3;
    }

    return sum;
}



static int count_bit_short_noESC(int *ix, int *end, unsigned int table) 
{
    /* No ESC-words */
    int	sum = 0;
    unsigned char *hlen = ht[table].hlen;

    while (ix < end) {
	int i;
	for (i = 0; i < 3; i++) {
	    int y = *(ix + 3);
	    int x = *ix++;
	    if (x != 0) {
		sum++;
		x *= 16;
	    }

	    if (y != 0) {
		sum++;
		x += y;
	    }

	    sum += hlen[x];
	}
	ix += 3;
    }

    return sum;
}



/*************************************************************************/
/*	      new_choose table						 */
/*************************************************************************/

/*
  Choose the Huffman table that will encode ix[begin..end] with
  the fewest bits.

  Note: This code contains knowledge about the sizes and characteristics
  of the Huffman tables as defined in the IS (Table B.7), and will not work
  with any arbitrary tables.
*/

int choose_table_takehiro(int *ix, int *end, int * s)
{
    int max;
    int choice0, choice1;
    int sum0, sum1;

    max = ix_max_takehiro(ix, end);

    if (max <= 15)  {
	if (max == 0) {
	    return 0;
	}
	/* try tables with no linbits */
	for (choice0 = 1; choice0 < 14; choice0++) {
	    if (ht[choice0].xlen > max) {
		break;
	    }
	}

	sum0 = count_bit_noESC(ix, end, choice0);
	choice1 = choice0;

	switch (choice0) {
	case 7:
	case 10:
	    choice1++;
	    sum1 = count_bit_noESC(ix, end, choice1);
	    if (sum0 > sum1) {
		sum0 = sum1;
		choice0 = choice1;
	    }
	    /*fall*/
	case 2:
	case 5:
	    choice1++;
	    sum1 = count_bit_noESC(ix, end, choice1);
	    if (sum0 > sum1) {
		sum0 = sum1;
		choice0 = choice1;
	    }
	    break;

	case 13:
	    choice1 += 2;
	    sum1 = count_bit_noESC(ix, end, choice1);
	    if (sum0 > sum1) {
		sum0 = sum1;
		choice0 = choice1;
	    }
	    break;

	default:
	    break;
	}
    } else {
	/* try tables with linbits */
	max -= 15;

	for (choice0 = 15; choice0 < 24; choice0++) {
	    if (ht[choice0].linmax >= max) {
		break;
	    }
	}

	for (choice1 = 24; choice1 < 32; choice1++) {
	    if (ht[choice1].linmax >= max) {
		break;
	    }
	}

	sum0 = count_bit_ESC(ix, end, choice0);
	sum1 = count_bit_ESC(ix, end, choice1);

	if (sum0 > sum1)  {
	    sum0 = sum1;
	    choice0 = choice1;
	}
    }

    *s += sum0;
    return choice0;
}


int choose_table_takehiro_short(int *ix, int *end, int * s)
{
    int max;
    int choice0, sum0;
#define TAKEHIRO_SHORT
#ifdef TAKEHIRO_SHORT
    int choice1, sum1;
#endif

    max = ix_max_takehiro(ix, end);

    if (max <= 15)  {
	if (max == 0) {
	    return 0;
	}
	/* try tables with no linbits */
	for (choice0 = 1; choice0 < 14; choice0++) {
	    if (ht[choice0].xlen > max) {
		break;
	    }
	}

	sum0 = count_bit_short_noESC(ix, end, choice0);

#ifdef TAKEHIRO_SHORT
	choice1 = choice0;

	switch (choice0) {
	case 7:
	case 10:
	    choice1++;
	    sum1 = count_bit_short_noESC(ix, end, choice1);
	    if (sum0 > sum1) {
		sum0 = sum1;
		choice0 = choice1;
	    }
	    /*fall*/
	case 2:
	case 5:
	    choice1++;
	    sum1 = count_bit_short_noESC(ix, end, choice1);
	    if (sum0 > sum1) {
		sum0 = sum1;
		choice0 = choice1;
	    }
	    break;

	case 13:
	    choice1 += 2;
	    sum1 = count_bit_short_noESC(ix, end, choice1);
	    if (sum0 > sum1) {
		sum0 = sum1;
		choice0 = choice1;
	    }
	    break;

	default:
	    break;
	}
#endif
    } else {
	/* try tables with linbits */
	max -= 15;

	for (choice0 = 15; choice0 < 24; choice0++) {
	    if (ht[choice0].linmax >= max) {
		break;
	    }
	}
	sum0 = count_bit_short_ESC(ix, end, choice0);

#ifdef TAKEHIRO_SHORT
	for (choice1 = 24; choice1 < 32; choice1++) {
	    if (ht[choice1].linmax >= max) {
		break;
	    }
	}
	sum1 = count_bit_short_ESC(ix, end, choice1);

	if (sum0 > sum1)  {
	    sum0 = sum1;
	    choice0 = choice1;
	}
#endif
    }

    *s += sum0;
    return choice0;
}


static int count_bits_long(int ix[576], gr_info *gi)
{
    int i, a1, a2;
    int bits = 0;

    i=576;
    for (; i > 1; i -= 2) 
	if (ix[i - 1] | ix[i - 2])
	    break;

    /* Determines the number of bits to encode the quadruples. */
    gi->count1 = i;
    {
	int sum0 = 0, sum1 = 0;
	for (; i > 3; i -= 4) {
	    int p, v;
	    if ((unsigned int)(ix[i-1] | ix[i-2] | ix[i-3] | ix[i-4]) > 1)
		break;

	    v = ix[i-1];
	    p = v;
	    bits += v;

	    v = ix[i-2];
	    if (v != 0) {
		p += 2;
		bits++;
	    }

	    v = ix[i-3];
	    if (v != 0) {
		p += 4;
		bits++;
	    }

	    v = ix[i-4];
	    if (v != 0) {
		p += 8;
		bits++;
	    }

	    sum0 += ht[32].hlen[p];
	    sum1 += ht[33].hlen[p];
	}

	if (sum0 < sum1) {
	    bits += sum0;
	    gi->count1table_select = 0;
	} else {
	    bits += sum1;
	    gi->count1table_select = 1;
	}
    }

    gi->big_values = i;
    if (i == 0)
	return bits;

    if (gi->block_type == 0) {
	int index;
	int scfb_anz = 0;

	while (scalefac_band_long[++scfb_anz] < i) 
	    ;

	index = subdv_table[scfb_anz].region0_count;
	while (scalefac_band_long[index + 1] > i)
	    index--;
	gi->region0_count = index;

	index = subdv_table[scfb_anz].region1_count;
	while (scalefac_band_long[index + gi->region0_count + 2] > i)
	    index--;
	gi->region1_count = index;

	a1 = scalefac_band_long[gi->region0_count + 1];
	a2 = scalefac_band_long[index + gi->region0_count + 2];
	gi->table_select[2] = choose_table_takehiro(ix + a2, ix + i, &bits);
    } else {
	gi->region0_count = 7;
	//gi->region1_count = SBMAX_l - 7 - 1;
	gi->region1_count = SFB_LMAX -1 - 7 - 1;
	a1 = scalefac_band_long[7 + 1];
	a2 = i;
	if (a1 > a2) {
	    a1 = a2;
	}
    }

    /* Count the number of bits necessary to code the bigvalues region. */
    gi->table_select[0] = choose_table_takehiro(ix, ix + a1, &bits);
    gi->table_select[1] = choose_table_takehiro(ix + a1, ix + a2, &bits);
    return bits;
}




int count_bits(int *ix,gr_info *cod_info)  
{
  int bits=0,i;
  for ( i = 0; i < 576; i++ )  {
     if ( ix[i] > 8191 + 14) return 100000;/* report unsuitable quantizer */
  }


  if (cod_info->block_type==2) {
    cod_info->table_select[0] = choose_table_takehiro_short(ix, ix + 36, &bits);
    cod_info->table_select[1] = choose_table_takehiro_short(ix + 36, ix + 576, &bits);
  }else{
    bits=count_bits_long(ix, cod_info);
    cod_info->count1 = (cod_info->count1 - cod_info->big_values) / 4;
    cod_info->big_values /= 2;
  }
  return bits;

}

