# an SSTP library module in Python
# Tamito KAJIYAMA <17 May 2001>
# $Id: sstplib.py,v 1.2 2003/07/25 03:16:21 shy Exp $

import mimetools
import re
import select
import socket
import SocketServer
import string
import sys
import time

class SSTPServer(SocketServer.TCPServer):
    pass

class AsynchronousSSTPServer(SSTPServer):
    def handle_request(self, timeout=0):
        r, w, e = select.select([self.socket], [], [], timeout)
        if not r:
            return
        SSTPServer.handle_request(self)

class DirectSSTPServer(SocketServer.UnixStreamServer):
    pass

class AsynchronousDirectSSTPServer(DirectSSTPServer):
    def handle_request(self, timeout=0):
        r, w, e = select.select([self.socket], [], [], timeout)
        if not r:
            return
        DirectSSTPServer.handle_request(self)

class BaseSSTPRequestHandler(SocketServer.StreamRequestHandler):
    responses = {
        200: "OK",
        204: "No Content",
        210: "Break",
        400: "Bad Request",
        408: "Request Timeout",
        409: "Conflict",
        420: "Refuse",
        501: "Not Implemented",
        503: "Service Unavailable",
        510: "Not Local IP",
        511: "In Black List",
        512: "Invisible",
        }
    MessageClass = mimetools.Message
    re_requestsyntax = re.compile("^([A-Z]+) SSTP/([0-9]\\.[0-9])$")
    def parse_request(self, requestline):
        if requestline[-2:] == '\r\n':
            requestline = requestline[:-2]
        elif requestline[-1:] == '\n':
            requestline = requestline[:-1]
        self.requestline = requestline
        match = self.re_requestsyntax.match(requestline)
        if not match:
            self.requestline = "-"
            self.send_error(400, "Bad Request %s" % repr(requestline))
            return 0
        self.command, self.version = match.groups()
        self.headers = self.MessageClass(self.rfile, 0)
        return 1
    def handle(self):
        self.error = self.version = None
        if not self.parse_request(self.rfile.readline()):
            return
        name = "do_%s_%s_%s" % (self.command, self.version[0], self.version[2])
        if not hasattr(self, name):
            self.send_error(501, "Not Implemented (%s/%s)" % (
                self.command, self.version))
            return
        method = getattr(self, name)
        method()
    def send_error(self, code, message=None):
        self.error = code
        self.log_error(message or self.responses[code])
        self.send_response(code, self.responses[code])
    def send_response(self, code, message=None):
        self.log_request(code, message)
        self.wfile.write("SSTP/%s %d %s\r\n\r\n" % (
            self.version or "1.0", code, self.responses[code]))
    def log_error(self, message):
        sys.stderr.write("[%s] %s\n" % (self.timestamp(), message))
    def log_request(self, code, message=None):
        if self.requestline == "-":
            request = self.requestline
        else:
            request = '"' + self.requestline + '"'
        sys.stderr.write('%s [%s] %s %d %s\n' % (
            self.client_hostname(), self.timestamp(),
            request, code, message or self.responses[code]))
    def client_hostname(self):
        try:
            host, port = self.client_address
        except:
            path = self.client_address
            return 'localhost'
        try:
            hostname, aliaslist, ipaddrlist = socket.gethostbyaddr(host)
        except socket.error:
            hostname = host
        return hostname
    month_names = ["Jan", "Feb", "Mar", "Apr", "May", "Jun",
                   "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"]
    def timestamp(self):
        t = time.localtime(time.time())
        m = self.month_names[t[1]-1]
        return "%02d/%s/%d:%02d:%02d:%02d %+05d" % (
            t[2], m, t[0], t[3], t[4], t[5], -time.timezone/36)

def test(ServerClass = SSTPServer,
         HandlerClass = BaseSSTPRequestHandler,
         port = 9801): # or 11000
    sstpd = ServerClass(('', port), HandlerClass)
    print "Serving SSTP on port %d ..." % port
    sstpd.serve_forever()

if __name__ == '__main__':
    test()
