# Tamito KAJIYAMA <15 June 2001>
# $Id: home.py,v 1.12 2003/07/23 00:08:50 shy Exp $

import os
import re
import string
import sys

import ninix.config
import ninix.alias
import ninix.dll

import locale

NINIX_HOME = "~/.ninix"
NINIX_USER = None

locale_charset = locale.nl_langinfo(locale.CODESET)

def print_error(message):
    sys.stderr.write(message + "\n")

def get_ninix_home():
    return os.path.expanduser(NINIX_HOME)

def get_ninix_user():
    if NINIX_USER is None:
        return get_ninix_home()
    return os.path.expanduser(NINIX_USER)

def get_gtkrc():
    path = os.path.join(get_ninix_user(), "gtkrc")
    if os.path.exists(path):
        return path
    return os.path.join(get_ninix_home(), "gtkrc")

def get_pango_fontrc():
    return os.path.join(get_ninix_user(), "pango_fontrc")

def get_preferences():
    return os.path.join(get_ninix_user(), "preferences")

def load_config():
    home_dir = get_ninix_home()
    if not os.path.exists(home_dir):
        return None
    ghosts, shells = search_ghosts(home_dir)
    balloons = search_balloons(home_dir)
    ##for desc, shiori_dir, use_makoto, surface_set, balloon, prefix, shiori_dll, shiori_name in ghosts:
    ##    if balloon:
    ##        balloons.append(balloon)
    ##for shell_name, surface_set, balloon in shells:
    ##    if balloon:
    ##        balloons.append(balloon)
    return ghosts, shells, balloons, search_plugins(home_dir)

def get_shiori(path):
    table = {}
    shiori_lib = ninix.dll.Library('shiori', path, saori_lib=None)
    for file in os.listdir(path):
        if is_readable(os.path.join(path, file)):
            name = None
            if file[-3:] == '.py':
                name =  file[:-3]
            elif file[-4:] == '.pyc':
                name =  file[:-4]
            if name and not table.has_key(name):
                shiori = shiori_lib.request(('', name))
                if shiori:
                    table[name] = shiori
    return table

def search_ghosts(home_dir, target=None):
    ghosts = []
    shells = []
    if target:
        dirlist = []
        dirlist.extend(target)
    else:
        try:
            dirlist = os.listdir(os.path.join(home_dir, "ghost"))
        except OSError:
            dirlist = []
    shiori_table = get_shiori(os.path.join(os.environ['PYTHONPATH'], 'ninix/dll'))
    for subdir in dirlist:
        prefix = os.path.join(home_dir, "ghost", subdir)
        ghost_dir = os.path.join(prefix, "ghost", "master")
        desc = read_descript_txt(ghost_dir)
        if desc is None:
            desc = ninix.config.null_config()
        shiori_dll = desc.get('shiori')
        # find a pseudo AI, shells, and a built-in balloon
        candidate = {'name': '',
                     'score': 0}
        # SHIORI compatible modules
        for name in shiori_table.keys():
            score = int(shiori_table[name].find(ghost_dir, shiori_dll))
            if score > candidate['score']:
                candidate['name'] = name
                candidate['score'] = score
        shell_name, surface_set = find_surface_set(prefix)
        balloon = find_balloon(prefix)
        if candidate['score'] == 0:
            if surface_set:
                # use as a shell
                shells.append((shell_name, surface_set, balloon))
            continue
        shiori_name = candidate['name']
        if desc.get("name") == "default":
            pos = 0
        else:
            pos = len(ghosts)
        use_makoto = find_makoto_dll(ghost_dir)
        ghosts.insert(pos, (
            desc, ghost_dir, use_makoto, surface_set, balloon, prefix,
            shiori_dll, shiori_name))
    return ghosts, shells

def search_balloons(home_dir):
    balloon_dir = os.path.join(home_dir, "balloon")
    try:
        dirlist = os.listdir(balloon_dir)
    except OSError:
        dirlist = []
    buffer = []
    for subdir in dirlist:
        subdir = os.path.join(balloon_dir, subdir)
        if not os.path.isdir(subdir):
            continue
        desc = read_descript_txt(subdir) # REQUIRED
        if not desc:
            continue
        balloon_info = read_balloon_info(subdir) # REQUIRED
        if not balloon_info:
            continue
        if desc.get("name") == "default":
            pos = 0
        else:
            pos = len(buffer)
        buffer.insert(pos, (desc, balloon_info))
    return buffer

def search_plugins(home_dir):
    buffer = []
    dir = os.path.join(home_dir, "plugin")
    try:
        dirlist = os.listdir(dir)
    except OSError:
        dirlist = []
    for subdir in dirlist:
        plugin = read_plugin_txt(os.path.join(dir, subdir))
        if plugin is None:
            continue
        buffer.append(plugin)
    return buffer

def read_descript_txt(dir):
    path = os.path.join(dir, "descript.txt")
    if is_readable(path):
        return ninix.config.open(path)
    return None

def read_install_txt(dir):
    path = os.path.join(dir, "install.txt")
    if is_readable(path):
        return ninix.config.open(path)
    return None

def read_alias_txt(dir):
    path = os.path.join(dir, "alias.txt")
    if is_readable(path):
        return ninix.alias.open(path)
    return None

def find_makoto_dll(dir):
    if is_readable(os.path.join(dir, "makoto.dll")):
        return 1
    return 0

def find_surface_set(dir):
    desc = read_descript_txt(os.path.join(dir, "ghost", "master"))
    if desc:
        shell_name = desc.get("name")
    else:
        shell_name = None
    if not shell_name:
        inst = read_install_txt(dir)
        if inst:
            shell_name = inst.get("name")
    surface_set = []
    shell_dir = os.path.join(dir, "shell")
    for name, desc, subdir in find_surface_dir(shell_dir):
        surface_dir = os.path.join(shell_dir, subdir)
        surface_info, alias = read_surface_info(surface_dir)
        if surface_info and \
           surface_info.has_key("surface0") and \
           surface_info.has_key("surface10"):
            if alias is None:
                alias = read_alias_txt(surface_dir)
            surface_set.append((name, desc, alias, surface_info))
    return shell_name, surface_set

def find_surface_dir(dir):
    buffer = []
    path = os.path.join(dir, "surface.txt")
    if os.path.exists(path):
        config = ninix.config.open(path)
        for name, subdir in config.itemlist:
            desc = read_descript_txt(os.path.join(dir, string.lower(subdir)))
            if desc is None:
                desc = ninix.config.null_config()
            buffer.append((name, desc, string.lower(subdir)))
    else:
        try:
            dirlist = os.listdir(dir)
        except OSError:
            dirlist = []
        for subdir in dirlist:
            desc = read_descript_txt(os.path.join(dir, subdir))
            if desc is None:
                desc = ninix.config.null_config()
            name = desc.get("name", subdir)
            buffer.append((name, desc, subdir))
    return buffer

re_surface = re.compile("surface([0-9]+)\.(png|dgp)")

def read_surface_info(surface_dir):
    surface = {}
    try:
        filelist = os.listdir(surface_dir)
    except OSError:
        filelist = []
    filename_alias = {}
    path = os.path.join(surface_dir, "alias.txt")
    if os.path.exists(path):
        dict = ninix.alias.open(path)
        for basename, alias in dict.items():
            if basename[:7] == "surface":
                filename_alias[alias] = basename
    # find png image and associated configuration file
    for filename in filelist:
        basename, suffix = os.path.splitext(filename)
        if filename_alias.has_key(basename):
            match = re_surface.match(filename_alias[basename] + suffix)
        else:
            match = re_surface.match(filename)
        if not match:
            continue
        img = os.path.join(surface_dir, filename)
        if not is_readable(img):
            continue
        key = "surface" + str(int(match.group(1)))
        txt = os.path.join(surface_dir, basename + "s.txt")
        if is_readable(txt):
            config = ninix.config.open(txt)
        else:
            config = ninix.config.null_config()
        txt = os.path.join(surface_dir, basename + "a.txt")
        if is_readable(txt):
            config.update(ninix.config.open(txt))
        surface[key] = (img, config)
    # find surfaces.txt
    alias = None
    for key, config in read_surfaces_txt(surface_dir):
        if key == "__alias__":
            alias = config
        elif key[:7] == "surface":
            try:
                img, null_config = surface[key]
            except KeyError:
                img = None
            surface[key] = (img, config)
    # find surface elements
    for key in surface.keys():
        img, config = surface[key]
        for key, method, filename, x, y in list_surface_elements(config):
            filename = string.lower(filename)
            basename, suffix = os.path.splitext(filename)
            if not surface.has_key(basename):
                surface[basename] = (os.path.join(surface_dir, filename),
                                     ninix.config.null_config())
    return surface, alias

def read_surfaces_txt(surface_dir):
    list = []
    path = os.path.join(surface_dir, "surfaces.txt")
    try:
        file = open(path)
    except IOError:
        return list
    alias_buffer = []
    while 1:
        line = file.readline()
        if not line:
            break
        if line[0] == "#" or line[:2] == "//":
            continue
        key = string.strip(line)
        if not key:
            continue
        try:
            while 1:
                line = file.readline()
                if not line:
                    raise ValueError, "unexpected end of file"
                line = string.strip(string.replace(line, "\x81\x40", ""))
                if not line:
                    continue
                elif line == "{":
                    break
                key = line # ignore the preceding key
            buffer = []
            while 1:
                line = file.readline()
                if not line:
                    raise ValueError, "unexpected end of file"
                line = string.strip(string.replace(line, "\x81\x40", ""))
                if not line:
                    continue
                elif line == "}":
                    break
                buffer.append(line)
            if key in ["sakura.surface.alias", "kero.surface.alias"]:
                alias_buffer.append(key)
                alias_buffer.append("{")
                alias_buffer.extend(buffer)
                alias_buffer.append("}")
            elif key[:7] == "surface":
                try:
                    key = key[:7] + str(int(key[7:]))
                except ValueError:
                    pass
                list.append((key, ninix.config.new_config(buffer)))
        except ValueError, error:
            print_error("%s: %s (parsing not completed)" % (path, error))
            break
    if alias_buffer:
        list.append(("__alias__", ninix.alias.new_alias(alias_buffer)))
    return list

def list_surface_elements(config):
    buffer = []
    for n in range(256):
        key = "element" + str(n)
        if not config.has_key(key):
            break
        spec = map(string.strip, string.split(config[key], ","))
        try:
            method, filename, x, y = spec
            x = int(x)
            y = int(y)
        except ValueError:
            print_error("invalid element spec for %s: %s" % (key, config[key]))
            continue
        buffer.append((key, method, filename, x, y))
    return buffer

def find_balloon(dir):
    inst = read_install_txt(dir)
    if inst:
        balloon_dir = inst.get("balloon.directory")
        if balloon_dir:
            path = os.path.join(
                dir, "ghost", "master", string.lower(balloon_dir))
            desc = read_descript_txt(path)
            info = read_balloon_info(path)
            if desc and info:
                return (desc, info)
    return None

re_balloon = re.compile("balloon([skc][0-9]+)\.(png)")
re_annex   = re.compile("(arrow[01]|sstp)\.(png)")

def read_balloon_info(balloon_dir):
    balloon = {}
    try:
        filelist = os.listdir(balloon_dir)
    except OSError:
        filelist = []
    for filename in filelist:
        match = re_balloon.match(filename)
        if not match:
            continue
        img = os.path.join(balloon_dir, filename)
        if match.group(2) != "png" and \
           is_readable(img[-3:] + "png"):
                continue
        if not is_readable(img):
            continue
        key = match.group(1)
        txt = os.path.join(balloon_dir, "balloon%ss.txt" % key)
        if is_readable(txt):
            config = ninix.config.open(txt)
        else:
            config = ninix.config.null_config()
        balloon[key] = (img, config)
    for filename in filelist:
        match = re_annex.match(filename)
        if not match:
            continue
        img = os.path.join(balloon_dir, filename)
        if not is_readable(img):
            continue
        key = match.group(1)
        config = ninix.config.null_config()
        balloon[key] = (img, config)
    return balloon

def read_plugin_txt(plugin_dir):
    path = os.path.join(plugin_dir, "plugin.txt")
    try:
        file = open(path)
    except IOError:
        return None
    charset = 'EUC-JP' # default
    plugin_name = startup = None
    menu_items = []
    error = None
    lineno = 0
    for line in file.readlines():
        lineno = lineno + 1
        if not string.strip(line) or line[0] == "#":
            continue
        pos = string.find(line, ":")
        if pos < 0:
            error = "line %d: syntax error" % lineno
            break
        name, value = string.strip(line[:pos]), string.strip(line[pos+1:])
        if name == "charset":
            charset = value
        elif name == "name":
            plugin_name = unicode(value, charset, 'ignore')
        elif name == "startup":
            list = string.split(value, ",")
            list[0] = os.path.join(plugin_dir, list[0])
            if not os.path.exists(list[0]):
                error = "line %d: invalid program name" % lineno
                break
            startup = list
        elif name == "menuitem":
            list = string.split(unicode(value, charset, 'ignore'), ",")
            if len(list) < 2:
                error = "line %d: syntax error" % lineno
                break
            list[1] = os.path.join(plugin_dir, list[1])
            if not os.path.exists(list[1]):
                error = "line %d: invalid program name" % lineno
                break
            menu_items.append((list[0], list[1:]))
        else:
            error = "line %d: syntax error" % lineno
            break
    else:
        if plugin_name is None:
            error = 'the "name" header field is required'
        elif not startup and not menu_items:
            error = 'either "startup" or "menuitem" header field is required'
    file.close()
    if error:
        sys.stderr.write("Error: %s\n%s (skipped)\n" % (error, path))
        return None
    return plugin_name, plugin_dir, startup, menu_items

def is_readable(path):
    try:
        open(path).read(64)
    except IOError:
        return 0
    return 1
    
###   TEST   ###

def test():
    import locale
    locale.setlocale(locale.LC_ALL, '')
    global NINIX_HOME
    try:
        NINIX_HOME = os.environ["NINIX_HOME"]
    except KeyError:
        pass
    config = load_config()
    if config is None:
        sys.stderr.write("Home directory not found.\n")
        sys.exit(1)
    ghosts, shells, balloons, plugins = config
    # ghosts
    for desc, shiori_dir, use_makoto, surface_set, balloon, prefix, shiori_dll, shiori_name in ghosts:
        print "GHOST", "=" * 50
        print prefix
        print str(desc).encode(locale_charset, 'ignore')
        print shiori_dir
        print shiori_dll
        print shiori_name
        print "use_makoto =", use_makoto
        if surface_set:
            for name, desc, alias, surface in surface_set:
                print "-" * 50
                print "surface:", name.encode(locale_charset, 'ignore')
                print str(desc).encode(locale_charset, 'ignore')
                for k, v in surface.items():
                    print k, "=", v[0]
                    print str(v[1]).encode(locale_charset, 'ignore')
                if alias:
                    buffer = []
                    for k, v in alias.items():
                        if k in ["sakura.surface.alias", "kero.surface.alias"]:
                            print k + ":"
                            for id, list in v.items():
                                print id, "= [" + string.join(list, ", ") + "]"
                            print
                        else:
                            buffer.append((k, v))
                    if buffer:
                        print "filename alias:"
                        for k, v in buffer:
                            print k, "=", v
                        print
        if balloon:
            print "-" * 50
            desc, balloon = balloon
            print str(desc).encode(locale_charset, 'ignore')
            for k, v in balloon.items():
                print k, "=", v[0]
                print str(v[1]).encode(locale_charset, 'ignore')
    # shells
    for shell_name, surface_set, balloon in shells:
        print "SHELL", "=" * 50
        print shell_name.encode(locale_charset, 'ignore')
        for name, desc, alias, surface in surface_set:
            print "-" * 50
            print "surface:", name.encode(locale_charset, 'ignore')
            print str(desc).encode(locale_charset, 'ignore')
            for k, v in surface.items():
                print k, "=", v[0]
                print str(v[1]).encode(locale_charset, 'ignore')
            if alias:
                buffer = []
                for k, v in alias.items():
                    if k in ["sakura.surface.alias", "kero.surface.alias"]:
                        print k + ":"
                        for id, list in v.items():
                            print id, "= [" + string.join(list, ", ") + "]"
                        print
                    else:
                        buffer.append((k, v))
                if buffer:
                    print "filename alias:"
                    for k, v in buffer:
                        print k, "=", v
                    print
        if balloon:
            print "-" * 50
            desc, balloon = balloon
            print str(desc).encode(locale_charset, 'ignore')
            for k, v in balloon.items():
                print k, "=", v[0]
                print str(v[1]).encode(locale_charset, 'ignore')
    # balloons
    for desc, balloon in balloons:
        print "BALLOON", "=" * 50
        print str(desc).encode(locale_charset, 'ignore')
        for k, v in balloon.items():
            print k, "=", v[0]
            print str(v[1]).encode(locale_charset, 'ignore')
    # plugins
    for plugin_name, plugin_dir, startup, menu_items in plugins:
        print "PLUGIN", "=" * 50
        print "name =", plugin_name.encode(locale_charset, 'ignore')
        if startup:
            print "startup =", '["' + string.join(startup, '", "') + '"]'
        for label, argv in menu_items:
            print 'menuitem "%s" =' % label.encode(locale_charset, 'ignore'),
            print '["' + string.join(argv, '", "') + '"]'

if __name__ == "__main__":
    test()
