;;; mew-virtual.el --- Virtual mode for Mew

;; Author:  Kazu Yamamoto <Kazu@Mew.org>
;; Created: Oct  2, 1996

;;; Code:

(require 'mew)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Virtual info
;;;

(defvar mew-vinfo-list '("count" "func" "top" "db" "thread-p" "column"))

(mew-blinfo-defun 'mew-vinfo mew-vinfo-list)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Virtual mode
;;;

(defun mew-virtual-mode ()
  "Mew Virtual mode:: major mode to visualize messages in a virtual folder.
For more information, see the document of 'mew-summary-mode'."
  (interactive)
  (setq major-mode 'mew-virtual-mode)
  (setq mode-line-buffer-identification mew-mode-line-id)
  (use-local-map mew-summary-mode-map)
  (setq buffer-read-only t)
  (setq truncate-lines t)
  ;;
  (make-local-variable 'tab-width)
  (make-local-variable 'search-invisible)
  (setq search-invisible nil)
  (cond
   (mew-gemacs-p
    (unless (mew-thread-p)
      (require 'font-lock) ;; xxx should be removed Emacs 21.0.104
      (jit-lock-register 'mew-summary-cook-region)))
   (t
    (make-local-hook 'window-scroll-functions)
    (add-hook 'window-scroll-functions 'mew-summary-cook-window nil 'local)))
  (mew-sinfo-set-disp-msg t)
  ;;
  (mew-summary-mode-name mew-mode-name-virtual)
  (mew-summary-setup-mode-line)
  (mew-summary-setup-decoration)
  (mew-highlight-cursor-line)
  (run-hooks 'mew-virtual-mode-hook))

(defun mew-virtual-folder-message ()
  "Display the real folder of this message."
  (interactive)
  (looking-at "^ *\\([0-9]+\\).*\r\\(.*\\)$")
  (message "%s" (mew-match 2)))

(defun mew-summary-virtual (&optional ext)
  "Create Virtual mode with inputed pattern.
If called with '\\[universal-argument]',
execute 'mew-summary-virtual-with-external'. Otherwise, 
'mew-summary-virtual-with-internal' is called."
  (interactive "P")
  (if (not (mew-summary-or-virtual-p))
      (message "This command cannot be used in this mode")
    (if ext
	(if mew-summary-virtual-with-external-function
	    (mew-summary-virtual-with-external)
	  (message "mew-summary-virtual-with-external-function is nil"))
      (mew-summary-virtual-with-internal))))

(defun mew-summary-virtual-with-internal ()
  "Create Virtual mode with mewls."
  (let* ((str (mew-input-string "Virtual folder name %s(%s): " "" "virtual"))
	 (vfolder (concat "++" str)) ;; xxx
	 (flds (mew-input-folders (mew-summary-folder-name)))
	 (pattern (mew-input-pick-pattern))
	 args)
    (mew-summary-switch-to-folder vfolder)
    (mew-erase-buffer)
    (mew-vinfo-set-count 1)
    (mew-vinfo-set-func nil)
    (while flds
      (setq args (cons (car flds) (cons "-s" args)))
      (setq flds (cdr flds)))
    (setq args (cons "-a" (cons "-p" (cons pattern (nreverse args)))))
    (mew-scan args nil 'virtual)))

(defvar mew-summary-virtual-with-external-function
  'mew-summary-virtual-with-grep
  "*A function to be called by '\\[universal-argument] \\<mew-summary-mode-map>\\[mew-summary-virtual]'.
This function MUST returns a file name which contains message numbers.")

(defun mew-summary-virtual-with-external ()
  "Create Virtual mode with 'mew-summary-virtual-with-external-function'."
  (interactive)
  (let* ((str (mew-input-string "Virtual folder name %s(%s): " "" "virtual"))
	 (vfolder (concat "++" str)) ;; xxx
	 (buf (generate-new-buffer mew-buffer-prefix))
	 (flds (mew-input-folders (mew-summary-folder-name)))
	 (pattern (mew-input-grep-pattern))
	 file)
    (save-excursion
      (set-buffer buf)
      (mew-erase-buffer)
      (mew-piolet
       mew-cs-text-for-read mew-cs-text-for-write
       (let ((file-name-coding-system nil)) ;; for XEmacs
	 (setq file (funcall mew-summary-virtual-with-external-function
			     flds pattern)))))
    (mew-remove-buffer buf)
    (mew-summary-switch-to-folder vfolder)
    (mew-erase-buffer)
    (mew-vinfo-set-count 1)
    (mew-vinfo-set-func
     `(lambda () (if (file-exists-p ,file) (delete-file ,file))))
    (mew-scan (list "-i" file) nil 'virtual)))

(defun mew-summary-virtual-with-grep (flds pattern)
  (setq pattern (mew-cs-encode-arg pattern))
  (let ((file (mew-make-temp-name))
	picks fld dir msgs nxt)
    (while flds
      (setq dir (mew-expand-folder (car flds)))
      (let ((default-directory dir))
	(setq msgs (mew-dir-messages "."))
	(if (= (length msgs) 1) (setq msgs (cons "/dev/null" msgs)))
	(cd dir)
	(mew-erase-buffer)
	(message "Picking in %s..." (car flds))
	(while msgs
	  (goto-char (point-max))
	  (setq nxt (nthcdr mew-prog-grep-max-msgs msgs))
	  (if nxt (setcdr (nthcdr (1- mew-prog-grep-max-msgs) msgs) nil))
	  (apply (function call-process)
		 mew-prog-vgrep nil t nil
		 (append mew-prog-vgrep-opts (list pattern) msgs))
	  (setq msgs nxt))
	(message "Picking in %s...done" (car flds)))
      (goto-char (point-min))
      (setq msgs nil)
      (while (not (eobp))
	(if (looking-at mew-regex-message-files2)
	    (setq msgs (cons (mew-match 0) msgs)))
	(forward-line))
      (setq msgs (mew-uniq-list msgs))
      (setq msgs (mapcar (function string-to-int) msgs))
      (setq msgs (sort msgs (function <)))
      (setq msgs (mapcar (function int-to-string) msgs))
      (if (null msgs)
	  ()
	(setq picks (cons (cons dir msgs) picks)))
      (setq flds (cdr flds)))
    (mew-erase-buffer)
    (setq picks (nreverse picks))
    (while picks
      (setq fld (car (car picks)))
      (if (string-match
	   (concat "^" (file-name-as-directory
			(expand-file-name mew-mail-path)))
	   fld)
	  (setq fld (concat "+" (substring fld (match-end 0) )))) ;; xxx
      (insert "CD: " fld "\n")
      (setq msgs (cdr (car picks)))
      (while msgs
	(insert (car msgs) "\n")
	(setq msgs (cdr msgs)))
      (setq picks (cdr picks)))
    (mew-frwlet
     mew-cs-text-for-read mew-cs-text-for-write
     (write-region (point-min) (point-max) file nil 'nomsg))
    file))

(provide 'mew-virtual)

;;; Copyright Notice:

;; Copyright (C) 1996-2001 Mew developing team.
;; All rights reserved.

;; Redistribution and use in source and binary forms, with or without
;; modification, are permitted provided that the following conditions
;; are met:
;; 
;; 1. Redistributions of source code must retain the above copyright
;;    notice, this list of conditions and the following disclaimer.
;; 2. Redistributions in binary form must reproduce the above copyright
;;    notice, this list of conditions and the following disclaimer in the
;;    documentation and/or other materials provided with the distribution.
;; 3. Neither the name of the team nor the names of its contributors
;;    may be used to endorse or promote products derived from this software
;;    without specific prior written permission.
;; 
;; THIS SOFTWARE IS PROVIDED BY THE TEAM AND CONTRIBUTORS ``AS IS'' AND
;; ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
;; IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
;; PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE TEAM OR CONTRIBUTORS BE
;; LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
;; CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
;; SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
;; BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
;; WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
;; OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN
;; IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

;;; mew-virtual.el ends here
