/*
 *	LAME MP3 encoding engine
 *
 *	Copyright (c) 1999 Mark Taylor
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <assert.h>

#ifdef HAVEGTK
#include "gtkanal.h"
#include <gtk/gtk.h>
#endif
#include "util.h"
#include "globalflags.h"
#include "psymodel.h"
#include "fastmdct.h"
#include "quantize.h"
#include "l3bitstream.h"
#include "reservoir.h"
#include "formatBitstream.h"
#include "version.h"
#include "VbrTag.h"
#include "id3tag.h"
#include "get_audio.h"



/* Global flags.  defined extern in globalflags.h */
/* default values set in lame_init() */
int allow_diff_short;
int ATHonly;
int autoconvert;
int experimentalX;
int experimentalY;
int experimentalZ;
int force_ms;
int fast_mode;
long int frameNum;
int gtkflag;
int g_bWriteVbrTag;
int gpsycho;
int highq;
sound_file_format input_format;
int lame_nowrite;
double resample_ratio;
int sfb21;
int silent;
int swapbytes;           /* force byte swapping */
long totalframes;        /* frames: 0..totalframes-1 */
int VBR;
int VBR_q;
int VBR_min_bitrate;   /* 32kbs */
int VBR_max_bitrate;  /* 256kbs */
int voice_mode;



/* Global variable definitions for "musicin.c" */
ID3TAGDATA id3tag;
struct stat sb;
Bit_stream_struc   bs;
III_side_info_t l3_side;
frame_params fr_ps;
char    inPath[MAX_NAME_SIZE];
char    outPath[MAX_NAME_SIZE];
int target_bitrate;
char               *programName;
unsigned long num_samples;
static layer info;

#ifdef BRHIST
#include <string.h>
#include <termcap.h>
void brhist_init(int br_min, int br_max);
void brhist_add_count(void);
void brhist_disp(void);
void brhist_disp_total(void);
extern long brhist_temp[15];
int disp_brhist = 1;
#endif



/************************************************************************
*
* usage
*
* PURPOSE:  Writes command line syntax to the file specified by #stderr#
*
************************************************************************/

void lame_usage(char *name)  /* print syntax & exit */
{
  fprintf(stderr,"\n");
  fprintf(stderr,"USAGE   :  %s [options] <infile> [outfile]\n",name);
  fprintf(stderr,"\n<infile> and/or <outfile> can be \"-\", which means stdin/stdout.\n");
  fprintf(stderr,"\n");
  fprintf(stderr,"OPTIONS :\n");
  fprintf(stderr,"    -m mode         (s)tereo, (j)oint, (f)orce or (m)ono  (default %c)\n",DFLT_MOD);
  fprintf(stderr,"                    force = force ms_stereo on all frames. Faster and\n");
  fprintf(stderr,"                    uses special Mid & Side masking thresholds\n");
  fprintf(stderr,"    -b <bitrate>    set the bitrate, default 128kbps\n");
  fprintf(stderr,"                    (for VBR, this sets the allowed minimum bitrate)\n");
  fprintf(stderr,"    -s sfreq        sampling frequency of input file(kHz) - default %4.1f\n",DFLT_SFQ);
  fprintf(stderr,"  --resample sfreq  sampling frequency of output file(kHz)- default=input sfreq\n");
  fprintf(stderr,"  --mp3input        input file is a MP3 file\n");
  fprintf(stderr,"  --voice           experimental voice mode\n");
  fprintf(stderr,"\n");
  fprintf(stderr,"    -v              use variable bitrate (VBR)\n");
  fprintf(stderr,"    -V n            quality setting for VBR.  default n=%i\n",VBR_q);
  fprintf(stderr,"                    0=high quality,bigger files. 9=smaller files\n");
  fprintf(stderr,"    -t              disable Xing VBR informational tag\n");
  fprintf(stderr,"    --nohist        disable VBR histogram display\n");
  fprintf(stderr,"\n");
  fprintf(stderr,"    -h              use (maybe) quality improvements\n");
  fprintf(stderr,"    -f              fast mode (low quality)\n");
  fprintf(stderr,"    -k              disable sfb=21 cutoff\n");
  fprintf(stderr,"    -d              allow channels to have different blocktypes\n");
  fprintf(stderr,"  --athonly         only use the ATH for masking\n");
  fprintf(stderr,"\n");
  fprintf(stderr,"    -r              input is raw pcm\n");
  fprintf(stderr,"    -x              force byte-swapping of input\n");
  fprintf(stderr,"    -a              downmix from stereo to mono file for mono encoding\n");
  fprintf(stderr,"    -e emp          de-emphasis n/5/c  (obsolete)\n");
  fprintf(stderr,"    -p              error protection.  adds 16bit checksum to every frame\n");
  fprintf(stderr,"                    (the checksum is computed correctly)\n");
  fprintf(stderr,"    -c              mark as copyright\n");
  fprintf(stderr,"    -o              mark as non-original\n");
  fprintf(stderr,"    -S              don't print progress report, VBR histograms\n");
  fprintf(stderr,"\n");
  fprintf(stderr,"  Specifying any of the following options will add an ID3 tag\n");
  fprintf(stderr,"     --tt <title>     title of song (max 30 chars)\n");
  fprintf(stderr,"     --ta <artist>    artist who did the song (max 30 chars)\n");
  fprintf(stderr,"     --tl <album>     album where it came from (max 30 chars)\n");
  fprintf(stderr,"     --ty <year>      year in which the song/album was made (max 4 chars)\n");
  fprintf(stderr,"     --tc <comment>   additional info (max 30 chars)\n");
  fprintf(stderr,"\n");
#ifdef HAVEGTK
  fprintf(stderr,"    -g              run graphical analysis on <infile>\n");
#endif
  display_bitrates(2);
  exit(1);
}






/************************************************************************
*
* parse_args
*
* PURPOSE:  Sets encoding parameters to the specifications of the
* command line.  Default settings are used for parameters
* not specified in the command line.
*
* If the input file is in WAVE or AIFF format, the sampling frequency is read
* from the AIFF header.
*
* The input and output filenames are read into #inpath# and #outpath#.
*
************************************************************************/
void lame_parse_args(int argc, char **argv)
{
  FLOAT srate;
  int   brate;
  layer *info = fr_ps.header;
  int   err = 0, i = 0;
  int mode_given=0;
  int num_channels=2;   /* number of channels of INPUT file */
  int samplerate,resamplerate=0;
  int framesize;
  double compression_ratio;

  /* preset defaults */
  programName = argv[0]; 
  inPath[0] = '\0';   outPath[0] = '\0';
  info->lay = DFLT_LAY;
  switch(DFLT_MOD) {
  case 's': info->mode = MPG_MD_STEREO; info->mode_ext = MPG_MD_LR_LR; break;
  case 'd': info->mode = MPG_MD_DUAL_CHANNEL; info->mode_ext=MPG_MD_LR_LR; break;
  case 'j': info->mode = MPG_MD_JOINT_STEREO; info->mode_ext=MPG_MD_LR_LR; break;
  case 'm': info->mode = MPG_MD_MONO; info->mode_ext = MPG_MD_LR_LR; break;
  default:
    fprintf(stderr, "%s: Bad mode dflt %c\n", programName, DFLT_MOD);
    abort();
  }
  samplerate = 1000*DFLT_SFQ;
  if((info->sampling_frequency = SmpFrqIndex((long)samplerate, &info->version)) < 0) {
    fprintf(stderr, "%s: bad sfrq default %.2f\n", programName, DFLT_SFQ);
    abort();
  }

  info->bitrate_index = 9;
  brate = 0;
  switch(DFLT_EMP) {
  case 'n': info->emphasis = 0; break;
  case '5': info->emphasis = 1; break;
  case 'c': info->emphasis = 3; break;
  default: 
    fprintf(stderr, "%s: Bad emph dflt %c\n", programName, DFLT_EMP);
    abort();
  }
  info->copyright = 0;
  info->original = 1;
  info->error_protection = FALSE;
  
#ifndef _BLADEDLL
  id3_inittag(&id3tag);
  id3tag.used = 0;
#endif

  /* process args */
  while(++i<argc && err == 0) {
    char c, *token, *arg, *nextArg;
    int  argUsed;
    
    token = argv[i];
    if(*token++ == '-') {
      if(i+1 < argc) nextArg = argv[i+1];
      else           nextArg = "";
      argUsed = 0;
      if (! *token) {
	/* The user wants to use stdin and/or stdout. */
	if(inPath[0] == '\0')       strncpy(inPath, argv[i],MAX_NAME_SIZE);
	else if(outPath[0] == '\0') strncpy(outPath, argv[i],MAX_NAME_SIZE);
      } 
      if (*token == '-') {
	/* GNU style */
	token++;

	if (strcmp(token, "resample")==0) {
	  argUsed=1;
	  srate = atof( nextArg );
	  /* samplerate = rint( 1000.0 * srate ); $A  */
	  resamplerate =  (( 1000.0 * srate ) + 0.5);
	  if (srate  < 1) {
	    fprintf(stderr,"Must specify samplerate with --resample\n");
	    exit(1);
	  }
	}
	else if (strcmp(token, "mp3input")==0) {
	  input_format=sf_mp3;
	}
	else if (strcmp(token, "voice")==0) {
	  voice_mode=1;
	}
	else if (strcmp(token, "athonly")==0) {
	  ATHonly=1;
	}
	else if (strcmp(token, "nohist")==0) {
#ifdef BRHIST
	  disp_brhist = 0;
#endif
	}
	/* options for ID3 tag */
 	else if (strcmp(token, "tt")==0) {
 		id3tag.used=1;      argUsed = 1;
  		strncpy(id3tag.title, nextArg, 30);
 		}
 	else if (strcmp(token, "ta")==0) {
 		id3tag.used=1; argUsed = 1;
  		strncpy(id3tag.artist, nextArg, 30);
 		}
 	else if (strcmp(token, "tl")==0) {
 		id3tag.used=1; argUsed = 1;
  		strncpy(id3tag.album, nextArg, 30);
 		}
 	else if (strcmp(token, "ty")==0) {
 		id3tag.used=1; argUsed = 1;
  		strncpy(id3tag.year, nextArg, 4);
 		}
 	else if (strcmp(token, "tc")==0) {
 		id3tag.used=1; argUsed = 1;
  		strncpy(id3tag.comment, nextArg, 30);
 		}
	else
	  {
	    fprintf(stderr,"%s: unrec option --%s\n",
		    programName, token);
	  }
	i += argUsed;
	
      } else  while( (c = *token++) ) {
	if(*token ) arg = token;
	else                             arg = nextArg;
	switch(c) {
	case 'm':        argUsed = 1;
	  mode_given=1;
	  if (*arg == 's')
	    { info->mode = MPG_MD_STEREO; info->mode_ext = MPG_MD_LR_LR; }
	  else if (*arg == 'd')
	    { info->mode = MPG_MD_DUAL_CHANNEL; info->mode_ext=MPG_MD_LR_LR; }
	  else if (*arg == 'j')
	    { info->mode = MPG_MD_JOINT_STEREO; info->mode_ext=MPG_MD_LR_LR; }
	  else if (*arg == 'f')
	    { info->mode = MPG_MD_JOINT_STEREO; force_ms=1; info->mode_ext=MPG_MD_LR_LR; }
	  else if (*arg == 'm')
	    { info->mode = MPG_MD_MONO; info->mode_ext = MPG_MD_LR_LR; }
	  else {
	    fprintf(stderr,"%s: -m mode must be s/d/j/f/m not %s\n",
		    programName, arg);
	    err = 1;
	  }
	  break;
	case 'V':        argUsed = 1;   VBR = 1;
	  if (*arg == '0')
	    { VBR_q=0; }
	  else if (*arg == '1')
	    { VBR_q=1; }
	  else if (*arg == '2')
	    { VBR_q=2; }
	  else if (*arg == '3')
	    { VBR_q=3; }
	  else if (*arg == '4')
	    { VBR_q=4; }
	  else if (*arg == '5')
	    { VBR_q=5; }
	  else if (*arg == '6')
	    { VBR_q=6; }
	  else if (*arg == '7')
	    { VBR_q=7; }
	  else if (*arg == '8')
	    { VBR_q=8; }
	  else if (*arg == '9')
	    { VBR_q=9; }
	  else {
	    fprintf(stderr,"%s: -V n must be 0-9 not %s\n",
		    programName, arg);
	    err = 1;
	  }
	  break;
	  
	case 's':
	  argUsed = 1;
	  srate = atof( arg );
	  /* samplerate = rint( 1000.0 * srate ); $A  */
	  samplerate =  (( 1000.0 * srate ) + 0.5);
	  break;
	case 'b':        
	  argUsed = 1;
	  brate = atoi(arg); 
	  break;
	case 't':  /* dont write VBR tag */
		g_bWriteVbrTag=0;
	  break;
	case 'r':  /* force raw pcm input file */
#ifdef LIBSNDFILE
	  fprintf(stderr,"WARNING: libsndfile may ignore -r and perform fseek's on the input.\n");
	  fprintf(stderr,"Compile without libsndfile if this is a problem.\n");
#endif
	  input_format=sf_raw;
	  break;
	case 'x':  /* force byte swapping */
	  swapbytes=TRUE;
	  break;
	case 'p': /* (jo) error_protection: add crc16 information to stream */
	  info->error_protection = 1; 
	  break;
	case 'a': /* autoconvert input file from stereo to mono - for mono mp3 encoding */
	  autoconvert = TRUE;
	  break;
	case 'h': 
	  highq = TRUE;
	  break;
	case 'k': 
	  sfb21 = FALSE;
	  break;
	case 'd': 
	  allow_diff_short = 1;
	  break;
	case 'v': 
	  VBR = 1;
	  break;
	case 'S': 
	  silent = TRUE;
	  break;
	case 'X':        argUsed = 1;   experimentalX = 0;
	  if (*arg == '0')
	    { experimentalX=0; }
	  else if (*arg == '1')
	    { experimentalX=1; }
	  else if (*arg == '2')
	    { experimentalX=2; }
	  else if (*arg == '3')
	    { experimentalX=3; }
	  else if (*arg == '4')
	    { experimentalX=4; }
	  else if (*arg == '5')
	    { experimentalX=5; }
	  else {
	    fprintf(stderr,"%s: -X n must be 0-5 not %s\n",
		    programName, arg);
	    err = 1;
	  }
	  break;


	case 'Y': 
	  experimentalY = TRUE;
	  break;
	case 'Z': 
	  experimentalZ = TRUE;
	  break;
	case 'f': 
	  fast_mode = 1;
	  break;
#ifdef HAVEGTK
	case 'g': /* turn on gtk analysis */
	  gtkflag = TRUE;
	  break;
#endif
	case 'e':        argUsed = 1;
	  if (*arg == 'n')                    info->emphasis = 0;
	  else if (*arg == '5')               info->emphasis = 1;
	  else if (*arg == 'c')               info->emphasis = 3;
	  else {
	    fprintf(stderr,"%s: -e emp must be n/5/c not %s\n",
		    programName, arg);
	    err = 1;
	  }
	  break;
	case 'c':       info->copyright = 1; break;
	case 'o':       info->original  = 0; break;
	default:        fprintf(stderr,"%s: unrec option %c\n",
				programName, c);
	err = 1; break;
	}
	if(argUsed) {
	  if(arg == token)    token = "";   /* no more from token */
	  else                ++i;          /* skip arg we used */
	  arg = ""; argUsed = 0;
	}
      }
    }
    else {
      if(inPath[0] == '\0')       strncpy(inPath, argv[i], MAX_NAME_SIZE);
      else if(outPath[0] == '\0') strncpy(outPath, argv[i], MAX_NAME_SIZE);
      else {
	fprintf(stderr,"%s: excess arg %s\n", programName, argv[i]);
	err = 1;
      }
    }
  }

  // Do not write VBR tag if VBR flag is not specified
  if (g_bWriteVbrTag==1 && VBR==0) g_bWriteVbrTag=0;

  if(err || inPath[0] == '\0') lame_usage(programName);  /* never returns */
  if (inPath[0]=='-') silent=1;  /* turn off status - it's broken for stdin */
  if(outPath[0] == '\0') {
    if (inPath[0]=='-') {
      /* if input is stdin, default output is stdout */
      strcpy(outPath,"-");
    }else {
      strncpy(outPath, inPath, MAX_NAME_SIZE - strlen(DFLT_EXT));
      strncat(outPath, DFLT_EXT, strlen(DFLT_EXT) );
    }
  }
  /* some file options not allowed with stdout */
  if (outPath[0]=='-') {
    g_bWriteVbrTag=0; /* turn off VBR tag */
    if (id3tag.used) {
      id3tag.used=0;         /* turn of id3 tagging */
      fprintf(stderr,"id3tag ignored: id3 tagging not supported for stdout.\n");
    }
  }


  /* if user did not explicitly specify input is mp3, check file name */
  if (input_format != sf_mp3)
    if (!(strcmp((char *) &inPath[strlen(inPath)-4],".mp3")))
      input_format = sf_mp3;

  /* default guess for number of channels */
  if (autoconvert) num_channels=2; 
  else if (info->mode == MPG_MD_MONO) num_channels=1;
  else num_channels=2;
  
#ifdef _BLADEDLL
  num_samples=0;
#else
  /* open the input file */
  OpenSndFile(inPath,info,samplerate,num_channels);  
  /* if GetSndSampleRate is non zero, use it to overwrite the default */
  if (GetSndSampleRate()) samplerate=GetSndSampleRate();
  if (GetSndChannels()) num_channels=GetSndChannels();
  num_samples = GetSndSamples();
#endif

  if (autoconvert==TRUE) {
    info->mode = MPG_MD_MONO; 
    info->mode_ext = MPG_MD_LR_LR;
  }
  if (num_channels==1) {
    autoconvert = FALSE;  /* avoid 78rpm emulation mode from downmixing mono file! */
    info->mode = MPG_MD_MONO;
    info->mode_ext = MPG_MD_LR_LR;
  }      
  /* if user specified mono and there are 2 channels, autoconvert */
  if ((num_channels==2) && (info->mode == MPG_MD_MONO))
    autoconvert = TRUE;
  
  if ((num_channels==1) && (info->mode!=MPG_MD_MONO)) {
    fprintf(stderr,"Error: mono input, stereo output not supported. \n");
    exit(1);
  }
  if (autoconvert==TRUE) {
    fprintf(stderr, "Autoconverting from stereo to mono. Setting encoding to mono mode.\n");
  }



  /* set the output sampling rate, and resample options if necessary 
     samplerate = input sample rate
     resamplerate = ouput sample rate
  */
  if (resamplerate==0) {
    /* user did not specify output sample rate */
    resamplerate=samplerate;   /* default */
  }

  info->sampling_frequency = SmpFrqIndex((long)resamplerate, &info->version);
  if( info->sampling_frequency < 0) {
    display_bitrates(2);
    exit(1);
  }

#ifndef _BLADEDLL
  /* estimate total frames.  must be done after setting sampling rate so
   * we know the framesize.  */
  totalframes=0;
  framesize = (info->version==0) ? 576 : 1152;
  if (num_samples == MAX_U_32_NUM) { 
    stat(inPath,&sb);  /* try file size, assume 2 bytes per sample */
    if (input_format == sf_mp3) {
      double totalseconds = (sb.st_size*8.0/(1000.0*GetSndBitrate()));
      double framespersecond =  (double)samplerate/(double)framesize;
      totalframes = 1+(totalseconds*framespersecond);
    }else{
      totalframes = 2+(sb.st_size/(2*framesize*num_channels));
    }
  }else{
    totalframes = 2+(num_samples/framesize);
  }
#endif /* _BLADEDLL */



  if ( brate == 0 ) {
    info->bitrate_index=9;
    brate = bitrate[info->version][info->lay-1][info->bitrate_index];
  } else {
    if( (info->bitrate_index = BitrateIndex(info->lay, brate, info->version,resamplerate)) < 0) {
      display_bitrates(2);
      exit(1);
    }
    /* a bit rate was specified.  for VBR, take this to be the minimum */
    VBR_min_bitrate=info->bitrate_index;
  }

  resample_ratio=0;
  if (resamplerate != samplerate) {
    fprintf(stderr,"Resampling:  input=%iHz  output=%iHz\n",
	    (int)samplerate,(int)resamplerate);
    resample_ratio = (double)samplerate/(double)resamplerate;
  }
  compression_ratio = resamplerate*16*num_channels/(1000.0*brate);
  if (compression_ratio > 12 ) {
    /* print message suggesting downsampling or low-pass filter */
  }


  /* default is STEREO for higher bitrates, not JSTEREO, unless of course
   * the user specified a mode */
  if ((!mode_given) && (compression_ratio <= 8 )) {
    if (info->mode != MPG_MD_MONO){
      info->mode = MPG_MD_STEREO; info->mode_ext = MPG_MD_LR_LR;
    }
  }
  if ((!mode_given) && (VBR_q <= 3 )) {
    if (info->mode != MPG_MD_MONO){
      info->mode = MPG_MD_STEREO; info->mode_ext = MPG_MD_LR_LR;
    }
  }

	

  if (info->bitrate_index==14) VBR=0;  /* dont bother with VBR at 320kbs */
  if (VBR) {
    /* default max bitrate is 256kbs */
    /* we do not normally allow 320bps frams with VBR, unless: */
    if (info->bitrate_index==13) VBR_max_bitrate=14;  
    if (VBR_q == 0) VBR_max_bitrate=14;   /* allow 320kbs */
    if (VBR_q >= 5) VBR_max_bitrate=12;   /* max = 224kbs */
    if (VBR_q >= 8) VBR_max_bitrate=9;    /* low quality, max = 128kbs */
    if (voice_mode) VBR_max_bitrate=10;
    highq = TRUE;
  }


  /* Should we disable the scalefactor band 21 cutoff? */
  /* NOTE: dont *enable* it here, since that could override the users' 
   * -k option, which disables sfb21 cufoff */
  /* remove coefficients in scalefactor band 21 (12 for short blocks)
   * FhG does this for bps <= 128kbs, so we will too.  
   * This amounds to a 16kHz low-pass filter.  If that offends you, you
   * probably should not be encoding at 128kbs!
   * There is no ratio[21] or xfsf[21], so when these coefficients are
   * included they are just quantized as is.  mt 5/99
   */
  /* disable sfb21 cutoff for high quality VBR */
  if (VBR && (VBR_q <4) ) sfb21 = 0;
  /* 44.1kHz at 128kbs: compression factor of 11 */
  /* 44.1kHz at 160kbs: compression factor of 8.82 */
  if (compression_ratio<9.0) sfb21=0;

  /* dont allow forced mid/side stereo for mono output */
  if (info->mode == MPG_MD_MONO) force_ms=0;  

  if (gtkflag) {
    lame_nowrite=1;    /* disable all file output */
    g_bWriteVbrTag=0;  /* disable Xing VBR tag */
  }

  if (info->version==0) {
    g_bWriteVbrTag=0;      /* no MPEG2 Xing VBR tags yet */
  }


  /* open the output file */
  /* if gtkflag, no output.  but set outfile = stdout */
  open_bit_stream_w(&bs, outPath, BUFFER_SIZE,lame_nowrite);  

  /* copy some header information */
  fr_ps.actual_mode = info->mode;
  fr_ps.stereo = (info->mode == MPG_MD_MONO) ? 1 : 2;


#ifdef BRHIST
  if (VBR) {
    if (disp_brhist)
      brhist_init(VBR_min_bitrate, VBR_max_bitrate);
  } else
    disp_brhist = 0;
#endif

  if (g_bWriteVbrTag)
    {
      // Write initial VBR Header to bitstream
      InitVbrTag(&bs,info->version-1,info->mode,info->sampling_frequency);
      /* flush VBR header frame to mp3 file */
      if (!lame_nowrite) 
	  {
		write_buffer(&bs);
		empty_buffer(&bs);
	  }
      /* note: if lame_nowrite==0, we do not empty the buffer, the VBR header will
       * remain in the bit buffer and the first mp3 frame will be
       * appeneded.  this way, lame_encode() will return to the calling
       * program the VBR header along with the first mp3 frame */
    }

  return;
}




/************************************************************************
 *
 * print_config
 *
 * PURPOSE:  Prints the encoding parameters used
 *
 ************************************************************************/
void lame_print_config(void)
{
  layer *info = fr_ps.header;
  char *mode_names[4] = { "stereo", "j-stereo", "dual-ch", "single-ch" };

  if (gtkflag) {
    fprintf(stderr, "Analyzing %s \n",inPath);
  }
  else {
    fprintf(stderr, "Encoding %s to %s\n",
	    (strcmp(inPath, "-")? inPath : "stdin"),
	    (strcmp(outPath, "-")? outPath : "stdout"));
    if (VBR)
      fprintf(stderr, "Encoding as %.1fkHz VBR(q=%i) %s MPEG%i LayerIII file\n",
	      s_freq[info->version][info->sampling_frequency],
	      VBR_q,mode_names[info->mode],2-info->version);
    else
      fprintf(stderr, "Encoding as %.1f kHz %d kbps %s MPEG%i LayerIII file\n",
	      s_freq[info->version][info->sampling_frequency],
	      bitrate[info->version][info->lay-1][info->bitrate_index],
	      mode_names[info->mode],2-info->version);
  }
  fflush(stderr);
}



 
#ifdef BRHIST

#define BRHIST_BARMAX 50

long brhist_count[15];
long brhist_temp[15];
int brhist_vbrmin;
int brhist_vbrmax;
long brhist_max;
char brhist_bps[15][5];
char brhist_backcur[200];
char brhist_bar[BRHIST_BARMAX+10];
char brhist_spc[BRHIST_BARMAX+1];

char stderr_buff[BUFSIZ];


void brhist_init(int br_min, int br_max)
{
  int i;
  char term_buff[1024];
  layer *info = fr_ps.header;
  char *termname;
  char *tp;
  char tc[10];

  for(i = 0; i < 15; i++)
    {
      sprintf(brhist_bps[i], "%3d:", bitrate[info->version][info->lay-1][i]);
      brhist_count[i] = 0;
      brhist_temp[i] = 0;
    }

  brhist_vbrmin = br_min;
  brhist_vbrmax = br_max;

  brhist_max = 0;

  memset(&brhist_bar[0], '*', BRHIST_BARMAX);
  brhist_bar[BRHIST_BARMAX] = '\0';
  memset(&brhist_spc[0], ' ', BRHIST_BARMAX);
  brhist_spc[BRHIST_BARMAX] = '\0';
  brhist_backcur[0] = '\0';

  if ((termname = getenv("TERM")) == NULL)
    {
      fprintf(stderr, "can't get TERM environment string.\n");
      disp_brhist = 0;
      return;
    }

  if (tgetent(term_buff, termname) != 1)
    {
      fprintf(stderr, "can't find termcap entry: %s\n", termname);
      disp_brhist = 0;
      return;
    }

  tc[0] = '\0';
  tp = &tc[0];
  tgetstr("up", &tp);
  brhist_backcur[0] = '\0';
  for(i = br_min-1; i <= br_max; i++)
    strcat(brhist_backcur, tc);

  /*
    tc[0] = '\0';
    tp = &tc[0];
    tgetstr("ce", &tp);
    strcat(brhist_bar, tc);
  */

  setbuf(stderr, stderr_buff);
}

void brhist_add_count(void)
{
  int i;

  for(i = brhist_vbrmin; i <= brhist_vbrmax; i++)
    {
      brhist_count[i] += brhist_temp[i];
      if (brhist_count[i] > brhist_max)
	brhist_max = brhist_count[i];
      brhist_temp[i] = 0;
    }
}

void brhist_disp(void)
{
  int i;
  long full;
  int barlen;

  full = (brhist_max < BRHIST_BARMAX) ? BRHIST_BARMAX : brhist_max;
  fputc('\n', stderr);
  for(i = brhist_vbrmin; i <= brhist_vbrmax; i++)
    {
      barlen = (brhist_count[i]*BRHIST_BARMAX+full-1) / full;
      fputs(brhist_bps[i], stderr);
      fputs(&brhist_bar[BRHIST_BARMAX - barlen], stderr);
      fputs(&brhist_spc[barlen], stderr);
      fputc('\n', stderr);
    }
  fputs(brhist_backcur, stderr);
  fflush(stderr);
}

void brhist_disp_total(void)
{
  int i;
  double ave;
  layer *info = fr_ps.header;

  for(i = brhist_vbrmin; i <= brhist_vbrmax; i++)
    fputc('\n', stderr);

  ave=0;
  for(i = brhist_vbrmin; i <= brhist_vbrmax; i++)
    ave += bitrate[info->version][info->lay-1][i]*
      (double)brhist_count[i] / totalframes;
  fprintf(stderr, "\naverage: %2.0f kbs\n",ave);
    
#if 0
  fprintf(stderr, "----- bitrate statistics -----\n");
  fprintf(stderr, " [kbps]      frames\n");
  for(i = brhist_vbrmin; i <= brhist_vbrmax; i++)
    {
      fprintf(stderr, "   %3d  %8ld (%.1f%%)\n",
	      bitrate[info->version][info->lay-1][i],
	      brhist_count[i],
	      (double)brhist_count[i] / totalframes * 100.0);
    }
#endif
  fflush(stderr);
}

#endif










/************************************************************************
* 
* encodeframe()
*
* SEMANTICS:  One overlapping frame of audio of up to 2 channels are
* processed at a time in the following order:
* (associated routines are in parentheses)
*
* 1.  Filter sliding window of data to get 32 subband
* samples per channel.
* (window_subband,filter_subband)
*
* 2.  Calculate scalefactors for the frame, and if layer 2,
* also calculate scalefactor select information.
* (*_scale_factor_calc)
*
* 3.  Calculate psychoacoustic masking levels using selected
* psychoacoustic model.
* (*_Psycho_One, psycho_anal)
*
* 4.  Perform iterative bit allocation for subbands with low
* mask_to_noise ratios using masking levels from step 4.
* (*_main_bit_allocation)
*
* 5.  Pack bit allocation and scalefactors onto bitstream.
* (*_encode_bit_alloc,*_encode_scale)
*
* 6.  Quantize subbands and pack them into bitstream
* (*_subband_quantization, *_sample_encoding)
*
************************************************************************/
int lame_encode(short int Buffer[2][1152],char *mpg123bs)
{
  static unsigned long frameBits;
  static unsigned long bitsPerSlot;
  static double frac_SpF;
  static double slot_lag;
  static int mode_gr;
  static int old_bitrate;
  static unsigned long sentBits = 0;
  static L3SBS     l3_sb_sample;
  /* set to 56 to sync with FhG */
#define EXTRADELAY 56
  static int samplesPerFrame;
  static short int mfbuf[2][1152+576+EXTRADELAY];   /* 3 granule internal buffer */
  double xr[2][2][576];
  double win_que[64];
  int l3_enc[2][2][576];
  int mpg123count;
  III_psy_ratio ratio;
  III_scalefac_t scalefac;

  int j,ch,gr,mean_bits;
  int bitsPerFrame;
  double pe[2][4];
  double pe_use[2][2];
  double pe_sum_old=0;
  static double pe_sum=0;
  short *win_buf[2];
  int stereo;
  layer *info;
  int i;
  int check_ms_stereo;
  int bit_rate;
  double samp;
  double ms_ener_ratio[2];

  stereo = fr_ps.stereo;
  info = fr_ps.header;

  info->mode_ext = MPG_MD_LR_LR; 

  /* use m/s stereo? */
  check_ms_stereo =   ((info->mode == MPG_MD_JOINT_STEREO) && 
		       (!force_ms) && (info->version == 1) &&
		       (stereo==2));


  if (frameNum==0) old_bitrate=info->bitrate_index;
  info->bitrate_index = old_bitrate;


  bit_rate = bitrate[info->version][info->lay-1][info->bitrate_index];
  samp =      s_freq[info->version][info->sampling_frequency];
  
  if (frameNum==0)  {    
    double avg_slots_per_frame;
    int whole_SpF;
    sentBits = 0;
    memset((char *) mfbuf, 0, sizeof(mfbuf));

    /* Figure average number of 'slots' per frame. */
    /* Bitrate means TOTAL for both channels, not per side. */
    bitsPerSlot = 8;
    samplesPerFrame = info->version == 1 ? 1152 : 576;
    avg_slots_per_frame = ((double)samplesPerFrame /samp) *
      ((double)bit_rate /  (double)bitsPerSlot);
    whole_SpF = (int) avg_slots_per_frame;
    frac_SpF  = avg_slots_per_frame - (double)whole_SpF;
    slot_lag  = -frac_SpF;
    info->padding = 1;
    if (fabs(frac_SpF) < 1e-9) info->padding = 0;
    mode_gr = (info->version == 1) ? 2 : 1;  /* mode_gr = 2 */
  }


  /* shift in new samples, delayed by 576+EXTRADELAY */
   for (ch=0; ch<stereo; ch++)
     for (i=0; i<576+EXTRADELAY; i++)
       mfbuf[ch][i]=mfbuf[ch][i+samplesPerFrame];
   for (ch=0; ch<stereo; ch++)
     for (i=0; i<samplesPerFrame; i++)
       mfbuf[ch][i+576+EXTRADELAY]=Buffer[ch][i];




  /********************** padding *****************************/
  if (VBR) {
  /* leave info_padding as it was set above */
  } else {
    if (frac_SpF != 0) {
      if (slot_lag > (frac_SpF-1.0) ) {
	slot_lag -= frac_SpF;
	info->padding = 0;
      }
      else {
	info->padding = 1;
	slot_lag += (1-frac_SpF);
      }
    }
  }

#ifndef _BLADEDLL
  /********************** status display  *****************************/
  if (!gtkflag && !silent) {
    int mod = info->version == 0 ? 100 : 20;
    if (frameNum%mod==0) {
      timestatus(info,frameNum,totalframes);
#ifdef BRHIST
      if (disp_brhist)
	{
	  brhist_add_count();
	  brhist_disp();
	}
#endif
    }
  }

#endif /* _BLADEDLL */

  /********************** process InputBuffer *****************************/


  if (force_ms)
    for (i=0; i< samplesPerFrame; i++) {
      /* dont overflow the short int */
      double mid = ((int)mfbuf[0][576+i]+(int)mfbuf[1][576+i])/(2.0);
      double side= ((int)mfbuf[0][576+i]-(int)mfbuf[1][576+i])/(2.0);
      mfbuf[0][576+i]=mid;
      mfbuf[1][576+i]=side;
    }


  
  /***************************** Layer 3 **********************************
   * mfbuf contains 3 granules:  [0 1 2 ] 
   * encoder will encode granules 0 1 
   * psy-model will be fed granules 1 2, and because of its 1 granule delay 
   * it will return thresholds for granules 0 1 */
  
  win_buf[0] = &mfbuf[0][0];
  win_buf[1] = &mfbuf[1][0];

  
  if (!fast_mode) {  
    /* psychoacoustic model 
     * psy model adds a 544 delay to sync with the filterbanks
     * in addition to this delay, it also adds a 1 granule (576) delay
     * that we must compensate for (mt 6/99). 
     */
    short int *bufp[2];  /* address of beginning of left & right granule */
    int blocktype[2];
    
    for (gr=0; gr < mode_gr ; gr++) {
      for ( ch = 0; ch < stereo; ch++ ) bufp[ch] = &mfbuf[ch][576 + gr*576];
      L3psycho_anal( bufp, stereo, gr, info,
         s_freq[info->version][info->sampling_frequency] * 1000.0,
	 check_ms_stereo,&ms_ener_ratio[gr],
         ratio.l[gr], ratio.s[gr], pe[gr], blocktype);
      for ( ch = 0; ch < stereo; ch++ ) 
	l3_side.gr[gr].ch[ch].tt.block_type=blocktype[ch];
    }
  }else{
    for (gr=0; gr < mode_gr ; gr++) 
      for ( ch = 0; ch < stereo; ch++ ) {
	l3_side.gr[gr].ch[ch].tt.block_type=NORM_TYPE;
	pe[gr][ch]=700;
      }
  }

  pe_sum_old=pe_sum;
  pe_sum=0;
  for( gr = 0; gr < mode_gr; gr++ ) 
    for ( ch = 0; ch < stereo; ch++ ) 
      pe_sum+=pe[gr][ch];
  


  /* for VBR, what bitrate should we try first? */
  if (VBR){
    info->bitrate_index=old_bitrate-1;
    if (abs(pe_sum_old - pe_sum)>200){
      if (pe_sum_old>pe_sum)
        info->bitrate_index--;
    }

    if (info->bitrate_index < VBR_min_bitrate)
      info->bitrate_index=VBR_min_bitrate;
    if (info->bitrate_index > VBR_max_bitrate)
      info->bitrate_index=VBR_max_bitrate;
  }


  /* block type flags */
  for( gr = 0; gr < mode_gr; gr++ ) {
    for ( ch = 0; ch < stereo; ch++ ) {
      gr_info *cod_info = &l3_side.gr[gr].ch[ch].tt;
      cod_info->mixed_block_flag = 0;     /* never used by this model */
      if (cod_info->block_type == NORM_TYPE )
	cod_info->window_switching_flag = 0;
      else
	cod_info->window_switching_flag = 1;
    }
  }


  /* polyphase filtering  */
  for( gr = 0; gr < mode_gr; gr++ )
    for ( ch = 0; ch < stereo; ch++ )
      for ( j = 0; j < 18; j++ )   {
	window_subband( &win_buf[ch], win_que, ch );
	filter_subband( win_que,  &(l3_sb_sample)[ch][gr+1][j][0] ); }
  
  /* apply mdct to the polyphase outputs */
  mdct_sub( l3_sb_sample, xr, stereo, &l3_side, mode_gr );

  /* data was scaled by 1/2.  fix so effectively it was scaled by 1/sqrt(2) */
  if (force_ms) {
    for ( gr = 0; gr < mode_gr; gr++ )
      for ( ch = 0; ch < stereo; ch++ ) 
	for (i =0 ; i< 576; i++) 
	  xr[gr][ch][i] *= SQRT2;
  }

  /* set a default for ms_ener_ratio since it was not computed in these cases */
  if (force_ms || fast_mode) {
    ms_ener_ratio[0]=.25;
    ms_ener_ratio[1]=.25;
  }


  if (check_ms_stereo) {
    /* make sure block type is the same in each channel */
    check_ms_stereo = 
      (l3_side.gr[0].ch[0].tt.block_type==l3_side.gr[0].ch[1].tt.block_type) &&
      (l3_side.gr[1].ch[0].tt.block_type==l3_side.gr[1].ch[1].tt.block_type);
  }
  if (check_ms_stereo) {
    if ( (ms_ener_ratio[0] + ms_ener_ratio[1])<.70) 
           info->mode_ext = MPG_MD_MS_LR;
    ms_ener_ratio[0]=Min(ms_ener_ratio[0],.5);
    ms_ener_ratio[1]=Min(ms_ener_ratio[1],.5);
  }

#ifdef HAVEGTK
  if (gtkflag) { 
    for ( gr = 0; gr < mode_gr; gr++ )
      for ( ch = 0; ch < stereo; ch++ ) {
	pinfo->ms_ratio[gr]=ms_ener_ratio[gr];
	pinfo->blocktype[gr][ch]=
	  l3_side.gr[gr].ch[ch].tt.block_type;
	for ( j = 0; j < 576; j++ ) pinfo->xr[gr][ch][j]=xr[gr][ch][j];
      }
  }
#endif

  /* if MS stereo, switch to MS psy data */
  for ( gr = 0; gr < mode_gr; gr++ ) {
    for ( ch = 0; ch < stereo; ch++ ) {
      if (highq && (info->mode_ext==MPG_MD_MS_LR)) {
	pe_use[gr][ch]=pe[gr][ch+2];
	memcpy(ratio.l[gr][ch],ratio.l[gr][ch+2],sizeof(ratio.l[gr][ch]));
	memcpy(ratio.s[gr][ch],ratio.s[gr][ch+2],sizeof(ratio.s[gr][ch]));
#ifdef HAVEGTK
       if (gtkflag) {
	 pinfo->ers[gr][ch]=pinfo->ers[gr][ch+2];
	 pinfo->pe[gr][ch]=pinfo->pe[gr][ch+2];
         memcpy(pinfo->energy[gr][ch],pinfo->energy[gr][ch+2],
		sizeof(pinfo->energy[gr][ch]));
	 memcpy(pinfo->thr[gr][ch],pinfo->thr[gr][ch+2],
		sizeof(pinfo->thr[gr][ch]));
	 memcpy(pinfo->en[gr][ch],pinfo->en[gr][ch+2],
		sizeof(pinfo->en[gr][ch]));
	 memcpy(pinfo->thr_s[gr][ch],pinfo->thr_s[gr][ch+2],
                sizeof(pinfo->thr_s[gr][ch]));
	 memcpy(pinfo->en_s[gr][ch],pinfo->en_s[gr][ch+2],
		sizeof(pinfo->en_s[gr][ch]));
       }
#endif
      }else{
	pe_use[gr][ch]=pe[gr][ch];
      }
    }
  }

  /* bit and noise allocation */
  if (VBR) 
    VBR_iteration_loop( pe_use, ms_ener_ratio, xr, &ratio, &l3_side, l3_enc, 
		    &scalefac, &fr_ps);
  else
    iteration_loop( pe_use, ms_ener_ratio, xr, &ratio, &l3_side, l3_enc, 
		    &scalefac, &fr_ps);

#ifdef BRHIST
  brhist_temp[info->bitrate_index]++;
#endif

  /* flag for our ms_stereo with psy-model on mid & side channels */
  if (force_ms) info->mode_ext = MPG_MD_MS_LR;

  /*  write the frame to the bitstream  */
  old_bitrate = info->bitrate_index;
  getframebits(info,stereo,&bitsPerFrame,&mean_bits);
  III_format_bitstream( bitsPerFrame, &fr_ps, l3_enc, &l3_side, 
			&scalefac, &bs, xr, NULL, 0 );


  frameBits = bs.totbit - sentBits;

  
  if ( frameBits % bitsPerSlot )   /* a program failure */
    fprintf( stderr, "Sent %ld bits = %ld slots plus %ld\n",
	     frameBits, frameBits/bitsPerSlot,
	     frameBits%bitsPerSlot );
  sentBits += frameBits;

  /* copy mp3 bit buffer into array */
  mpg123count = copy_buffer(mpg123bs,&bs);
  if (!lame_nowrite) write_buffer(&bs);  /* ouput to mp3 file */
  empty_buffer(&bs);  /* empty buffer */

  if (g_bWriteVbrTag) AddVbrFrame((int)(sentBits/8));

#ifdef HAVEGTK 
  if (gtkflag) { 
    int adj=0;
    if (info->version == 0) adj=576;
    for ( ch = 0; ch < stereo; ch++ ) {
      for ( j = 0; j < WINDELAY; j++ )
	pinfo->pcmdata[ch][j] = pinfo->pcmdata[ch][j+1152-adj];
      for ( j = 0; j < 1152-adj; j++ ) {
	pinfo->pcmdata[ch][j+WINDELAY] = mfbuf[ch][j];
      }
    }
  }
#endif
  //  fprintf(stderr,"finished frameNum=%i \n",frameNum);
  frameNum++;
  return mpg123count;
}


#ifndef _BLADEDLL
int lame_readframe(short int Buffer[2][1152])
{
  int iread,ifreq,ofreq;
  static int eof;

  if (frameNum==0) eof=0;
  iread = get_audio(Buffer,fr_ps.stereo,fr_ps.header);
  /* note: if input is stereo and output is mono, get_audio() 
   * will return stereo=1 and .5*(L+R) in channel 0, 
   * and nothing in channel 1. */
  /* DOWNSAMPLE, ETC */
  ofreq=1000*s_freq[info.version][info.sampling_frequency];  /* output */
  ifreq=ofreq*resample_ratio;                                /* input */

#ifdef HAVEGTK
  if (gtkflag) {
    pinfo->frameNum = frameNum;
    pinfo->sampfreq=ofreq;
    pinfo->framesize=(fr_ps.header->version==0) ? 576 : 1152;
    pinfo->stereo = fr_ps.stereo;
  }
#endif

  if (resample_ratio>0) {
    fprintf(stderr,"Error: resample code not yet written! \n");
    exit(1);
  }

  /* check to see if we overestimated/underestemated totalframes */
  if (!eof) {
    if (iread==0) {
      /* this is the last frame */
      totalframes=frameNum+1;  
      eof=1;
    }else{
      if (frameNum > (totalframes-1)) totalframes = frameNum;
    }
  }



  return iread;
}
#endif /* _BLADEDLL */

/* initialize mp3 encoder */
void lame_init(int nowrite)
{

#ifdef __FreeBSD__
#include <floatingpoint.h>
  {
  /* seet floating point mask to the Linux default */
  fp_except_t mask;
  mask=fpgetmask();
  /* if bit is set, we get SIGFPE on that error! */
  fpsetmask(mask & ~(FP_X_INV|FP_X_DZ));
  /*  fprintf(stderr,"FreeBSD mask is 0x%x\n",mask); */
  }
#endif
#ifdef ABORTFP
  {
#include <fpu_control.h>
  unsigned int mask;
  _FPU_GETCW(mask);
  /* Set the Linux mask to abort on most FPE's */
  /* if bit is set, we _mask_ SIGFPE on that error! */
  //   mask &= ~( _FPU_MASK_IM | _FPU_MASK_ZM | _FPU_MASK_OM | _FPU_MASK_UM );
   mask &= ~( _FPU_MASK_IM | _FPU_MASK_ZM | _FPU_MASK_OM );
  _FPU_SETCW(mask);
  }
#endif


#ifndef _BLADEDLL
  fprintf(stderr,"LAME version %s (www.sulaco.org/mp3) \n",get_lame_version());
  fprintf(stderr,"GPSYCHO: GPL psycho-acoustic model version %s. \n",get_psy_version());
#ifdef LIBSNDFILE
  fprintf(stderr,"Input handled by libsndfile (www.zip.com.au/~erikd/libsndfile)\n");
#endif
#endif /* _BLADEDLL */

  /* Global flags.  set defaults here */
  allow_diff_short=0;
  ATHonly=0;
  autoconvert=FALSE;
  experimentalX = 0;
  experimentalY = 0;
  experimentalZ = 0;
  force_ms=FALSE;
  fast_mode=0;
  frameNum=0;
  g_bWriteVbrTag=1;
  gtkflag=0;
  gpsycho=1;
  highq=0;
  input_format=sf_unknown;
  lame_nowrite=nowrite;
  resample_ratio=0;
  sfb21=1;
  silent=0;
  swapbytes=0;
  totalframes=0;
  VBR=0;
  VBR_q=4;
  VBR_min_bitrate=1;   /* 32kbs */
  VBR_max_bitrate=13;  /* 256kbs */
  voice_mode=0;


  // Clear info structure
  memset(&info,0,sizeof(info));
  memset(&bs, 0, sizeof(Bit_stream_struc));
  memset(&fr_ps, 0, sizeof(frame_params));
  memset(&l3_side,0x00,sizeof(III_side_info_t));
  
  fr_ps.header = &info;
  fr_ps.tab_num = -1;             /* no table loaded */
  fr_ps.alloc = NULL;
  info.version = MPEG_AUDIO_ID;   /* =1   Default: MPEG-1 */
  info.extension = 0;       

  InitFormatBitStream();
  InitReservoir();

}


#ifndef _BLADEDLL
int lame_cleanup(char *mpg123bs)
{
  int mpg123count;
  frameNum--;
  if (!gtkflag && !silent) {
      timestatus(&info,frameNum,totalframes);
      fprintf(stderr,"\n");
#ifdef BRHIST
      if (disp_brhist)
	{
	  brhist_add_count();
	  brhist_disp();
	}
#endif
#ifdef BRHIST
      if (disp_brhist)
	brhist_disp_total();
#endif
      fflush(stderr);
  }


  CloseSndFile();


  III_FlushBitstream();
  mpg123count = copy_buffer(mpg123bs,&bs);

  if (!lame_nowrite)
  {
    write_buffer(&bs);  /* ouput to mp3 file */
    empty_buffer(&bs);  /* empty buffer */

    // Close the file
    fclose(bs.pt);
  
    // Deallocate all buffers
    desalloc_buffer(&bs);
  
    
    // AF ADDED
    if (g_bWriteVbrTag)
	{
		// Calculate relative quality of VBR stream 
		// 0=best, 100=worst
		int nQuality=VBR_q*100/9;

		// Write Xing header again
		PutVbrTag(outPath,nQuality);
	}
    // AF END
    
    // write an ID3 tag 
    if(id3tag.used) {
      id3_buildtag(&id3tag);
      id3_writetag(outPath, &id3tag);
    }
  }
  return mpg123count;
}
#endif /* _BLADEDLL */
