/*////////////////////////////////////////////////////////////////////////
Copyright (c) 1994-1999 Yutaka Sato
Copyright (c) 1994-1999 Electrotechnical Laboratry (ETL), AIST, MITI

Permission to use, copy, and distribute this material for any purpose
and without fee is hereby granted, provided that the above copyright
notice and this permission notice appear in all copies.
ETL MAKES NO REPRESENTATIONS ABOUT THE ACCURACY OR SUITABILITY OF THIS
MATERIAL FOR ANY PURPOSE.  IT IS PROVIDED "AS IS", WITHOUT ANY EXPRESS
OR IMPLIED WARRANTIES.
/////////////////////////////////////////////////////////////////////////
Content-Type:	program/C; charset=US-ASCII
Program:	mount.c (mount URL to/from)
Author:		Yutaka Sato <ysato@etl.go.jp>
Description:

	MOUNT="virtualUrl realUrl optionList"

History:
	940806	created
	9907xx	full-URL in the left-hand (host name/addr matching)
		wild-card hostList like http://192.31.200.* in left-hand
		MOUNT="/-_-* * dstproto={protoList},dst={siteList}"
		MOUNT="/path/* http://* dst={siteList}"
ToDo:

	- proto1://site1/path1 proto2://site2/path2
	  where site1 is a "name of a hostList" (or Site in terms of URL)
	- proto1://+siteList/path1 proto2://site2/path2 vhost=siteList
	- MOUNT="http:// * /path1 http:// * /path2"
	- reverse matching in left hand (ex. *.gif)

//////////////////////////////////////////////////////////////////////#*/
#include "ystring.h"
extern char *CTX_clif_hostport();
extern char *file_hostpath();
extern char *getv();
extern char *gethostaddr();
extern char *gethostaddr_fromcache();
#define Strcpy(d,s)	(strcpy(d,s), d+strlen(d))


#ifndef NULL
#define NULL 0L
#endif

#ifndef Verbose
extern int LOG_VERBOSE;
#define Verbose LOG_VERBOSE==0 ? 0 : sv1vlog
#endif

#ifndef MTAB_SIZE
#define MTAB_SIZE	256
#endif

#ifdef DONT_SORT_MTAB
#define DO_MTAB_SORT 0
#else
#define DO_MTAB_SORT 1
#endif

#define ASIS		"="

#define C_CLHOST	0x000001
#define C_CLFROM	0x000002
#define C_CLVIA		0x000004
#define C_CLPATH	0x000008
#define C_CLCONDS	0x00000F
#define C_METHOD	0x000010
#define C_WITHQUERY	0x000020
#define C_ASPROXY	0x000040
#define C_DSTPROTO	0x000100
#define C_VHOST		0x000200 /* destination host in the request URL */
#define C_RHOST		0x000400 /* destination host in the response */
#define C_DSTHOST	(C_VHOST|C_RHOST)
#define C_DEFAULT	0x000800 /* default mount derived from SERVER param. */
#define C_ALL		0x000FFF

#define U_MOUNT		0x001000
#define U_MOVED_TO	0x002000
#define U_USE_PROXY	0x004000
#define U_DISPPORT	0x010000
#define U_PORTMAP	0x020000
#define U_RESOLV	0x040000
#define U_INDIRECT	0x080000
#define U_CGI		0x100000
#define U_ALL		0xFFF000

typedef struct {
	char	*u_src;
	int	 u_fullurl;
	int	 u_remain;
	char	*u_format;
	char	*u_prefix;
	int	 u_emptysite;
	int	 u_path2site;	/* "/path/* scheme://*"	*/
	int	 u_any;		/* "/path/* *"		*/

	char	*u_proto;
	char	*u_userpass;
	char	*u_user;
	char	*u_pass;

/* caching and matching of host/addr should be done in hostlist.c */
	int	 u_hostList;
	char	*u_hostn;
	char	*u_hosta;
	int	 u_iport;
	int	 u_stdport;
	char	*u_portmap;
	int	 u_iport_mapped;

	char	*u_path;
	int	 u_plen;
	int	 u_asis;
} Url;

typedef struct {
	int	 u_serno;
	int	 u_compiled;
	int	 u_disabled;
	int	 u_flags;

	int	 u_conds;
	int	 u_conds_value;
	char	*u_conds_param;

	char	*u_Strings[8];
	int	 u_PathLs[8];
	Url	 Src;
	Url	 Dst;
	int	 u_xcond;
	int	 u_timeout;
} Mtab;

#define	L_VIA		1
#define L_PATH		2
#define L_CLIF		3
#define L_FROM		4
#define L_DST		5
#define viaList		u_PathLs[L_VIA]
#define pathList	u_PathLs[L_PATH]
#define clifList	u_PathLs[L_CLIF]
#define fromList	u_PathLs[L_FROM]
#define dstList		u_PathLs[L_DST]

#define S_OPTIONS	0
#define S_USEPROXY	1
#define S_METHOD	2
#define S_DSTPROTO	3
#define S_FTOCL		4
#define S_WHOLE		5

#define u_opts		u_Strings[S_OPTIONS]
#define u_useproxy	u_Strings[S_USEPROXY]
#define u_method	u_Strings[S_METHOD]
#define u_dstproto	u_Strings[S_DSTPROTO]
#define u_ftocl		u_Strings[S_FTOCL]
#define u_whole		u_Strings[S_WHOLE]

static struct {
	int	 m_do_sort;
	Mtab	*m_mtab[MTAB_SIZE];
	int	 m_mtabN;
	int	 m_init_from;
	int	 m_last_hit;
} mount = {
	DO_MTAB_SORT,
	{0}, 0, 0, 0
};

#define mtab		mount.m_mtab
#define mtabN		mount.m_mtabN
#define do_MTAB_SORT	mount.m_do_sort
#define init_from	mount.m_init_from
#define last_hit	mount.m_last_hit

clear_mtab(){
	mtabN = 0;
	init_from = 0;
	last_hit = 0;
	if( mtab[0] ){
		mtab[0] = 0;
	}
}

#define INDIRECT_URLEXT "%0/indirect.cfi?_?%1-"
struct {
	char	*o_name;
	int	 o_flags;
	int	 o_value;
	int	 o_listx;
} mount_opts[] = {
	{"default",	C_DEFAULT},
	{"dstproto",	C_DSTPROTO,	S_DSTPROTO	},
	{"dst",		C_DSTHOST,	0,L_DST		},
	{"vhost",	C_VHOST,	0,L_DST		},
	{"rhost",	C_RHOST,	0,L_DST		},
	{"from",	C_CLFROM,	0,L_FROM	},
	{"host",	C_CLHOST,	0,L_CLIF	},
	{"path",	C_CLPATH,	0,L_PATH	},
	{"src",		C_CLVIA,	0,L_VIA		},
	{"via",		C_CLVIA,	0,L_VIA		},
	{"method",	C_METHOD,	S_METHOD	},
	{"asproxy",	C_ASPROXY,			},
	{"withquery",	C_WITHQUERY,			},
	{"ident",	}, /* client's user name got from Ident server */

	{"moved",	U_MOVED_TO,			},
	{"useproxy",	U_USE_PROXY,	S_USEPROXY	},
	{"indirect",	U_INDIRECT,			},
	{"resolv",	U_RESOLV,			},

	{"htmlconv",		},/* META, SSI, DHTML */
	{"uriconv",		},

	{"ftocl",	0,		S_FTOCL		},
	{"ftosv",		},
	{"ro",			},/* read only */
	{"rw",			},/* read & write (FTP) */

	{"owner",		},/* dynamic mount from remote */

	{"delay",		},/* scheduling, priority, ... */

	/* cache control */
	{"expire",		},
	{"cache",		},

	/* routing by MOUNT ? */
	{"master",		},
	{"proxy",		},

	/* HTTP */
	{"cgi",		U_CGI	},
	{"genvhost",		},/* generate Host: in forwarding req. */
	{"robots",		},/* /robots.txt control */
	{"charcode",		},
	{"rcode",		},/* response code */
	{"rhead",		},/* response header in MIME format */

	/* NNTP */
	{"hide",		},/* NNTP group mask */
	{"upact",		},
	{"pathhost",		},
	0
};
static scanopt(opt,mt)
	char *opt;
	Mtab *mt;
{	char buf[1024],*val,*nam1,what[32];
	int oi,flag1,val1,list1;

	strcpy(buf,opt);
	if( val = strchr(buf,'=') )
		*val++ = 0;
	else	val = "";

	if( streq(buf,"indirect") )
		mt->Dst.u_format = strdup(INDIRECT_URLEXT);

	for( oi = 0; nam1 = mount_opts[oi].o_name; oi++ ){
		if( !streq(buf,nam1) )
			continue;
		if( flag1 = (mount_opts[oi].o_flags & U_ALL) )
			mt->u_flags |= flag1;
		if( flag1 = (mount_opts[oi].o_flags & C_ALL) )
			mt->u_conds |= flag1; 

		if( val1 = mount_opts[oi].o_value ){
			mt->u_Strings[val1] = strdup(val);
		}
		if( list1 = mount_opts[oi].o_listx ){
			sprintf(what,"MOUNT.%s",buf);
			mt->u_PathLs[list1] = makePathList(what,strdup(val));
		}
		break;
	}
	return 0;
}
mount_disable_lasthit(){
	if( last_hit ){
		mtab[last_hit-1]->u_disabled = 1;
		return last_hit;
	}
	return 0;
}
mount_enable(last){
	if( last )
		mtab[last-1]->u_disabled = 0;
}
mount_nodefaults(iproto,on)
	char *iproto;
{	int mi;
	Mtab *mt;

	for( mi = 0; mi < mtabN; mi++ ){
		mt = mtab[mi];
		if( mt->u_conds & C_DEFAULT ){
			mt->u_disabled = on;
		}
	}
}
Mounted()
{	int mi,mn;

	mn = 0;
	for( mi = 0; mi < mtabN; mi++ )
		if( !mtab[mi]->Dst.u_asis )
			mn++;
	return mn;
}
MountedConditional()
{	int mi,mn;
	Mtab *mt;

	mn = 0;
	for( mi = 0; mi < mtabN; mi++ ){
		mt = mtab[mi];
		if( mt->u_conds & C_CLCONDS )
			mn++;
	}
	return mn;
}
scan_MOUNT(Conn,spec)
	char *Conn;
	char *spec;
{	char src[1024],dst[1024],opts[1024];
	char xdst[1024];

	opts[0] = 0;
	if( sscanf(spec,"%s %s %s",src,dst,opts) < 2 )
	{
		sv1log("MOUNT=\"%s\" ? (missing right hand)\n",spec);
		return;
	}
	Verbose("%s\n",spec);
	set_MOUNT(Conn,src,dst,opts);
}
char *getMountSrcByDst(dst)
	char *dst;
{	Mtab *mt;
	int mi;

	for( mi = 0; mi < mtabN; mi++ ){
		mt = mtab[mi];
		if( streq(mt->Dst.u_src,dst) )
			return mt->Src.u_src;
	}
	return NULL;
}

set_MOUNT(Conn,src,dst,opts)
	char *Conn;
	char *src,*dst,*opts;
{	Mtab *mt;
	int mi;
	int len;
	char dstbuf[1024];
	char whole[1024];

	if( mtab[0] == NULL ){
		mtab[0] = (Mtab*)-1;

		/* These patterns should be able to be re-defined by user ... */
		set_MOUNT(Conn,"/-*",ASIS,"");
		set_MOUNT(Conn,"/=*",ASIS,"");
		if( streq(src,"/-*") && streq(dst,ASIS) && opts[0]==0 ) return;
		if( streq(src,"/=*") && streq(dst,ASIS) && opts[0]==0 ) return;
	}

	sprintf(whole,"%s %s %s",src,dst,opts);
	for( mi = 0; mi < mtabN; mi++ ){
		mt = mtab[mi];
		if( streq(mt->u_whole,whole) ){
			Verbose("IGNORE DUPLICATE MOUNT[%d] %s\n",mi,whole);
			return;
		}
	}

	mt = (Mtab*)calloc(sizeof(Mtab),1);
	if( mt == NULL )
		return;
	mtab[mtabN++] = mt;
	mt->u_whole = strdup(whole);
	mt->u_serno = mtabN;
	mt->u_xcond = getCondition(Conn);

	mt->Src.u_src = strdup(src);
	mt->Src.u_fullurl = isFullURL(src);
	len = strlen(src);

	if( mt->Src.u_src[len-1] == '*' ){
		mt->Src.u_src[len-1] = 0;
		mt->Src.u_remain = 1;
		if( 1 < len && mt->Src.u_src[len-2] == ']' ){
			char *dp;
			if( dp = strrchr(mt->Src.u_src,'[') ){
				mt->Src.u_format = strdup(dp);
				*dp = 0;
			}
		}
	}

	if( *dst == '/' ){
		sprintf(dstbuf,"file://%s",dst);
		dst = dstbuf;
	}
	mt->Dst.u_src = strdup(dst);
	len = strlen(dst);
	if( mt->Dst.u_src[len-1] == ']' ){
		char *dp;
		if( dp = strrchr(mt->Dst.u_src,'[') )
		if( mt->Dst.u_src < dp && dp[-1] == '*' ){
			*dp = 0;
			len = strlen(mt->Dst.u_src);
			mt->Dst.u_format = strdup(dp+1);
			*strchr(mt->Dst.u_format,']') = 0;
		}
	}
	if( mt->Dst.u_src[len-1] == '*' ){
		mt->Dst.u_src[len-1] = 0;
		mt->Dst.u_remain = 1;
	}

	mt->u_opts = strdup(opts);
	scan_commaList(opts,1,scanopt,mt);
}
set_MOUNT_ifndef(Conn,src,dst,opts)
	char *Conn;
	char *src,*dst,*opts;
{	int mi;
	char srcbuf[1024];
	int len,remain;

	strcpy(srcbuf,src);
	len = strlen(src);
	if( srcbuf[len-1] == '*' ){
		srcbuf[len-1] = 0;
		remain = 1;
	}else	remain = 0;

	for( mi = 0; mi < mtabN; mi++ ){
		if( strcmp(mtab[mi]->Src.u_src,srcbuf) == 0 )
		if( strcmp(mtab[mi]->u_opts,opts) == 0 )
			/* if ``cond'' are different */
			return;
	}
	if( *opts == 0 )
		opts = "default";
	set_MOUNT(Conn,src,dst,opts);
}

/*
 *  - A rule with a left hand ``Path1*'' should be after the rule with
 *    a left hand ``Path1Path2''.
 *  - In rules of the same left hand ``Path1'' with each other should be
 *    sorted (or overwritten if without any condition) in reverse order
 *    of the definition.
 *
 *  return S1_S2;  s1 should be before than s2
 *  return S2_S1;  s2 should be before than s1
 */

#define S2_S1	 1
#define S1_S2	-1

static mtabcomp(mt1p,mt2p)
	Mtab **mt1p,**mt2p;
{	Mtab *mt1,*mt2;
	char *s1,*s2;
	int r1,r2;
	int comp;

	mt1 = *mt1p; if( mt1 == NULL ) return S2_S1;
	mt2 = *mt2p; if( mt2 == NULL ) return S1_S2;

	s1 = mt1->Src.u_src;
	s2 = mt2->Src.u_src;
	r1 = mt1->Src.u_remain;
	r2 = mt2->Src.u_remain;

	/*if( mt1->Dst.u_asis && mt2->Dst.u_asis )*/
/*
	if( streq(s1,ASIS) && streq(s2,ASIS) )
	{
		s1 = mt1->Dst.u_src;
		s2 = mt2->Dst.u_src;
		r1 = mt1->Dst.u_remain;
		r2 = mt2->Dst.u_remain;
	}
*/

	comp = strcmp(s1,s2);

	if( comp == 0 ){
		if( r1 && !r2 ) return S2_S1;
		if( r2 && !r1 ) return S1_S2;

		/* rule with rexp-prefix first */
		if(  mt1->Src.u_format && !mt2->Src.u_format ) return S1_S2;
		if( !mt1->Src.u_format &&  mt2->Src.u_format ) return S2_S1;

		/* default rule last */
		if( mt1->u_conds & C_DEFAULT ) return S2_S1;
		if( mt2->u_conds & C_DEFAULT ) return S1_S2;

		/* rule with src=xxx first */
		if( mt1->u_conds > mt2->u_conds ) return S1_S2;
		if( mt1->u_conds < mt2->u_conds ) return S2_S1;

		/* reverse entry order for overwriting */
		if( mt1->u_serno < mt2->u_serno )
			return S2_S1;
		else	return S1_S2;
	}else{	
		if( strstr(s1,s2) && r2) return S1_S2;
		if( strstr(s2,s1) && r1) return S2_S1;

		/* entry order */
		if( mt1->u_serno < mt2->u_serno )
			return S1_S2;
		else	return S2_S1;
	}
}
static sort_mtab()
{
	Bsort(mtab,mtabN,sizeof(Mtab*),mtabcomp);
}
static dump_mtab()
{	int mi;
	Mtab *mt;

	for( mi = 0; mi < mtabN; mi++ ){
		mt = mtab[mi];
		sv1log("MOUNT[%d]%s[%d] %s%s %s%s %s\n",
			mi,
			mi != mt->u_serno-1 ? "X":"=",
			mt->u_serno-1,
			mt->Src.u_src,mt->Src.u_remain?"*":"",
			mt->Dst.u_src,mt->Dst.u_remain?"*":"",
			mt->u_opts);
	}
}
init_mtab(){
	int mi;
	char *ssrc,*dsrc;
	char proto[64],hostport[256],host[256],port[256],path[1024],prefix[64];
	int plen;
	char *addr,*dpath;
	char login[256],*dp,userpass[256],user[64],pass[64];
	char loginpath[1024];
	int iport;
	Mtab *mt;
	char *opts;
	char dproto[256];
	char *upath;

	if( mtabN <= init_from )
		return;
	mi = init_from;
	init_from = mtabN;

	if( do_MTAB_SORT )
		sort_mtab();
	dump_mtab();

	for(mi = 0; mi < mtabN; mi++ ){
		mt = mtab[mi];
		if( mt->u_compiled )
			continue;
		mt->u_compiled = 1;

		ssrc = mt->Src.u_src;
		if( streq(ssrc,ASIS) )
			mt->Src.u_asis = 1;
		else
		if( isFullURL(ssrc) ){
			char norm_url[1024],hostport[1024];
			decomp_absurl(ssrc,proto,login,path,sizeof(path));
			mt->Src.u_proto = strdup(proto);
			iport = scan_hostport(proto,login,host);
			sprintf(hostport,"%s:%d",host,iport);
			mt->Src.u_hostList = makePathList("MOUNT.vhost",
				strdup(hostport));
			mt->Src.u_path = strdup(path);
			mt->Src.u_plen = strlen(path);
			if( IsResolvable(host) ){
				sethostcache(host,1);
				sethostcache_predef(host);
			}
			sv1log("## MOUNT FULL-URL-SRC [%s]://[%s:%d]/[%s]\n",
				proto,host,iport,path);
			/* normalize URL string ... */
		}

		proto[0] = path[0] = 0;
		dpath = path;
		dsrc = mt->Dst.u_src;

		dproto[0] = 0;
		scan_URI_scheme(dsrc,dproto,sizeof(dproto));

		if( mt->u_opts )
			opts = mt->u_opts;
		else	opts = "";

		if( streq(dsrc,ASIS) ){
			mt->Dst.u_asis = 1;
			continue;
		}else
		if( dsrc[0] == 0 && mt->Dst.u_remain ){
			mt->Dst.u_any = 1;
			continue;
		}else
		if( plen = isURN(dsrc) ){
			strcpy(proto,dproto);
			login[0] = 0;
			strcpy(path,dsrc+strlen(proto)+1);
			dpath = path;
		}else
		if( mt->Dst.u_remain && streq(dsrc+strlen(dproto),"://") ){
			mt->Dst.u_path2site = 1;
			strcpy(proto,dproto);
			login[0] = 0;
			dpath = "";
		}else
		if( localPathProto(dproto) ){
			if( upath = file_hostpath(dsrc,proto,login) ){
				if( login[0] == 0 )
					strcpy(login,"localhost");
				if( upath[0] == '/' )
					strcpy(path,upath+1);
				else
				if( !isFullpath(upath) ){
					getcwd(path,sizeof(path));
					chdir_cwd(path,upath,0);
					strcat(path,"/");
					if( *path == '/' )
						strcpy(path,path+1);
				}
				else	strcpy(path,upath);
				dpath = path;
			}
		}else
		if( sscanf(dsrc,"%[^:]://%s",proto,loginpath) == 2 ){
			if( loginpath[0] == '/' || loginpath[0] == '?' ){
				mt->Dst.u_emptysite = 1;
				login[0] == 0;
				strcpy(path,loginpath+1);
			}else{
				sscanf(loginpath,"%[^/?]%s",login,path);
				if( path[0] == '/' )
					dpath = path+1;
				else	dpath = path;
			}
		}else{
			Verbose("MOUNT ? %s (right hand should be full-URL)\n",
				mt->Dst.u_src);
			continue;
		}

		decomp_URL_siteX(login,userpass,user,pass,hostport,host,port);
		mt->Dst.u_userpass = strdup(userpass);
		mt->Dst.u_user = strdup(user);
		mt->Dst.u_pass = strdup(pass);

		url_rmprefix(proto,prefix);
		mt->Dst.u_prefix = strdup(prefix);

		if( login[0] ){
			iport = scan_hostport(proto,hostport,host);
			if( addr = gethostaddr(host) )
				mt->Dst.u_hosta = strdup(addr);
			else	addr = "";
			Verbose("MOUNT HOST %s=%s\n",host,addr);
			/* multiple address should be cared ... */
		}else{
			host[0] = 0;
			iport = 0;
			addr = "";
		}

		mt->Dst.u_hostList = makePathList("MOUNT.rhost",
			strdup(hostport));

		mt->Dst.u_proto = strdup(proto);
		mt->Dst.u_hostn = strdup(host);
		mt->Dst.u_stdport = serviceport(proto);
		mt->Dst.u_iport = iport;
		if( dp = strchr(hostport,':') ){
			mt->u_flags |= U_DISPPORT;
			if( dp[1] == '+' || dp[1] == '-' ){
				mt->u_flags |= U_PORTMAP;
				mt->Dst.u_portmap = strdup(dp+1);
			}
		}

		mt->Dst.u_path = strdup(dpath);
		mt->Dst.u_plen = strlen(dpath);

		if( login[0] )
			Verbose("[%d] MOUNT=%s %s%s://%s[%s]:%d%s %s\n",
				mi,mt->Src.u_src,prefix,proto,
				host,addr,iport,path,opts);
		else	Verbose("[%d] MOUNT=%s %s%s:%s %s\n",
				mi,mt->Src.u_src,prefix,proto,path,opts);


		if( IsResolvable(host) ){
			sethostcache(host,1);
			sethostcache_predef(host);
			/* preload cache from all of RESOLV=dns,nis,file ... */
		}
	}
}
static char *rewrite_path(ctx,out,fmt,in)
	void *ctx;
	char *out,*fmt,*in;
{	char *op,*fp,fc;
	char urlbuf[4096],*uv[256];
	int un,u1,u2,ui,tailis;

	op = out;
	strcpy(urlbuf,in);
	tailis = strtailchr(urlbuf) == '/';
	un = stoV(urlbuf,256,uv,'/');

	for( fp = fmt; fc = *fp; fp++ ){
		if( fc != '%' ){
			*op++ = fc;
			continue;
		}
		fc = *++fp;
		if( fc == 0 )
			break;

		if( '0' <= fc && fc <= '9' ){
			u1 = fc - '0';
			u2 = u1;
			if( fp[1] == '-' ){
				fp++;
				u2 = un;
			}
			ui = u1;
			if( ui < un ) do {
				if( u1 < ui )
					*op++ = '/';
				strcpy(op,uv[ui]);
				op += strlen(op);
			} while ( ++ui < u2 );
			if( u2 == un && tailis )
				*op++ = '/';
		}
	}
	*op = 0;
	return op;
}

static dstNoMatch(mt,clif,srcurl)
	Mtab *mt;
	char *clif,*srcurl;
{	char site[256],proto[256],host[256],port[256];
	int porti;
	int got;

	if( mt->Dst.u_path2site ){
		got = 0;
		scan_URI_site(srcurl+strlen(mt->Src.u_src),site,sizeof(site));
		if( *site ){
			got = 1;
			strcpy(proto,mt->Dst.u_proto);
			decomp_URL_site(site,host,port);
		}
	}else{
		got = scan_protositeport(srcurl,proto,host,port);
	}
	if( got ){
		if( *port == 0 )
			porti = serviceport(proto);
		else	porti = atoi(port);
		if( !matchPath1(mt->dstList,"-",host,porti) )
			return 1;
	}
	return 0;
}
static evalCond1(ctx,mt,clif,method,srcurl,rhost,rport)
	void *ctx;
	Mtab *mt;
	char *clif;
	char *method;
	char *srcurl;
	char *rhost;
{
	if( mt->fromList && !matchFROM(ctx,mt->fromList,NULL) )
		return 0;

	if( mt->clifList && clif != NULL ){
		char host[256];
		int port;
		port = scan_hostport1(clif,host);
		if( !matchPath1(mt->clifList,"-",host,port) )
			return 0;
	}

	if( mt->dstList && srcurl != NULL ){
		if( (mt->u_conds & C_VHOST) == 0 )
			return 0;
		if( dstNoMatch(mt,clif,srcurl) )
			return 0;
	}

	if( mt->dstList && rhost != NULL ){
		int nomatch,co;
		if( (mt->u_conds & C_RHOST) == 0 )
			return 0;
		co = RES_CACHEONLY(1);
		nomatch = !matchPath1(mt->dstList,"-",rhost,rport);
		RES_CACHEONLY(co);
		if( nomatch )
			return 0;
	}

	if( mt->u_method && method != NULL ){
		/*sv1log("### MOUNT METHOD [%s][%s]\n",mt->u_method,method);*/
		if( strcmp(mt->u_method,"*") != 0 )
		if( strcasestr(mt->u_method,method) == NULL )
			return 0;
	}

	if( mt->pathList && !CTX_matchPATHs(ctx,mt->pathList,NULL) )
		return 0;

	if( mt->viaList && !CTX_matchPATH1(ctx,mt->viaList,NULL,"-") )
		return 0;

	if( mt->u_xcond && evalCondition(ctx,mt->u_xcond) == 0 )
		return 0;

	return 1;
}
static evalCond(ctx,mt,clif,dstproto,method,srcurl,rhost,rport)
	void *ctx;
	Mtab *mt;
	char *clif;
	char *dstproto;
	char *method;
	char *srcurl;
	char *rhost;
{	int value;
	char param[1024],*pp;

	if( mt->u_dstproto && dstproto ){
		if( !wordIsinList(mt->u_dstproto,dstproto) )
			return 0;
	}

	param[0] = 0;
	pp = param;
	if( clif && mt->clifList ){ pp = Strcpy(pp,clif); *pp++ = ' '; }
	if( rhost && mt->dstList ){ pp = Strcpy(pp,rhost); *pp++ = ' '; }
	*pp = 0;

	/*
	if( method ){ strcat(param," "); strcat(param,method); }
	if( srcurl ){ strcat(param," "); strcat(param,srcurl); }
	*/

	if( mt->u_conds_value )
	if( strcmp(param,mt->u_conds_param) == 0 )
		return 0 < mt->u_conds_value ? 1 : 0;
		/* should check if input parameters (clif,method,srcurl)
		 * is identical with the previous evaluation...
		 */

	value = evalCond1(ctx,mt,clif,method,srcurl,rhost,rport);
	if( clif ){
		if( value )
			mt->u_conds_value = 1;
		else	mt->u_conds_value = -1;
		Strdup(&mt->u_conds_param,param);
	}
	return value;
}
reset_MOUNTconds()
{	int mi;

	for( mi = 0; mi < mtabN; mi++ ){
		mtab[mi]->u_conds_value = 0;
		mtab[mi]->Dst.u_iport_mapped = 0;
	}
}
static portmap(ctx,mt)
	void *ctx;
	Mtab *mt;
{	int mport;
	Url *dst;

	dst = &mt->Dst;
	if( mport = dst->u_iport_mapped )
		return mport;
	mport = portMap(ctx,dst->u_proto,dst->u_hostn,dst->u_portmap);
	dst->u_iport_mapped = mport;
	return mport;
}

CTX_evalMountCond(ctx,opts,user,chost,cport,ihost,iport)
	void *ctx;
	char *opts,*user,*chost,*ihost;
{	int List;
	int match = 0;
	char path[1024];

	sprintf(path,"%s:%d!%s:%d",ihost,iport,chost,cport);

	if( strncmp(opts,"from=",5) == 0 ){
		List = makePathList("SERVER.from",opts+5);
		match = matchPath1(List,user,chost,cport);
	}else
	if( strncmp(opts,"host=",5) == 0 ){
		List = makePathList("SERVER.clif",opts+5);
		match = matchPath1(List,user,ihost,iport);
	}else
	if( strncmp(opts,"path=",5) == 0 ){
		List = makePathList("SERVER.path",opts+5);
		match = CTX_matchPATHs(ctx,List,path);
	}else{
		if( strncmp(opts,"via=",4)==0 || strncmp(opts,"src=",4)==0 )
			opts += 4;
		List = makePathList("SERVER.via",opts);
		match = CTX_matchPATH1(ctx,List,path,user);
	}
	return match;
}
static matchHost(up,host)
	Url *up;
	char *host;
{	char *ahostn,*ahosta,*hosta;

	ahostn = up->u_hostn;
	ahosta = up->u_hosta;
	if( streq(host,ahostn) || ahosta && streq(host,ahosta) ) 
		return 1;

	if( ahosta )
	if( hosta = gethostaddr_fromcache(host) )
		if( streq(hosta,ahosta) )
			return 1;

	/* when the host have several addresses ...
	 * `matchhostaddr_incache(host,ahosta)' is necessary ?
	 */
	return 0;
}
static matchURL(up,url)
	Url *up;
	char *url;
{	char proto[128],login[128],host[128],upath[1024];
	int iport;
	int hlen,plen;

	hlen = plen = 0;
	decomp_absurl(url,proto,login,upath,sizeof(upath));
	iport = scan_hostport(proto,login,host);

	if( streq(proto,up->u_proto) )
	if( matchPath1(up->u_hostList,"-",host,iport) )
	{
		hlen = strlen(url) - strlen(upath);
		plen = url_strstr(upath,up->u_path);
		Verbose("## MATCH FULL-URL-SRC [%s][%s] %d+%d\n",
			up->u_src,url,hlen,plen);
	}
	return hlen + plen;
}
static xmatch(mt,patn,url,lenp)
	Mtab *mt;
	char *patn,*url;
	int *lenp;
{
	*lenp = 0;
	if( *patn == 0 || (*lenp = url_strstr(url,patn)) ){
		if( mt->Src.u_remain || url[*lenp] == 0 )
			return 1;
	}
	return 0;
}
static char *mount_url_toX(ctx,myhostport,method,url,qtype,rmt)
	void *ctx;
	char *myhostport;
	char *method,*url;
	Mtab **rmt;
{	char *up,iurl[16*1024],*rp,src_rest[1024];
	char *iupath;
	int isfull;
	char *ourl;
	int match;
	int ai;
	char *addr;
	int len;
	Mtab *mt;
	int delput;
	char *dstproto,protob[128],*dp;

	if( myhostport == NULL )
		myhostport = CTX_clif_hostport(ctx);

	strcpy(iurl,url);
	if( rp = strpbrk(iurl," \t\r\n") ){
		strcpy(src_rest,rp);
		*rp = 0;
	}else	src_rest[0] = 0;

	isfull = isFullURL(iurl);
	/* must care full-path MOUNT ...
	if( isfull ){
		iupath = ...
	}else
	 */
	iupath = iurl;
	if( urlpath_normalize(iupath,iupath) )
		Verbose("MOUNT url-path normalized [%s]>[%s]\n",url,iurl);

	for( ai = 0; ai < mtabN; ai++ ){
		mt = mtab[ai];
		if( mt->Src.u_asis || mt->u_disabled )
			continue;

		match = xmatch(mt,mt->Src.u_src,iurl,&len);
		if( !match && mt->Src.u_proto && isfull ){
			int hlen;
			if( hlen = matchURL(&mt->Src,iurl) )
			if( match = xmatch(mt,mt->Src.u_path,iurl+hlen,&len) )
				len += hlen;
		}

		if( !match )
			continue;

		if( mt->Dst.u_any ){
			ourl = iurl + strlen(mt->Src.u_src);
			scan_URI_scheme(ourl,protob,sizeof(protob));
			dstproto = protob;
		}else{
			ourl = iurl;
			dstproto = NULL;
		}
		if( evalCond(ctx,mt,myhostport,dstproto,method,ourl,NULL,0)==0 )
			continue;

		if( mt->Src.u_format )
			if( !RexpMatch(&iurl[len],mt->Src.u_format) )
				continue;

		if( mt->u_conds & C_WITHQUERY )
		if( strchr(iurl,'?') == NULL )
			continue;

		if( mt->u_conds & C_ASPROXY )
		if( !isfull )
			continue;

		/*
		 * input URL matched with the URL pattern of this MOUNT point
		 */

		if( (qtype & U_MOVED_TO ) && (mt->u_flags & U_MOVED_TO ) == 0
		 || (qtype & U_USE_PROXY) && (mt->u_flags & U_USE_PROXY) == 0
		){
			Verbose("[%d] URL Matched but not for MovedTo[%x]:%s\n",
				ai,qtype,iurl);
			/* must exit without rewriting */
			goto FOUND;
		}

		if( mt->Dst.u_asis ){
			Verbose("[%d] MOUNT ASIS: %s\n",ai,mt->Src.u_src);
			if( mt->u_opts && mt->u_opts[0] )
				goto FOUND;
			break;
		}

		if( mt->Dst.u_any ){
			up = Strcpy(url,iurl+len);
			up = Strcpy(up,src_rest);
			goto FOUND;
			break;
		}

		if((mt->u_flags & U_RESOLV) && mt->Dst.u_hosta)
			addr = mt->Dst.u_hosta;
		else	addr = mt->Dst.u_hostn;

		up = Strcpy(url,mt->Dst.u_prefix);
		up = Strcpy(up,mt->Dst.u_proto);
		*up++ = ':';

		if( addr[0] ){ 
			up = Strcpy(up,"//");
			if( *mt->Dst.u_userpass ){
				up = Strcpy(up,mt->Dst.u_userpass);
				up = Strcpy(up,"@");
			}
			if( addr[0] == '-' && addr[1] == 0 ){
				up = Strcpy(up,myhostport);
			}else{
				up = Strcpy(up,addr);
				if( mt->u_flags & U_DISPPORT ){
					int mport;
					if( mt->u_flags & U_PORTMAP )
						mport = portmap(ctx,mt);
					else	mport = mt->Dst.u_iport;
					sprintf(up,":%d",mport);
					up += strlen(up);
				}
			}
			delput = 0;
		}else
		if( mt->Dst.u_emptysite ){
			up = Strcpy(up,"//");
			delput = 0;
		}else
		if( mt->Dst.u_path2site ){
			up = Strcpy(up,"//");
			delput = 1;
		}else{
			delput = 1;
		}

		if( mt->Dst.u_path ){
			if( !delput ){
				up = Strcpy(up,"/");
				delput = 1;
			}
			up = Strcpy(up,mt->Dst.u_path);
			delput = 1;
		}

		if( iurl[len] )
		if( mt->Dst.u_remain ){
			if( !delput ){
				up = Strcpy(up,"/");
				delput = 1;
			}
			if( mt->Dst.u_format )
				up = rewrite_path(ctx,up,mt->Dst.u_format,iurl+len);
			else	up = Strcpy(up,iurl+len);
		}

		Verbose("*** %s MOUNTED TO[%d] %s ***\n",
			mt->Src.u_src,ai,mt->Dst.u_src);
		sv1log("*** %s => %s ***\n",iurl,url);

		if( src_rest[0] )/* not a part of the URL */
			up = Strcpy(up,src_rest);
		last_hit = ai+1;
		goto FOUND;
	}
	return 0;

FOUND:
	/*
	 * MOVED_TO and USE_PROXY are not matching conditions about URL
	 * but the interpretations of the rule when the URL is matched.
	 * So they are here.
	 */
	if( qtype & U_MOVED_TO ){
		if((mt->u_flags & U_MOVED_TO) == 0)
			return 0;
	}else
	if( qtype & U_USE_PROXY ){
		if((mt->u_flags & U_USE_PROXY) == 0)
			return 0;
	}else{
		if( mt->u_flags & (U_MOVED_TO|U_USE_PROXY) )
			return 0;
	}
	if( rmt ) *rmt = mt;
	return mt->u_opts;
}
char *CTX_mount_url_to(ctx,myhostport,method,url)
	void *ctx;
	char *myhostport,*method,*url;
{
	return mount_url_toX(ctx,myhostport,method,url,U_MOUNT,NULL);
}
char *CTX_moved_url_to(ctx,myhostport,method,url)
	void *ctx;
	char *myhostport,*method,*url;
{
	return mount_url_toX(ctx,myhostport,method,url,U_MOVED_TO,NULL);
}
char *CTX_changeproxy_url_to(ctx,myhostport,method,url,proxy)
	void *ctx;
	char *myhostport,*method,*url,*proxy;
{	Mtab *mt;
	char *opt;

	*proxy = 0;
	if( opt = mount_url_toX(ctx,myhostport,method,url,U_USE_PROXY,&mt) )
	if( mt->u_useproxy )
		strcpy(proxy,mt->u_useproxy);
	return opt;
}

char *CTX_mount_url_fromL(ctx,url,proto,hostport,path,search,dproto,delegate)
	void *ctx;
	char *url,*proto,*hostport,*path,*search,*dproto,*delegate;
{	char *up,*dp1,*dp2,dhost[128];
	int ai;
	int plen;
	Mtab *mt;
	char host[128];
	int iport;
	int match;
	int mport;

	up = url;

	if( proto == 0 || hostport == 0 )
		return 0;
	if( path == 0 )
		path = "";

	if( dproto == 0 || dproto[0] == 0 )
		dproto = "http";

	if( hostport[0] == 0 ){
		host[0] = 0;
		iport = 0;
	}else	iport = scan_hostport(proto,hostport,host);

	for( ai = 0; ai < mtabN; ai++ ){
		mt = mtab[ai];
		if( mt->Dst.u_asis || mt->u_disabled )
			continue;

		if( mt->u_flags & (U_MOVED_TO|U_USE_PROXY) )
			continue;

		plen = mt->Dst.u_plen;

		match = 0;
		if( mt->Dst.u_any )
			match = 1;
		else
		if( streq(mt->Dst.u_proto,proto) ){
			if( mt->u_flags & U_PORTMAP )
				mport = portmap(ctx,mt);
			else	mport = mt->Dst.u_iport;
			if( mport == 0 || mport == iport )
			if( mt->Dst.u_hostn[0] == 0 || matchHost(&mt->Dst,host) )
			if( plen == 0 || strncmp(path,mt->Dst.u_path,plen) == 0 )
				match = 1;
		}

		if( !match )
			continue;

		if( evalCond(ctx,mt,delegate,proto,NULL,NULL,host,iport)==0 )
			continue;

		if( !mt->Dst.u_remain && path[plen] != 0 )
			continue;

		if( mt->Src.u_asis )
			break;

		Verbose("** %s UNMOUNTED FROM %s **\n",
			mt->Src.u_src,mt->Dst.u_src);

		if( dp1 = strchr(delegate,':') ){
			if( atoi(dp1+1) == serviceport(dproto) ){
				strcpy(dhost,delegate);
				dp1 = strchr(dhost,':');
				if( dp2 = strchr(dp1,'/') )
					strcpy(dp1,dp2);
				else	*dp1 = 0;
				delegate = dhost;
			}
		}

		if( !mt->Src.u_fullurl ){
			sprintf(up,"%s://%s",dproto,delegate);
			up += strlen(up);
		}
		up = Strcpy(up,mt->Src.u_src);
		if( mt->Dst.u_proto == 0 ){
			sprintf(up,"%s://%s/",proto,hostport);
			up += strlen(up);
		}
		if( mt->Dst.u_path2site ){
			sprintf(up,"%s/",hostport);
			up += strlen(up);
		}

		if( streq(proto,"gopher") ){
			if( up[-1] == '/' )
				up = Strcpy(up,"1");
			else	up = Strcpy(up,"/1");
		}

		if( plen )
			path += plen;
		up = Strcpy(up,path);

		if( search ){
			*up++ = '?';
			up = Strcpy(up,search);
		}
		last_hit = ai+1;
		return mt->u_opts;
	}
	return 0;
}
CTX_scan_mtab(ctx,func,arg)
	void *ctx;
	int (*func)();
	char *arg;
{	Mtab *mt;
	int mi;
	char *clif;

	clif = CTX_clif_hostport(ctx);
	for( mi = 0; mi < mtabN; mi++ ){
		mt = mtab[mi];

		if( evalCond(ctx,mt,clif,NULL,NULL,NULL,NULL,0) == 0 )
			continue;

if( mt->Dst.u_hostn )
/* scanned but not compiled, caused by duplicat set of MOUNT list X-< */
		(*func)(arg,
			mt->Src.u_src,
			mt->Dst.u_proto,
			mt->Dst.u_user,
			mt->Dst.u_pass,
			mt->Dst.u_hostn,
			mt->Dst.u_iport,
			mt->Dst.u_path,
			mt->u_opts);
	}
}

trans_url(nurl,ourl,fmt)
	char *nurl,*ourl,*fmt;
{
	strcpy(nurl,fmt);
	sv1log("Transform-URL: %s -> %s\n",ourl,nurl);
}
