/*////////////////////////////////////////////////////////////////////////
Copyright (c) 1994-1999 Yutaka Sato
Copyright (c) 1994-1999 Electrotechnical Laboratry (ETL), AIST, MITI

Permission to use, copy, and distribute this material for any purpose
and without fee is hereby granted, provided that the above copyright
notice and this permission notice appear in all copies.
ETL MAKES NO REPRESENTATIONS ABOUT THE ACCURACY OR SUITABILITY OF THIS
MATERIAL FOR ANY PURPOSE.  IT IS PROVIDED "AS IS", WITHOUT ANY EXPRESS
OR IMPLIED WARRANTIES.
/////////////////////////////////////////////////////////////////////////
Content-Type:	program/C; charset=US-ASCII
Program:	delegated (DeleGate Server)
Author:		Yutaka Sato <ysato@etl.go.jp>
Description:
History:
	950525	extracted from delegated.c
	950525	reformed to be independent of DeleGate
//////////////////////////////////////////////////////////////////////#*/
#include <stdio.h>
#include "ystring.h"

#include "log.h"
#define NOLOG	((FILE*)-1)
#define RIGHTNOW	-1

#define LINESIZE	0x4000
#define PATHLENG	1024

int LOG_type;

char *(*LOG_stdlogfile)();
int   (*LOG_substfile)();

int LOG_sockio[2] = {-1,-1};
int LOG_sock_enable;
int LOG_public;
int LOG_center = -1;

char LP_NOTTY[] = "NoTTY";
char LF_LOGFILE[]  = "LOGFILE";
char LF_ERRORLOG[] = "ERRORLOG";
char LF_TRACELOG[] = "TRACELOG";
char LF_PROTOLOG[] = "PROTOLOG";
char LF_ABORTLOG[] = "ABORTLOG";

#define LOG_buffered(logF)	(logF->l_buff && logF->l_buff[0])

static log_flush(logF,fp)
	Logfile *logF;
	FILE *fp;
{	int rcode;

	if( logF == NULL )
		return 0;

	if( logF->l_abandon || fp == NULL )
		rcode = 0;
	else	rcode = fputlog(0,logF->l_buff,fp);

	free(logF->l_buff);
	logF->l_buff = NULL;
	logF->l_size = 0;
	logF->l_leng = 0;
	return rcode;
}

extern Logfile *LOG_create();
static Logfile *LogAbort;
setAbortLog(form)
	char *form;
{
	if( form ){
		LogAbort = LOG_create("abortlog",LF_ABORTLOG,"-",form,"a",0);
		LogAbort->l_abandon = 1;
	}
}
AbortLog(){
	int abandon;

	if( LogAbort ){
		LOG_write(LogAbort,"\n",1);
		abandon = LogAbort->l_abandon;
		LogAbort->l_abandon = 0;
		LOG_flush(LogAbort,time(0));
		LogAbort->l_abandon = abandon;
	}
}
clrAbortLog(){
	log_flush(LogAbort,NULL);
}

/*
static int *stack0,*stackm;
checkStackSize(arg)
{

	if( stack0 == NULL || (unsigned int)stack0 < ((unsigned int)&arg) ){
		stack0 = &arg;
		if( stackm == NULL )
			stackm = stack0;
	}
	if( ((unsigned int)&arg) < ((unsigned int)stackm) ){
		stackm = &arg;
		fprintf(stderr,"#### STACK SIZE #### %8d\n%s\n",
			(unsigned int)stack0-(unsigned int)stackm,arg);
		sleep(3);
	}
}
*/

static fputlog(public,str,fp)
	char *str;
	FILE *fp;
{	int rcode;
	int len;
	int wcc;

/*
	checkStackSize(str);
*/
	len = strlen(str);

	if( 0 < len && fp != NULL && fp != NOLOG ){
		fseek(fp,0,2);
		fputs(str,fp);
		rcode = fflush(fp);
	}else	rcode = 0;

	if( 0 < len ){
		if( LOG_sock_enable )
			write(LOG_sockio[1],str,len);

		if( (public || LOG_public) && 0 <= LOG_center ){
			wcc = write(LOG_center,str,len);
			Verbose("sendlog[%d] %s",wcc,str);
		}

		if( LogAbort )
			LOG_write(LogAbort,str,len);
	}
	return rcode;
}


int LOG_VERBOSE;
int LOG_GENERIC;

logVERBOSE(){ return LOG_VERBOSE; }

extern int START_TIME;
extern char *TIMEFORM_mdHMS;
extern char *TIMEFORM_HTTPD;
extern FILE *dirfopen();

int CHILD_SERNO;
int CHILD_SERNO_MULTI;
int CHILD_SERNO_SINGLE;
SERNO(){ return CHILD_SERNO; }
SERNO_MINOR(){ return CHILD_SERNO_MULTI; }
MySeqNum(){ return (CHILD_SERNO<<16) + (CHILD_SERNO_MULTI+CHILD_SERNO_SINGLE); }

static Logfile LogFiles[32];
static int numLogFiles = 1;
#define LogF0	LogFiles[0]

#define LOGFILE_ACTIVE(fp) (fp!=NULL && fp!=NOLOG && fp!=stderr)
#define LOGFILE_OPENED(fp) (fp!=NULL && fp!=NOLOG)
#define Lfileno(fp) (LOGFILE_OPENED(fp) ? fileno(fp) : -1)

static locked_fclose(logF)
	Logfile *logF;
{	FILE *fp;
	FILE *fp1;
	int rcode,lkfd,lkrcode,elapsed;
	int logfd;
	Logfile *logf;
	int li;

	fp = logF->l_fp;
	rcode = 0;

	if( !LOGFILE_ACTIVE(fp) )
		goto EXIT;

	for( li = 0; li < numLogFiles; li++ ){
		logf = &LogFiles[li];
		fp1 = logf->l_fp;
		if( fp1 == fp && logF != logf )
			goto EXIT;
	}

	logfd = fileno(fp);
	lkfd = -1;
	if( logF->l_dolock ){
		if( (lkfd = logF->l_lockfd) < 0 )
			lkfd = logfd;
		if( 0 <= lkfd )
			lkrcode = lock_exclusiveTO(lkfd,5*1000,&elapsed);
	}

	/* explicit unlock seems necessary
	 * when another process is openning the file ... */
	if( lkfd == logfd && lkrcode == 0 )
		lock_unlock(lkfd);

	rcode = fclose(fp);

	if( logF->l_dolock ){
		if( 0 <= logF->l_lockfd ){
			close(logF->l_lockfd);
			logF->l_lockfd = -1;
		}
	}

EXIT:
	logF->l_fp = NULL;
	return rcode;
}
static locked_fflush(logF)
	Logfile *logF;
{	FILE *fp;
	int rcode,lkfd,lkrcode,elapsed;

	fp = logF->l_fp;
	lkfd = -1;
	if( logF->l_dolock && LOG_buffered(logF) ){
		if( (lkfd = logF->l_lockfd) < 0 )
			lkfd = fileno(fp);
		if( 0 <= lkfd )
			lkrcode = lock_exclusiveTO(lkfd,5*1000,&elapsed);
	}

	if( logF->l_buff )
		rcode = log_flush(logF,fp);
	else	rcode = fflush(fp);

	if( 0 <= lkfd )
		lock_unlock(lkfd);

	return rcode;
}

static setLogf(logF,fp)
	Logfile *logF;
	FILE *fp;
{	FILE *Fp;

	if( logF == NULL )
		return;
	if( logF->l_fp == fp )
		return;

	Fp = logF->l_fp;
	if( LOGFILE_OPENED(Fp) ){
		fprintf(stderr,
		"DeleGate[%d] overwrite log entry! %x[%d]->%x[%d] %s\n",
			getpid(),Fp,Lfileno(Fp),fp,Lfileno(fp),
			logF->l_pform?logF->l_pform:"(NoFileName)");
		locked_fclose(logF);
	}
	logF->l_fp = fp;
}
static Lclose(logF)
	Logfile *logF;
{
	if( LOGFILE_ACTIVE(logF->l_fp) )
		locked_fclose(logF);
	logF->l_fp = NULL;
}
static closeAgedLog1(Logf,now)
	Logfile *Logf;
{
	if( LOGFILE_ACTIVE(Logf->l_fp) )
	if( now == RIGHTNOW || Logf->l_until != 0 && Logf->l_until <= now )
	{
/*
should not write such a debug info into the logfile like http.log ...
if( SERNO_MINOR() == 0 ){
fseek(Logf->l_fp,0,2);
putTime(Logf->l_fp);
 fprintf(Logf->l_fp,"Check logfile age: %s\n",Logf->l_path);
}
*/
		Lclose(Logf);
		return 1;
	}
	return 0;
}

static substDate1(file,link,tag,date)
	char *file,*link,*tag;
{	char *fp,*dp,*np,*tp,format[256];

	while( dp = strstr(link,tag) ){
		if( tp = strchr(link,']') )
			strcpy(dp,tp+1);
		else{
			fprintf(stderr,"missing ']' for %s...\n",tag);
			break;
		}
	}
	while( dp = strstr(file,tag) ){
		np = dp + strlen(tag);
		strcpy(format,np);
		if( tp = strchr(format,']') ){
			*tp++ = 0;
			StrftimeLocal(dp,32,format,date,0);
			strcat(dp,tp);
		}else{
			fprintf(stderr,"missing ']' for %s...\n",tag);
			break;
		}
	}
}
static substDate(file,link,now,start)
	char *file,*link;
{
	strcpy(link,file);
	substDate1(file,link,"[date+",now);
	substDate1(file,link,"[start+",start);
	if( strcmp(file,link) != 0 )
		return 1;
	else	return 0;
}
StrSubstDate(str)
	char *str;
{	char tmp[1024];

	return substDate(str,tmp,time(0),START_TIME);
}

/*
 * assuming that log file name is valid at least in the same minute.
 */
static valid_until(filefmt,now)
	char *filefmt;
{	int until,step,stepi;
	char file1[PATHLENG],file2[PATHLENG],current[PATHLENG];

	strcpy(file1,filefmt); substDate(file1,current,now,START_TIME);
	step = 24 * 3600 * 7;
	until = now;

	step /= 2;
	until += step;
	for( stepi = 0; 0 < step; stepi++ ){
		strcpy(file2,filefmt);
		substDate(file2,current,until,START_TIME);
		step /= 2;
		if( strcmp(file1,file2) == 0 )
			until += step;
		else	until -= step;
	}
	if( strcmp(file1,file2) == 0 )
		until += 60;
	if( until < now + 60 )
		until = now + 60;
	until = (until / 60) * 60;
	return until;
}

static makeAbspath(file,base)
	char *file,*base;
{	char tmp[PATHLENG];

	if( !isBoundpath(file) ){
		strcpy(tmp,file);
		file[0] = 0; strcats3(file,base,"/",tmp,NULL);
	}
}

/*
 * Unlinking an opened NFS file (inode) will produce a .nfsXXXX file.
 * This will happen when one of possible multiple hard links of the file
 * is opened.  Moreover on Solaris2.5, repetitive link & unlink will
 * produce multiple .nfsXXXX files...
 * - the link to the current file should be symbolic link on Unix ?
 * - previous one can be moved to LOGDIR/prev/LOGFILE if it is a symlink.
 * - unlinking could be done by rename() safely ?
 * - should wait until all referers of the file will close it ?
 * - old "current" link with only one link count must be backed up.
 */
static FILE *agefile(ofp,file,current,mode,tmpdir)
	FILE *ofp;
	char *file,*current,*mode,*tmpdir;
{	char agelock[PATHLENG],lkpath[PATHLENG];
	int was_active;
	int lockfd = -1;
	FILE *nfp;

	was_active = LOGFILE_ACTIVE(ofp);

loglog("agefile(%x,%s,%s,%s,%s) %d\n",ofp,file,current,mode,tmpdir,was_active);

	if( File_cmp(file,current) == 0 ){
		if( was_active ){
			/* if the active file is obsoleted, reopen it... */
			if( file_ino(fileno(ofp)) != File_ino(current) )
				goto REOPEN;
		}
		return ofp;
	}

	sprintf(agelock,"%s.aging",current);
	lockfd = getLockFile(tmpdir,agelock,lkpath);
loglog("#### agelock=%s %s [%d]\n",agelock,lkpath,lockfd);
	if( 0 <= lockfd && lock_exclusiveTO(lockfd,20*1000,NULL) != 0 )
		goto REOPEN;

	/* another process may aged the file during waiting for lock */
	if( File_cmp(file,current) == 0 )
		goto REOPEN;

loglog("AGEFILE(%x,%s,%s,%s,%s) %d\n",ofp,file,current,mode,tmpdir,was_active);

	if( File_is(current) && 0 < File_size(file) ){
		char *getenv();
		char aged[PATHLENG],*aged_ext,ext[256];
		int rcode;
		aged_ext = getenv("AGEDFILEEXT");
		if( aged_ext == 0 || *aged_ext == 0 )
			aged_ext = "-%Y%m%d%H%M%S.old";
		StrftimeLocal(ext,sizeof(ext),aged_ext,File_mtime(file));
		sprintf(aged,"%s%s",file,ext);
		rcode = rename(file,aged);
		File_touch(file);
loglog("AGEFILE rename(%s,%s)=%d\n",file,aged,rcode);
	}

	unlink(current);
	linkRX(file,current); /* should use symbolic link ? */

REOPEN:
	nfp = fopen(file,mode);
	if( nfp == NULL ){
		extern char *getusernames();
		char names[1024],msg[1024];
		FILE *tfp;

		sprintf(msg,"Can't fopen(\"%s\",\"%s\") by %s",file,mode,
			getusernames(names));

		if( was_active )
			fprintf(ofp,"#### %s\n",msg);
		else
		if( tfp = fopen(current,"a") ){
			fprintf(tfp,"#### %s\n",msg);
			fclose(tfp);
		}else{
			fprintf(stderr,"DeleGate: %s\n",msg);
		}
	}
	if( was_active )
		fclose(ofp);

	if( 0 <= lockfd )
		close(lockfd);

	return nfp;
}

static FILE *open_logtmpfile(Logf,istmp,form,mode,proto)
	Logfile *Logf;
	char istmp,*form,*mode,*proto;
{	char *dp;
	char file[PATHLENG],current[PATHLENG],filefmt[PATHLENG];
	char vardir[PATHLENG],logdir[PATHLENG],tmpdir[PATHLENG];
	int now;
	int linked;
	FILE *fp;
	char lkpath[PATHLENG];

	if( form == NULL || *form == 0 )
		return NOLOG;
	if( strcmp(form,"-stderr") == 0 )
		return stderr;

	strcpy(file,form);
	if( LOG_substfile != NULL )
		(*LOG_substfile)(file,proto,vardir,logdir,tmpdir);
	if( istmp )
		makeAbspath(file,tmpdir);
	else	makeAbspath(file,logdir);
	now = time(0);
	strcpy(filefmt,file);
	linked = substDate(file,current,now,START_TIME);

	if( Logf->l_fp == NULL )
		setLogf(Logf,NOLOG);

	fp = dirfopen("LOG-ACT",file,mode);
	if( lARGDUMP() ){
		fprintf(stderr,"open:   %x %s (%s,%s)=%x\n",
			Logf,form,file,mode,fp==NULL?-1:fileno(fp));
		sleep(1);
	}
	if( fp == NULL ){
		/*ERRMSG("Warning: cannot open logfile %s\n",file);*/
		fp = NOLOG;
	}

	if( linked ){
		makeAbspath(current,logdir);
		fp = agefile(fp,file,current,mode,tmpdir);
		if( fp == NULL )
			fp = NOLOG;
		Logf->l_until = valid_until(filefmt,now);

{
char next[64];
StrftimeLocal(next,sizeof(next),TIMEFORM_HTTPD,Logf->l_until);
loglog("valid_until: next=[%s] cur=[%s] fp=%x\n",next,file,fp);
}
	}

	if( fp != NOLOG && Logf != NULL ){
		Strdup(&Logf->l_pform,form);
		Strdup(&Logf->l_path,file);
		Strdup(&Logf->l_mode,mode);
		setLogf(Logf,fp);
		if( Logf->l_dolock && Logf->l_lockfd < 0 ){
			Logf->l_lockfd = getLocalLock(fp,tmpdir,current,lkpath);
			if( 0 <= Logf->l_lockfd ){
				if( lARGDUMP() )
					fprintf(stderr,"##REMOTE:%s\n",current);
				Strdup(&Logf->l_lkpath,lkpath);
			}
		}
	}
	return fp;
}

static FILE *open_tmpfile(Logf,form,mode,proto)
	Logfile *Logf;
	char *form,*mode,*proto;
{	FILE *fp;

	fp = open_logtmpfile(Logf,1,form,mode,proto);
	if( fp == NOLOG )
		return NULL;
	else	return fp;
}
FILE *open_logfile(Logf,form,mode,proto)
	Logfile *Logf;
	char *form,*mode,*proto;
{	FILE *fp;

	if( lTTY() && !Logf->l_notty )
		fp = stderr;
	else	fp = open_logtmpfile(Logf,0,form,mode,proto);
	setLogf(Logf,fp);
	return fp;
}

Logfile *LOG_which(proto,filter1,options)
	char *proto,*filter1;
{	int li;
	Logfile *logF;
	char *proto1,*filters;

	for( li = 0; li < numLogFiles; li++ ){
		logF = &LogFiles[li];
		proto1 = logF->l_proto;

		if( proto1 != 0 )
		if( streq(proto1,"*") || strcaseeq(proto1,proto) )
		if( filters = logF->l_filters ){
			if( strcmp(filter1,filters) == 0 )
				goto EXIT;
			if( (options & LW_EXMATCH) == 0 ){
				if( strcmp(filters,"*") == 0 )
					goto EXIT;
				if( strstr(filters,filter1) )
					goto EXIT;
			}
		}
	}
	if( options & LW_CREATE )
		logF = &LogFiles[numLogFiles++];
	else	logF = NULL;
EXIT:
	if( lARGDUMP() )
	    fprintf(stderr,"[%d] >> LOG_which(%s,%s) = %s\n",
		getpid(),proto,filter1,
		(logF&&logF->l_path)?logF->l_path:"<not-enabled>");
	return logF;
}
char *LOG_format(logF)
	Logfile *logF;
{
	return logF->l_lform;
}
char *LOG_buffer(logF)
	Logfile *logF;
{
	return logF->l_buff;
}

static setLogParams(logF,proto,filters,logform,mode,pathform,dolock)
	Logfile *logF;
	char *proto,*filters,*logform,*mode,*pathform;
{

	if( lARGDUMP() )
	    fprintf(stderr,"[%d] LOG(%-8s, %-8s, %1s, %1s, %-26s) %x %x\n",
		getpid(),proto,filters,logform,mode,pathform,logF,logF->l_fp);

	Strdup(&logF->l_proto,  proto);
	Strdup(&logF->l_pform,  pathform);
	Strdup(&logF->l_filters,filters);
	Strdup(&logF->l_lform,  logform);
	Strdup(&logF->l_mode,   mode);
	logF->l_dolock = dolock;
	logF->l_notty = strcmp(proto,LP_NOTTY) == 0;
	logF->l_lockfd = -1;
}
Logfile *LOG_create(proto,filters,logform,pathform,mode,dolock)
	char *proto,*filters,*logform,*pathform,*mode;
{	Logfile *logF;

	logF = LOG_which(proto,filters,LW_EXMATCH|LW_CREATE);
	if( LOGFILE_ACTIVE(logF->l_fp) )
		Lclose(logF);
	setLogParams(logF,proto,filters,logform,mode,pathform,dolock);
	LOG_GENERIC = 1;
	return logF;
}
FILE *LOG_open(logF)
	Logfile *logF;
{
	if( logF->l_pform == NULL ){
		return NULL;
	}
	return open_logfile(logF,logF->l_pform,logF->l_mode,logF->l_proto);
}

extern char *Malloc();
#define MAXLENG		0x10000
#define LOGCHUNK	(MAXLENG/8)

LOG_write(LogF,str,leng)
	Logfile *LogF;
	char *str;
{	char *buff;
	int inc;

	buff = LogF->l_buff;
	if( LogF->l_size < LogF->l_leng + leng + 1 ){
		inc = (((leng+1)/LOGCHUNK) + 1) * LOGCHUNK;
		if( MAXLENG < LogF->l_size + inc )
			LogF->l_leng = 0;
		else{
			LogF->l_size += inc;
			LogF->l_buff = Malloc(LogF->l_buff,LogF->l_size);
		}
	}
	if( LogF->l_buff == NULL ){
		if( LOGFILE_OPENED(LogF->l_fp) ){
			fprintf(LogF->l_fp,"#### FATAL: Malloc(%x,%d) failed.\n",
				buff,LogF->l_size);
			fflush(LogF->l_fp);
		}
		LogF->l_size = 0;
		return;
	}
	Strncpy(&LogF->l_buff[LogF->l_leng],str,leng+1);
	LogF->l_leng += leng;
}
LOG_printf(logF,fmt,a,b,c,d,e,f,g,h,i,j,k,l,m,n)
	Logfile *logF;
	char *fmt,*a,*b,*c,*d,*e,*f,*g,*h,*i,*j,*k,*l,*m,*n;
{	MemFile MemF; char msg[LINESIZE];

	str_sopen(MemF,"LOG_printf",msg,sizeof(msg),0,"w");
	str_sprintf(MemF,fmt,a,b,c,d,e,f,g,h,i,j,k,l,m,n);
	LOG_write(logF,msg,strlen(msg));
}

static LOG_flush(logF,now)
	Logfile *logF;
{	FILE *fp;
	int rcode;

	if( !LOG_buffered(logF) )
		return;

	closeAgedLog1(logF,now);
	if( logF->l_fp == NULL )
		LOG_open(logF);
	fp = logF->l_fp;
	if( fp == NULL || fp == NOLOG )
		return;

	rcode = locked_fflush(logF);

	if( rcode == EOF ){
		fprintf(stderr,"DeleGate-LOG_flush: EOF[%d]\n",fileno(fp));
		perror("DeleGate-LOG_flush");
	}
}
LOG_flushall()
{	int li;
	Logfile *logF;
	int now;

	now = time(0);
	for( li = 0; li < numLogFiles; li++ ){
		logF = &LogFiles[li];
		LOG_flush(logF,now);
	}
}
LOG_closeall()
{	int li;
	Logfile *logF;
	int now;

	now = time(0);
	for( li = 0; li < numLogFiles; li++ ){
		logF = &LogFiles[li];
		LOG_flush(logF,now);
		Lclose(logF);
	}
}
LOG_openall()
{	int li;
	Logfile *logF;

	for( li = 0; li < numLogFiles; li++ ){
		logF = &LogFiles[li];
		if( LOGFILE_OPENED(logF->l_fp) ){
			LOG_flush(logF,time(0));
			Lclose(logF);
		}
		LOG_open(logF);
	}
}

reopenLogFile()
{	char path[PATHLENG],vardir[PATHLENG],logdir[PATHLENG],tmpdir[PATHLENG];

	if( LOG_stdlogfile != NULL )
	if( LogF0.l_path != NULL )
	{
		strcpy(path,(*LOG_stdlogfile)());
		(*LOG_substfile)(path,LogF0.l_proto,vardir,logdir,tmpdir);
		if( strcmp(path,LogF0.l_path) != 0 ){
			if( lARGDUMP() )
			    fprintf(stderr,"LOG file switched: %s -> %s\n",
				LogF0.l_path,path);
			Strdup(&LogF0.l_path,path);
			Lclose(&LogF0);
		}
	}
}
FILE *openLogFile(now)
{
	if( LogF0.l_filters == NULL ){
		/* temporary registration to be found in LOG_which... */
		setLogParams(&LogF0,"delegate",LF_LOGFILE,"","","",0);
	}

	closeAgedLog1(&LogF0,now);

	/* lTTY() might be turn on after NOLOG was set */
	if( LogF0.l_fp == NOLOG && lTTY() )
		LogF0.l_fp = NULL;

	if( LogF0.l_fp == NULL ){
		setLogf(&LogF0,NOLOG); /* suppress log during opening */
		if( LOG_stdlogfile != NULL )
			open_logfile(&LogF0,(*LOG_stdlogfile)(),"a","delegate");
	}
	if( LogF0.l_fp == NOLOG )
		return NULL;

	if( LogF0.l_fp != stderr ) /* if not a tty exactly X-< */
		fseek(LogF0.l_fp,0,2);

	return LogF0.l_fp;
}
fdopenLogFile(fd)
{	FILE *fp;

	if( LogF0.l_fp == NULL || LogF0.l_fp == NOLOG ){
		if( fd == fileno(stderr) )
			fd = dup(fd);
		if( fp = fdopen(fd,"a") )
			setLogf(&LogF0,fp);
	}
}
curLogFd()
{	int logfd;

	if( LogF0.l_fp == NULL || LogF0.l_fp == NOLOG )
		logfd = -1;
	else	logfd = fileno(LogF0.l_fp);
	return logfd;
}
FILE *curLogFp()
{
	if( LogF0.l_fp == NULL || LogF0.l_fp == NOLOG )
		return NULL;
	else	return LogF0.l_fp;
}

logTimeout()
{	int timeout = 0;
	int now = time(0);
	int to;
	int li;
	Logfile *Logf;

	for( li = 0; li < numLogFiles; li++ ){
		Logf = &LogFiles[li];
		if( 0 < Logf->l_until && LOGFILE_ACTIVE(Logf->l_fp) ){
			to = Logf->l_until - now;
			if( 0<to && (timeout==0 || (0<timeout && to<timeout))  )
				timeout = to;
		}
	}
	return timeout;
}
LOG_checkAged(renew)
{	int now;
	int checknow;
	int li;
	Logfile *logF;

	if( renew )
		now = RIGHTNOW;
	else{
		now = time(0);
		checknow = 0;
		for( li = 0; li < numLogFiles; li++ ){
			logF = &LogFiles[li];
			if( 0 < logF->l_until && logF->l_until <= now ){
				checknow = 1;
				break;
			}
		}
		if( !checknow )
			return;
	}

	for( li = 0; li < numLogFiles; li++ ){
	    logF = &LogFiles[li];
	    if( logF->l_pform && (logF->l_fp!=NOLOG && logF->l_fp!=stderr) ){
		if( logF->l_fp != NULL ){
			if( closeAgedLog1(logF,now) )
				LOG_open(logF); /* to unlink aged log ... */
		}else{
		    open_logfile(logF,logF->l_pform,"r",logF->l_proto);
		    if( LOGFILE_ACTIVE(LogF0.l_fp) ){
			if( closeAgedLog1(logF,now) )
				LOG_open(logF);
		    }
		    Lclose(logF);
		}
	    }
	}
}

extern int REQUEST_SERNO;
extern int SERVREQ_SERNO;
static makeTime(MemF,now,usec)
	void *MemF;
{	char tms[1024];

	StrftimeLocal(tms,sizeof(tms),TIMEFORM_mdHMS,now);
	str_sprintf(MemF,"%s.%02d [%d] %d+%d",
		tms,usec/10000,Getpid(),SERNO(),SERNO_MINOR());
	if( REQUEST_SERNO || SERVREQ_SERNO )
		str_sprintf(MemF,"/%d",REQUEST_SERNO);
	if( SERVREQ_SERNO )
		str_sprintf(MemF,"/%d",SERVREQ_SERNO);
	str_sprintf(MemF,": ");
}
putTime(fp)
	FILE *fp;
{	int now,usec;
	char tms[1024];

	now = Gettimeofday(&usec);
	StrftimeLocal(tms,sizeof(tms),TIMEFORM_mdHMS,now);
	fprintf(fp,"%s.%02d [%d] %d+%d",
		tms,usec/10000,getpid(),SERNO(),SERNO_MINOR());
	if( REQUEST_SERNO )
		fprintf(fp,"/%d",REQUEST_SERNO);
	fputs(": ",fp);
}

static char *publog;
publiclog(sel,fmt,a,b,c,d,e,f,g,h,i,j,k)
	char *sel,*fmt,*a,*b,*c,*d,*e,*f,*g,*h,*i,*j,*k;
{	char vmsg[LINESIZE];
	int leng;

	vmsg[0] = 0;
	sprintf(vmsg,fmt,a,b,c,d,e,f,g,h,i,j,k);
	if( publog == NULL )
		publog = strdup(vmsg);
	else{
		leng = strlen(publog);
		publog = Malloc(publog,leng+strlen(vmsg)+1);
		strcpy(publog+leng,vmsg);
	}
}
clear_publiclog()
{
	if( publog ){
		free(publog);
		publog = NULL;
	}
}
have_publiclog()
{
	return publog != NULL;
}
flush_publiclog(route)
	char *route;
{	char vmsg[LINESIZE];

	if( publog ){
		if( route ){
			sprintf(vmsg,"=<!%s ",route);
			strcat(vmsg,publog);
			fputlog(1,vmsg,NULL);
		}else	fputlog(1,publog,NULL);
		free(publog);
		publog = NULL;
	}
}
put_publiclog(sel,fmt,a,b,c,d,e,f,g,h,i,j,k)
	char *sel,*fmt,*a,*b,*c,*d,*e,*f,*g,*h,*i,*j,*k;
{	char vmsg[LINESIZE];

	sprintf(vmsg,fmt,a,b,c,d,e,f,g,h,i,j,k);
	fputlog(1,vmsg,NULL);
}

daemonlog(sel,fmt,a,b,c,d,e,f,g,h,i,j,k)
	char *sel,*fmt,*a,*b,*c,*d,*e,*f,*g,*h,*i,*j,*k;
{
	if( lSILENT() )
		return;

	if( sel != NULL ){
		if( *sel == 'D' && !lVERB() && !lSYNC() )
			return;
		if( *sel != 'E' && *sel != 'F' && *sel != 'T' && lTERSE() )
			return;
	}
	if( sel != NULL )
		FatalTraceLog(sel,fmt,a,b,c,d,e,f,g,h,i,j,k);
	lputLog(&LogF0,fmt,a,b,c,d,e,f,g,h,i,j,k);
}
lputLog(logF,fmt,a,b,c,d,e,f,g,h,i,j,k)
	Logfile *logF;
	char *fmt,*a,*b,*c,*d,*e,*f,*g,*h,*i,*j,*k;
{	MemFile MemF; char msg[LINESIZE];
	char vmsg[LINESIZE];
	int now,usec;

	now = Gettimeofday(&usec);

	if( logF == &LogF0 )
	if( openLogFile(now) == NULL )
	if( LOG_sock_enable == 0 )
		return;

	str_sopen(MemF,"lputLog",msg,sizeof(msg),0,"w");
	makeTime(MemF,now,usec);
	str_sprintf(MemF,fmt,a,b,c,d,e,f,g,h,i,j,k);
	Str2vstr(msg,str_stell(MemF),vmsg,sizeof(vmsg));
	fputlog(0,vmsg,logF->l_fp);
	send_syslog(vmsg);
}

static char myport[64];
static Logfile *errorTraceLog[2];
#define LF_FATAL	0
#define LF_TRACE	1
#define ETLogfile	errorTraceLog[LF_which]
#define ETLogid		(LF_which == LF_FATAL ? LF_ERRORLOG : LF_TRACELOG)

static FatalTraceLog(which,fmt,a,b,c,d,e,f,g,h,i,j,k)
	char *which;
	char *fmt,*a,*b,*c,*d,*e,*f,*g,*h,*i,*j,*k;
{	int LF_which;
	char *path;
	int now,usec;
	MemFile MemF; char msg[LINESIZE];

	switch( *which ){
		case 'F': LF_which = LF_FATAL; break;
		case 'T': LF_which = LF_TRACE; break;
		default:  return;
	}
	if( ETLogfile == NULL )
		ETLogfile = LOG_which(LP_NOTTY,ETLogid,0);

	if( ETLogfile == NULL )
		return;

	if( myport[0] == 0 )
		printPrimaryPort(myport);

	if( LOGFILE_OPENED(ETLogfile->l_fp) == 0 )
		LOG_open(ETLogfile);

	if( LOGFILE_OPENED(ETLogfile->l_fp) == 0 )
		return;

	now = Gettimeofday(&usec);
	StrftimeLocal(msg,sizeof(msg),TIMEFORM_mdHMS,now,usec);
	str_sopen(MemF,"FatalTraceLog",msg,sizeof(msg),strlen(msg),"w");
	str_sprintf(MemF," [%d]-P%s ",Getpid(),myport);
	str_sprintf(MemF,fmt,a,b,c,d,e,f,g,h,i,j,k);
	fputlog(0,msg,ETLogfile->l_fp);
}
TraceLog(fmt,a,b,c,d,e,f,g,h,i,j,k)
	char *fmt,*a,*b,*c,*d,*e,*f,*g,*h,*i,*j,*k;
{	char xfmt[1024];

	sprintf(xfmt,"#{TR}# %s",fmt);
	daemonlog("T",xfmt,a,b,c,d,e,f,g,h,i,j,k);
}

svlog(fmt,a,b,c,d,e,f,g,h,i,j,k)
	char *fmt,*a,*b,*c,*d,*e,*f,*g,*h,*i,*j,*k;
{
	daemonlog(NULL,fmt,a,b,c,d,e,f,g,h,i,j,k);
}
sv0log(fmt,a,b,c,d,e,f,g,h)
	char *fmt,*a,*b,*c,*d,*e,*f,*g,*h;
{
	closeAgedLog1(&LogF0,time(0));
	svlog(fmt,a,b,c,d,e,f,g,h);
}
svvlog(fmt,a,b,c,d,e,f,g,h)
	char *fmt,*a,*b,*c,*d,*e,*f,*g,*h;
{
	if( !lTERSE() )
	if( lSYNC() || lVERB() )
		svlog(fmt,a,b,c,d,e,f,g,h);
}
sv1log(fmt,a,b,c,d,e,f,g,h,i,j,k)
	char *fmt,*a,*b,*c,*d,*e,*f,*g,*h,*i,*j,*k;
{
	if( !lTERSE() )
		svlog(fmt,a,b,c,d,e,f,g,h,i,j,k);
}
sv1vlog(fmt,a,b,c,d,e,f,g,h,i,j,k)
	char *fmt,*a,*b,*c,*d,*e,*f,*g,*h,*i,*j,*k;
{
	if( !lTERSE() )
	if( lSYNC() || lVERB() )
		svlog(fmt,a,b,c,d,e,f,g,h,i,j,k);
}
sv1tlog(fmt,a,b,c,d,e,f,g,h)
	char *fmt,*a,*b,*c,*d,*e,*f,*g,*h;
{
	svlog(fmt,a,b,c,d,e,f,g,h);
}
dbg(fmt,a,b,c,d,e,f,g,h,i,j,k)
	char *fmt,*a,*b,*c,*d,*e,*f,*g,*h,*i,*j,*k;
{
	svlog(fmt,a,b,c,d,e,f,g,h,i,j,k);
}

ERRMSG(fmt,a,b,c,d,e,f,g,h)
	char *fmt,*a,*b,*c,*d,*e,*f,*g,*h;
{	MemFile MemF; char msg[LINESIZE];

	str_sopen(MemF,"ERRMSG",msg,sizeof(msg),0,"w");
	str_sprintf(MemF,"-delegated[%d]- ",getpid());
	str_sprintf(MemF,fmt,a,b,c,d,e,f,g,h);
	fputlog(0,msg,stderr);
}
DBGMSG(fmt,a,b,c,d,e,f,g,h)
	char *fmt,*a,*b,*c,*d,*e,*f,*g,*h;
{
	if( lSYNC() || lVERB() )
		ERRMSG(fmt,a,b,c,d,e,f,g,h);
}

static FILE *StatFp = NULL;
FILE *openStatusFile(pathform)
	char *pathform;
{	FILE *fp;

	if( StatFp == NULL ){
		if( pathform == NULL )
			StatFp = NOLOG;
		else	StatFp = open_tmpfile(NULL,pathform,"w","");
	}
	if( StatFp != NOLOG && StatFp != NULL )
		return StatFp;
	else	return NULL;
}

static Logfile CCF;
FILE *fopenCC(server,mode,path)
	char *server,*mode,*path;
{	char form[1024];
	FILE *fp;

	sprintf(form,"servers/cc/%s",server);
	fp = open_tmpfile(&CCF,form,mode,"servers");
	if( fp != NULL && CCF.l_path != NULL )
		strcpy(path,CCF.l_path);
	else	path[0] = 0;
	return fp;
}
FILE *fcloseCC()
{
	Lclose(&CCF);
	return NULL;
}



static char *actdir;
static char *ACTDIR(){
	char path[1024];
	if( actdir == 0 ){
		strcpy(path,"${ACTDIR}");
		(*LOG_substfile)(path,"",NULL,NULL,NULL);
		actdir = strdup(path);
	}
	return actdir;
}
static char *admdir;
static char *ADMDIR(){
	char path[1024];
	if( admdir == 0 ){
		strcpy(path,"${ADMDIR}");
		(*LOG_substfile)(path,"",NULL,NULL,NULL);
		admdir = strdup(path);
	}
	return admdir;
}

static Logfile tmpF;
FILE *LOG_openLogFile(form,mode)
	char *form,*mode;
{	FILE *fp;

	bzero(&tmpF,sizeof(Logfile));
	fp = open_logtmpfile(&tmpF,0,form,mode,"*");
	if( fp == NOLOG )
		return NULL;
	else	return fp;
}

#define NUMLOCKS 32
static Logfile PortF;
typedef struct {
	int	l_pid; /* the owner of this lock file */
	int	l_fd;
	int	l_group;
	char   *l_port;
	char   *l_path;
} Lock;
static Lock locks[NUMLOCKS];

rmPortLocks()
{	int lid,pid;
	char *path;

	pid = getpid();
	for( lid = 0; lid < NUMLOCKS; lid++ )
	if( pid == locks[lid].l_pid && 0 <= locks[lid].l_fd ){
		path = locks[lid].l_path;
		Verbose("LOCK: unlink %s\n",path);
		unlink(path);
		free(path);
		locks[lid].l_pid = 0;
	}
}
PortLocks(port,group,path)
	char *port;
	char *path;
{	char pathbuf[1024],ports[1024];
	int lid,pid,fd;
	FILE *fp;

	pid = getpid();
	for( lid = 0; lid < NUMLOCKS; lid++ ){
		if( locks[lid].l_port == NULL )
			break;
		if( locks[lid].l_group == group )
		if( strcmp(locks[lid].l_port,port) == 0 ){
			if( locks[lid].l_pid == pid && 0 <= locks[lid].l_fd ){
				if( path )
					strcpy(path,locks[lid].l_path);
				return locks[lid].l_fd;
			}
			free(locks[lid].l_port);
			free(locks[lid].l_path);
			close(locks[lid].l_fd);
			locks[lid].l_pid = 0;
			locks[lid].l_group = 0;
			break;
		}
	}

	if( path == NULL )
		path = pathbuf;
	sprintf(path,"%s/locks/PORT/%s.%d",ACTDIR(),port,group);

	fp = dirfopen("PortLock",path,"w+");
	if( fp != NULL ){
		printServPort(ports,"",1);
		fprintf(fp,"%s %d\n",ports,pid);
		fd = dup(fileno(fp));
		fclose(fp);
		setCloseOnExec(fd);
		locks[lid].l_pid = pid;
		locks[lid].l_fd = fd;
		locks[lid].l_group = group;
		locks[lid].l_port = strdup(port);
		locks[lid].l_path = strdup(path);
		return fd;
	}
	return -1;
}

get_LOCKFILE(path)
	char *path;
{
	sprintf(path,"%s/locks/LOCKFILE",ACTDIR());
}

get_delaysock(file,path)
	char *file,*path;
{
	sprintf(path,"%s/delay/%s",ACTDIR(),file);
}

LOG_createPortFile(file,stayopen)
	char *file;
{	FILE *fp;
	int ai;
	char *arg;

	fp = open_tmpfile(&PortF,file,"w+","");
	if( fp != NOLOG && fp != NULL ){
		fprintf(fp,"%d\n",getpid());
		fflush(fp);

		if( stayopen ){
		sv1log("Stay open PIDFILE for accept() lock[fd=%d]\n",
			fileno(fp));
		}else
		if( fp != stderr ){
			Lclose(&PortF);
		}
	}
	if( fp == NULL ){
		fprintf(stderr,"DeleGate: cannot create %s\n",file);
		return -1;
	}
	return 0;
}
LOG_deletePortFile()
{	FILE *fp;
	char pid[256];

	if( PortF.l_path ){
		if( fp = dirfopen("PortFile",PortF.l_path,"r") ){
			fgets(pid,sizeof(pid),fp);
			fclose(fp);
			Lclose(&PortF);
			if( atoi(pid) == getpid() )
				unlink(PortF.l_path);
		}
	}
}
PortLockReopen()
{
	if( PortF.l_fp != NULL && PortF.l_fp != NOLOG ){
		fclose(PortF.l_fp);
		PortF.l_fp = NULL;
		setLogf(&PortF,dirfopen("PortFile",PortF.l_path,"r+"));
		if( PortF.l_fp != NULL )
			return fileno(PortF.l_fp);
	}
	return -1;
}
PortLockFd()
{	FILE *fp;

	fp = PortF.l_fp;
	if( fp != NULL && fp != NOLOG )
		return fileno(PortF.l_fp);
	else	return -1;
}

extern char *fgetsTimeout();
extern FILE *TMPFILE();
static get_serverlog(dst,end,timeout)
	FILE *dst;
	char *end;
{	FILE *log;
	char buff[0x20000],*ep,*bgn;
	int rcc;
	char line[LINESIZE];
	int leng = 0;

	rcc = RecvPeek(LOG_sockio[0],buff,sizeof(buff)-1);
	buff[rcc] = 0;
	bgn = buff;
	if( end ){
		while( ep = strstr(bgn,end) ){
			ep += strlen(end);
			if( strstr(ep,end) )
				bgn = ep;
			else	break;
		}
	}
	log = TMPFILE("ServerLog");
	fputs(bgn,log);
	fflush(log);
	fseek(log,0,0);
	while( fgetsTimeout(line,sizeof(line),log,timeout) != NULL ){ 
		leng += strlen(line);
		fputs(line,dst);
		if( end != NULL && strstr(line,end) != NULL )
			break;
	}
	fflush(dst);
	fclose(log);
	return leng;
}
get_serverinitlog(dst)
	FILE *dst;
{	int leng;

	fprintf(dst,"--BEGIN--\r\n");
	leng = get_serverlog(dst,"--INITIALIZATION DONE--",4);
	fprintf(dst,"--END--\r\n");
	return leng;
}
restart_server(dst)
	FILE *dst;
{	int leng;

	leng = get_serverlog(dst,NULL,1);
	if( 0 < leng )
		fputs("--\n",dst);
	Kill(getppid(),1);
	sleep(2); /* wait until restart done ... */
	return ftell(dst) + get_serverinitlog(dst);
}


ftp_xferlog(start,chost,size,path,bin,in,anon,user,auser,cstat)
	char *chost,*path,*user,*auser,*cstat;
{	Logfile *LogF;
	char log[LINESIZE];

	xferlog(log,start,chost,size,path,bin,in,anon,user,auser);
	sv1log("XFERLOG: %s %s\n",log,cstat);
	if( LogF = LOG_which("ftp",LF_PROTOLOG,0) ){
		LOG_printf(LogF,"%s %s\n",log,cstat);
		LOG_flush(LogF,time(0));
	}
}

extern char *DELEGATE_MTAB;
MtabFileName(user,path)
	char *user,*path;
{
	sprintf(path,"%s/%s",DELEGATE_MTAB,user);
	if( LOG_substfile )
		(*LOG_substfile)(path,"",NULL,NULL,NULL);
}

extern char *DELEGATE_NEWSLIB;
NewsLibdir(path,spec)
	char *path,*spec;
{
	if( spec[0] == '/' )
		strcpy(path,spec);
	else	sprintf(path,"%s/%s",DELEGATE_NEWSLIB,spec);
	(*LOG_substfile)(path,"",NULL,NULL,NULL);
}

extern char *DELEGATE_SOCKETS;
UnixSocketDir(path)
	char *path;
{	char dir[PATHLENG];

	strcpy(dir,DELEGATE_SOCKETS);
	(*LOG_substfile)(dir,"",NULL,NULL,NULL);

	if( File_mtime(dir) < 0 )
	if( mkdirRX(dir) != 0 ){
		sprintf(dir,"/tmp/delegate.sockets.%d",geteuid());
		if( mkdirRX(dir) != 0 )
			strcpy(dir,"/tmp");
	}
	strcpy(path,dir);
	strcat(path,"/");
}

#define CLRCNT	0
#define INCCNT	1
#define ADDPID	2
#define DELPID	3
#define DECCNT	4

static int getclpids(fp,pids,npids,pid)
	FILE *fp;
	int pids[];
{	int count,rcc,xpid;
	char procs[0x10000],*lp,*np;

	count = 0;
	rcc = fread(procs,1,sizeof(procs)-1,fp);
	procs[rcc] = 0;

	for( lp = procs; lp && *lp; lp = np ){
		if( *lp == '#' )
			break;
		if( np = strchr(lp,'\n') )
			*np++ = 0;
		xpid = atoi(lp);

		if( xpid == 0 )
			sv1log("#### countUp: found PID==0\n");
/*
		else
		if( !aliveProc(xpid) )
			sv1log("#### countUp: dead %d\n",xpid);
*/
		else
		if( xpid != pid ){
			pids[count++] = xpid;
			if( npids <= count-1 )
				break;
		}
	}
	return count;
}

static char *unlinkv[256];
static int unlinks;
static addUnlinks(path)
	char *path;
{	int ui;

	for( ui = 0; ui < unlinks; ui++ )
		if( strcmp(path,unlinkv[ui]) == 0 )
			return;

	unlinkv[unlinks++] = strdup(path);
}
static canUnlinks(path)
	char *path;
{	int ui,uj;

	for( ui = 0; ui < unlinks; ui++ ){
		if( strcmp(path,unlinkv[ui]) == 0 ){
			free(unlinkv[ui]);
			for( uj = ui; uj < unlinks; uj++ )
				unlinkv[uj] = unlinkv[uj+1];
			unlinks--;
			break;
		}
	}
}
static exeUnlinks()
{	int ui;
	char line[32];
	FILE *fp;
	int pid;

	for( ui = 0; ui < unlinks; ui++ ){
		if( fp = fopen(unlinkv[ui],"r") ){
			fgets(line,sizeof(line),fp);
			fclose(fp);
			if( sscanf(line,"#%d",&pid) ){
				if( pid == getpid() )
					unlink(unlinkv[ui]);
			}
		}
		free(unlinkv[ui]);
	}
	unlinks = 0;
}

#define MAX_CLPROCS 1024
countUp(file,istmp,op,pid,lmtime,path)
	char *file;
	long *lmtime;
	char *path;
{	char pathb[PATHLENG],line[128];
	FILE *fp;
	int xpid,pids[MAX_CLPROCS],count,px;
	int now;

	if( path == NULL )
		path = pathb;

	path[0] = 0;
	if( istmp )
		strcats3(path,ACTDIR(),"/",file,NULL);
	else	strcats3(path,ADMDIR(),"/",file,NULL);

	fp = dirfopen("ClientCounter",path,"r+");
	if( lmtime != NULL ){
		if( fp == NULL )
			*lmtime = 0;
		else	*lmtime = file_mtime(fileno(fp));
	}
	if( fp == NULL ){
		if( op == DELPID ){
			sv1log("#### countUp can't open r+ [%s]\n",path);
			return -1;
		}
		fp = dirfopen("ClientCounter",path,"w+");
		if( fp == NULL ){
			sv1log("#### countUp can't open w+ [%s]\n",path);
			return -1;
		}
	}
/*
 * unlink too old counter file on count-up ...
 */

	if( lock_exclusiveTO(fileno(fp),10*1000,NULL) != 0 ){
		sv1log("#### countUp can't lock [%s]\n",path);
		count = -1;
	}else{
		if( op == INCCNT || op == DECCNT || op == CLRCNT ){
			now = time(0);
			count = 0;
			if( fgets(line,sizeof(line),fp) != NULL )
				count = atoi(line);
			if( op == INCCNT ) count++; else
			if( op == DECCNT ) count--; else
			if( op == CLRCNT ) count=0;
			fseek(fp,0,0);
			fprintf(fp,"%d %d\n",count,now);
			fflush(fp);
			Ftruncate(fp,0,1);
		}else{
			count = getclpids(fp,pids,MAX_CLPROCS,pid);
			if( op == ADDPID ){
				canUnlinks(path);
				pids[count++] = pid;
			}

			clearerr(fp);
			fseek(fp,0,0);
			if( 0 < count ){
				for( px = 0; px < count; px++ )
					fprintf(fp,"%8d\r\n",pids[px]);
				fputs("########\r\n",fp);
			}else{
				fprintf(fp,"#%7d\r\n",pid);
			}
			fflush(fp);
		}
		lock_unlock(fileno(fp));
	}
	fclose(fp);

	/* unlink after close for Windows */
	if( count == 0 )
		addUnlinks(path);

	return count;
}

static int  ClientCountFilePID;
static char ClientCountFile[256];

extern unsigned int FQDN_hash();

ClientCountUp(user,host,addr,port)
	char *user,*host,*addr;
{	char file[128];
	int ccount;
	int pid;

	pid = Getpid();
	sprintf(file,"clients/%02d/%s:%s",FQDN_hash(host)%32,addr,host);
	ccount = countUp(file,1,ADDPID,pid,NULL,NULL);
	if( 0 < ccount ){
		strcpy(ClientCountFile,file);
		ClientCountFilePID = pid;
	}
	return ccount;
}
ClientCountDown()
{	char *file,*dp;
	int count;
	int pid;

	pid = Getpid();
	if( pid != ClientCountFilePID )
		return -1;
	ClientCountFilePID = 0;

	file = ClientCountFile;
	if( file[0] == 0 )
		return -1;

	count = countUp(file,1,DELPID,pid,NULL,NULL);
/*
	{
		for( dp = file; *dp; dp++ )
			if( *dp == '.' )
				*dp = '/';
		countUp(file,0,INCCNT,pid,NULL,NULL);
	}
*/
	file[0] = 0;
	return count;
}

BeforeExec(){
	ClientCountDown();
}
static finish(code)
{
	BeforeExit();
	ClientCountDown();
	exeUnlinks();
	DO_FINALIZE(code);
}
Finish(code)
{
	finish(code);
	exit(code);
	fprintf(stderr,"\n[%d] exit(%d) INTERRUPTED\n",getpid(),code);
	_exit(code);
}
_Finish(code)
{
	finish(code);
	_exit(code);
}

typedef void (*VFUNC)();
typedef struct {
	char	*x_what;
	VFUNC	 x_func;
	void	*x_arg;
	int	 x_pid;
	int	 x_done;
} XFunc;
static XFunc before_exits[16];
static int before_exitX;
addBeforeExit(what,func,arg)
	char *what;
	VFUNC func;
	void *arg;
{	int fi;
	XFunc *xp;

	sv1log("[%d] ADD BeforeExit[%d] %s\n",getpid(),before_exitX,what);
	xp = &before_exits[before_exitX++];
	xp->x_what = what;
	xp->x_func = func;
	xp->x_arg = arg;
	xp->x_pid = getpid();
}
BeforeExit(){
	int fi;
	XFunc *xp;
	int pid;

	pid = getpid();
	for( fi = 0; fi < before_exitX; fi++ ){
		xp = &before_exits[fi];
		if( xp->x_pid == pid && xp->x_done == 0 ){
			sv1log("[%d] DO BeforeExit[%d] %s\n",pid,fi,xp->x_what);
			xp->x_done = 1;
			(*xp->x_func)(xp->x_what,xp->x_arg);
		}
	}
}


putRejectList(what,dproto,dhost,dport,dpath,referer,
		sproto,shost,sport,suser,auser,apass,reason)
	char *what,*dproto,*dhost,*dpath,*referer,
		*sproto,*shost,*suser,*auser,*apass,*reason;
{	FILE *fp;
	char path[2048];
	char *rp,rc;
	char hostport[256];
	char stime[256];

	HostPort(hostport,dproto,dhost,dport);
	sprintf(path,"%s/rejects/%s/%s",ADMDIR(),dproto,hostport);

	fp = dirfopen("RejectList",path,"r+");
	if( fp == NULL )
		fp = dirfopen("RejectList",path,"w+");
	if( fp == NULL ){
		/*fp = stderr;*/
		/* no reject list is available */
		return;
	}

	StrftimeLocal(stime,sizeof(stime),TIMEFORM_HTTPD,time(NULL));

	lock_exclusiveTO(fileno(fp),10*1000,NULL);
	fseek(fp,0,2);
	fprintf(fp,"[%s] %s %s://%s:%s@%s:%d/%s %s %s://%s:%s@%s:%d/%s ",
		stime,what,
		dproto,auser,apass,dhost,dport,dpath, referer,
		sproto,suser,"",   shost,sport,""
	);
	putc('"',fp);
	for( rp = reason; rc = *rp; rp++ ){
		if( ' ' <= rc && rc < 0x7F && rc != '"' && rc != '%' )
			putc(rc,fp);
		else if( rc == '\n' ) fputs("\\n",fp);
		else if( rc == '\r' ) fputs("\\r",fp);
		else if( rc == '\t' ) fputs("\\t",fp);
		else if( rc == '\f' ) fputs("\\f",fp);
		else if( rc == '\\' ) fputs("\\\\",fp);
		else	fprintf(fp,"%%%02x",rc);
	}
	putc('"',fp);
	fputs("\r\n",fp);
	lock_unlock(fileno(fp));
	fclose(fp);
}

extern char *MAILGATE;

FILE *openMbox();
FILE *openMuid(create,muid,mbox)
	char *muid,*mbox;
{	char path[1024];
	FILE *fp;

	sprintf(path,"%s/%s/muid/%s",ADMDIR(),MAILGATE,muid);
	if( create ){
		if( fp = dirfopen("MBOXMUID",path,"w+") )
			fputs(mbox,fp);
		return fp;
	}else{
		if( fp = dirfopen("MBOXMUID",path,"r+") ){
			fgets(mbox,256,fp);
			fclose(fp);
			return openMbox(0,mbox,muid);
		}
		return NULL;
	}
}
FILE *openMbox(create,mbox,muid)
	char *mbox,*muid;
{	char path[1024];
	FILE *fp,*ufp;

	sprintf(path,"%s/%s/mbox/%s",ADMDIR(),MAILGATE,mbox);
	fp = dirfopen("MBOXMBOX",path,"r+");
	if( fp == NULL && create ){
		fp = dirfopen("MBOXUSER",path,"w+");
		if( fp != NULL )
			if( ufp = openMuid(1,muid,mbox) )
				fclose(ufp);
	}
	return fp;
}
FILE *openGACL(create,muid)
	char *muid;
{	char path[1024];
	FILE *fp;

	sprintf(path,"%s/admin/%s",ADMDIR(),muid);
	fp = dirfopen("MUIDGACL",path,"r+");
	if( fp == NULL && create )
		fp = dirfopen("MUIDGACL",path,"w+");
	return fp;
}

FILE *openAclRecord(create,url,path)
	char *url,*path;
{	char proto[128],login[128],upath[1024];
	FILE *fp;

	decomp_absurl(url,proto,login,upath,sizeof(upath));
	sprintf(path,"%s/record/%s/%s/%s",ADMDIR(),proto,login,upath);

	fp = dirfopen("ACL",path,"r+");
	if( fp == NULL && create )
		fp = dirfopen("ACL",path,"w+");
	return fp;
}
FILE *openAclFile(create,proto,host,port,upath)
	char *proto,*host,*port,*upath;
{	char hostport[1024],aclpath[2048];
	FILE *fp;

	HostPort(hostport,proto,host,port);
	sprintf(aclpath,"%s/acl/%s/%s/%s",ADMDIR(),proto,hostport,upath);

	fp = dirfopen("ACL",aclpath,"r+");
	if( fp == NULL && create )
		fp = dirfopen("ACL",aclpath,"w+");
	return fp;
}

local_lockTO(ex,path,fp,timeout,elapsedp,lkfdp)
	char *path;
	FILE *fp;
	int *elapsedp;
	int *lkfdp;
{	char dir[1024],lkpath[1024];
	int lkfd,lkrcode;

	if( path[0] ){
		strcpy(dir,ACTDIR());
		lkfd = getLocalLock(fp,dir,path,lkpath);
		if( lkfd < 0 ){
			lkfd = fileno(fp);
			strcpy(lkpath,path);
		}
	}else{
		lkfd = fileno(fp);
		strcpy(lkpath,"[tmpfile]");
	}

	if( ex )
		lkrcode = lock_exclusiveTO(lkfd,timeout,elapsedp);
	else	lkrcode = lock_sharedTO(lkfd,timeout,elapsedp);

	if( lkrcode != 0 && lkfd != fileno(fp) ){
		close(lkfd);
		lkfd = -1;
	}

	*lkfdp = lkfd;
	return lkrcode;
}

/*
#include <sys/syscall.h>
open(path,flags,mode)
	char *path;
{	int fd;
	char msg[1024];

	fd = syscall(SYS_open,path,flags,mode);
	sprintf(msg,"-- %d open(%s,%x)\n",fd,path,flags);
	write(2,msg,strlen(msg));
	return fd;
}
unlink(path)
	char *path;
{	int code;
	char msg[1024];

	code = syscall(SYS_unlink,path);
	sprintf(msg,"-- %d unlink(%s)\n",code,path);
	write(2,msg,strlen(msg));
	return code;
}
*/
