/*////////////////////////////////////////////////////////////////////////
Copyright (c) 1997-1999 Yutaka Sato
Copyright (c) 1997-1999 Electrotechnical Laboratry (ETL), AIST, MITI

Permission to use, copy, and distribute this material for any purpose
and without fee is hereby granted, provided that the above copyright
notice and this permission notice appear in all copies.
ETL MAKES NO REPRESENTATIONS ABOUT THE ACCURACY OR SUITABILITY OF THIS
MATERIAL FOR ANY PURPOSE.  IT IS PROVIDED "AS IS", WITHOUT ANY EXPRESS
OR IMPLIED WARRANTIES.
//////////////////////////////////////////////////////////////////////////
Content-Type:	program/C; charset=US-ASCII
Program:	windows.c
Author:		Yutaka Sato <ysato@etl.go.jp>
Description:
History:
	970202	created
//////////////////////////////////////////////////////////////////////#*/

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
extern char *getenv();

#include "config.h"
#include "log.h"
#define LE (LOGLEVEL<0) ? 0:porting_dbg
#define LT (LOGLEVEL<1) ? 0:porting_dbg
#define LV (LOGLEVEL<2) ? 0:porting_dbg
#define LW (LOGLEVEL<3) ? 0:porting_dbg
#define DBGWRITE	write
#define SYST		"UNIX"

#ifdef _MSC_VER
#include "vsignal.h"
#undef DBGWRITE
#define DBGWRITE	__write
#undef SYST
#define SYST		"WIN"

#ifdef __EMX__
#undef SYST
#define SYST		"OS/2"
#include <os2emx.h>
#endif

#include <winsock.h>
#include <sys/timeb.h> /* for ftime() */
#include <share.h> /* _SH_DENYNO */

typedef union {
 struct sockaddr_in	_sin;
	char		_sab[64];
} VSAddr;
typedef struct sockaddr *SAP;

#define MAXHANDLE	4096

static __write(fd,buf,size)
	char *buf;
{	int oh;
	int ok,wcc;

	oh = _get_osfhandle(fd);
	ok = WriteFile((HANDLE)oh,buf,size,&wcc,NULL);
	return wcc;
}
static __read(fd,buf,size)
	char *buf;
{	int oh;
	int ok,rcc;
	
	oh = _get_osfhandle(fd);
	ok = ReadFile((HANDLE)oh,buf,size,&rcc,NULL);
	return rcc;
}

/*######## CLOCK ########*/
gettimeofday(tp, tzp)
	struct timeval *tp;
	struct timezone *tzp;
{	struct timeb timeb;

	ftime(&timeb);
	tp->tv_sec = timeb.time;
	tp->tv_usec = timeb.millitm * 1000;
	LW("gettimeofday(%x,%x) = [%d,%d]",tp,tzp,tp->tv_sec,tp->tv_usec);
	return 0;
}

/*######## SIGNAL ########*/
sigsetmask(){
	LV("sigsetmask() not supported");
	return -1;
}
sigblock(){
	LV("sigblock() not supported");
	return -1;
}
kill(pid,sig){
	HANDLE ph;
	int xcode;

	ph = pid;
	/* it should be ph = OpenProcess(pid), but because the pid
	 * argument given is a process handle which is got at spawnvp()...
	 */

	if( !GetExitCodeProcess(ph,&xcode) )
		return -1;
	if( xcode != STILL_ACTIVE )
		return -1;
	if( sig == SIGTERM ){
		if( TerminateProcess(ph,0) )
			return 0;
		else	return -1;
	}
	LE("kill(%d,%d) not supported",pid,sig);
	return -1;
}

alarm(sec){
	LE("alarm(%d) not supported",sec);
	return -1;
}
sleep(sec){
	LW("sleep(%d)",sec);
	_sleep(sec*1000);
}

/*######## USER and GROUP ########*/
#include "passwd.h"
static struct passwd passwd;
setpwent(){
	LV("setpwent()");
	return 0;
}
endpwent(){
	LV("endpwent()");
	return 0;
}
struct passwd *getpwuid(uid){
	LV("getpwuid(%d)",uid);
	passwd.pw_name = "windows";
	passwd.pw_passwd = "";
	passwd.pw_uid = 0;
	passwd.pw_gid = 0;
	passwd.pw_quota = 0;
	passwd.pw_comment = "";
	passwd.pw_gecos = "Windows";
	passwd.pw_dir = "/users/default";
	passwd.pw_shell = "";
	return &passwd;
}
getgrnam(name)
	char *name;
{
	LV("getgrnam(%s)",name);
	return 0;
}
getpwnam(name)
	char *name;
{
	LV("getpwnam(%s)",name);
	return 0;
}
getgrgid(gid){
	LV("getgrgid(%d)",gid);
	return 0;
}

/* ######## PROCESS OWNER ######## */
getegid(){
	LW("getegid() = 0");
	return 0;
}
geteuid(){
	LW("geteuid() = 0");
	return 0;
}
getuid(){
	LW("getuid() = 0");
	return 0;
}
getgid(){
	LW("getgid() = 0");
	return 0;
}
setuid(uid){
	int rcode;
	if( uid == 0 )
		rcode = 0;
	else	rcode = -1;
	LW("setuid(%d) = %d",uid,rcode);
	return rcode;
}
setgid(gid){
	int rcode;
	if( gid == 0 )
		rcode = 0;
	else	rcode = -1;
	LW("setgid(%d) = %d",gid,rcode);
	return rcode;
}

/*######## FILE ########*/

#include <WINDOWS.H>
#include <WINBASE.H>

static HANDLE _SELF;
static HANDLE SELF()
{
	if( _SELF == 0 ){
		_SELF = OpenProcess(PROCESS_ALL_ACCESS,0,getpid());
		LV("PID=%d SELF_HANDLE=%d",getpid(),_SELF);
	}
	return _SELF;
}

duphandle(sproc,shandle,dproc,closesrc,noinherit)
	HANDLE sproc;
	HANDLE shandle;
	HANDLE dproc;
{	HANDLE dhandle;
	DWORD access;
	BOOL inherit;
	DWORD options;

	access = 0;
	inherit = !noinherit; /* FALSE may be good for some sockets... */
	options = DUPLICATE_SAME_ACCESS;
	if( closesrc )
		options |= DUPLICATE_CLOSE_SOURCE;

	if( sproc == NULL ) sproc = SELF();
	if( dproc == NULL ) dproc = SELF();

	dhandle = NULL;
	DuplicateHandle(sproc,shandle,dproc,&dhandle,
		access,inherit,options);

	if( dhandle == NULL )
	LT("DUPHANDLE: %d,%d => %d,%d",sproc,shandle,dproc,dhandle);
	return (int)dhandle;
}

/*######## SOCKET ########*/
static WORD wVersionRequested;
static WSADATA wsaData;
void socket_init(){
	int err;

	if( wsaData.szDescription[0] != 0 )
		return;
	wVersionRequested = MAKEWORD(1,1);
	err = WSAStartup(wVersionRequested,&wsaData);
}
static sockInfo()
{
	if( getppid() == 0 ){
		LT("WINSOCK INFO.");
		LT("Desc: %s",wsaData.szDescription);
		LT("Stat: %s",wsaData.szSystemStatus);
		LT("Maxs: %d, Vend: %s",wsaData.iMaxSockets,wsaData.lpVendorInfo);
	}
}
static ISSOCK(sock)
{	int len,type;

	len = sizeof(type);
	if( getsockopt(sock,SOL_SOCKET,SO_TYPE,(char*)&type,&len) == 0 )
		return 1 <= type;
	return 0;
}

static int NO_SOCKOPEN = 1;
/* bad if this value is zero in CHARCODE filter from FTP/HTTP on WindowsNT
 * where the client SOCKET is inherited twice...
 */

static short open_osfhandle[MAXHANDLE]; /* get_osfhandle(fd) emulation */
static short isdup_handles[MAXHANDLE];
static short opened_sockcnt[MAXHANDLE];

static int coes[64];
static int coen;
setCloseOnExecSocket(fd)
{	int sock;
	int ci;

	if( 0 <= fd )
	if( sock = SocketOf(fd) ){
		for( ci = 0; ci < coen; ci++ )
			if( coes[ci] == sock )
				return 0;

		LV("set CloseOnExecSocket[%d] = %d/%d",coen,sock,fd);
		coes[coen++] = sock;
		return 0;
	}
	return -1;
}
clearCloseOnExecSocket(fd)
{	int sock;
	int ci,cj;

	if( 0 <= fd )
	if( sock = SocketOf(fd) ){
		for( ci = 0; ci < coen; ci++ ){
			if( coes[ci] == sock ){
				for( cj = ci; cj < coen; cj++ )
					coes[cj] = coes[cj+1];
				LV("clr CloseOnExec[%d] = %d/%d",ci,sock,fd);
				coen--;
				return 0;
			}
		}
	}
	return -1;
}
setInheritance(ifd,inherit)
{	HANDLE ih,nh;
	int nfd;

	ih = _get_osfhandle(ifd);
	nh = duphandle(0,ih,0,0,!inherit);
	close(ifd);
	nfd = _open_osfhandle(nh,0);
	if( nfd != ifd ){
		dup2(nfd,ifd);
		close(nfd);
	}
	LE("setInferitance(%d,%d) %d %d %d",ifd,inherit,ih,nh,nfd);
	return ifd;
}

static issock(fd)
{	struct stat st0,st1;
	char *bp0,*bp1;
	int bi,diff;

	if( fstat(fd,&st1) != 0 ){
		LV("issock(%d) CANT OPEN1",fd);
		return 0;
	}
	if( fstat(sessionfd(),&st0) != 0 ){
		LE("######## issock(%d) CANT OPEN2",fd);
		exit(1);
	}
	if( st0.st_mode != st1.st_mode ){
		LV("DIFF %d:%d %d:%d",fd,st1.st_mode,sessionfd(),st0.st_mode);
		return 0;
	}
	return 1;
}

static closeSocket(fd,sock)
{	int rcode1,rcode2,isdup;

	clearCloseOnExecSocket(fd);

	if( fd < 0 ){
		rcode1 = -1;
		isdup = 0;
	}else{
		rcode1 = _close(fd);
		isdup = isdup_handles[fd];
		isdup_handles[fd] = 0;
		open_osfhandle[fd] = 0;
	}

	if( 0 < sock )
	if( 0 < opened_sockcnt[sock] )
		opened_sockcnt[sock] -= 1;

	if( sock < 0 )
		rcode2 = -1;
	else
	if( 0 < opened_sockcnt[sock] )
		rcode2 = 0;
	else{
		rcode2 = closesocket(sock);
	}

	LV("-- SOCKET %d*(%3d/%2d) = %d,%d close() dup=%d",
		opened_sockcnt[sock],sock,fd,rcode1,rcode2,isdup);

	return rcode1;

}
static openSocket(tfd,sock,isdup,wfmt,a1,a2)
	char *wfmt,*a1,*a2;
{	int fd;
	char msg[256];

	sprintf(msg,wfmt,a1,a2);
	if( sock <= 0 ){
		LE("%s -- openSocket(%d) bad SOCKET handle",msg,sock);
		return -1;
	}
	fd = -1;
	if( !NO_SOCKOPEN ){
		/*
		 * how should it be treated when 0 <= tfd ... dup2() case ?
		 */
		fd = _open_osfhandle(sock,0);
		if( fd < 0 ){
			LE("NO OPEN_OSFHANDLE(SOCKET)");
			NO_SOCKOPEN = 1;
		}
	}
	if( NO_SOCKOPEN ){
		if( 0 <= tfd ){
			if( _dup2(sessionfd(),tfd) == 0 )
				fd = tfd;
			else{
				LE("%s -- failed dup2(x,%d)\n",tfd);
				return -1;
			}
		}else	fd = _dup(sessionfd());
		open_osfhandle[fd] = sock;
	}
	if( isdup ){
		isdup_handles[fd] = sock;
		opened_sockcnt[sock] += 1;
	}else{
		isdup_handles[fd] = 0;
		opened_sockcnt[sock]  = 1;
	}

	LV("-- SOCKET %d*(%3d/%2d) opened %s",
		opened_sockcnt[sock],sock,fd,msg);
	return fd;
}
SocketOf(fd)
{	int sock,oh;

	if( fd < 0 || MAXHANDLE <= fd ){
		LW("SocketOf(%d)?",fd);
		return 0;
	}

	oh = _get_osfhandle(fd);

	if( NO_SOCKOPEN ){
		if( 0 < (sock = open_osfhandle[fd]) )
		if( opened_sockcnt[sock] <= 0 || !issock(fd) ){
			open_osfhandle[fd] = 0;
			isdup_handles[fd] = 0;
			if( 0 < opened_sockcnt[sock] ){
				opened_sockcnt[sock] -= 1;
				if( opened_sockcnt[sock] <= 0 )
					closesocket(sock);
			}
			LV("-- SOCKET %d*(%3d/%2d) closed (detected)",
				opened_sockcnt[sock],sock,fd);
			sock = 0;
		}
	}else	sock = oh;

	LW("SocketOf(%d/%d) ISSOCK=%d",sock,fd,ISSOCK(sock));

	if( sock && !ISSOCK(sock) )
		sock = 0;

	return sock;
}
getsockHandle(fd){ return (0 < SocketOf(fd)) ? SocketOf(fd) : -1; }

_SOCKET(dom,type,proto)
{	int sock,fd;

	socket_init();
	sock = socket(dom,type,proto);
	if( sock < 0 ){
		if( dom == AF_UNIX )
			LV("socket(AF_UNIX) = -1, NOT SUPPORTED");
		return sock;
	}
	for( fd = 0; fd < MAXHANDLE; fd++ ){
		if( open_osfhandle[fd] == sock ){
			LE("dangling socket descriptor: %d/%d",sock,fd);
			open_osfhandle[fd] = -1;
		}
	}
	fd = openSocket(-1,sock,0,"socket()");
	LV("socket() = %d/%d",sock,fd);
	return fd;
}
_BIND(fd,addr,len)
	SAP addr;
{	int sock = SocketOf(fd);
	int rcode;

	rcode = bind(sock,addr,len);
	LV("bind(%d/%d) = %d",sock,fd,rcode);
	return rcode;
}
_LISTEN(fd,nblog)
{	int sock = SocketOf(fd);
	int rcode;

	rcode = listen(sock,nblog);
	LV("listen(%d/%d) = %d",sock,fd,rcode);
	return rcode;
}
_ACCEPT(fd,addr,len)
	SAP addr;
	int *len;
{	int sock = SocketOf(fd);
	int csock,nfd;

	csock = accept(sock,addr,len);
	if( 0 <= csock )
		nfd = openSocket(-1,csock,0,"accept(%d)",fd);
	else	nfd = -1;
	LV("accept(%d/%d) = %d/%d",sock,fd,csock,nfd);
	return nfd;
}
_CONNECT(fd,addr,len)
	SAP addr;
{	int sock = SocketOf(fd);
	int rcode;
	int wserrno;

	rcode = connect(sock,addr,len);
	wserrno = WSAGetLastError();
	if( wserrno ){
		LV("connect(%d/%d) = %d errno=(%d <- %d)",sock,fd,rcode,
			errno,wserrno);
		errno = wserrno;
	}else	LV("connect(%d/%d) = %d",sock,fd,rcode);
	return rcode;
}
_SELECT(sz,x,y,z,tv)
	struct fd_set *x,*y,*z;
	struct timeval *tv;
{	int rcode;
	int wserrno;

	LV("select(%d,%d) start",sz,tv?tv->tv_sec:0);
	rcode = select(sz,x,y,z,tv);
	wserrno = WSAGetLastError();
	LV("select() = %d errno=(%d / %d)",rcode,errno,wserrno);
	return rcode;
}
_SEND(fd,buf,len,flags)
	char *buf;
{	int sock = SocketOf(fd);
	int wcc;

	wcc = send(sock,buf,len,flags);
	LW("send(%d/%d) = %d",sock,fd,wcc);
	return wcc;
}
_RECV(fd,buf,len,flags)
	char *buf;
{	int sock = SocketOf(fd);
	int rcc;

	rcc = recv(sock,buf,len,flags);
	LW("recv(%d/%d) = %d",sock,fd,rcc);
	return rcc;
}
_SENDTO(fd,buf,len,flags,to,tolen)
	char *buf;
	SAP to;
{	int sock = SocketOf(fd);
	int wcc;

	wcc = sendto(sock,buf,len,flags,to,tolen);
	LW("sendto(%d/%d) = %d",sock,fd,wcc);
	return wcc;
}
_RECVFROM(fd,buf,len,flags,from,fromlen)
	char *buf;
	SAP from;
	int *fromlen;
{	int sock = SocketOf(fd);
	int rcc;

	rcc = recvfrom(sock,buf,len,flags,from,fromlen);
	LW("recvfrom(%d/%d) = %d",sock,fd,rcc);
	return rcc;
}
_GETSOCKNAME(fd,addr,len)
	SAP addr;
	int *len;
{	int sock = SocketOf(fd);
	int rcode;

	rcode = getsockname(sock,addr,len);
	LW("getsockname(%d/%d) = %d",sock,fd,rcode);
	return rcode;
}
_GETPEERNAME(fd,addr,len)
	SAP addr;
	int *len;
{	int sock = SocketOf(fd);
	int rcode;

	rcode = getpeername(sock,addr,len);
	LW("getpeername(%d/%d) = %d",sock,fd,rcode);
	return rcode;
}
_SETSOCKOPT(fd,lev,op,val,len)
	char *val;
{	int sock = SocketOf(fd);
	int rcode;

	rcode = setsockopt(sock,lev,op,val,len);
	LW("setsockopt(%d/%d) = %d",sock,fd,rcode);
	return rcode;
}
_GETSOCKOPT(fd,lev,op,val,len)
	char *val;
	int *len;
{	int sock = SocketOf(fd);
	int rcode;

	rcode = getsockopt(sock,lev,op,val,len);
	LW("getsockopt(%d/%d) = %d",sock,fd,rcode);
	return rcode;
}
_GETHOSTNAME(name,size)
	char *name;
{
	socket_init();
	return gethostname(name,size);
}

int socketpair(d,type,protocol,fsv)
	int fsv[2];
{	VSAddr ina;
	int inalen,port;
	int sv[2];
	int asock,len;
	int rcode = 0;

	sv[0] = sv[1] = -1;
	asock = socket(d,type,protocol);
	if( asock < 0 )
		return -1;

	inalen = VSA_atosa(&ina,0,"127.0.0.1");

	rcode |= bind(asock,(SAP)&ina,inalen);
	rcode |= listen(asock,2);

	len = sizeof(VSAddr);
	getsockname(asock,(SAP)&ina,&len);
	port = VSA_port(&ina);

	sv[1] = socket(d,type,protocol);
	inalen = VSA_atosa(&ina,port,"127.0.0.1");
	rcode |= connect(sv[1],(SAP)&ina,inalen);

	len = sizeof(VSAddr);
	sv[0] = accept(asock,(SAP)&ina,&len);
	closesocket(asock);

	fsv[0] = openSocket(-1,sv[0],0,"socketpair(0)");
	fsv[1] = openSocket(-1,sv[1],0,"socketpair(1)");

	LV("socketpair() = %d [%d/%d,%d/%d]",rcode,sv[0],fsv[0],sv[1],fsv[1]);
	return rcode;
}

setNonblockingSocket(fd,on)
{	int sock;

	if( sock = SocketOf(fd) )
		return ioctlsocket(sock,FIONBIO,&on);
	return -1;
}
setNonblockingPipe(pv)
	int pv[];
{	HANDLE phandle;
	int mode,ok;

	mode = PIPE_READMODE_BYTE | PIPE_NOWAIT;
	phandle = _get_osfhandle(pv[0]);
	ok = SetNamedPipeHandleState(phandle,&mode,NULL,NULL);
	LE("setNonblockingPipe(%d/%d) ok=%d",phandle,pv[0],ok);
}
static ISPIPE(pfd)
{	int ok,ninst;
	HANDLE phandle;

	phandle = _get_osfhandle(pfd);
	if( GetFileType(phandle) == FILE_TYPE_PIPE )
		return 1;

	ninst = -1;
	ok = GetNamedPipeHandleState(phandle,NULL,&ninst,NULL,NULL,NULL,0);
	if( ok == TRUE && 0 < ninst )
		return 1;
	else	return 0;
}
pollPipe(pfd,slpmsec)
{	HANDLE phandle;
	int nready,rem,slept,sleep1,s1;

	if( SocketOf(pfd) )
		return -1;
	if( !ISPIPE(pfd) )
		return -1;

	phandle = _get_osfhandle(pfd);
	if( isWindows95() ){
		/* PeekNamedPipe of Win95/98 seems never fail on EOF ... */
		if( 15000 < slpmsec ){
			LT("pollPipe(%d,%d->15000) for Win95/98...",slpmsec);
			slpmsec = 15000;
		}
	}
	for( slept = 0; slpmsec == 0 || slept <= slpmsec; slept += s1 ){
		nready = -1;
		if( PeekNamedPipe(phandle,NULL,0,NULL,&nready,NULL) == FALSE )
			return -1;

		if( nready != 0 )
			return nready;
		if( slpmsec != 0 && slpmsec <= slept )
			break;

		rem = slpmsec - slept;
		if( slept <  500 ) sleep1 =   5; else
		if( slept < 5000 ) sleep1 =  50; else
				   sleep1 = 500;

		if( slpmsec == 0 || sleep1 < rem )
			s1 = sleep1;
		else	s1 = rem;
		usleep(s1*1000);
	}
	return 0;
}

/* ######## NIS ######## */
yp_get_default_domain(domp)
	char **domp;
{
	LV("yp_get_default_domain() = -1");
	*domp = NULL;
	return -1;
}
yp_match(dom,map,name,len,vp,vlen)
	char *dom;
{
	LV("yp_match() = -1");
	return -1;
}

/* ######## STDIO ######## */

dup2(fd1,fd2)
{	int rcode;
	int sock,nsock,nfd;

	if( close(dup2) == 0 )
		LE("closed(%d) before dup2(%d,%d)",fd2,fd1,fd2);

	if( sock = SocketOf(fd1) ){
		if( NO_SOCKOPEN )
			nsock = sock;
		else	nsock = duphandle(0,sock,0,0,0);
		nfd = openSocket(fd2,nsock,1,"dup2(%d/%d)",sock,fd2);
	}else{
		rcode = _dup2(fd1,fd2);
		if( rcode == 0 )
			nfd = fd2;
		else	nfd = -1;
	}
	return nfd;
}
dup(fd)
{	int sock,nsock,nfd;

	if( sock = SocketOf(fd) ){
		if( NO_SOCKOPEN )
			nsock = sock;
		else	nsock = duphandle(0,sock,0,0,0);
		nfd = openSocket(-1,nsock,1,"dup(%d/%d)",sock,fd);
	}else	nfd = _dup(fd);
	return nfd;
}
close(fd)
{	int sock,rcode;

	if( sock = SocketOf(fd) )
		rcode = closeSocket(fd,sock);
	else	rcode = _close(fd);
	return rcode;
}

_read(fd,buf,size)
	char *buf;
{	int rcc;
	int sock;
	HANDLE oh;

	if( sock = SocketOf(fd) ){
		rcc = recv(sock,buf,size,0);
		if( rcc < 0 )
		LV("-- SOCKET recv(%d/%d,%x,%d,0) = %d",sock,fd,buf,size,rcc);
	}else{
		rcc = __read(fd,buf,size);
		if( rcc < 0 )
		LW("-- read(%d,%x,%d) = %d",fd,buf,size,rcc);
	}
	return rcc;
}
_write(fd,buf,size)
	char *buf;
{	int wcc;
	int sock;
	int xerr;

	if( sock = SocketOf(fd) ){
		wcc = send(sock,buf,size,0);
		if( xerr = WSAGetLastError() ){
			LE("-- SOCKET send(%d/%d,%x,%d,0) = %d %d",
				sock,fd,buf,size,wcc,xerr);

			if( getenv("SIGPIPE_TERM") )
			if( xerr == WSAENETRESET
			 || xerr == WSAECONNRESET || xerr == WSAECONNABORTED ){
				LE("send() -- cause SIGTERM for SIGPIPE");
				raise(SIGTERM);
			}
		}else
		LV("-- SOCKET send(%d/%d,%x,%d,0) = %d",sock,fd,buf,size,wcc);
	}else{
		wcc = __write(fd,buf,size);
		LW("-- write(%d,%x,%d) = %d",fd,buf,size,wcc);
	}
	return wcc;
}

static int sleepsock = -1;
usleep(timeout)
{ 	struct timeval tvbuf,*tv;
	struct fd_set mask;
	int port;

	if( timeout ){
		tv = &tvbuf;
		tv->tv_sec  = timeout / 1000000;
		tv->tv_usec = timeout % 1000000;
	}else	tv = NULL;

	if( sleepsock < 0 ){
		sleepsock = bindsocket(-1,&port);
		if( sleepsock < 0 )
			return -1;
	}
	FD_ZERO(&mask);
	FD_SET(sleepsock,&mask);
	return select(128,&mask,NULL,NULL,tv);
}
Usleep(usec){ usleep(usec); }

uname(name)
	char *name;
{
	return -1;
}
Uname(name)
	char *name;
{
	if( isWindows95() )
		strcpy(name,"Windows95");
	else	strcpy(name,"WindowsNT");
	return 0;
}
isWindows95(){ return getpid() < 0; }

#include <time.h>
static utime2ftime(tv,ftime)
	struct timeval *tv;
	FILETIME *ftime;
{	int year,mon,day,hour,min,sec;
	struct tm *tm;
	WORD ddate,dtime;

	tm = gmtime(&tv->tv_sec);
	year = tm->tm_year + 1900 - 1980;
	mon  = tm->tm_mon  +  1;
	day  = tm->tm_mday;
	hour = tm->tm_hour;
	min  = tm->tm_min;
	sec  = tm->tm_sec / 2;
	ddate = (year <<  8) | (mon << 5) | day;
	dtime = (hour << 11) | (min << 5) | sec;
	DosDateTimeToFileTime(ddate,dtime,ftime);
}
utimes(path,tvp)
	char *path;
	struct timeval *tvp;
{	HANDLE fh;
	FILETIME atime,mtime;
	BOOL ok;
	struct tm *tm;

	fh = CreateFile(path,GENERIC_READ|GENERIC_WRITE,
		0,NULL,OPEN_EXISTING,FILE_ATTRIBUTE_NORMAL,NULL);
	if( fh == NULL )
		return -1;

	utime2ftime(&tvp[0],&atime);
	utime2ftime(&tvp[1],&mtime);

	ok = SetFileTime(fh,NULL,&atime,&mtime);
	CloseHandle(fh);
	if( ok )
		return 0;
	else	return -1;
}


/* ################### */
struct iovec {
	char *iov_base;
	int   iov_len;
}
writev(fd, iov, iovcnt)
	struct iovec *iov;
{
	LE("writev() NOT SUPPORTED");
	exit(1);
}
sigpause(){
	LE("sigpause() NOT SUPPORTED");
	exit(1);
}

/*
 *	parent-child relationship is not implemented in Windows
 */

typedef struct {
	int	p_ppid;		/* parent's process handle */
	int	p_pmode;	/* spawn mode from the parent {P_NOWAIT} */
	int	p_inhfd[4];
	int	p_inhsock[4];	/* socket handle connected to the client */
	int	p_pinhsock[4];	/* parent's clsock inherited if in WindowsNT */
	int	p_toparent;	/* socket for sync and exist status code */
	int	p_nosockopen;
	struct {
	int	p_sessionfd[2];
	} p_common;
	int	p_closeOnExecN;
	int	p_closeOnExec[8];
} ProcEnv;

static ProcEnv MyProcEnv;
static ProcEnv ProcEnv0;

static int argc;
static char **argv;
extern char **environ;

#define SPAWNENV	"SPAWN_ENVIRON"

extern int _fileinfo;

static int children[MAXHANDLE];
static int nalive;
static int ntotal;

spawnvpe(pmode,path,av,environ)
	char *path;
	char *av[];
	char *environ[];
{	char *nenviron[256],spenv[128];
	int ei;
	int ci,cj,dsock,isock[3];
	int cphandle;
	int asock,port,csock,wcc,rcc;
	ProcEnv cpe;
	char ststat[1];
	char *env;
	int len,nei;

	_fileinfo = -1;

	asock = bindsocket(-1,&port);
	sprintf(spenv,"%s=%d",SPAWNENV,port);
	len = strlen(SPAWNENV) + 1;

	nei = 0;
	for( ei = 0; env = environ[ei]; ei++ )
		if( strncmp(env,spenv,len) != 0 )
			nenviron[nei++] = env;
	nenviron[nei++] = spenv;
	nenviron[nei] = NULL;

	cphandle = _spawnvpe(pmode,path,av,nenviron);
	if( cphandle == -1 ){
		char cwd[1024];
		getcwd(cwd,sizeof(cwd));
		LE("failed spawn(%s) at %s",path,cwd);
		fprintf(stderr,"failed spawn(%s) at %s\n",path,cwd);
		return cphandle;
	}

	csock = acceptsocket(asock);
	closesocket(asock);

	cpe = ProcEnv0;
	cpe.p_pmode = pmode;
	cpe.p_ppid = getpid();
	cpe.p_nosockopen = NO_SOCKOPEN;
	cpe.p_common = MyProcEnv.p_common;

	inheritsock(0,cphandle,&cpe);
	inheritsock(1,cphandle,&cpe);
	inheritsock(2,cphandle,&cpe);

	cj = 0;
	isock[0] = cpe.p_inhsock[0];
	isock[1] = cpe.p_inhsock[1];
	isock[2] = cpe.p_inhsock[2];
	for( ci = 0; ci < coen; ci++ ){
		dsock = coes[ci];
		if( ISSOCK(dsock) )
		if( isock[0] != dsock && isock[1] != dsock && isock[2] != dsock )
			cpe.p_closeOnExec[cj++] = coes[ci];
	}
	cpe.p_closeOnExecN = cj;

	wcc = send(csock,(char*)&cpe,sizeof(cpe),0);

	ntotal++;
	nalive++;
	children[csock] = cphandle;

	LE("spawn() = %d, children(alive=%d,total=%d)",
		cphandle,nalive,ntotal);
	return cphandle;
}
static sessionfd(){
	int fd;

	fd = MyProcEnv.p_common.p_sessionfd[0];
	return fd;
}
SessionFd(){ return sessionfd(); }

/* Set it on to emulate systems without SOCKET inheritance (Windows95) */
static int forceDUP = 1;

CFI_init(ac,av) char *av[]; {
	int clsock,svsock;
	int sfd,efd,fd;

	DO_INITIALIZE(ac,av);
	env2arg("CFI_");

	clsock = getclientsock();
	svsock = getserversock();

	if( clsock < 0 || svsock < 0 )
		LE("CFI_init: clsock=%d svsock=%d\n",clsock,svsock);

	if( clsock != 0 ) dup2(clsock,0);
	if( svsock != 1 ) dup2(svsock,1);

	sfd = sessionfd();
	efd = fileno(stderr);
	for( fd = 3; fd < 32; fd++ ){
		if( fd != sfd && fd != efd )
			close(fd);
	}
	return ac;
}
DO_INITIALIZE(ac,av)
	char *av[];
{	int pin;
	int rcc;
	int pport,psock;
	int ei,nei;
	int ci;
	char *env;
	int len;
	int ppid;
	int wcc;

	setBinaryIO();
	WINthread();

	argc = ac;
	argv = av;
	socket_init();

	psock = -1;
	nei = 0;
	len = strlen(SPAWNENV);

	for( ei = 0; env = environ[ei]; ei++ ){
		if( strncmp(env,SPAWNENV,len) == 0 && env[len] == '=' ){
			pport = atoi(&env[len+1]);
			psock = connectsocket(-1,pport);
			rcc = recv(psock,(char*)&MyProcEnv,sizeof(ProcEnv),0);
			if( rcc != sizeof(ProcEnv) ){
				LT("CANNOT READ ARGUMENTS [%d/%d]: %d/%d",
					pport,psock,rcc,sizeof(ProcEnv));
				exit(-1);
			}
			MyProcEnv.p_toparent = psock;
			LV("STARTUP: %s, got[%d], ppid=%d,pmode=%d",
				env,rcc, MyProcEnv.p_ppid, MyProcEnv.p_pmode);
		}
	}

	ppid = MyProcEnv.p_ppid;

	if( ppid == 0 ){
		pipe(MyProcEnv.p_common.p_sessionfd);
	}else{
		NO_SOCKOPEN = MyProcEnv.p_nosockopen;
	}

	for( ci = 0; ci < MyProcEnv.p_closeOnExecN; ci++ ){
		int sock,rcode;
		sock = MyProcEnv.p_closeOnExec[ci];
		rcode = closesocket(sock);
		LV("do CloseOnExec[%d]=%d, do close=%d",ci,sock,rcode);
	}

	if( 0 <= psock ){
		HANDLE pphandle;
		pphandle = OpenProcess(PROCESS_ALL_ACCESS,0,ppid);
		inheritSOCK(ppid,pphandle,pport,0);
		inheritSOCK(ppid,pphandle,pport,1);
		inheritSOCK(ppid,pphandle,pport,2);
	}else{
		if( start_service(ac,av) )
			DO_FINALIZE(0);
	}
}
inheritSOCK(ppid,pphandle,pport,si){
	int pclsock,nclsock,fd;
	VSAddr ina;
	int len;

	pclsock = MyProcEnv.p_pinhsock[si];
	nclsock = MyProcEnv.p_inhsock[si];

	LV("PPID=%d/%d CLSOCK=[%d->%d]",ppid,pphandle,pclsock,nclsock);

	/* CLSOCK possibly inherited (WindowsNT) */
	if( !isWindows95() )
	if( 0 <= pclsock && pclsock != nclsock )
	{
		int sx,inherited;
		inherited = 0;
		for( sx = 0; sx < 3; sx++ )
			if( MyProcEnv.p_inhsock[sx] == pclsock )
				inherited = 1;
		if( inherited == 0 ){
		LV("inheritSOCK -- closesocket(%d)",pclsock);
		closesocket(pclsock);
		}
	}

	if( 0 <= nclsock ){
		if( !ISSOCK(nclsock) ){
			LE("FATAL: inherited handle[%d] %d is not socket",si,nclsock);
			exit(-1);
		}
		MyProcEnv.p_inhfd[si] = fd = openSocket(-1,nclsock,1,"fork()");
		if( fd < 0 )
			exit(-1);

		len = sizeof(VSAddr);
		bzero(&ina,len);
		getpeername(nclsock,(SAP)&ina,&len);

		LV("PARENT=%d, PORT=%d, CLSOCK[%d]=%d/%d:%d [%s]",
			ppid,pport,si,
			nclsock,fd,ISSOCK(nclsock),
			VSA_ntoa(&ina));
	}
}

static bindsocket(sock,portp)
	int *portp;
{	int port;
	int rcode;
	VSAddr ina;
	int inalen,len;

	if( sock < 0 )
		sock = socket(AF_INET,SOCK_STREAM,0);

	inalen = VSA_atosa(&ina,0,"127.0.0.1");
	rcode = bind(sock,(SAP)&ina,inalen);
	rcode = listen(sock,1);

	if( portp ){
		len = sizeof(VSAddr);
		getsockname(sock,(SAP)&ina,&len);
		*portp = VSA_port(&ina);
	}
	return sock;
}
static connectsocket(sock,port)
	short port;
{	VSAddr ina;
	int inalen,rcode;

	if( sock < 0 )
		sock = socket(AF_INET,SOCK_STREAM,0);

	inalen = VSA_atosa(&ina,port,"127.0.0.1");
	rcode = connect(sock,(SAP)&ina,inalen);

	LV("connect(%d,%d) = %d",sock,port,rcode);
	return sock;
}
static acceptsocket(asock)
{	int csock;
	VSAddr ina;
	int len;
	int myport;

	len = sizeof(VSAddr);
	getsockname(asock,(SAP)&ina,&len);
	myport = VSA_port(&ina);

	len = sizeof(VSAddr);
	csock = accept(asock,(SAP)&ina,&len);
	LV("accept(%d,%d) = %d [%s:%d]",asock,myport,csock,
		VSA_ntoa(&ina),VSA_port(&ina));
	return csock;
}

static struct InheritSock {
	int	is_fd;
	int	is_sock;
	int	is_set;
	int	is_close_src;
} INHERIT_SOCK[4];
setclosesource(si){
	if( isWindows95() )
		INHERIT_SOCK[si].is_close_src = 1;
}

static int dummy_in = -1;
static dummy_input()
{	int io[2];
	int in2;

	if( dummy_in < 0 ){
		pipe(io);
		dummy_in = io[0];
		if( dummy_in == 0 ){
			in2 = _dup(dummy_in);
			_close(dummy_in);
			dummy_in = in2;
		}
		close(io[1]);
	}
	return dummy_in;
}

static inheritsock(si,cphandle,cpe)
	ProcEnv *cpe;
{	struct InheritSock *ihs;
	int cs,rcode,dummy;

	ihs = &INHERIT_SOCK[si];
	if( ihs->is_set ){
		ihs->is_set = 0;
		cs = ihs->is_close_src;
		ihs->is_close_src = 0;
		cpe->p_pinhsock[si] = ihs->is_sock;
		cpe->p_inhsock[si] = duphandle(0,ihs->is_sock,cphandle,cs,0);
		if( cs ){
			dummy = dummy_input();
			rcode = closeSocket(ihs->is_fd,ihs->is_sock);
			_dup2(dummy,ihs->is_fd);
		}
		LV("INHERIT: %d->%d (close_source=%d,%d/%d)",
		ihs->is_sock,cpe->p_inhsock[si],cs,ihs->is_sock,ihs->is_fd);
	}else{
		cpe->p_pinhsock[si] = -1;
		cpe->p_inhsock[si] = -1;
	}
}
passsock(si,sock)
	int sock;
{
	if( sock < 0 )
		return sock;

	LV("set INHERIT_SOCK=%d/%d",SocketOf(sock),sock);
	INHERIT_SOCK[si].is_fd = sock;
	INHERIT_SOCK[si].is_sock = SocketOf(sock);
	INHERIT_SOCK[si].is_set = 1;
	return sock;
}
recvsock(si)
{	int fd;

	fd = MyProcEnv.p_inhfd[si];
	LV("get INHERIT_SOCK=%d/%d",SocketOf(fd),fd);
	return fd;
}
closedups(si){
	int fd,sock;
	int closed;

	closed = 0;
	sock = MyProcEnv.p_inhsock[si];
	for( fd = 0; fd < MAXHANDLE; fd++ )
		if( isdup_handles[fd] == sock ){
			LE("closedups(%d/%d)",sock,fd);
			closeSocket(fd,sock);
			closed++;
		}
	return closed;
}
setclientsock(sock){ return passsock(0,sock); }
getclientsock(){ return recvsock(0); }
setserversock(sock){ return passsock(1,sock); }
getserversock(){ return recvsock(1); }
setcontrolsock(sock){ return passsock(2,sock); }
getcontrolsock(){ return recvsock(2); }
WithSocketFile(){ return 0; }
file_isselectable(fd){
	if( SocketOf(fd) )
		return 1;
	if( ISPIPE(fd) )
		return 1;
	return 0;
}

extern int WAIT_WNOHANG;

waitpid(pid,statusp,options)
	int *statusp;
{
	LE("waitpid() = -1 (not supported)");
	return -1;
}
wait3(statusp,options,rusage)
	int *statusp;
	char *rusage;
{	int chsock,pid;
	int nready;
	int xpid,xstatus;
	struct fd_set chset;
	struct timeval tvb,*tv = &tvb;

	xpid = -1;
	xstatus = -1;

	FD_ZERO(&chset);
	for( chsock = 0; chsock < MAXHANDLE; chsock++ )
		if( children[chsock] )
			FD_SET(chsock,&chset);

	if( options == WAIT_WNOHANG ){
		tv->tv_sec = 0;
		tv->tv_usec = 0;
	}else	tv = NULL;

	nready = select(MAXHANDLE,&chset,NULL,NULL,tv);

	if( nready )
	for( chsock = 0; chsock < MAXHANDLE; chsock++ ){
		if( FD_ISSET(chsock,&chset) ){
			pid = children[chsock];
			children[chsock] = 0;
			closesocket(chsock);
			xpid = _cwait(&xstatus,pid,0);
			nalive--;
			if( 0 < xpid )
				break;
		}
	}

	if( 0 <= xpid )
	LE("wait3() = %d, status=%d, children(alive=%d,total=%d)",
		xpid,xstatus,nalive,ntotal);

	if( statusp )
		*statusp = xstatus;
	return xpid;
}

static checkarg(path,av,nav,nab)
	char *path,*av[],*nav[],*nab;
{	char *arg,*ap;
	int ai;

	ap = nab;
	for( ai = 0; arg = av[ai]; ai++ ){
		if( strpbrk(arg," \t") && *arg != '"' ){
			nav[ai] = ap;
			sprintf(ap,"\"%s\"",arg);
			ap += strlen(ap) + 1;
		}else	nav[ai] = arg;
	}
	nav[ai] = NULL;
}
spawnvp(pmode,path,av)
	char *path,*av[];
{	char *nav[MAX_ARGC],nab[MAX_ARGB];

	checkarg(path,av,nav,nab);
	return spawnvpe(pmode,path,nav,environ);
}
execvp(path,av)
	char *path,*av[];
{	char *nav[MAX_ARGC],nab[MAX_ARGB];

	checkarg(path,av,nav,nab);
	return _execvp(path,nav);
}

wait(statusp)
	int *statusp;
{	int pid;

	LE("wait(%x) = ...",statusp,pid);
	pid = wait3(NULL,0,NULL);
	LE("wait(%x) = %d",statusp,pid);
	return pid;
}
getppid(){
	int ppid;
	HANDLE ph;

	ppid = MyProcEnv.p_ppid;
	if( ppid == 0 )
		return ppid;
	ph = OpenProcess(PROCESS_QUERY_INFORMATION,0,ppid);
	if( ph == NULL )
		return 0;
	CloseHandle(ph);
	LV("getppid() = %d",ppid);
	return ppid;
}
setsid(){
	LT("setsid() = -1 (not supported)");
	return -1;
}

DO_FINALIZE(code){
	int fd,sock;

	LV("PID=%d exit(%d) ...",getpid(),code);

	fcloseall();

	if( NO_SOCKOPEN )
	for( fd = 0; fd < MAXHANDLE; fd++ )
		if( sock = open_osfhandle[fd] )
			closeSocket(fd,sock);

	deltmpfiles();
	_rmtmp();

	LV("PID=%d exit(%d)",getpid(),code);
}

#else

setclosesource(si){ }
setclientsock(sock){ }
getclientsock(){ return -1; }
setserversock(sock){ }
getserversock(){ return -1; }
setcontrolsock(sock){ }
getcontrolsock(){ return -1; }

SocketOf(sock){ return sock; }
closedups(){ }
setNonblockingPipe(pv) int pv[]; { setNonblockingIO(pv[0],1); }
setNonblockingSocket(fd,on){ setNonblockingIO(fd,on); }
file_isselectable(fd){ return 1; }
pollPipe(pfd,msec){ return -1; }

create_service()
{
	return 0;
}

CFI_init(ac,av) char *av[]; {
	DO_INITIALIZE(ac,av);
	env2arg("CFI_");
	return ac;
}
SessionFd(){ return -1; }
setCloseOnExecSocket(fd){ return setCloseOnExec(fd); }
clearCloseOnExecSocket(fd){ return -1; }
setInheritance(ifd,inherit){ return -1; }

#ifdef __EMX__
putenv_sockhandle(fd,env)
	char *env;
{	int shandle;

	shandle = _getsockhandle(fd);
	if( shandle != -1 ){
		sprintf(env,"EMX_SOCKHANDLE%d=%d",fd,shandle);
		putenv(env);
	}
	return shandle;
}

getenv_sockhandle(fd){
	char name[32],*env;
	int shandle,fdx;

	sprintf(name,"EMX_SOCKHANDLE%d",fd);
	if( (env = getenv(name)) && *env ){
		shandle = atoi(env);
		if( 0 <= shandle ){
			for( fdx = 0; fdx < 32; fdx++ ){
				if( _getsockhandle(fdx) == shandle ){
					dup2(fdx,fd);
					return fd;
				}
			}
			fdx = _impsockhandle(shandle,0);
			if( fdx != fd ){
				dup2(fdx,fd);
				close(fdx);
			}
			return fd;
		}
	}
	return -1;
}

DO_FINALIZE(){
	fcloseall();
	deltmpfiles();
	_rmtmp();
}

DO_INITIALIZE(ac,av)
	char *av[];
{	int fd;
	unsigned long rel = 0, cur;
 
	DosSetRelMaxFH(&rel, &cur);
	if( cur < 48 ){
		LV("increase MaxFH: %d -> %d",cur,48);
		DosSetMaxFH(48);
	}

	for( fd = 0; fd < 32; fd++ )
		getenv_sockhandle(fd);

	setBinaryIO();
}
extern int SPAWN_P_WAIT;
execvp(path,argv)
	char *path,*argv[];
{	int stat;
	int fd;
	char envs[32][32];

	for( fd = 0; fd < 32; fd++ )
		putenv_sockhandle(fd,envs[fd]);

	stat = spawnvp(SPAWN_P_WAIT,path,argv);
	if( stat == -1 )
		return -1;
	else	exit(stat);
}
WithSocketFile(){ return 0; }
getsockHandle(fd){ return _getsockhandle(fd); }
#else
DO_INITIALIZE(ac,av)
	char *av[];
{
	setBinaryIO();
}
DO_FINALIZE(){ }
WithSocketFile(){ return 1; }
getsockHandle(fd){ return -1; }
#endif /* !__EMX__ */

#endif


porting_dbg(fmt,a,b,c,d,e,f,g,h,i,j)
	char *fmt;
	char *a,*b,*c,*d,*e,*f,*g,*h,*i,*j;
{	char buf[4096];
	int now = time(0L);
	int logfd;

	sprintf(buf,"(%s) %02d:%02d [%3d] ",SYST,(now%3600)/60,now%60,getpid());
	sprintf(buf+strlen(buf),fmt,a,b,c,d,e,f,g,h,i,j);
	strcat(buf,"\n");
	if( lFG() || curLogFd() < 0 )
		logfd = fileno(stderr);
	else	logfd = curLogFd();
	DBGWRITE(logfd,buf,strlen(buf));
}

arg2env(prefix)
	char *prefix;
{	char env[1024],tmp[1024];

	sprintf(env,"%sLOGFD=%d",prefix,curLogFd());
	putenv(strdup(env));
}
env2arg(prefix)
	char *prefix;
{	char name[1024],*env;

	sprintf(name,"%sLOGFD",prefix);
	if( env = getenv(name) )
		dup2(atoi(env),fileno(stderr));
}


#include "windows0.c"

/*////////////////////////////////////////////////////////////////////////
Content-Type:	program/C; charset=US-ASCII
Program:	fcntl.c
Author:		Yutaka Sato <ysato@etl.go.jp>
Description:
History:
	961231	extracted from file.c
	990309	merged to windows0.c
//////////////////////////////////////////////////////////////////////#*/

setNonblockingIO(fd,on)
{	int flags;
	int rcode;

	if( F_SETFL == -1 || O_NDELAY == -1 ){
		rcode = setNonblockingSocket(fd);
		if( rcode < 0 ){
			syslog_ERROR("Non-Blocking I/O not supported\n");
			return -1;
		}else	return rcode;
	}

	flags = fcntl(fd,F_GETFL,0);
	if( on )
		flags |=  O_NDELAY;
	else	flags &= ~O_NDELAY;
	return fcntl(fd,F_SETFL,flags);
}
