# $Id: tc_string.rb,v 1.8 2002/09/11 03:26:38 knu Exp $

require 'test/unit'
begin
  require 'features/ruby18/string'
rescue LoadError; end

class TC_String < Test::Unit::TestCase
  def test_s_new
    assert_equal("", String.new)
    assert_equal("abc", String.new("abc"))

    assert_raises(TypeError) { String.new(nil) }
  end

  def test_aref
    s = 'deja VROOOM'

    assert_raises(ArgumentError) { s[/(.)O+/, 1, 2] }

    assert_equal('ROOO', s[/(.)O+/])
    assert_equal('R', s[/(.)O+/, 1])
    assert_nil(s[/(.)Q+/])
    assert_nil(s[/(.)Q+/, 1])
  end

  def test_aset
    s = 'deja VROOOM'

    assert_raises(ArgumentError) { s[/(.)O+/, 1, 2] = '***' }

    assert_raises(IndexError) { s[/(.)Q+/] = '***' }
    assert_raises(IndexError) { s[/(.)Q+/, 1] = '***' }

    assert_equal('?', s[/(.)O+/, 1] = '?' )
    assert_equal('deja V?OOOM', s)
    assert_equal('***', s[/(.)O+/] = '***' )
    assert_equal('deja V***M', s)
  end

  def test_casecmp
    assert_equal(true, 'A'.casecmp('z') < 0)
    assert_equal(0, 'aBc'.casecmp('AbC'))
    assert_equal(true, 'z'.casecmp('A') > 0)
  end

  def test_chomp
    s = "ruby17\r\n"
    assert_equal("ruby17", s.chomp!)
    assert_nil(s.chomp!)

    assert_equal("ruby17", "ruby17\r\n".chomp)
    assert_equal("ruby17", "ruby17\r".chomp)
    assert_equal("ruby17", "ruby17\n".chomp)
    assert_equal("ruby17\n", "ruby17\n\r".chomp)

    assert_equal("ruby17", "ruby17\r\n".chomp("\r\n"))
    assert_equal("ruby", "ruby17\r\n".chomp("17\r\n"))
  end

  def test_insert
    a = 'adef'

    b = a.insert(1, 'bc')

    assert_same(a, b)
    assert_equal('abcdef', b)
  end

  def test_lstrip
    s = "\t  A \r\n B  \r\n"

    t = s.lstrip

    assert_not_same(s, t)
    assert_equal("A \r\n B  \r\n", t)

    t = s.lstrip!

    assert_same(s, t)
    assert_equal("A \r\n B  \r\n", t)

    t = s.lstrip!

    assert_nil(t)
  end

  def test_match
    m = 'abc'.match(/(.)(..)/)

    assert_kind_of(MatchData, m)
    assert_equal('bc', m[2])

    m = 'abc'.match('(.)(..)')

    assert_kind_of(MatchData, m)
    assert_equal('bc', m[2])
  end

  def test_rstrip
    s = "\t  A \r\n B  \r\n"

    t = s.rstrip

    assert_not_same(s, t)
    assert_equal("\t  A \r\n B", t)

    t = s.rstrip!

    assert_same(s, t)
    assert_equal("\t  A \r\n B", t)

    t = s.rstrip!

    assert_nil(t)
  end

  def test_slice
    s = 'deja VROOOM'

    assert_raises(ArgumentError) { s.slice(/(.)O+/, 1, 2) }

    assert_equal('ROOO', s.slice(/(.)O+/))
    assert_equal('R', s.slice(/(.)O+/, 1))
    assert_nil(s.slice(/(.)Q+/))
    assert_nil(s.slice(/(.)Q+/, 1))
  end

  def test_slice!
    s = 'deja VROOOM'

    assert_raises(ArgumentError) { s.slice!(/(.)O+/, 1, 2) }

    assert_nil(s.slice!(/(.)Q+/))
    assert_nil(s.slice!(/(.)Q+/, 1))

    assert_equal('R', s.slice!(/(.)O+/, 1))
    assert_equal('deja VOOOM', s)
    assert_equal('VOOO', s.slice!(/(.)O+/))
    assert_equal('deja M', s)
  end

  def test_to_i
    assert_raises(TypeError) { "0".to_i(nil) }
    assert_raises(ArgumentError) { "0".to_i(4) }

    assert_equal(0xbeef, "beef".to_i(16))
    assert_equal(0xbeef, "0xbeef".to_i(16))
    assert_equal(0, "z1".to_i(16))

    assert_equal(2112, "2112".to_i())
    assert_equal(2112, "2112".to_i(10))

    assert_equal(0753, "753".to_i(8))
    assert_equal(0753, "0753".to_i(8))
    assert_equal(0, "81".to_i(8))

    assert_equal(0b1011_1011_1100, "101110111100".to_i(2))
    assert_equal(0b1011_1011_1100, "0b101110111100".to_i(2))
    assert_equal(0b1011_1011_1100, "1011_1011_1100".to_i(2))
    assert_equal(0b1011_1011_1100, "0b1011_1011_1100".to_i(2))
    assert_equal(0, "21".to_i(2))
  end
end
