require "misen/node"
require "misen/parse"
require "misen/style-sgml"
require "runit/testcase"


class TESTCASE_MisenParser < RUNIT::TestCase
  def test_simple
    parsed = parse_sgml('
<misen:block>
text1<misen:inline />text2
</misen:block>
')
    assert_equals "\n", parsed.shift

    block = parsed.shift
    begin
      assert_equals "block", block.identifier
      children = block.children

      assert_equals "\ntext1", children.shift

      inline = children.shift
      assert_equals "inline", inline.identifier

      assert_equals "text2\n", children.shift
      assert children.empty?
    end
    assert_equals "\n", parsed.shift
    assert parsed.empty?
  end

  def test_nested
    parsed = parse_sgml_without_whitespace('
<misen:parent>
  <misen:child>
    granchild
  </misen:child>
</misen:parent>
')

    parent = Misen.new_block_node("parent") {
      child = Misen.new_block_node("child") {
        ["granchild"]
      }
      [child]
    }
    expected = [parent]
    assert_equals expected, parsed
  end

  def test_block_repeat
    parsed = parse_sgml_without_whitespace('
<misen:b1></misen:b1><misen:b2></misen:b2>
')
    expected = []
    expected << Misen.new_block_node("b1")
    expected << Misen.new_block_node("b2")
    assert_equals expected, parsed
  end

  def test_extinline
    parsed = parse_sgml('<tag attr="@@attr@@">')
    expected = []
    expected << '<tag attr="'
    inline_style = Misen.inline_style_for_name(Misen::STYLE_SGML_EX,
                                               "sgml_attr")
    expected << Misen.new_inline_node("attr", inline_style)
    expected << '">'
    assert_equals expected, parsed
  end
  
  def parse_sgml_without_whitespace(text)
    parse_sgml(text.gsub(/\s+/, ""))
  end

  def parse_sgml(text)
    parser = Misen.new_parser(Misen::STYLE_SGML_EX)
    parser.parse_text(text)
  end

end #/TESTCASE_MisenParser
