require 'thread'
require 'monitor'

module Dpklib
  class QueueWorker
    include MonitorMixin

    attr_accessor :worker_proc, :on_clean_proc
    attr_reader :worker_thread, :work_queue

    WorkerError = Class.new(StandardError)
    WorkerNotRunningError = Class.new(WorkerError)
    WorkerGoingShutdownError = Class.new(WorkerError)

    SHUTDOWN_REQUEST = Object.new
    ONCLEAN_REQUEST_PROC = Class.new(Proc)

    def initialize(&worker_proc)
      @worker_proc = worker_proc
      @work_queue = Queue.new
      @working = false
      @shutdown = false
    end

    def execute
      begin
        @working = true
        while true do
          work = @work_queue.pop

          break if work.equal?( SHUTDOWN_REQUEST )
          if work.kind_of?( ONCLEAN_REQUEST_PROC ) then
            work.call
            next
          end

          @worker_proc[work]
        end
      ensure
        @shutdown = false
        @working = false
      end
    end

    def working?
      @working
    end

    def shutdown
      return unless working?
      return if @shutdown
      @shutdown = true
      @work_queue.push SHUTDOWN_REQUEST
    end

    def wait_clean(*args, &block)
      raise(WorkerNotRunningError) unless working?
      
      queue = Queue.new
      work = ONCLEAN_REQUEST_PROC.new {
        queue.push( block ? block.call(*args) : nil )
      }
      push work
      queue.pop
    end

    def push(work)
      raise(WorkerGoingShutdownError) if @shutdown
      @work_queue.push work
    end

  end #/QueueWorker

  class BGQueueWorker
    attr_reader :worker, :worker_thread

    def initialize(&worker_proc)
      @worker = QueueWorker.new(&worker_proc)

      @worker_thread = nil
    end

    def wait_clean(*args, &block)
      @worker.wait_clean(*args, &block)
    end

    def push(work)
      @worker.push(work)
    end

    def start
      @worker_thread = Thread.new {
        @worker.execute
      }
    end

    def shutdown
      return unless running?
      @worker.shutdown
      @worker_thread.join
      @worker_thread = nil
    end

    def running?
      @worker_thread != nil
    end

  end #/BGQueueWorker

end #/Dpklib
