=begin
    mo.rb - A simple class for operating GNU MO file.
    Copyright (C) 2002  Masahiro Sakai, Masao Mutoh
    Copyright (C) 2001  Masahiro Sakai

        Masahiro Sakai                  <s01397ms@sfc.keio.ac.jp>
        Masao Mutoh                     <mutoh@highway.ne.jp>

    You can redistribute this file and/or modify it under the same term
    of Ruby.  License of Ruby is included with Ruby distribution in
    the file "README".

    $Id: mo.rb,v 1.4 2002/10/22 11:24:29 mutoh Exp $
=end

begin
  require 'iconv'
  $__MO_NO_ICONV = false
rescue LoadError
  $__MO_NO_ICONV = true
end

class MOFile < Hash
  class InvalidFormat < RuntimeError; end;

  Header = Struct.new(:magic,
                      :revision,
                      :nstrings,
                      :orig_table_offset,
                      :translated_table_offset,
                      :hash_table_size,
                      :hash_table_offset)

  MAGIC_BIG_ENDIAN    = "\x95\x04\x12\xde"
  MAGIC_LITTLE_ENDIAN = "\xde\x12\x04\x95"

  def self.open(arg = nil, output_charset = nil)
    result = self.new(output_charset)
    case arg
    when String
      result.load_from_file(arg)
    when IO
      result.load_from_stream(arg)
    end
    result
  end

  def initialize(output_charset = nil)
    @little_endian = false
    @output_charset = output_charset
    super()
  end

  def load_from_stream(io)
    magic = io.read(4)
    case magic
    when MAGIC_BIG_ENDIAN
      @little_endian = false
    when MAGIC_LITTLE_ENDIAN
      @little_endian = true
    else
      raise InvalidFormat.new(sprintf("Unknown signature %s", magic.dump))
    end

    header = Header.new(magic, *(io.read(4 * 6).unpack(@little_endian ? 'V6' : 'N6')))
    raise InvalidFormat.new(sprintf("file format revision %d isn't supported", header.revision)) if header.revision > 0

    io.pos = header.orig_table_offset
    orig_table_data = io.read((4 * 2) * header.nstrings).unpack(@little_endian ? 'V*' : 'N*')

    io.pos = header.translated_table_offset
    trans_table_data = io.read((4 * 2) * header.nstrings).unpack(@little_endian ? 'V*' : 'N*')

    original_strings = Array.new(header.nstrings)
    for i in 0...header.nstrings
      io.pos = orig_table_data[i * 2 + 1]
      original_strings[i] = io.read(orig_table_data[i * 2 + 0])
    end

    clear
    for i in 0...header.nstrings
      io.pos = trans_table_data[i * 2 + 1]
      str = io.read(trans_table_data[i * 2 + 0])

      if original_strings[i] == ""
        if str
          @charset = nil
          @nplurals = nil
          @plural = nil
          str.each_line{|line|
            if /^Content-Type:/i =~ line and /charset=((?:\w|-)+)/i =~ line
              @charset = $1
            elsif /^Plural-Forms:\s*nplurals\s*\=\s*(\d*);\s*plural\s*\=\s*([^;]*)\n?/ =~ line
              @nplurals = $1
              @plural = $2
            end
            break if @charset and @nplurals
          }
        end
      else
        if @output_charset
          begin
            unless $__MO_NO_ICONV
              str = Iconv.iconv(@output_charset, @charset, str).join
            end
          rescue Iconv::IllegalSequence
            if $DEBUG
              print "@charset = ", @charset, "\n"
              print "@output_charset = ", @output_charset, "\n"
              print "msgid = ", original_strings[i], "\n"
              print "msgstr = ", str, "\n"
            end
          end
        end
      end
      self[original_strings[i]] = str
    end
    self
  end

  def save_to_stream(io)
    header_size = 4 * 7
    table_size  = 4 * 2 * size

    header = Header.new(
      @little_endian ? MAGIC_LITTLE_ENDIAN : MAGIC_BIG_ENDIAN, # magic
      0,                            # revision
      size,                         # nstrings
      header_size,                  # orig_table_offset
      header_size + table_size,     # translated_table_offset
      header_size + table_size * 2, # hash_table_offset
      0                             # hash_table_size
    )
    io.write(header.to_a.pack('a4' + (@little_endian ? 'V6' : 'N6')))

    ary = to_a
    ary.sort!{|a, b| a[0] <=> b[0]} # sort by original string

    pos = header.hash_table_offset + header.hash_table_size

    orig_table_data = Array.new()
    ary.each{|item, _|
      orig_table_data.push(item.size)
      orig_table_data.push(pos)
      pos += item.size + 1 # +1 is <NUL>
    }
    io.write(orig_table_data.pack(@little_endian ? 'V*' : 'N*'))

    trans_table_data = Array.new()
    ary.each{|_, item|
      trans_table_data.push(item.size)
      trans_table_data.push(pos)
      pos += item.size + 1 # +1 is <NUL>
    }
    io.write(trans_table_data.pack(@little_endian ? 'V*' : 'N*'))

    ary.each{|item, _| io.write(item); io.write("\0") }
    ary.each{|_, item| io.write(item); io.write("\0") }

    self
  end

  def load_from_file(filename)
    File.open(filename, 'rb'){|f| load_from_stream(f)}
  end

  def save_to_file(filename)
    File.open(filename, 'wb'){|f| save_to_stream(f)}
  end

  attr_accessor :little_endian
  attr_reader :charset, :nplurals, :plural
end


# Test

if $0 == __FILE__
  if (ARGV.include? "-h") or (ARGV.include? "--help")
    STDERR.puts("mo.rb [filename.mo ...]")
    exit
  end

  ARGV.each{ |item|
    mo = MOFile.open(item)
    puts("------------------------------------------------------------------")
    puts("charset = " + mo.charset) if mo.charset
    puts("nplurals = " + mo.nplurals) if mo.nplurals
    puts("plural = " + mo.plural) if mo.plural
    puts("------------------------------------------------------------------")
    mo.each{
      |key, value|
      puts("original message = \"#{key}\"")
      puts("translated message = \"#{value}\"")
      puts("--------------------------------------------------------------------")
    }
  }
end
