require "dpklib/shadow"
require "dpklib/update-monitor"
require "webrick/httpauth"
require "webrick/httpserver"

module Dpklib_WEBrick
  class SingleUserAuthenticator < Struct.new(:realm, :user, :pass)
    def initialize
      super
      self.realm = ENV["HTTP_AUTH"]
      self.user = ENV["HTTP_AUTH_USER"]
      self.pass = ENV["HTTP_AUTH_PASS"]
    end

    def authorize(req, res)
      if realm then
        WEBrick::HTTPAuth.basic_auth(req, res, realm) { |req_user, req_pass|
          (user == req_user) && (pass == req_pass)
        }
      end
    end
  end #/SingleUserAuthenticator

  class ShadowAuthenticator < Struct.new(:realm, :shadow_manager)
    def initialize
      super()
      self.realm = ENV["HTTP_AUTH"]
      
      shadow_manager = Dpklib::ShadowPasswordManager.new
      shadow_manager.digest_class = shadow_digest_class
      self.shadow_manager = shadow_manager

      passwd_file = shadow_password_file
      if passwd_file
        @shadow_file_monitor = Dpklib::FileUpdateMonitor.new(passwd_file)
        @shadow_file_monitor.error_on_lost = true
        update_shadow_manager
      end
    end
    
    def authorize(req, res)
      if realm then
        WEBrick::HTTPAuth.basic_auth(req, res, realm) { |req_user, req_pass|
          update_shadow_manager
          shadow_manager.verify(req_user, req_pass)
        }
      end
    end

    def update_shadow_manager
      if @shadow_file_monitor
        @shadow_file_monitor.with_update do
          shadow_manager.clear
          shadow_manager.add_from_file(@shadow_file_monitor.monitored_file)
        end
      end
    end
    
    def shadow_digest_class
      nil
    end
    
    def shadow_password_file
      ENV["HTTP_SHADOW"]
    end
  end #/ShadowAuthenticator
  
  module AuthHttpServerModule
    attr_writer :authenticator_class
    def authenticator_class
      @authenticator_class || SingleUserAuthenticator
    end
    
    def authenticator
      @authenticator ||= authenticator_class.new
    end

    def service(req, res)
      authenticator = self.authenticator
      authenticator.authorize(req, res)
      super
    end
  end #/AuthHttpServerModule

  class AuthHttpServer < WEBrick::HTTPServer
    include AuthHttpServerModule
  end #/AuthHttpServer

  class ShadowAuthHttpServer < WEBrick::HTTPServer
    include AuthHttpServerModule
    def authenticator_class
      ShadowAuthenticator
    end
  end #/ShadowAuthHttpServer
  
end #/Dpklib_WEBrick
