require "dpklib/class"
require "dpklib/strutils"
require "misen/node"
require "misen/style"


module Misen
  ParseError = Class.new(StandardError)

  class BlockParser
    def initialize(text, style)
      @children_stack = nil
      @text, @style = text, style

      inline_regexps = style.inline_styles.collect { |inline_style|
        inline_style.regexp
      }
      @inline_pattern_regexp = Dpklib.join_regexp(*inline_regexps)
    end

    def parse
      @children_stack = [[]]

      block_begin_re = @style.block_begin_regexp
      block_end_re = @style.block_end_regexp
      block_pattern_re =
        Dpklib.join_regexp(block_begin_re, block_end_re)
      rest_text = @text

      while block_pattern_re =~ rest_text
        concat_with_inline_parsing($~.pre_match)
        rest_text = $~.post_match
        case $~[0]
        when block_begin_re
          node =
            Misen.new_block_node(@style.block_get_identifier($~.to_a))
          concat_node(node)
          children_begin
        when block_end_re
          children_end
        end
      end
      concat_with_inline_parsing(rest_text)

      fail("Unterminated block exists.") if has_unterminated?
      @children_stack[0]
    end

    def concat_with_inline_parsing(text)
      while @inline_pattern_regexp =~ text
        concat_text($~.pre_match)
        matched = $~[0]
        text = $~.post_match

        for inline_style in @style.inline_styles
          if inline_style.regexp =~ matched
            identifier = inline_style.get_identifier($~.to_a)
            node = Misen.new_inline_node(identifier, inline_style)
            concat_node(node)
            break
          end
        end
      end
      concat_text(text)
    end

    def concat_text(text)
      text.empty? || current_nodes << text
    end

    def concat_node(node)
      current_nodes << node
    end

    def children_begin
      @children_stack << []
    end

    def children_end
      children = @children_stack.pop
      current_nodes[-1].children.replace(children)
    end

    def current_nodes
      @children_stack[-1] || fail("Unexpected end of children.")
    end

    def has_unterminated?
      @children_stack.size != 1
    end

    def fail(msg)
      raise ParseError, msg
    end
  end #/BlockParser

  module ParserMixin
    def parse_text(source)
      Dpklib.implemented_by_subclass
    end

    def parse_input(input)
      parse_text(input.read)
    end
  end #/ParserMixin

  class Parser
    include ParserMixin

    def initialize(style)
      @style = style
    end

    def parse_text(source)
      block_parser = BlockParser.new(source, @style)
      block_parser.parse
    end
  end #/Parser

  class << self
    def new_parser(style)
      Parser.new(style)
    end
  end #/<< self
end #/Misen
